#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.MlagShowCli as MlagShowCli
import CliPlugin.MlagTunnelCli as MlagTunnelCli
import CliPlugin.EbraEthIntfCli as EbraEthIntfCli
import CliPlugin.MlagWarningCli as MlagWarningCli
from CliPlugin.BridgingCliModel import UnicastMacAddressTable
from CliPlugin.BridgingCli import MacAddrTableExprForShow
from CliPlugin.MlagModel import Interfaces, Mlag, ConfigSanity, IssuWarnings, \
      IssuCompatibility
from CliPlugin.MlagTunnelModel import MlagTunnel
from CliPlugin.MacAddr import macAddrMatcher
from CliPlugin.VlanCli import vlanIdMatcher
import Intf.IntfRange as IntfRange
from Intf.IntfRange import IntfRangeMatcher
from MultiRangeRule import MultiRangeMatcher
import ShowCommand
import Tac
import Url

VxlanAutoIntfType = IntfRange.GenericRangeIntfType(
   lambda: ( 1, 1 ),
   "Vxlan", "Vx", "Vxlan Tunnel Interface" )
matcherMlag = CliMatcher.KeywordMatcher( 'mlag', helpdesc='MLAG status' )
nodeMlag = CliCommand.Node( matcher=matcherMlag, 
      guard=MlagShowCli.mlagSupportedGuard )
matcherInterfaces = CliMatcher.KeywordMatcher( 'interfaces', 
      helpdesc='MLAG interfaces' )
matcherStates = CliMatcher.KeywordMatcher( 'states',
      helpdesc='MLAG states' )
MlagNum = Tac.Type( "Mlag::MlagNum" )
matcherInterfaceId = MultiRangeMatcher( lambda: ( MlagNum.min, MlagNum.max ),
      False, 'MLAG Interface ID' )
matcherDiff = CliMatcher.KeywordMatcher( 'diff',
      helpdesc='MAC addresses only in peer or local' )
nodeMlagPeer = CliCommand.guardedKeyword( 'mlag-peer',
      helpdesc='MAC addresses learned from MLAG peer',
      guard=MlagShowCli.mlagSupportedGuard )
matcherIntfRange = IntfRangeMatcher( 
      explicitIntfTypes=EbraEthIntfCli.switchPortIntfTypes )
matcherVxIntfRange = IntfRangeMatcher( 
      explicitIntfTypes=( [ VxlanAutoIntfType ] ) )
matcherIssu = CliMatcher.KeywordMatcher( 'issu', helpdesc='MLAG ISSU information' )
matcherUrl = Url.UrlMatcher( helpdesc='Software image URL', 
      fsFunc=lambda fs: fs.scheme in [ 'file:', 'flash:' ] )
mlagPoStates = {
      'active-partial': 'Include active-partial interfaces',
      'disabled': 'Include disabled interfaces',
      'inactive': 'Include inactive interfaces',
      'configured': 'Include configured interfaces',
      'active-full': 'Include active-full interfaces',
      }
macEntryType = {
      'static': 'Static entry type',
      'dynamic': 'Dynamic entry type',
      'unicast': 'Unicast entry type',
      'configured': 'Configured MAC entries',
      }

#--------------------------------------------------------------------------------
# show mlag [ detail ]
#--------------------------------------------------------------------------------
class MlagCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mlag [ detail ]'
   data = {
      'mlag': nodeMlag,
      'detail': 'With details for debugging',
   }
   handler = MlagShowCli.showMlag
   cliModel = Mlag

BasicCli.addShowCommandClass( MlagCmd )

#--------------------------------------------------------------------------------
# show mlag config-sanity [ all ]
#--------------------------------------------------------------------------------
class MlagConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mlag config-sanity [ all ]'
   data = {
      'mlag': nodeMlag,
      'config-sanity': 'Check peer MLAG configuration sanity',
      'all': 'Including matches and defaults',
   }
   handler = MlagShowCli.showMlagConfSanity
   cliModel = ConfigSanity

BasicCli.addShowCommandClass( MlagConfigSanityCmd )

#--------------------------------------------------------------------------------
# show mlag interfaces [ MLAG_IDS ] [ states ( { active-partial | disabled |
# inactive | configured | active-full } ) ][ detail ]
#--------------------------------------------------------------------------------
class MlagInterfacesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mlag interfaces [ MLAG_IDS ] '
              '[ states STATES ][ detail ]' )
   data = {
      'mlag': nodeMlag,
      'interfaces': matcherInterfaces,
      'MLAG_IDS': matcherInterfaceId,
      'states': matcherStates,
      'STATES' : CliCommand.setCliExpression( mlagPoStates, name='STATES' ),
      'detail': 'With details for debugging',
   }
   handler = MlagShowCli.showMlagInt
   cliModel = Interfaces

BasicCli.addShowCommandClass( MlagInterfacesCmd )

#--------------------------------------------------------------------------------
# show mlag tunnel [ counters ]
#--------------------------------------------------------------------------------
class MlagTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mlag tunnel [ counters ]'
   data = {
      'mlag': nodeMlag,
      'tunnel': 'MLAG tunnel information',
      'counters': 'MLAG tunnel counters',
   }
   handler = MlagTunnelCli.showMlagTunnel
   cliModel = MlagTunnel

BasicCli.addShowCommandClass( MlagTunnelCmd )

#--------------------------------------------------------------------------------
# show mlag interfaces [ INTERFACES ] members
#--------------------------------------------------------------------------------
class MlagInterfacesMembersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mlag interfaces [ MLAG_IDS ] members'
   data = {
      'mlag': nodeMlag,
      'interfaces': matcherInterfaces,
      'MLAG_IDS': matcherInterfaceId,
      'members': 'MLAG members',
   }
   handler = MlagShowCli.showMlagMembers
   cliModel = Interfaces

BasicCli.addShowCommandClass( MlagInterfacesMembersCmd )

#--------------------------------------------------------------------------------
# show ( ( mac address-table ) | mac-address-table ) mlag-peer diff local-only
#--------------------------------------------------------------------------------
class MlagPeerDiffLocalOnlyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show MAC_ADDR_TABLE mlag-peer diff local-only'
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'mlag-peer': nodeMlagPeer,
      'diff': matcherDiff,
      'local-only': 'MAC addresses only in local',
   }
   handler = MlagShowCli.showMacTableOnlyInLocal
   cliModel = UnicastMacAddressTable

BasicCli.addShowCommandClass( MlagPeerDiffLocalOnlyCmd )

#--------------------------------------------------------------------------------
# show ( ( mac address-table ) | mac-address-table ) mlag-peer diff peer-only
#--------------------------------------------------------------------------------
class MlagPeerDiffPeerOnlyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show MAC_ADDR_TABLE mlag-peer diff peer-only'
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'mlag-peer': nodeMlagPeer,
      'diff': matcherDiff,
      'peer-only': 'MAC addresses only in peer',
   }
   handler = MlagShowCli.showMacTableOnlyInPeer
   cliModel = UnicastMacAddressTable

BasicCli.addShowCommandClass( MlagPeerDiffPeerOnlyCmd )

#--------------------------------------------------------------------------------
# show ( ( mac address-table ) | mac-address-table ) mlag-peer [ ENTRY_TYPE ]
# [ address MACADDR ] [ interface INTF | VXINTF ] [ vlan VLANID ]
#--------------------------------------------------------------------------------
class MlagPeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show MAC_ADDR_TABLE mlag-peer '
              '[ ENTRY_TYPE ] [ address MACADDR ] '
              '[ interface INTF | VXINTF ] [ vlan VLANID ]' )
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'mlag-peer': nodeMlagPeer,
      'ENTRY_TYPE': CliMatcher.EnumMatcher( macEntryType ),
      'address': 'Ethernet address',
      'MACADDR': macAddrMatcher,
      'interface': 'Interface',
      'INTF': matcherIntfRange,
      'VXINTF': matcherVxIntfRange,
      'vlan': 'VLAN Identifier',
      'VLANID': vlanIdMatcher,
   }
   handler = MlagShowCli.showMacTable
   cliModel = UnicastMacAddressTable

BasicCli.addShowCommandClass( MlagPeerCmd )

#--------------------------------------------------------------------------------
# show mlag issu compatibility URL
#--------------------------------------------------------------------------------
class MlagIssuCompatibilityUrlCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mlag issu compatibility URL'
   data = {
      'mlag': nodeMlag,
      'issu': matcherIssu,
      'compatibility': 'MLAG ISSU compatibility check',
      'URL': matcherUrl,
   }
   handler = MlagWarningCli.showMlagIssuCompatibility
   cliModel = IssuCompatibility

BasicCli.addShowCommandClass( MlagIssuCompatibilityUrlCmd )

#--------------------------------------------------------------------------------
# show mlag issu warnings
#--------------------------------------------------------------------------------
class MlagIssuWarningsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mlag issu warnings'
   data = {
      'mlag': nodeMlag,
      'issu': matcherIssu,
      'warnings': 'MLAG ISSU warning messages',
   }
   handler = MlagWarningCli.showMlagIssuWarnings
   cliModel = IssuWarnings

BasicCli.addShowCommandClass( MlagIssuWarningsCmd )
