#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliModel import Model, Str, Enum, Dict, Submodel, Int, Bool, Float, List
from IntfModels import Interface
from ArnetModel import Ip4Address, MacAddress
from TableOutput import TableFormatter, Format, FormattedCell, createTable
from MlagShared import NEG_STATUS_ALL
from Ark import timestampToStr
import CliPlugin.IntfModel as IntfModel
import CliPlugin.IntfCli as IntfCli
import re
import LacpLib
import os
import Toggles.MlagToggleLib
from textwrap import fill
from Arnet import intfNameKey
from LagModel import LacpSysIdBase
from LacpLib import dot43sysid

def showMlagTableFormat( detail=False ):
   t = TableFormatter()
   f = Format( justify="left", noBreak=True, terminateRow=False )
   f2 = Format()
   f3 = Format( maxWidth=40, wrap=True ) if detail else Format()
   f4 = Format()
   f.noPadLeftIs( True )
   f.padLimitIs( True ) # makes the table compact
   f2.noPadLeftIs( True )
   f2.padLimitIs( True )
   f3.noPadLeftIs( True )
   f3.padLimitIs( True )
   f3.noTrailingSpaceIs( True )
   f4.noTrailingSpaceIs( True )
   t.formatColumns( f, f2, f3, f4 )
   return t

def negStatusStrToEnum( st ):
   # "Domain mismatch"->domainMismatch, for example.
   return re.sub( " ([a-z])", lambda m: m.group(1).capitalize(), st.lower() )

def negStatusEnumToStr( st ):
   # domainMismatch->"Domain mismatch", for example.
   return re.sub( "([A-Z])", lambda m: " " + m.group(1).lower(), st ).capitalize()

negStatuses = [ negStatusStrToEnum( s ) for s in NEG_STATUS_ALL ]

# Cli Model for "show mlag [detail]"
class Mlag( Model ):
   # Configuration
   domainId = Str( help="MLAG domain ID", optional=True )
   localInterface = Interface( help="Layer 3 interface used for communication "
                               "with MLAG peer", optional=True )
   peerAddress = Ip4Address( help="IP address of the other MLAG peer",
                             optional=True )
   peerLink = Interface( help="Layer 2 interface used for communication "
                         "with MLAG peer", optional=True )
   heartbeatPeerAddress = Ip4Address( help="IP address of the other MLAG peer "
                                       "used for UDP heartbeats (if different "
                                       "from peerAddress)",
                                       optional=True )
   heartbeatPeerVrf = Str( help="VRF for the heartbeat peer address",
                            optional=True )
   reloadDelay = Int( help="How long (in seconds) the non-peer-link ports "
                      "are errdisabled after a reboot. -1 means inifinite." )
   reloadDelayNonMlag = Int( help="How long (in seconds) the non-peer-link, "
                             "non-MLAG ports are errdisabled after a reboot" )
   configSanity = Enum( values=('consistent', 'inconsistent'),
                        help="Peer configuration consistency check",
                        optional=True )
   # Status
   state = Enum( values=('disabled', 'disabling', 'inactive', 'active'),
                 help="MLAG state" )
   negStatus = Enum( values=negStatuses, help="Negotiation status",
                     optional=True )
   peerLinkStatus = Enum( values=IntfModel.intfOperStatuses,
                          help="Operational state of the MLAG peer link",
                          optional=True )
   localIntfStatus = Enum( values=IntfModel.intfOperStatuses,
                           help="Operational state of the MLAG local interface",
                           optional=True )
   systemId = MacAddress( help="MLAG System ID", optional=True )
   mlagPorts = Dict( valueType=int, help="Number of MLAGs having each state. "
                     "Key values are: Disabled, Configured, Inactive, "
                     "Active-partial, Active-full" )
   portsErrdisabled = Bool(
      help="True if ports are disabled because reload delay is active" )
   portsErrdisabledTime = Float( help="Time when the ports were errdisabled",
                                 optional=True )
   dualPrimaryDetectionState = Enum( values=('configured', 'running',
                                             'detected', 'disabled' ),
                                      help="Dual-primary detection state",
                                      optional=True )
   dualPrimaryPortsErrdisabled = Bool(
      help="True if ports are errdisabled because dual-primary"
      " is detected or recovery delay is active" )

   class Detail( Model ):
      mlagState = Enum( values=Tac.Type( 'Mlag::MlagState' ).attributes,
                        help="Specific MLAG state" )
      peerMlagState = Enum( values=Tac.Type(
         'Mlag::PeerMlagState::MlagState' ).attributes, help="Peer MLAG state" )
      stateChanges = Int( help="Number of MLAG state transitions" )
      lastStateChangeTime = Float(
         help="Time of the last MLAG state transition", optional=True )
      mlagHwReady = Bool(
         help="True if the switch hardware is ready to be configured as an "
         "MLAG peer", optional=True )
      failover = Bool( help="True if this MLAG peer is in Failover mode" )
      failoverCauseList = List( valueType=str, help="List of causes for Failover" )
      failoverInitiated = Bool( help="True if failover is initiated by this MLAG "
                                "peer due to forceful failover. Examples : "
                                "Maintenance Mode, Fast Boot Reload" )
      lastFailoverChangeTime = Float(
         help="Time of the last failover transition", optional=True )
      secondaryFromFailover = Bool(
         help="True if this MLAG peer's state is secondary because the "
         "other peer assumed the primary state due to failover" )
      primaryPriority = Int( help="Used to determine which MLAG peer becomes "
                             "primary when negotiating from scratch" )
      _primaryPriorityDefault = Int( 
         help="The default MLAG primaryPriority value" ) 
      peerPrimaryPriority = Int(
         help="The other MLAG peer's primaryPriority value" )
      peerMacAddress = MacAddress(
         help="The other MLAG peer's bridge MAC address" )
      peerMacRoutingSupported = Bool(
         help="True if the switch hardware can route using peer's bridge MAC "
         "address", optional=True )
      peerPortsErrdisabled = Bool(
         help="True if the other MLAG peer's ports are disabled because reload "
         "delay is active" )
      lacpStandby = Bool( help="True if LACP standby is used to disable the "
                          "non-peer-link, MLAG ports during the reload delay "
                          "window" )
      heartbeatInterval = Int(
         help="Configured time (in milliseconds) between the sending of each "
               "MLAG heartbeat" )
      effectiveHeartbeatInterval = Int(
         help="Effective time (in milliseconds) between the sending of each "
               "MLAG heartbeat" )
      heartbeatTimeout = Int(
         help="Time (in milliseconds) of no received MLAG heartbeats before "
         "a timeout is declared" )
      lastHeartbeatTimeout = Float(
         help="Time of the last MLAG heartbeat timeout", optional=True )
      heartbeatTimeoutsSinceReboot = Int(
         help="Number of MLAG heartbeat timeouts that occurred "
         "since the last reload" )
      udpHeartbeatAlive = Bool( help="True if we received UDP heartbeats from "
                                "peer within the last timeout period" )
      udpHeartbeatsReceived = Int(
         help="Number of MLAG UDP heartbeat received" )
      udpHeartbeatsSent = Int(
         help="Number of MLAG UDP heartbeat sent" )
      peerMonotonicClockOffset = Float(
         help="System monotonic time difference (in seconds) "
         "between the MLAG peers", optional=True )
      enabled = Bool( help="True if the MLAG feature is enabled" )
      mountChanges = Int( help="Number of MLAG peer mount state transitions" )
      fastMacRedirectionEnabled = Bool(
         help="True if fast MAC redirection is enabled for "
         "faster L2/L3 convergence", optional=True )

      dualPrimaryDetectionDelay = Int( help="How long (in seconds) to delay "
                                        "the detection of dual-primary after "
                                        "failover.",
                                        optional=True )
      dualPrimaryAction = Enum( values=('none', 'errdisableAllInterfaces', ),
                                 help="Action to take when dual primary is "
                                 "detected.",
                                 optional=True )
      dualPrimaryMlagRecoveryDelay = Int( help="How long (in seconds) to delay "
                                      "the recovery of MLAG interfaces after "
                                      " dual primary is resolved",
                                      optional=True )
      dualPrimaryNonMlagRecoveryDelay = Int(
         help="How long (in seconds) to delay the recovery of non-MLAG interfaces "
         " after dual primary is resolved", optional=True )

   detail = Submodel( valueType=Detail, help="More details of MLAG status",
                      optional=True )

   def render( self ):
      t = showMlagTableFormat()
      headerFormat = Format( justify="left", noBreak=True, terminateRow=True )
      headerFormat.noPadLeftIs( True )
      headerFormat.padLimitIs( True )
      headerFormat.noTrailingSpaceIs( True )
      t.newRow( FormattedCell( content="MLAG Configuration:", nCols=1,
                               format=headerFormat ) )

      def _getStr( val, default="" ):
         return str( val ) if val else default

      def _getCapStr( val, default="" ):
         return str( val ).capitalize() if val else default

      def _getStringValue( val, default="" ):
         return val.stringValue if val else default

      for ( label, value ) in (
         ( "domain-id", _getStr( self.domainId ) ),
         ( "local-interface", _getStringValue( self.localInterface ) ),
         ( "peer-address", _getStr( self.peerAddress, '0.0.0.0' ) ),
         ( "peer-link", _getStringValue( self.peerLink ) ),
         ( "hb-peer-address", self.heartbeatPeerAddress ),
         ( "hb-peer-vrf", self.heartbeatPeerVrf ),
         ( "peer-config", _getStr( self.configSanity ) ),
         ):
         if value is not None:
            t.newRow( label, ":  ", value )

      t.newRow( "" )
      t.newRow( FormattedCell( content="MLAG Status:", nCols=1,
                               format=headerFormat ) )
      if self.portsErrdisabled:
         if self.reloadDelay == -1 or self.reloadDelayNonMlag == -1:
            timeLeft = 'infinity'
         elif self.portsErrdisabledTime:
            maxReloadDelay = max( self.reloadDelay, self.reloadDelayNonMlag )
            import datetime
            timeLeft = '%s left' % datetime.timedelta(
               seconds=int( self.portsErrdisabledTime
                            + maxReloadDelay - Tac.now() ) )
         reloadState = "%s/Reload" % str( self.state ).capitalize()
         t.newRow( "state", ":  ", reloadState, " (%s)" % timeLeft )
      else:
         t.newRow( "state", ":  ", str( self.state ).capitalize() )
      for ( label, value ) in (
         ( "negotiation status",
           negStatusEnumToStr( _getStr( self.negStatus ) ) ),
         ( "peer-link status", _getCapStr( self.peerLinkStatus ) ),
         ( "local-int status", _getCapStr( self.localIntfStatus ) ),
         ( "system-id", _getStr( self.systemId, '00:00:00:00:00:00' ) ) ):
         t.newRow( label, ":  ", value )
      dpState = self.dualPrimaryDetectionState
      dpPortsErrdisabled = self.dualPrimaryPortsErrdisabled
      if dpState is not None:
         t.newRow( "dual-primary detection", ":  ", dpState.capitalize() )
         if Toggles.MlagToggleLib.toggleRecoveryDelayEnabled():
            t.newRow( "dual-primary interface errdisabled", ":  ",
                      dpPortsErrdisabled )
      t.newRow( "" )
      t.newRow( FormattedCell( content="MLAG Ports:", nCols=1,
                               format=headerFormat ) )
      for k in ( 'Disabled', 'Configured', 'Inactive',
                 'Active-partial', 'Active-full' ):
         t.newRow( k, ":  ", str( self.mlagPorts[k] ) )

      print t.output()

      if not self.detail:
         return

      def _agoOrInfinity( val ):
         return "Infinity" if val == -1 else "%d seconds" % val

      d = self.detail
      # pylint: disable-msg=E1101
      def _portsErrdisabledTuple():
         if d.mlagState == "primary":
            return ( "Peer ports errdisabled", d.peerPortsErrdisabled )
         else:
            return ( "Ports errdisabled", self.portsErrdisabled )

      def dualPrimaryActionStr( action ):
         # make it shorter
         if action == 'errdisableAllInterfaces':
            return 'errdisable-all'
         return action

      def getDetailedMlagStates( state, pState, cause, failoverInitiated ):
         '''
         This function takes input the current MLAG state, MLAG state of the peer,
         the list of failover causes, and generates detailed output of MLAG state
         for both the peers.
         - It describes the intermediate MLAG states in different situations.
         - Consider the examples :
         *** Peer link is down ***
           o If the peer link is down, it has the upper hand in deciding the detailed
             MLAG states and we will ignore other failover causes.
             In this case, both MLAG peers would try to become primary.
           o In case the state is already primary, it will remain as it is.
           o In cases the state is disabled/disabling, we don't consider that
             it will be able to switch states in those cases, so again the output
             remains as it is.
           o In cases the state is either secondary or inactive, they are considered
             to be able to switch their state, so the detailed state would include
             the statement that it is "waiting to become primary" because the
             peer link is down.

         *** Peer link is NOT down ***
           o If the peer link is up, then we would consider other failover causes
             to determine the detailed states.
           o Consider the case where there is no failover due to Mlag ASU2 or
             Mlag Maintenance Mode. In this case everything is working just fine
             and there won't be any additional detail to display, so the output
             remains as it is.
           o Now, suppose failover is there due to either Maintenance Mode/ASU2.
             Then we would want the targeted peer to become Secondary in a failover
             and the other one to become Primary.
           o In case the state of targeted peer is already secondary, the output
             will remain as it is. But if the targeted peer is not secondary, its
             output would include the statement that it is "waiting to become
             secondary" due to failover.
           o Also for the other peer, if it's not already primary, the output
             would include the statement that it is "waiting to become primary"
             due to failover.
         '''
         failoverCauseName = Tac.Type( 'Mlag::FailoverCauseName' )
         pLinkDown = failoverCauseName.peerLinkDown in cause
         maintMode = failoverCauseName.maintMode in cause
         mlagAsu = failoverCauseName.mlagAsu in cause
         unknown = not ( maintMode or mlagAsu or pLinkDown )

         # Fetch mlag state constants from TAC model.
         MlagState = Tac.Type( 'Mlag::MlagState' )
         PeerMlagState = Tac.Type( 'Mlag::PeerMlagState::MlagState' )

         detailState = ""
         detailPeerState = ""

         # Detail strings
         waitPrimaryFailoverDesc = ( ' ( waiting to become primary '
                                     'due to failover )' )
         waitPrimaryPeerLinkDownDesc = ( ' ( waiting to become primary '
                                         'because peer link is down )' )
         waitSecondaryFailoverDesc = ( ' ( waiting to become secondary '
                                       'due to failover )' )

         # If there is no failover cause at all (Maintenance Mode or ASU2) and
         # the peer link is also up, then there would be no transition in the states.
         # We should not show detailed states on the MLAG peer which has not
         # initiated the failover.
         if unknown or not failoverInitiated:
            detailState = state
            detailPeerState = pState
         # If the peer link is down, it has the upper hand in deciding the detailed
         # MLAG states and we will ignore other failover causes.
         # In this case, both MLAG peers would try to become primary.
         # This is known as Split Brain.
         # This is like a breakup between a couple, they will live their own life
         # independently as they don't know if this is permanent or temporary. :P
         elif pLinkDown:
            if state in [ MlagState.inactive, MlagState.secondary ]:
               detailState = state + waitPrimaryPeerLinkDownDesc

               if pState == PeerMlagState.primary:
                  detailPeerState = pState
               else:
                  detailPeerState = pState + waitPrimaryPeerLinkDownDesc

            else:
               detailState = state
               if pState == PeerMlagState.primary:
                  detailPeerState = pState
               else:
                  detailPeerState = pState + waitPrimaryPeerLinkDownDesc

         else:
            if state == MlagState.secondary:
               detailState = state
               if pState == PeerMlagState.primary:
                  detailPeerState = pState
               else:
                  detailPeerState = pState + waitPrimaryFailoverDesc

            else:
               detailState = state + waitSecondaryFailoverDesc
               if pState == PeerMlagState.primary:
                  detailPeerState = pState
               else:
                  detailPeerState = pState + waitPrimaryFailoverDesc

         return ( detailState, detailPeerState )

      detailState, detailPeerState = getDetailedMlagStates(
         d.mlagState, d.peerMlagState, d.failoverCauseList,
         d.failover or d.failoverInitiated )
      print "MLAG Detailed Status:"
      t = showMlagTableFormat( detail=True )
      for ( label, value ) in (
         ( "State", detailState ),
         ( "Peer State", detailPeerState ),
         ( "State changes", d.stateChanges ),
         ( "Last state change time", timestampToStr( d.lastStateChangeTime ) ),
         ( "Hardware ready", d.mlagHwReady ),
         ( "Failover", d.failover or d.failoverInitiated ),
         ( "Failover Cause(s)", ", ".join( d.failoverCauseList ) ),
         ( "Last failover change time",
           timestampToStr( d.lastFailoverChangeTime ) ),
         ( "Secondary from failover", d.secondaryFromFailover ),
         ( "primary-priority", d.primaryPriority ),
         ( "Peer primary-priority", d.peerPrimaryPriority ),
         ( "Peer MAC address", d.peerMacAddress ),
         ( "Peer MAC routing supported", d.peerMacRoutingSupported ),
         ( "Reload delay", _agoOrInfinity( self.reloadDelay ) ),
         ( "Non-MLAG reload delay", _agoOrInfinity( self.reloadDelayNonMlag ) ),
         _portsErrdisabledTuple(),
         ( "Lacp standby", d.lacpStandby ),
         ( "Configured heartbeat interval", "disabled" if d.heartbeatInterval == 0
           else "%d ms" % d.heartbeatInterval ),
         ( "Effective heartbeat interval", "disabled"
            if d.effectiveHeartbeatInterval == 0
            else "%d ms" % d.effectiveHeartbeatInterval ),
         ( "Heartbeat timeout", "disabled" if d.heartbeatInterval == 0 else
           "%d ms" % d.heartbeatTimeout ),
         ( "Last heartbeat timeout", timestampToStr( d.lastHeartbeatTimeout ) ),
         ( "Heartbeat timeouts since reboot", d.heartbeatTimeoutsSinceReboot ),
         ( "UDP heartbeat alive", d.udpHeartbeatAlive ),
         ( "Heartbeats sent/received", "%d/%d" % 
            ( d.udpHeartbeatsSent, d.udpHeartbeatsReceived ) ),
         ( "Peer monotonic clock offset",
           "unknown" if not d.peerMonotonicClockOffset else
           "%f seconds" % d.peerMonotonicClockOffset ),
         ( "Agent should be running", d.enabled ),
         ( "P2p mount state changes", d.mountChanges ),
         ( "Fast MAC redirection enabled", d.fastMacRedirectionEnabled ),
         ( "Dual-primary detection delay", d.dualPrimaryDetectionDelay ),
         ( "Dual-primary action", dualPrimaryActionStr( d.dualPrimaryAction ) ),
         ( "Dual-primary recovery delay", d.dualPrimaryMlagRecoveryDelay ),
         ( "Dual-primary non-mlag recovery delay",
           d.dualPrimaryNonMlagRecoveryDelay )
      ):

         # pylint: disable-msg=W0212
         if ( "primary-priority" in label ) and ( 
            d.primaryPriority == d._primaryPriorityDefault ):
            pass
         elif ( "Dual-primary" in label and value is None ):
            pass
         elif ( "recovery delay" in label and
                not Toggles.MlagToggleLib.toggleRecoveryDelayEnabled() ):
            pass
         else:
            t.newRow( label, ":  ", value )

      # pylint: enable-msg=E1101

      print t.output()

intfEnabledValues = ( 'ena', 'dis' )

# Cli Model for "show mlag interfaces..."
class Interfaces( Model ):

   class MlagInterface( Model ):
      localInterface = Interface(
         help="Local interface corresponding to the MLAG", optional=True )
      localInterfaceDescription = Str(
         help="Description of the local interface "
         "corresponding to the MLAG", optional=True )
      localInterfaceStatus = Enum(
         values=IntfModel.intfOperStatuses,
         help="Operational state of the local MLAG interface", optional=True )
      peerInterface = Interface(
         help="Peer interface corresponding to the MLAG", optional=True )
      peerInterfaceStatus = Enum(
         values=IntfModel.intfOperStatuses,
         help="Operational state of the peer MLAG interface", optional=True )
      status = Enum( values=('n/a', 'disabled', 'configured', 'inactive',
                             'active-partial', 'active-partial-local-up',
                             'active-partial-remote-up', 'active-full'),
                     help="Status of MLAG", optional=True )

      class Detail( Model ):
         localInterfaceEnabled = Enum(
            values=intfEnabledValues,
            help="Enabled state of the local MLAG interface", optional=True )
         peerInterfaceEnabled = Enum(
            values=intfEnabledValues,
            help="Enabled state of the peer MLAG interface", optional=True )
         lastChangeTime = Float( help="Time of the last status transition",
                                 optional=True )
         changeCount = Int( help="Number of status transitions",
                            optional=True )

      detail = Submodel( valueType=Detail,
                         help="More details of MLAG interface", optional=True )

      members = List( valueType=Interface,
                      help="MLAG interface members (from both peers)",
                      optional=True )

   interfaces = Dict( valueType=MlagInterface,
                      help="MLAG interfaces, keyed by MLAG ID" )

   def renderStr( self ):
      if len( self.interfaces ) == 0:
         return "No MLAG ports configured\n"

      def _newTable( intf ):
         assert not intf.members
         if intf.detail:
            return createTable( ( "mlag", "state", "local", "remote",
                                  ( "local/remote", ( "oper", "config" )),
                                  "last change", "changes" ) )
         else:
            table = createTable( ( "mlag", "desc", "state", "local",
                                   "remote", ("local/remote", ("status",)) ) )
            table.formatColumns( None, Format( justify='left', maxWidth=30 ),
                                 None, None, None, None )
            return table

      def _shortenIntfName( n ):
         if n is None:
            return 'n/a'
         if n.stringValue == '':
            return '-'
         return IntfCli.Intf.getShortname( n.stringValue )

      def _intfStatusStr( intfStatus, unknown ):
         return unknown if intfStatus == 'unknown' else intfStatus

      def _intfEnabledStr( enabled, default ):
         return enabled if enabled else default

      def _newRow( table, mlagId, intf ):
         table.newRow(
            mlagId, intf.localInterfaceDescription if
            intf.localInterfaceDescription else '', intf.status,
            _shortenIntfName( intf.localInterface ),
            _shortenIntfName( intf.peerInterface ),
            "%s/%s" % ( _intfStatusStr( intf.localInterfaceStatus, '?' ),
                        _intfStatusStr( intf.peerInterfaceStatus, '-' ) ) )

      def _newRowDetail( table, mlagId, intf ):
         linkStatus = "%s/%s" % \
             ( _intfStatusStr( intf.localInterfaceStatus, '?' ),
               _intfStatusStr( intf.peerInterfaceStatus, '-' ) )
         linkConfig = "%s/%s" % \
             ( _intfEnabledStr( intf.detail.localInterfaceEnabled, '?' ),
               _intfEnabledStr( intf.detail.peerInterfaceEnabled, '-' ) )
         table.newRow( mlagId, intf.status,
                       _shortenIntfName( intf.localInterface ),
                       _shortenIntfName( intf.peerInterface ),
                       linkStatus, linkConfig,
                       timestampToStr( intf.detail.lastChangeTime ),
                       '0' if not intf.detail.changeCount
                       else intf.detail.changeCount )

      def _membersStr( mlagId, intf ):
         ret = ''
         ret += "Mlag%s is %s\n" % ( mlagId, intf.localInterface.stringValue )
         if intf.members:
            ret += LacpLib.printList( intf.members, "  Active Ports: ", 74,
                                      returnValue=True ) + '\n'
         return ret

      table = None
      output = ''
      # First print MLAGs interfaces that were requested but not
      # configured. Then display the table of requested, configured
      # MLAGs.
      for mlagId, intf in \
             sorted( self.interfaces.iteritems(), # pylint: disable-msg=E1101
                     key=lambda x: int( x[0] ) + 9999
                     if x[1].localInterface else int( x[0] ) ):
         if not intf.localInterface:
            output += "MLAG port %s not configured\n" % mlagId
         else:
            if intf.members is not None:
               output += _membersStr( mlagId, intf )
            else:
               if not table:
                  table = _newTable( intf )
               if intf.detail:
                  _newRowDetail( table, mlagId, intf )
               else:
                  _newRow( table, mlagId, intf )

      if table:
         output += table.output()
      return output

   def render( self ):
      rs = self.renderStr()
      if rs:
         print rs,

# Cli Model for "show mlag issu warnings ..."
class IssuWarnings( Model ):
   issuWarnings = List( valueType=str, help="Warning messages after a reload "
                        "of an MLAG peer is requested" )
   def render( self ):
      worryStr = '\n\n'.join( self.issuWarnings )

      # We print some additional text for the purposes of test due to
      # limitations in cli expect
      testMode = os.environ.get( "MLAG_CLITEST" )

      if not testMode:
         if worryStr:
            print worryStr
      else:
         name = 'WORRY'
         if worryStr:
            print "MLAG_CLITEST_START_%s[" % name + worryStr + \
                "]MLAG_CLITEST_END_%s\n" % name
         else:
            print "MLAG_CLITEST_NO_%s" % name

# Cli Model for "show mlag config-sanity"
class ParamValue( Model ):
   localValue = Str( help="Local value", optional=True )
   peerValue = Str( help="Peer value", optional=True )
   
class InterfaceValue( Model ):
   interface = Dict( valueType=ParamValue,
                     help="Local and peer values for interface" )

class GlobalFeatureParam( Model ):
   # global-param per feature is unique for a configuration.
   globalParameters = Dict( valueType=ParamValue,
                            help="Local and peer values for parameter" )

class IntfFeatureParam( Model ):
   # interface-param per feature is not unique for a configuration.
   interfaceParameters = Dict( valueType=InterfaceValue,
                               help="Local and peer values for interface parameter" )

class ConfigSanity( Model ):

   globalConfiguration = Dict( valueType=GlobalFeatureParam,
                          help="Global configuration keyed on feature" )
   interfaceConfiguration = Dict( valueType=IntfFeatureParam,
                    help="Interface configuration keyed on feature" )
   detail = Bool( help="Will list matching peer-local value pairs too",
                 optional=True )
   mlagActive = Bool( help="Command is enabled if MLAG state is active" )

   def renderStr( self ):
      if not self.mlagActive:
         print "Command not applicable for inactive MLAG state.\n"
         return
      if not self.globalConfiguration:
         globalTablePresent = False
         print "No global configuration inconsistencies found.\n"
      else:
         globalTablePresent = True
      if not self.interfaceConfiguration:
         intfTablePresent = False
         print "No per interface configuration inconsistencies found.\n"
      else:
         intfTablePresent = True

      if not globalTablePresent and not intfTablePresent:
         return

      # param sorting tweaks to alphabetic paramName sort
      def _paramSortKey( ):
         # Sort by vlan id if string 'vlan %d' is present in paramName. This can
         # occur anywhere in the paramName, but currently it is at the end, as there
         # is no example of nested configuration within vlan config.
         alphaToNum = lambda param: int( param ) if param.isdigit() else param
         return lambda key : [ alphaToNum( vlanId ) for vlanId in re.split( \
               'vlan ([0-9]+)', key[ 0 ] ) ]

      def _intfParamSortKey( ):
         # intfParamName = [ vlanId | ip address ] paramName
         # Sort by paramName and then by vlan id or ip address if present
         ipPattern = r'\b\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}\b'
         def alphaToNum( alpha ):
            if alpha.isdigit():
               return int( alpha )
            elif re.match( ipPattern, alpha ):
               ipAddr = re.match( ipPattern, alpha ).group()
               return Tac.Value( "Arnet::IpAddr", stringValue=ipAddr ).value
            else:
               return alpha
         # Below function will return 
         # [ 'vlan-to-vni', 10 ] if key[ 0 ]  = '10 vlan-to-vni'
         # [ 'flood vtep list', 33686018 ] if key[ 0 ] = '2.2.2.2 flood vtep list'
         return lambda key : [ alphaToNum( word ) for word in re.split(
               r'\b(%s|\d+)\b\s' % ipPattern, key[ 0 ] ) ][ :: -1 ]

      def _newTable( ):
         table = createTable( ( "Feature", "Attribute", "Interfaces", "Local value",
                                "Peer value" ) )
         return table
      def _newGlobalTable( ):
         table = createTable( ( "Feature", "Attribute", "Local value",
                                "Peer value" ) )
         return table

      def _newRow( table, pluginName, paramName, intfName, paramValue ):
         if ( intfName.startswith( "Peer" ) ) :
            intfNameS = "P" + IntfCli.Intf.getShortname( intfName[ 4: ] )
         else:
            intfNameS = IntfCli.Intf.getShortname( intfName )
         localVal = paramValue.localValue if paramValue.localValue else '-'
         peerVal = paramValue.peerValue if paramValue.peerValue else '-'
         table.newRow( pluginName,
                       paramName,
                       intfNameS,
                       localVal,
                       peerVal )

      def _newRowGlobal( table, pluginName, paramName, paramValue ):
         localVal = paramValue.localValue if paramValue.localValue else '-'
         peerVal = paramValue.peerValue if paramValue.peerValue else '-'
         table.newRow( pluginName,
                       paramName,
                       localVal,
                       peerVal )

      headerFormat = Format( justify="left", noBreak=True, terminateRow=True )
      headerFormat.noPadLeftIs( True )
      headerFormat.padLimitIs( True )
      headerFormat.noTrailingSpaceIs( True )

      globalTable = None
      intfTable = None
      output = ''
      # pylint: disable-msg=E1101
      for featureName, featureValue in sorted( 
         self.globalConfiguration.iteritems() ):
         if not globalTable:
            globalTable = _newGlobalTable()
         for paramName, paramValue in sorted(
            featureValue.globalParameters.iteritems(), key=_paramSortKey() ):
            _newRowGlobal( globalTable, featureName, paramName, paramValue )
      # pylint: disable-msg=E1101
      for featureName, featureValue in sorted( 
         self.interfaceConfiguration.iteritems() ):
         if not intfTable:
            intfTable = _newTable()
         for intfParamName, intfParamValue in sorted(
            featureValue.interfaceParameters.iteritems(), key=_intfParamSortKey() ):
            for intfName, paramValue in sorted( intfParamValue.interface.iteritems(),
                                key=lambda kv: intfNameKey( kv[ 0 ] ) ):
               _newRow( intfTable, featureName, intfParamName, intfName, paramValue )

      if globalTable:
         if self.detail:
            output += "Global configuration:\n"
         else:
            output += "Global configuration inconsistencies:\n"
         output += globalTable.output()
         output += "\n"
      if intfTable:
         if self.detail:
            output += "Interface configuration:\n"
         else:
            output += "Interface configuration inconsistencies:\n"
         output += intfTable.output()
         output += "\n"
      return output

   def render( self ):
      rs = self.renderStr()
      if rs:
         print rs,

# Cli Model for "show mlag issu compatibility <image>"
compatCheckMessage = "Please use the MLAG ISSU Compatibility Check tool at "\
                     "https://www.arista.com/en/support/mlag-portal "\
                     "to verify ISSU compatibility."
# Generic error message
invalidTemplate = "%s is not a valid EOS system image."

class IssuCompatibility( Model ):
   __revision__ = 2
   currentVersion = Str(
         help="The EOS version of the current image" )
   newVersion = Str(
         help="The EOS version of the new image", optional=True )
   newImagePath = Str(
         help="The path to the new image" )
   isValid = Bool(
         help="True if the new image is valid" )
   detailedErrMessage = Str( 
         help="detailed error message for invalid image",
         optional=True )

   def renderStr( self ):
      def _getStr( val, default="" ):
         return str( val ) if val else default
      newImagePath = _getStr( self.newImagePath )
      if not self.isValid:
         detailedErrMessage = _getStr( self.detailedErrMessage )
         if detailedErrMessage != "":
            return self.detailedErrMessage
         else:
            return fill( invalidTemplate % newImagePath ) 
      else:
         return compatCheckMessage

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # suggestedVersion is no longer needed
         dictRepr[ 'suggestedVersion' ] = []

      return dictRepr

   def render( self ):
      print self.renderStr()
              
# FIXME: This class does not implement lacpSysId(), declared in LacpSysIdBase
# pylint: disable=abstract-method
class MlagLacpSysId( LacpSysIdBase ):
   systemId = Str( help="System Identifier" )
   priority = Int( help="System priority." )
   _brief = Bool( help="Detailed or brief" )
   _internal = Bool( help="Display LACP system identifier"
                          " in conjunction with other internal LACP information" )

   def render( self ):
      if self.systemId == '00:00:00:00:00:00':
         return
      if self._internal:
         # Even though this is similar to _brief, note that there is no
         # additional newline in the output
         print "MLAG System-identifier: %s" % dot43sysid( self.priority,
                                                          self.systemId )
         return
      if self._brief:
         print "MLAG System-identifier: %s\n" % dot43sysid( self.priority,
                                                            self.systemId )
      else:
         print "MLAG System-identifier: %s\n" % self.systemId
         print "  802.11.43 representation: %s\n" % dot43sysid( self.priority,
                                                                self.systemId )

