#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import CliCommand
import CliMatcher
import ConfigMount
import Toggles.MlagToggleLib
import CliPlugin.MlagConfigCli as MlagConfigCli
from CliPlugin.EbraEthIntfCli import switchPortMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.VlanCli import bridgingSupportedGuard
from CliPlugin.VirtualIntfRule import VirtualIntfMatcher
from CliPlugin.VlanIntfCli import VlanIntf
from CliPlugin.VrfCli import getAllVrfNames

mlagConfig = None
matcherDomainId = CliMatcher.StringMatcher( helpdesc='MLAG domain ID', 
      helpname='STRING' )
matcherDetection = CliMatcher.KeywordMatcher( 'detection', 
      helpdesc='Configure dual-primary detection' )
matcherDualPrimary = CliMatcher.KeywordMatcher( 'dual-primary', 
      helpdesc='Configure dual-primary parameters' )
matcherHeartbeatInterval = CliMatcher.KeywordMatcher( 'heartbeat-interval', 
      helpdesc='Time in milliseconds between MLAG heartbeat messages' )
matcherVersion = CliMatcher.IntegerMatcher( 1, 1000000, 
      helpdesc='Supported MLAG Version' )
matcherPeerAddress = CliMatcher.KeywordMatcher( 'peer-address', 
      helpdesc='IP address of MLAG peer' )
matcherDelay = CliMatcher.IntegerMatcher( 1, 1000, 
      helpdesc='Specify delay in seconds' )
matcherPeerLink = CliMatcher.KeywordMatcher( 'peer-link', 
      helpdesc='Interface connecting to MLAG peer' )
matcherReloadDelay = CliMatcher.KeywordMatcher( 'reload-delay', 
      helpdesc=( 'Delay (seconds) after reboot until all non peer-link '
                 'ports are enabled' ) )
matcherNonMlag = CliMatcher.KeywordMatcher( 'non-mlag', 
      helpdesc=( 'Delay (seconds) after reboot until ports that are not '
                 'part of an MLAG are enabled' ) )
matcherMlag = CliMatcher.KeywordMatcher( 'mlag', 
      helpdesc=( 'Delay (seconds) after reboot until non peer-link ports '
                 'that are part of an MLAG are enabled' ) )
matcherMode = CliMatcher.KeywordMatcher( 'mode', 
      helpdesc='MLAG behavior during reload-delay period' )
matcherPriority = CliMatcher.IntegerMatcher( 1, 32767, 
      helpdesc='Lower means higher priority' )
matcherDelay = CliMatcher.IntegerMatcher( 0, 86400, helpdesc='Seconds' )
matcherInterval = CliMatcher.IntegerMatcher( 1000, 30000, helpdesc='Milliseconds' )
matcherTimeout = CliMatcher.IntegerMatcher( 2500, 75000, helpdesc='Milliseconds' )
matcherThreshold = CliMatcher.IntegerMatcher( 0, 3600, helpdesc='Seconds' )
matcherVlan = VirtualIntfMatcher( 'Vlan', 1, 4094, 
                                  value=lambda mode, intf: VlanIntf( intf, mode ), 
                                  guard=bridgingSupportedGuard )
matcherRecovery = CliMatcher.KeywordMatcher( 'recovery',
      helpdesc=( 'Configure actions taken during dual-primary detection recovery' ) )
matcherRecoveryNonMlag = CliMatcher.KeywordMatcher( 'non-mlag',
      helpdesc=( 'Delay (seconds) after dual-primary detection resolves until ports '
                 'that are not part of an MLAG are enabled' ) )
matcherRecoveryMlag = CliMatcher.KeywordMatcher( 'mlag',
      helpdesc=( 'Delay (seconds) after dual-primary detection resolves until non '
                 'peer-link ports that are part of an MLAG are enabled' ) )
nodeLocalInactive = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'inactive',
         helpdesc='Configure actions taken when MLAG link is locally inactive' ),
      guard=MlagConfigCli.fastMacRedirectConfigurableGuard )
#--------------------------------------------------------------------------------
# [ no | default ] domain-id DOMAINID
#--------------------------------------------------------------------------------
class DomainIdCmd( CliCommand.CliCommandClass ):
   syntax = 'domain-id DOMAINID'
   noOrDefaultSyntax = 'domain-id ...'
   data = {
      'domain-id': 'Unique identifier for the MLAG domain',
      'DOMAINID': matcherDomainId,
   }
   handler = MlagConfigCli.setDomainId
   noOrDefaultHandler = handler

MlagConfigCli.ConfigMlagMode.addCommandClass( DomainIdCmd )

#--------------------------------------------------------------------------------
# ( no | default ) dual-primary detection
#--------------------------------------------------------------------------------
class DualPrimaryDetectionCmd( CliCommand.CliCommandClass ):
   syntax = 'dual-primary detection delay DELAY [ action errdisable all-interfaces ]'
   noOrDefaultSyntax = 'dual-primary detection ...'
   data = {
      'dual-primary': matcherDualPrimary,
      'detection': matcherDetection,
      'delay': 'Delay detection for <N> seconds',
      'DELAY': matcherDelay,
      'action': 'Specify action when dual-primary is detected',
      'errdisable': 'Errdisable interfaces',
      'all-interfaces': 'Disable all Ethernet interfaces except peer-link',
   }
   handler = MlagConfigCli.setDualPrimaryDetection
   noOrDefaultHandler = MlagConfigCli.noDualPrimaryDetection

MlagConfigCli.ConfigMlagMode.addCommandClass( DualPrimaryDetectionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-interval HEARTBEAT_INTERVAL
#--------------------------------------------------------------------------------
class HeartbeatIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'heartbeat-interval HEARTBEAT_INTERVAL'
   noOrDefaultSyntax = 'heartbeat-interval ...'
   data = {
      'heartbeat-interval': 'Time in milliseconds between MLAG heartbeat messages',
      'HEARTBEAT_INTERVAL': matcherInterval,
   }
   @staticmethod
   def handler( mode, args ):
      mlagConfig.heartbeatInterval = args.get( 'HEARTBEAT_INTERVAL', 
                                               mlagConfig.heartbeatIntervalDefault )
   noOrDefaultHandler = handler

MlagConfigCli.ConfigMlagMode.addCommandClass( HeartbeatIntervalCmd )

#----------------------------------------------------------------------------------
#[ no | default ] dual-primary recovery delay mlag DELAY1 non-mlag DELAY2
#----------------------------------------------------------------------------------
class DualPrimaryRecoveryDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'dual-primary recovery delay mlag DELAY1 non-mlag DELAY2'
   noOrDefaultSyntax = 'dual-primary recovery delay ...'
   data = {
      'dual-primary': matcherDualPrimary,
      'recovery': matcherRecovery,
      'delay': 'Delay interface recovery for mlag and non-mlag interfaces',
      'mlag': matcherRecoveryMlag,
      'DELAY1' : matcherDelay,
      'non-mlag': matcherRecoveryNonMlag,
      'DELAY2' : matcherDelay,
      }
   handler = MlagConfigCli.setDualPrimaryRecoveryDelay
   noOrDefaultHandler = MlagConfigCli.noDualPrimaryRecoveryDelay
if Toggles.MlagToggleLib.toggleRecoveryDelayEnabled():
   MlagConfigCli.ConfigMlagMode.addCommandClass(
      DualPrimaryRecoveryDelayCmd )

#--------------------------------------------------------------------------------
# heartbeat-interval disabled
#--------------------------------------------------------------------------------
class HeartbeatIntervalDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'heartbeat-interval disabled'
   data = {
      'heartbeat-interval': matcherHeartbeatInterval,
      'disabled': 'Disable sending heartbeats and checking heartbeat timeouts',
   }
   @staticmethod
   def handler( mode, args ):
      mlagConfig.heartbeatInterval = 0

MlagConfigCli.ConfigMlagMode.addCommandClass( HeartbeatIntervalDisabledCmd )

#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-timeout HEARTBEAT_TIMEOUT
#--------------------------------------------------------------------------------
class HeartbeatTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'heartbeat-timeout HEARTBEAT_TIMEOUT'
   noOrDefaultSyntax = 'heartbeat-timeout ...'
   data = {
      'heartbeat-timeout': ( 'Elapsed milliseconds since last heartbeat before '
                             'breaking MLAG' ),
      'HEARTBEAT_TIMEOUT': matcherTimeout,
   }
   hidden = True
   @staticmethod
   def handler( mode, args ):
      mlagConfig.heartbeatTimeout = args.get( 'HEARTBEAT_TIMEOUT', 0 )
   noOrDefaultHandler = handler

MlagConfigCli.ConfigMlagMode.addCommandClass( HeartbeatTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] local-interface VLAN_INTF
#--------------------------------------------------------------------------------
class LocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface VLAN_INTF'
   noOrDefaultSyntax = 'local-interface ...'
   data = {
      'local-interface': ( 'VLAN interface for accepting connections from '
                           'MLAG peer' ),
      'VLAN_INTF': matcherVlan,
   }
   handler = MlagConfigCli.setLocalInterface
   noOrDefaultHandler = MlagConfigCli.noLocalInterface

MlagConfigCli.ConfigMlagMode.addCommandClass( LocalInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] max-version MAX_VERSION
#--------------------------------------------------------------------------------
class MaxVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'max-version MAX_VERSION'
   noOrDefaultSyntax = 'max-version ...'
   data = {
      'max-version': 'Maximum supported MLAG version',
      'MAX_VERSION': matcherVersion,
   }
   hidden = True
   @staticmethod
   def handler( mode, args ):
      mlagConfig.maxVersion = args.get( 'MAX_VERSION', mlagConfig.maxVersionDefault )
   noOrDefaultHandler = handler

MlagConfigCli.ConfigMlagMode.addCommandClass( MaxVersionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] min-version MIN_VERSION
#--------------------------------------------------------------------------------
class MinVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'min-version MIN_VERSION'
   noOrDefaultSyntax = 'min-version ...'
   data = {
      'min-version': 'Minimum supported MLAG version',
      'MIN_VERSION': matcherVersion,
   }
   hidden = True
   @staticmethod
   def handler( mode, args ):
      mlagConfig.minVersion = args.get( 'MIN_VERSION', mlagConfig.minVersionDefault )
   noOrDefaultHandler = handler

MlagConfigCli.ConfigMlagMode.addCommandClass( MinVersionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] peer-address PEERADDRESS
#--------------------------------------------------------------------------------
class PeerAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'peer-address PEERADDRESS'
   noOrDefaultSyntax = 'peer-address ...'
   data = {
      'peer-address': 'IP address of MLAG peer',
      'PEERADDRESS': IpAddrMatcher( helpdesc="Peer's IP address" ),
   }
   handler = MlagConfigCli.setPeerAddress
   noOrDefaultHandler = MlagConfigCli.noPeerAddress

MlagConfigCli.ConfigMlagMode.addCommandClass( PeerAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] peer-address heartbeat HEARTBEAT [ vrf VRF ]
#--------------------------------------------------------------------------------
class PeerAddressHeartbeatCmd( CliCommand.CliCommandClass ):
   syntax = 'peer-address heartbeat HEARTBEAT [ vrf VRF ]'
   noOrDefaultSyntax = 'peer-address heartbeat ...'
   data = {
      'peer-address': matcherPeerAddress,
      'heartbeat': 'Optional separate peer-address for heartbeat',
      'HEARTBEAT': IpAddrMatcher( helpdesc="Peer's IP address" ),
      'vrf': 'Specify VRF name',
      'VRF': CliMatcher.DynamicNameMatcher( getAllVrfNames, 'VRF name' ),
   }
   handler = MlagConfigCli.setHeartbeatPeerAddress
   noOrDefaultHandler = MlagConfigCli.noHeartbeatPeerAddress

MlagConfigCli.ConfigMlagMode.addCommandClass( PeerAddressHeartbeatCmd )

#--------------------------------------------------------------------------------
# ( no | default ) peer-link ETHINTF
#--------------------------------------------------------------------------------
class PeerLinkCmd( CliCommand.CliCommandClass ):
   syntax = 'peer-link ETHINTF'
   noOrDefaultSyntax = 'peer-link ...'
   data = {
      'peer-link': matcherPeerLink,
      'ETHINTF': switchPortMatcher,
   }
   handler = MlagConfigCli.setPeerLink
   noOrDefaultHandler = MlagConfigCli.noPeerLink

MlagConfigCli.ConfigMlagMode.addCommandClass( PeerLinkCmd )

#--------------------------------------------------------------------------------
# [ no | default ] primary-priority PRIMARYPRIORITY
#--------------------------------------------------------------------------------
class PrimaryPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'primary-priority PRIMARYPRIORITY'
   noOrDefaultSyntax = 'primary-priority ...'
   data = {
      'primary-priority': 'Priority relative to MLAG peer',
      'PRIMARYPRIORITY': matcherPriority, 
   }
   hidden = True
   @staticmethod
   def handler( mode, args ):
      mlagConfig.primaryPriority = args.get( 'PRIMARYPRIORITY', 
                                             mlagConfig.primaryPriorityDefault )
   noOrDefaultHandler = handler

MlagConfigCli.ConfigMlagMode.addCommandClass( PrimaryPriorityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] recently-rebooted-threshold THRESHOLD
#--------------------------------------------------------------------------------
class RecentlyRebootedThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'recently-rebooted-threshold THRESHOLD'
   noOrDefaultSyntax = 'recently-rebooted-threshold ...'
   data = {
      'recently-rebooted-threshold': ( 'Duration of the recently-rebooted '
                                       'interval (seconds)' ),
      'THRESHOLD': matcherThreshold,
   }
   hidden = True
   @staticmethod
   def handler( mode, args ):
      pass
   noOrDefaultHandler = handler

MlagConfigCli.ConfigMlagMode.addCommandClass( RecentlyRebootedThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] reload-delay RELOAD_DELAY
#--------------------------------------------------------------------------------
class ReloadDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'reload-delay RELOAD_DELAY'
   noOrDefaultSyntax = 'reload-delay [ RELOAD_DELAY ]'
   data = {
      'reload-delay': matcherReloadDelay,
      'RELOAD_DELAY': matcherDelay,
   }
   handler = MlagConfigCli.setReloadDelay
   noOrDefaultHandler = MlagConfigCli.noReloadDelayMlag

MlagConfigCli.ConfigMlagMode.addCommandClass( ReloadDelayCmd )

#--------------------------------------------------------------------------------
# reload-delay infinity
#--------------------------------------------------------------------------------
class ReloadDelayInfinityCmd( CliCommand.CliCommandClass ):
   syntax = 'reload-delay infinity'
   noOrDefaultSyntax = 'reload-delay infinity'
   data = {
      'reload-delay': matcherReloadDelay,
      'infinity': 'Keep non peer-link ports disabled after reboot',
   }
   handler = MlagConfigCli.setReloadDelay
   noOrDefaultHandler = MlagConfigCli.noReloadDelayMlag

MlagConfigCli.ConfigMlagMode.addCommandClass( ReloadDelayInfinityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] reload-delay mlag RELOAD_DELAY
#--------------------------------------------------------------------------------
class ReloadDelayMlagCmd( CliCommand.CliCommandClass ):
   syntax = 'reload-delay mlag RELOAD_DELAY'
   noOrDefaultSyntax = 'reload-delay mlag ...'
   data = {
      'reload-delay': matcherReloadDelay,
      'mlag': matcherMlag,
      'RELOAD_DELAY': matcherDelay,
   }
   handler = MlagConfigCli.setReloadDelayMlag
   noOrDefaultHandler = MlagConfigCli.noReloadDelayMlag

MlagConfigCli.ConfigMlagMode.addCommandClass( ReloadDelayMlagCmd )

#--------------------------------------------------------------------------------
# reload-delay mlag infinity
#--------------------------------------------------------------------------------
class ReloadDelayMlagInfinityCmd( CliCommand.CliCommandClass ):
   syntax = 'reload-delay mlag infinity'
   data = {
      'reload-delay': matcherReloadDelay,
      'mlag': matcherMlag,
      'infinity': ( 'Keep non peer-link ports that are part of an MLAG '
                    'disabled after reboot' ),
   }
   handler = MlagConfigCli.setReloadDelayMlag

MlagConfigCli.ConfigMlagMode.addCommandClass( ReloadDelayMlagInfinityCmd )

#--------------------------------------------------------------------------------
# reload-delay mode lacp standby
#--------------------------------------------------------------------------------
class ReloadDelayModeCmd( CliCommand.CliCommandClass ):
   syntax = 'reload-delay mode lacp standby'
   noOrDefaultSyntax = 'reload-delay mode ...'
   data = {
      'reload-delay': matcherReloadDelay,
      'mode': matcherMode,
      'lacp': 'LACP behavior during reload-delay period',
      'standby': 'Force LACP to be in standby mode during reload-delay period',
   }
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mlagConfig.lacpStandby = mlagConfig.lacpStandbyDefault
   handler = MlagConfigCli.setLacpStandby

MlagConfigCli.ConfigMlagMode.addCommandClass( ReloadDelayModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] reload-delay non-mlag RELOAD_DELAY
#--------------------------------------------------------------------------------
class ReloadDelayNonMlagCmd( CliCommand.CliCommandClass ):
   syntax = 'reload-delay non-mlag RELOAD_DELAY'
   noOrDefaultSyntax = 'reload-delay non-mlag ...'
   data = {
      'reload-delay': matcherReloadDelay,
      'non-mlag': matcherNonMlag,
      'RELOAD_DELAY': matcherDelay,
   }
   handler = MlagConfigCli.setReloadDelayNonMlag
   noOrDefaultHandler = MlagConfigCli.noReloadDelayNonMlag

MlagConfigCli.ConfigMlagMode.addCommandClass( ReloadDelayNonMlagCmd )

#--------------------------------------------------------------------------------
# reload-delay non-mlag infinity
#--------------------------------------------------------------------------------
class ReloadDelayNonMlagInfinityCmd( CliCommand.CliCommandClass ):
   syntax = 'reload-delay non-mlag infinity'
   data = {
      'reload-delay': matcherReloadDelay,
      'non-mlag': matcherNonMlag,
      'infinity': ( 'Keep non peer-link ports that are not part of an MLAG '
                    'disabled after reboot' ),
   }
   handler = MlagConfigCli.setReloadDelayNonMlag

MlagConfigCli.ConfigMlagMode.addCommandClass( ReloadDelayNonMlagInfinityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown'
   data = {
      'shutdown': 'Disable the MLAG feature',
   }
   handler = MlagConfigCli.noEnabled
   noOrDefaultHandler = MlagConfigCli.setEnabled

MlagConfigCli.ConfigMlagMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] inactive mac-address destination peer-link [ direct | indirect ]
#--------------------------------------------------------------------------------
class FastMacRedirectCmd( CliCommand.CliCommandClass ):
   syntax = 'inactive mac-address destination peer-link TRAFFIC_DIRECT'
   noOrDefaultSyntax = 'inactive mac-address destination peer-link ...'
   data = {
      'inactive': nodeLocalInactive,
      'mac-address': ( 'Configurations for MAC addresses learnt on inactive '
                       'MLAG links' ),
      'destination': 'Set destination of MAC addresses',
      'peer-link': 'Set MAC addresses to peer link',
      'TRAFFIC_DIRECT': CliMatcher.EnumMatcher( {
            'direct': 'Move MAC addresses to peer link',
            'indirect': 'Enable fast MAC redirection',
      } ),
   }
   handler = MlagConfigCli.setFastMacRedirect
   noOrDefaultHandler = MlagConfigCli.defaultFastMacRedirectConfig

MlagConfigCli.ConfigMlagMode.addCommandClass( FastMacRedirectCmd )

def Plugin( entityManager ):
   global mlagConfig
   mlagConfig = ConfigMount.mount( entityManager, "mlag/config", "Mlag::Config",
                                   "w" )
