#!/usr/bin/env python
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''CLI commands for configuring MLAG.'''

import CliParser, BasicCli, LazyMount
import ConfigMount, Tracing, IntfCli
import LagCli
from CliMode.Mlag import MlagMode
import Tac
from .MlagWarningCli import reloadDelayLacpConfigMsg, addMlagShutdownWarnings
from TypeFuture import TacLazyType

mlagConfig = None
ipConfig = None
ethPhyIntfConfigDir = None
lagConfigDir = None
cliConfig = None
mlagConfigurationState = None
mlagHwStatus = None

tacFastMacRedirectionConfig = TacLazyType( "Mlag::FastMacRedirectionConfig" )

t0 = Tracing.trace0

def fastMacRedirectConfigurableGuard( mode, token ):
   if mlagHwStatus.fastMacRedirectionConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def updateMlagConfigState( mode ):
   # Update ConfigurationState to indicate whether Mlag has been configured and 
   # how the configuration was applied. This is to avoid going into reload delay 
   # when Mlag configuration is applied for the first time during runtime.
   if mode.session_.startupConfig() and mlagConfig.configured():
      mlagConfigurationState.state = "configuredStartup"
   elif mlagConfig.configured():
      mlagConfigurationState.state = "configuredCli"
   else:
      mlagConfigurationState.state = "unknown"

ReloadDelay = Tac.Type( "Mlag::ReloadDelay" )

#-------------------------------------------------------------------------------
# config-mlag mode
#-------------------------------------------------------------------------------
class ConfigMlagMode( MlagMode, BasicCli.ConfigModeBase ):
   name = "MLAG configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      MlagMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMlagMode( mode, args ):
   childMode = mode.childMode( ConfigMlagMode )
   mode.session_.gotoChildMode( childMode )

def noMlagMode( mode, args ):
   defaultMlagConfig = Tac.newInstance( 'Mlag::Config', '' )
   for attr in mlagConfig.tacType.attributeQ:
      # Skipping the below attributes:
      # 1. 'intfConfig': It is a collection of Port-Channel interfaces
      # which have 'mlag <num>' in their 'interface Port-Channel<num>'
      # configuration. Since 'interface Port-Channel<num>' configuration is 
      # outside of 'mlag configuration' we do not want to purge it here.   
      # 2. Tacc related attributes.
      if attr.name in ( 'intfConfig', 'parent', 'parentAttrName', 'isNondestructing',
            'entity', 'name' ):
         continue
      if attr.writable:
         if attr.isCollection:
            getattr( mlagConfig, attr.name ).clear()
         else:
            setattr( mlagConfig, attr.name,
                     getattr( defaultMlagConfig, attr.name ) )
   updateMlagConfigState( mode )

#-------------------------------------------------------------------------------
# [no|default] shutdown
#-------------------------------------------------------------------------------
def setEnabled( mode, args ):
   mlagConfig.enabled = True
   updateMlagConfigState( mode )

def noEnabled( mode, args ):
   addMlagShutdownWarnings( mode )
   mlagConfig.enabled = False
   updateMlagConfigState( mode )

#-------------------------------------------------------------------------------
# [no] domain-id
#-------------------------------------------------------------------------------
def setDomainId( mode, args ):
   mlagConfig.domainId = args.get( 'DOMAINID', '' )
   updateMlagConfigState( mode )

def isPeerAddressInPrimarySubnet( intfName, peerAddress=None ):
   ipIntfConfig = ipConfig.ipIntfConfig[ intfName ]

   if not ipIntfConfig:
      return False

   intfAddrWithMask = ipIntfConfig.addrWithMask

   if not peerAddress:
      peerAddress =  mlagConfig.peerAddress
   inSubnet = intfAddrWithMask.address != '0.0.0.0' and peerAddress != '0.0.0.0' and\
              intfAddrWithMask.contains( peerAddress )

   return inSubnet

# This function checks for peer address and returns error if one of the
# following is true
#  - peer address is interface's primary address
#  - peer address is interface's secondary address
#  - peer address does not belong to subnet of both primary and secondary
#    addresses
def mlagGetIntfError( intfName, peerAddress ):
   ipIntfConfig = ipConfig.ipIntfConfig[ intfName ]
   intfAddrWithMask = ipIntfConfig.addrWithMask

   # check if peer address is equal to local interface address
   if intfAddrWithMask.address == peerAddress:
      errMsg = "Peer address cannot be local interface address"
      return errMsg

   inSubnet = isPeerAddressInPrimarySubnet( intfName, peerAddress )

   for intfAddrWithMask in ipIntfConfig.secondaryWithMask:
      # check if peer address is equal to local interface secondary address
      if intfAddrWithMask.address == peerAddress:
         errMsg = "Peer address cannot be local interface address"
         return errMsg

      if not inSubnet: # if not found in any subnet yet, keep checking
         inSubnet = intfAddrWithMask.address != '0.0.0.0' and \
                    intfAddrWithMask.contains( peerAddress )

   # did not find in any subnets
   if not inSubnet:
      errMsg = "Peer address must be in local-interface's subnet"
      return errMsg

   return ""

#-------------------------------------------------------------------------------
# check if local interface has IP and peer address is in the  sunbet of local intf
#-------------------------------------------------------------------------------
def validatePeerAndLocalIntf( mode, localIntfId, peerAddress ):
   # Allow configuration to go through if:
   #  - No local interface is specified but peer address is valid
   #  - Local interface has valid IP address but no peer address given
   #  - Local interface has valid IP address and Peer address is not same as local
   #    interface primary/secondary address. Peer address belongs to the same subnet
   #    as local interface primary (or) secondary address
   if not localIntfId or peerAddress == '0.0.0.0':
      return True
   ipIntfConfig = ipConfig.ipIntfConfig.get( localIntfId )

   if not ipIntfConfig:
      mode.addError( "Local interface must have an IP address configured" )
      return False

   # see if there is any error for peerAddress and local interface
   errMsg = mlagGetIntfError( localIntfId, peerAddress )
   if errMsg:
      mode.addError( errMsg )
      return False
   return True

#-------------------------------------------------------------------------------
# [no] local-interface
#-------------------------------------------------------------------------------
def setLocalInterface( mode, args ):
   intf = args[ 'VLAN_INTF' ]
   if intf.name not in intf.intfConfigDir.intfConfig:
      mode.addError( "Interface %s not configured" % intf.name )
   else:
      if validatePeerAndLocalIntf( mode, intf.name, mlagConfig.peerAddress ):
         mlagConfig.localIntfId = intf.name
         updateMlagConfigState( mode )

def noLocalInterface( mode, args ):
   mlagConfig.localIntfId = ""
   updateMlagConfigState( mode )

#-------------------------------------------------------------------------------
# [no] peer-address
#-------------------------------------------------------------------------------
def setPeerAddress( mode, args):
   peerAddress = args[ 'PEERADDRESS' ]
   if validatePeerAndLocalIntf( mode, mlagConfig.localIntfId, peerAddress ):
      mlagConfig.peerAddress = peerAddress
      mlagConfig.peerAddressConfigured = peerAddress != '0.0.0.0'
      updateMlagConfigState( mode )

def noPeerAddress( mode, args ):
   mlagConfig.peerAddress = '0.0.0.0'
   mlagConfig.peerAddressConfigured = False
   updateMlagConfigState( mode )

#-------------------------------------------------------------------------------
# [no|default] peer-address heartbeat <IP> [vrf <VRF>]
#-------------------------------------------------------------------------------
def setHeartbeatPeerAddress( mode, args ):
   peerAddress = args[ 'HEARTBEAT' ]
   vrfName = args.get( 'VRF' )
   if vrfName == 'default':
      vrfName = ''
   mlagConfig.heartbeatPeerAddress = Tac.Value( "Mlag::PeerAddressAndVrf",
                                                peerAddress, vrfName or '' )
   mlagConfig.heartbeatPeerAddressConfigured = peerAddress != '0.0.0.0'

def noHeartbeatPeerAddress( mode, args ):
   mlagConfig.heartbeatPeerAddress = Tac.Value( "Mlag::PeerAddressAndVrf",
                                                "0.0.0.0", "" )
   mlagConfig.heartbeatPeerAddressConfigured = False

#-------------------------------------------------------------------------------
# dual-primary detection delay <SECONDS> [action errdisable all-interfaces]
# [no|default] dual-primary detection
#-------------------------------------------------------------------------------
def setDualPrimaryDetection( mode, args ):
   delay = args[ 'DELAY' ]
   action = 'dualPrimaryActionErrdisableAllInterfaces' if 'action' in args else None
   mlagConfig.dualPrimaryDetectionDelay = delay
   mlagConfig.dualPrimaryAction = action or 'dualPrimaryActionNone'

def noDualPrimaryDetection( mode, args ):
   mlagConfig.dualPrimaryAction = 'dualPrimaryActionNone'
   mlagConfig.dualPrimaryDetectionDelay = 0

#-------------------------------------------------------------------------------
# dual-primary recovery delay mlag <SECONDS> non-mlag <SECONDS>
# [no|default] dual-primary recovery delay
#-------------------------------------------------------------------------------
def setDualPrimaryRecoveryDelay( mode, args ):
   mlagDelay = args[ 'DELAY1' ]
   nonMlagDelay = args[ 'DELAY2' ]

   mlagConfig.dualPrimaryMlagRecoveryDelay = mlagDelay
   mlagConfig.dualPrimaryNonMlagRecoveryDelay = nonMlagDelay

def noDualPrimaryRecoveryDelay( mode, args ):
   mlagConfig.dualPrimaryMlagRecoveryDelay = 0
   mlagConfig.dualPrimaryNonMlagRecoveryDelay = 0

#-------------------------------------------------------------------------------
# [no] peer-link
#-------------------------------------------------------------------------------
def validPeerLink( intf, mode ):
   config = intf.config()
   if not config:
      mode.addError( "Interface %s not configured" % intf.name )
      return False

   # check if the proposed peer-link is not configured with
   # the available unidirectionalLinkMode's.
   if not Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( intf.name ):
      if config.unidirectionalLinkMode != 'uniLinkModeDisabled':
         if config.unidirectionalLinkMode == 'uniLinkModeSendOnly':
            linkMode = 'unidirectional send-only'
         elif config.unidirectionalLinkMode == 'uniLinkModeReceiveOnly':
            linkMode = 'unidirectional receive-only'
         else:
            linkMode = 'unidirectional send-receive'
         mode.addError( "%s is configured with %s" % ( intf.name, linkMode ) )
         return False

   # check if Port-Channel already been configured as mlag interface.
   intfConfig = mlagConfig.intfConfig
   name = intf.name
   if name in intfConfig:
      mode.addError( "%s already configured with MLAG %s" % ( name,
                                                              intfConfig[ name ] ) )
      return False

   # check if the proposed peer-link is already a member of a port-channel
   if name in lagConfigDir.phyIntf and lagConfigDir.phyIntf[ name ].lag:
      channel = lagConfigDir.phyIntf[ name ].lag.intfId
      mode.addError( "%s already configured as member of %s" % ( name, channel ) )
      return False

   if config.intfId in ethPhyIntfConfigDir:
      if ( config.rxFlowcontrol == 'flowControlConfigOn' or
         config.txFlowcontrol == 'flowControlConfigOn' ):
         mode.addError( '%s has flow control configured' % name )
         return False

   # MLAG: check that no member has flowControl configured
   flowPorts = []
   for member in LagCli.channelPorts( mode, config.intfId, lacpOnly=False,
                                      status=Tac.Type( "Lag::LagStatusFilter" ).\
                                         filterOnActiveAndInactive,
                                       useLagConfig=True ):
      ethPhyIntfConfig = ethPhyIntfConfigDir.intfConfig.get( member )
      if ethPhyIntfConfig and \
         ( ethPhyIntfConfig.rxFlowcontrol == 'flowControlConfigOn' or
           ethPhyIntfConfig.txFlowcontrol == 'flowControlConfigOn' ):
         flowPorts.append(member)
   if flowPorts:
      mode.addError( 'Flow control is configured on %s' % ', '.join( flowPorts ) )
      return False

   sic = cliConfig.switchIntfConfig.get( name )
   invalidModeMapping = { 'dot1qTunnel' : 'dot1q-tunnel',
                          'tap' : 'tap',
                          'tool' : 'tool' }
   if sic and sic.switchportMode in invalidModeMapping:
      mode.addError( '%s is configured in %s mode' % 
                     ( name, invalidModeMapping[ sic.switchportMode ] ) )
      return False

   return True

def setPeerLink( mode, args ):
   intf = args[ 'ETHINTF' ]
   if validPeerLink( intf, mode ):
      mlagConfig.peerLinkIntfId = intf.name
      updateMlagConfigState( mode )

def noPeerLink( mode, args ):
   mlagConfig.peerLinkIntfId = ""
   updateMlagConfigState( mode )

# This function checks the Mlag Configuration and adds warning if LACP Standby 
# is configured and the Non-Mlag reload delay < Mlag reload delay
def checkMlagReloadDelayConfig( mode, lacpStandby, reloadDelayMlag, 
                                reloadDelayNonMlag ):
   if( lacpStandby and ( reloadDelayNonMlag < reloadDelayMlag ) ):
      mode.addWarning( reloadDelayLacpConfigMsg %
                       ( reloadDelayNonMlag, reloadDelayMlag ) )

#-------------------------------------------------------------------------------
# reload-delay [mlag|non-mlag] {<seconds>|infinity}
#-------------------------------------------------------------------------------
def setReloadDelay( mode, args ):
   delay = args.get( 'RELOAD_DELAY', mlagConfig.reloadDelayInfinity )
   checkMlagReloadDelayConfig( mode, mlagConfig.lacpStandby, delay, 
                               mlagConfig.reloadDelayNonMlag.delay)
   mlagConfig.reloadDelay = ReloadDelay( "reloadDelayConfigured", delay )

def setReloadDelayMlag( mode, args ):
   setReloadDelay( mode, args )
   mlagConfig.reloadDelayMlagConfigured = True

def setReloadDelayNonMlag( mode, args ):
   delay = args.get( 'RELOAD_DELAY', mlagConfig.reloadDelayInfinity )
   checkMlagReloadDelayConfig( mode, mlagConfig.lacpStandby, 
                               mlagConfig.reloadDelay.delay, delay)
   mlagConfig.reloadDelayNonMlag = ReloadDelay( "reloadDelayConfigured", delay )

def noReloadDelayMlag( mode, args ):
   mlagConfig.reloadDelay = ReloadDelay()
   checkMlagReloadDelayConfig( mode, mlagConfig.lacpStandby,
                               mlagConfig.reloadDelay.delay,
                               mlagConfig.reloadDelayNonMlag.delay)
   mlagConfig.reloadDelayMlagConfigured = False

def noReloadDelayNonMlag( mode, args ):
   mlagConfig.reloadDelayNonMlag = ReloadDelay()
   checkMlagReloadDelayConfig( mode, mlagConfig.lacpStandby,
                               mlagConfig.reloadDelay.delay,
                               mlagConfig.reloadDelayNonMlag.delay)

#-------------------------------------------------------------------------------
# reload-delay mode lacp standby
#-------------------------------------------------------------------------------
def setLacpStandby( mode, args ):
   checkMlagReloadDelayConfig( mode, True, mlagConfig.reloadDelay.delay, 
                               mlagConfig.reloadDelayNonMlag.delay)
   mlagConfig.lacpStandby = True

#-------------------------------------------------------------------------------
# [ no | default ] inactive mac-address destination peer-link [ direct | indirect ]
#-------------------------------------------------------------------------------
def setFastMacRedirect( mode, args ):
   if args[ 'TRAFFIC_DIRECT' ] == 'indirect':
      mlagConfig.fastMacRedirectionConfig = \
         tacFastMacRedirectionConfig.fastMacRedirectionConfigured
   else:
      mlagConfig.fastMacRedirectionConfig = \
         tacFastMacRedirectionConfig.fastMacRedirectionUnconfigured
   
def defaultFastMacRedirectConfig( mode, args ):
   mlagConfig.fastMacRedirectionConfig = \
      tacFastMacRedirectionConfig.defaultFastMacRedirection

class MlagConfigIntf( IntfCli.IntfDependentBase ):
   def destroy( self ):
      if mlagConfig.peerLinkIntfId == self.intf_.name:
         t0( self.intf_.name, "destroyed, reset peer-link",
             mlagConfig.peerLinkIntfId )
         mlagConfig.peerLinkIntfId = ""
      if mlagConfig.localIntfId == self.intf_.name:
         t0( self.intf_.name, "destroyed, reset local-interface",
             mlagConfig.localIntfId )
         mlagConfig.localIntfId = ""

#-------------------------------------------------------------------------------
# Mounts
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global mlagConfig, ipConfig, lagConfigDir, cliConfig
   global ethPhyIntfConfigDir
   global mlagConfigurationState
   global mlagHwStatus
   mlagConfig = ConfigMount.mount( entityManager, "mlag/config", "Mlag::Config",
                                   "w" )
   mlagConfigurationState = LazyMount.mount( entityManager, 
                                             "mlag/configurationState",
                                             "Mlag::ConfigurationState", "w" )
   mlagHwStatus = LazyMount.mount( entityManager, "mlag/hardware/status",
                                   "Mlag::Hardware::Status", "r" )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ConfigMount.mount( entityManager, "interface/config/eth/phy/slice",
                    "Tac::Dir", "wi" )
   ethPhyIntfConfigDir = LazyMount.mount( entityManager,
                                          "interface/config/eth/phy/all",
                                          "Interface::AllEthPhyIntfConfigDir", "r" )
   lagConfigDir = LazyMount.mount( entityManager, "lag/input/config/cli",
                                   "Lag::Input::Config", "r" )
   cliConfig = LazyMount.mount( entityManager, "bridging/input/config/cli",
                                "Bridging::Input::CliConfig", "r" )

   IntfCli.Intf.registerDependentClass( MlagConfigIntf )
