#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.TapAggIntfCli as TapAggIntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.FruCli as FruCli
from CliPlugin.EbraEthIntfCli import switchPortMatcher
from CliToken.Mac import macMatcherForConfig
from MultiRangeRule import MultiRangeMatcher

matcherEncapsulation = CliMatcher.KeywordMatcher( 'encapsulation',
   helpdesc='Encapsulation protocols in packet header' )
matcherExclusive = CliMatcher.KeywordMatcher( 'exclusive',
   helpdesc='Exclusive tap aggregation mode' )
matcherFcsError = CliMatcher.KeywordMatcher( 'fcs-error',
   helpdesc='Configure the behavior regarding packets with FCS error' )
matcherFormat = CliMatcher.KeywordMatcher( 'format',
   helpdesc='Format configuration for timestamp header' )
matcherHeader = CliMatcher.KeywordMatcher( 'header',
   helpdesc='Header configuration for timestamp' )
matcherMixed = CliMatcher.KeywordMatcher( 'mixed',
   helpdesc='Mixed tap-agg and switched mode' )
matcherMode = CliMatcher.KeywordMatcher( 'mode',
   helpdesc='Configure tap aggregation mode' )
matcherModule = CliMatcher.KeywordMatcher( 'module', helpdesc='Configure module' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile',
   helpdesc='Configure TCAM profile' )
matcherReplace = CliMatcher.KeywordMatcher( 'replace',
   helpdesc='Replace field configuration for timestamp' )
matcherSourceMac = CliMatcher.KeywordMatcher( 'source-mac',
   helpdesc='Replace source MAC with 48-bit truncated IEEE 1588 timestamp' )
matcherStrip = CliMatcher.KeywordMatcher( 'strip',
   helpdesc='Remove a tag in packet header' )
matcherTapagg = CliMatcher.KeywordMatcher( 'tapagg',
   helpdesc='TapAgg type' )
matcherTimestamp = CliMatcher.KeywordMatcher( 'timestamp',
   helpdesc='Configuration to set global timestamp format in Ethernet header' )
matcherType = CliMatcher.KeywordMatcher( 'type', helpdesc='Specify type' )
matcherLinecard = MultiRangeMatcher(
   value=lambda mode, grList: [ ( 'Linecard%d' % i ) for i in grList.values() ],
   rangeFn=lambda: FruCli.rangeFn( 'Linecard' ), noSingletons=False,
   helpdesc='Linecard number', tagLong='Linecard' )
nodeEncapsulation = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'encapsulation', helpdesc='Encapsulation protocols in packet header' ),
   guard=TapAggIntfCli.brVnTagStripGuard )
nodeExclusive = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'exclusive', helpdesc='Exclusive tap aggregation mode' ),
   guard=TapAggIntfCli.tapaggGuard )
nodeFcsError = CliCommand.Node( matcher=matcherFcsError,
   guard=TapAggIntfCli.fcsErrorModeGuard )
nodeFcs = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'fcs', helpdesc='Configure FCS behavior' ),
   guard=TapAggIntfCli.fcsErrorModeGuard )
nodeHeader = CliCommand.Node( matcher=matcherHeader,
   guard=EthIntfCli.timestampHeaderSupportedGuard )
nodeMixed = CliCommand.Node( matcher=matcherMixed, guard=TapAggIntfCli.tapaggGuard )
nodeProfile = CliCommand.Node( matcher=matcherProfile,
   guard=TapAggIntfCli.tcamProfileGuard )
nodeReplace = CliCommand.Node( matcher=matcherReplace,
   guard=TapAggIntfCli.timestampReplaceSmacGuard )
nodeSourceMac = CliCommand.Node( matcher=matcherSourceMac,
   guard=TapAggIntfCli.timestampReplaceSmacGuard )
nodeStrip = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'strip',
   helpdesc='Remove a tag in packet header' ),
   guard=TapAggIntfCli.brVnTagStripGuard )
nodeTimestamp = CliCommand.Node( matcher=matcherTimestamp,
   guard=EthIntfCli.timestampHeaderSupportedGuard )

#--------------------------------------------------------------------------------
# [ no | default ] encapsulation dot1br strip
#--------------------------------------------------------------------------------
class EncapsulationDot1BrStripCmd( CliCommand.CliCommandClass ):
   syntax = 'encapsulation dot1br strip'
   noOrDefaultSyntax = syntax
   data = {
      'encapsulation': nodeEncapsulation,
      'dot1br': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'dot1br',
         helpdesc='802.1 BR tag' ), guard=TapAggIntfCli.brVnTagStripGuard ),
      'strip': nodeStrip,
   }
   handler = TapAggIntfCli.brTagStripConfig
   noOrDefaultHandler = TapAggIntfCli.noBrTagStripConfig

TapAggIntfCli.TapAggConfigMode.addCommandClass( EncapsulationDot1BrStripCmd )

#--------------------------------------------------------------------------------
# [ no | default ] encapsulation vn-tag strip
#--------------------------------------------------------------------------------
class EncapsulationVnTagStripCmd( CliCommand.CliCommandClass ):
   syntax = 'encapsulation vn-tag strip'
   noOrDefaultSyntax = syntax
   data = {
      'encapsulation': nodeEncapsulation,
      'vn-tag': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'vn-tag',
         helpdesc='VN-Tag' ), guard=TapAggIntfCli.brVnTagStripGuard ),
      'strip': nodeStrip,
   }
   handler = TapAggIntfCli.vnTagStripConfig
   noOrDefaultHandler = TapAggIntfCli.noVnTagStripConfig

TapAggIntfCli.TapAggConfigMode.addCommandClass( EncapsulationVnTagStripCmd )

#--------------------------------------------------------------------------------
# mac fcs-error ( correct | discard | pass-through )
# [ no | default ] mac fcs-error ...
#--------------------------------------------------------------------------------
class MacFcsErrorCmd( CliCommand.CliCommandClass ):
   syntax = 'mac fcs-error ( correct | discard | pass-through )'
   noOrDefaultSyntax = 'mac fcs-error ...'
   data = {
      'mac': macMatcherForConfig,
      'fcs-error': nodeFcsError,
      'correct': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'correct',
         helpdesc='Replace the FCS with a correct one (default behavior)' ),
         guard=TapAggIntfCli.fcsErrorModeGuard ),
      'discard': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'discard',
         helpdesc='Discard the packets' ), guard=TapAggIntfCli.fcsErrorModeGuard ),
      'pass-through': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'pass-through',
         helpdesc='Let the packets pass through with their original FCS' ),
         guard=TapAggIntfCli.fcsErrorModePassthroughGuard ),
   }
   handler = TapAggIntfCli.setFcsErrorMode
   noOrDefaultHandler = TapAggIntfCli.noFcsErrorMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacFcsErrorCmd )

#--------------------------------------------------------------------------------
# mac fcs append
# [ no | default ] mac fcs ...
#--------------------------------------------------------------------------------
class MacFcsCmd( CliCommand.CliCommandClass ):
   syntax = 'mac fcs append'
   noOrDefaultSyntax = 'mac fcs ...'
   data = {
      'mac': macMatcherForConfig,
      'fcs': nodeFcs,
      'append': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'append',
         helpdesc='Append a correct FCS after the original FCS' ),
         alias='ACTION',
         guard=TapAggIntfCli.fcsModeGuard ),
   }
   handler = TapAggIntfCli.setFcsMode
   noOrDefaultHandler = TapAggIntfCli.noFcsMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacFcsCmd )

#--------------------------------------------------------------------------------
# mac timestamp header format ( 64-bit | 48-bit )
# [ no | default ] mac timestamp header format ...
#--------------------------------------------------------------------------------
class MacTimestampHeaderFormatCmd( CliCommand.CliCommandClass ):
   syntax = 'mac timestamp header format ( 64-bit | 48-bit )'
   noOrDefaultSyntax = 'mac timestamp header format ...'
   data = {
      'mac': macMatcherForConfig,
      'timestamp': nodeTimestamp,
      'header': nodeHeader,
      'format': matcherFormat,
      '64-bit': '64-bit IEEE 1588 timestamp format',
      '48-bit': '48-bit truncated IEEE 1588 timestamp format',
   }
   handler = TapAggIntfCli.setTimestampHeaderFormat
   noOrDefaultHandler = TapAggIntfCli.setDefaultTimestampHeaderFormat

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacTimestampHeaderFormatCmd )

#--------------------------------------------------------------------------------
# mac timestamp replace source-mac
# [ no | default ] mac timestamp replace source-mac ...
#--------------------------------------------------------------------------------
class MacTimestampReplaceSourceMacCmd( CliCommand.CliCommandClass ):
   syntax = 'mac timestamp replace source-mac'
   noOrDefaultSyntax = 'mac timestamp replace source-mac ...'
   data = {
      'mac': macMatcherForConfig,
      'timestamp': nodeTimestamp,
      'replace': nodeReplace,
      'source-mac': nodeSourceMac,
   }
   handler = TapAggIntfCli.timestampReplaceSmacConfig
   noOrDefaultHandler = TapAggIntfCli.setDefaultTimestampReplaceSmac

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacTimestampReplaceSourceMacCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mode exclusive no-errdisable ETHINTF
#--------------------------------------------------------------------------------
class ModeExclusiveNoErrdisableEthintfCmd( CliCommand.CliCommandClass ):
   syntax = 'mode exclusive no-errdisable ETHINTF'
   noOrDefaultSyntax = syntax
   data = {
      'mode': matcherMode,
      'exclusive': nodeExclusive,
      'no-errdisable': 'Do not errdisable incompatible interfaces',
      'ETHINTF': switchPortMatcher,
   }
   handler = TapAggIntfCli.setTapAggNoErrdisableIntf
   noOrDefaultHandler = TapAggIntfCli.noTapAggNoErrdisableIntf

TapAggIntfCli.TapAggConfigMode.addCommandClass( ModeExclusiveNoErrdisableEthintfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] protocol lldp trap
#--------------------------------------------------------------------------------
class ProtocolLldpTrapCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol lldp trap'
   noOrDefaultSyntax = syntax
   data = {
      'protocol': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'protocol',
         helpdesc='Specify protocol of matched packets' ),
         guard=TapAggIntfCli.lldpReceiveConfigurableGuard ),
      'lldp': 'Set action on Link Layer Discovery Protocol (LLDP) packets',
      'trap': 'Enable protocol packet redirect to CPU',
   }

   @staticmethod
   def handler( mode, args ):
      TapAggIntfCli.lldpReceiveConfig( mode, no=None )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      TapAggIntfCli.lldpReceiveConfig( mode, no=True )

TapAggIntfCli.TapAggConfigMode.addCommandClass( ProtocolLldpTrapCmd )

#--------------------------------------------------------------------------------
# [ no | default ] service-policy type tapagg mac access-list match ip
#--------------------------------------------------------------------------------
class ServicePolicyTypeTapaggMacAccessListMatchIpCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy type tapagg mac access-list match ip'
   noOrDefaultSyntax = syntax
   data = {
      'service-policy': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'service-policy', helpdesc='Configure Service Policy' ),
         guard=TapAggIntfCli.macAclMatchIpConfigurableGuard ),
      'type': matcherType,
      'tapagg': matcherTapagg,
      'mac': 'MAC',
      'access-list': 'MAC access-list',
      'match': 'Specify packets to match',
      'ip': 'Match IP packets',
   }
   handler = TapAggIntfCli.setTrafficSteeringMacAclMatchIp
   noOrDefaultHandler = TapAggIntfCli.noTrafficSteeringMacAclMatchIp

TapAggIntfCli.TapAggConfigMode.addCommandClass(
   ServicePolicyTypeTapaggMacAccessListMatchIpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] truncation size ...
#--------------------------------------------------------------------------------

# Actual "truncation size" cmd is registered by platforms
# See SandDanzTapAggConfigMode.py
class TruncationSizeCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'truncation size ...'
   data = {
      'truncation': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'truncation',
         helpdesc='Global truncation settings' ),
         guard=TapAggIntfCli.truncationGlobalSizeGuard ),
      'size': 'Configure global truncation size',
   }
   noOrDefaultHandler = TapAggIntfCli.setDefaultGlobalTruncationSize

TapAggIntfCli.TapAggConfigMode.addCommandClass( TruncationSizeCmd )

#--------------------------------------------------------------------------------
# mode exclusive
# [ no | default ] mode [ exclusive ]
#--------------------------------------------------------------------------------
class ModeExclusiveCmd( CliCommand.CliCommandClass ):
   syntax = 'mode exclusive'
   noOrDefaultSyntax = 'mode [ exclusive ]'
   data = {
      'mode': 'Configure tap aggregation mode',
      'exclusive': nodeExclusive,
   }
   handler = TapAggIntfCli.setMode
   noOrDefaultHandler = TapAggIntfCli.noMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( ModeExclusiveCmd )

#--------------------------------------------------------------------------------
# mode exclusive profile ( TAPAGGPROFILES | USERPROFILE )
# ( no | default ) mode [ exclusive ] profile ( TAPAGGPROFILES | USERPROFILE )
#--------------------------------------------------------------------------------
class ModeExclusiveProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'mode exclusive profile ( TAPAGGPROFILES | USERPROFILE )'
   noOrDefaultSyntax = 'mode [ exclusive ] profile ( TAPAGGPROFILES | USERPROFILE )'
   data = {
      'mode': matcherMode,
      'exclusive': nodeExclusive,
      'profile': nodeProfile,
      'TAPAGGPROFILES': CliMatcher.DynamicKeywordMatcher(
         TapAggIntfCli.tapAggProfiles ),
      'USERPROFILE': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
         helpdesc='User-Defined TCAM Profile', helpname='WORD' ),
   }
   handler = TapAggIntfCli.setMode
   noOrDefaultHandler = TapAggIntfCli.noMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( ModeExclusiveProfileCmd )

#--------------------------------------------------------------------------------
# mode mixed module CARDNUMBERS
# ( no | default ) mode mixed module CARDNUMBERS ...
#--------------------------------------------------------------------------------
class ModeMixedModuleCardnumbersCmd( CliCommand.CliCommandClass ):
   syntax = 'mode mixed module CARDNUMBERS'
   noOrDefaultSyntax = 'mode mixed module CARDNUMBERS ...'
   data = {
      'mode': matcherMode,
      'mixed': nodeMixed,
      'module': matcherModule,
      'CARDNUMBERS': matcherLinecard,
   }
   handler = TapAggIntfCli.setMode
   noOrDefaultHandler = TapAggIntfCli.noMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( ModeMixedModuleCardnumbersCmd )

#--------------------------------------------------------------------------------
# mode mixed module CARDNUMBERS profile ( TAPAGGPROFILES | USERPROFILE )
#--------------------------------------------------------------------------------
class ModeMixedModuleCardnumbersProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'mode mixed module CARDNUMBERS profile ' + \
      '( TAPAGGPROFILES | USERPROFILE )'
   data = {
      'mode': matcherMode,
      'mixed': nodeMixed,
      'module': matcherModule,
      'CARDNUMBERS': matcherLinecard,
      'profile': nodeProfile,
      'TAPAGGPROFILES': CliMatcher.DynamicKeywordMatcher(
         TapAggIntfCli.tapAggProfiles ),
      'USERPROFILE': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
         helpdesc='User-Defined TCAM Profile', helpname='WORD' ),
   }
   handler = TapAggIntfCli.setMode

TapAggIntfCli.TapAggConfigMode.addCommandClass(
   ModeMixedModuleCardnumbersProfileCmd )
