#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliGlobal
import CliMatcher
import CliParser
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.EthIntfCli import EthPhyIntf
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import CliPlugin.PolicyMapCliLib as PolicyMapCliLib
import CliPlugin.TapAggIntfCli as TapAggIntfCli
import CliPlugin.TapAggPmapCliLib as TapAggPmapCliLib
import ConfigMount
import Intf.IntfRange
import LazyMount
import MultiRangeRule
from TypeFuture import TacLazyType
import Tracing

t0 = Tracing.trace0

# Tac types
UniqueId = TacLazyType( 'Ark::UniqueId' )
VlanIdTuple = TacLazyType( 'Bridging::VlanIdTuple' )

# Global variables
gv = CliGlobal.CliGlobal( dict(
   tapAggPmapConfig=None,
   tapAggHwStatus=None,
) )

#--------------------------------------------------------------------------------
# Guards
#--------------------------------------------------------------------------------
def guardPmapDot1qRemove( mode, token ):
   if ( gv.tapAggHwStatus.modeSupported
        and gv.tapAggHwStatus.pmapDot1qRemoveSupported ):
      return None
   return CliParser.guardNotThisPlatform

def guardPmapRemoveDot1qRange( mode, token ):
   if gv.tapAggHwStatus.dot1qRemoveInnerOnlySupported:
      return None
   # This platform doesn't support only removing the inner tag, the list must
   # include 1 (i.e. the outer tag)
   if isinstance( token, str ):
      outerTagPresent = '1' in token
   else: # token type is GenericRangeList, from MultiRangeRule
      outerTagPresent = 1 in token.values()
   if outerTagPresent:
      return None
   return CliParser.guardNotThisPlatform

#--------------------------------------------------------------------------------
# Matchers and nodes
#--------------------------------------------------------------------------------
matcherSet = CliMatcher.KeywordMatcher( 'set',
      helpdesc='Set values' )
matcherDot1Q = CliMatcher.KeywordMatcher( 'dot1q',
      helpdesc='Remove dot1q tag' )
matcherIdTag = CliMatcher.KeywordMatcher( 'id-tag',
      helpdesc='Port identity' )
matcherPortId = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Tap port identity tag' )
matcherInnerPortId = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Inner tap port identity tag' )
nodeInner = CliCommand.guardedKeyword( 'inner',
      helpdesc='Inner port identity',
      guard=TapAggIntfCli.qinqIdentityTaggingGuard )

#--------------------------------------------------------------------------------
# policy-map class mode
# command to enter the mode:
# [no] policy-map type tapagg <name>
#--------------------------------------------------------------------------------
class PolicyMapClassModeTapAgg( PolicyMapCliLib.PolicyMapClassMode ):
   name = 'Policy Map TapAgg Class Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      PolicyMapCliLib.PolicyMapClassMode.__init__( self, parent, session,
                                                   context, 'tapagg' )

   def _delAction( self, actionType ):
      policyRuleAction = self.pmapClassContext.policyRuleAction
      policyRuleAction.delAction( actionType )

   def _addAction( self, actionType, action ):
      policyRuleAction = self.pmapClassContext.policyRuleAction
      policyRuleAction.addAction( actionType, action )

   def _getAction( self, actionType ):
      policyRuleAction = self.pmapClassContext.policyRuleAction
      actions = policyRuleAction.actions()
      return actions.get( actionType )

   def delIdentityTag( self, args ):
      self._delAction( TapAggPmapCliLib.actSetIdentityTag )

   def setIdentityTag( self, args ):
      portId = args[ 'PORTID' ]
      innerPortId = args.get( 'INNERPORTID', 0 )
      self.delIdentityTag( args )
      idTagTuple = VlanIdTuple( portId, innerPortId )
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.idTag.newMember( self.cmapName,
                                                                     UniqueId(),
                                                                     idTagTuple )
      self._addAction( TapAggPmapCliLib.actSetIdentityTag, tapAggAct )

   def delAggGroupAct( self ):
      self._delAction( TapAggPmapCliLib.actSetAggregationGroup )

   def delAggGroup( self ):
      t0( "delAggGroup" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetAggregationGroup )
      if tapAggAct:
         intfList = tapAggAct.aggIntf.keys()
      else:
         intfList = []
      self.delAggGroupAct()
      groupKey = "{}_{}".format( self.pmapName, self.cmapName )
      if groupKey in gv.tapAggPmapConfig.group:
         del gv.tapAggPmapConfig.group[ groupKey ]
      if intfList:
         tapAggAct = gv.tapAggPmapConfig.tapAggActions.aggGroup.newMember(
               self.cmapName, UniqueId() )
         for intf in intfList:
            tapAggAct.aggIntf[ intf ] = True
         self._addAction( TapAggPmapCliLib.actSetAggregationGroup,
                          tapAggAct )

   def delAggIntf( self ):
      t0( "delAggIntf" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetAggregationGroup )
      if tapAggAct:
         currList = tapAggAct.aggGroup.keys()
      else:
         currList = []
      self.delAggGroupAct()
      if currList:
         tapAggAct = gv.tapAggPmapConfig.tapAggActions.aggGroup.newMember(
               self.cmapName, UniqueId() )
         for group in currList:
            tapAggAct.aggGroup[ group ] = True
         self._addAction( TapAggPmapCliLib.actSetAggregationGroup,
                           tapAggAct )

   def setAggGroup( self, allowedGroupOp, delete=False ):
      t0( "setAggGroup" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetAggregationGroup )
      if tapAggAct:
         currList = tapAggAct.aggGroup.keys()
         intfList = tapAggAct.aggIntf.keys()
      else:
         intfList = []
         currList = []
      self.delAggGroupAct()
      groupKey = "{}_{}".format( self.pmapName, self.cmapName )
      if groupKey in gv.tapAggPmapConfig.group.keys():
         del gv.tapAggPmapConfig.group[ groupKey ]
      groupList = allowedGroupOp
      newList = []
      if delete:
         for group in currList:
            if group not in groupList:
               newList.append( group )
      else:
         newList = currList + groupList
      if not newList and not intfList:
         return
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.aggGroup.newMember(
            self.cmapName, UniqueId() )
      pmapGroup = gv.tapAggPmapConfig.group.newMember( groupKey )
      for group in newList:
         tapAggAct.aggGroup[ group ] = True
         pmapGroup.groupName[ group ] = True
      for intf in intfList:
         tapAggAct.aggIntf[ intf ] = True
      self._addAction( TapAggPmapCliLib.actSetAggregationGroup,
                       tapAggAct )

   def setAggIntf( self, allowedRawIntfOp ):
      t0( "setAggIntf" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetAggregationGroup )
      if tapAggAct:
         currList = tapAggAct.aggIntf.keys()
         groupList = tapAggAct.aggGroup.keys()
      else:
         currList = []
         groupList = []
      self.delAggGroupAct()

      intfOp, intfList = getIntfOpAndList( allowedRawIntfOp )
      newList = []
      if intfOp == 'add':
         newList = currList + intfList
      elif intfOp == 'set':
         newList = intfList
      elif intfOp == 'remove':
         for intf in currList:
            if intf not in intfList:
               newList.append( intf )
      if not newList and not groupList:
         return
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.aggGroup.newMember(
            self.cmapName, UniqueId() )
      for intf in newList:
         tapAggAct.aggIntf[ intf ] = True
      for group in groupList:
         tapAggAct.aggGroup[ group ] = True
      self._addAction( TapAggPmapCliLib.actSetAggregationGroup,
                       tapAggAct )

   def delStripHdrBytes( self, args ):
      self._delAction( TapAggPmapCliLib.actStripHdrBytes )

   def setStripHdrBytes( self, args ):
      # VLAN_RANGE type is GenericRangeList, we use its string representation '1-2'
      hdrInfo = str( args[ 'VLAN_RANGE' ] )
      hdrType = TapAggPmapCliLib.stripHdrDot1q

      self.delStripHdrBytes( args )
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.stripHdrBytes.newMember(
            self.cmapName, UniqueId() )
      stripHdr = TapAggPmapCliLib.stripHdrAction( hdrType, hdrInfo )
      tapAggAct.stripHdrBytes = stripHdr
      self._addAction( TapAggPmapCliLib.actStripHdrBytes, tapAggAct )

#--------------------------------------------------------------------------------
# Helpers
#--------------------------------------------------------------------------------
def getIntfOpAndList( allowedRawIntfOp ):
   rawIntfOp, intfs = allowedRawIntfOp
   intfList = []
   if intfs:
      if isinstance( intfs, EthPhyIntf ):
         # Its a singleton object, not a list of intfs
         intfList = [ intfs.name ]
      elif isinstance( intfs, list ):
         intfList =  intfs
      else:
         assert isinstance( intfs, MultiRangeRule.IntfList )
         intfList += list( intfs.intfNames() )

   return rawIntfOp, intfList

#--------------------------------------------------------------------------------
# remove dot1q outer VLAN_RANGE
# ( no | default ) remove dot1q outer ...
#--------------------------------------------------------------------------------
class RemoveDot1QOuterCmd( CliCommand.CliCommandClass ):
   syntax = 'remove dot1q outer VLAN_RANGE'
   noOrDefaultSyntax = 'remove dot1q outer ...'
   data = {
      'remove' : CliCommand.guardedKeyword( 'remove',
         helpdesc='Remove a header',
         guard=guardPmapDot1qRemove ),
      'dot1q' : matcherDot1Q,
      'outer' : 'Remove outer tag',
      'VLAN_RANGE' : CliCommand.Node(
         matcher=MultiRangeRule.MultiRangeMatcher(
            rangeFn=lambda: ( 1, gv.tapAggHwStatus.dot1qRemoveMaxIndex ),
            helpdesc='Specify indices of VLAN tags to be removed',
            noSingletons=False ),
         guard=guardPmapRemoveDot1qRange ),
   }

   handler = PolicyMapClassModeTapAgg.setStripHdrBytes
   noOrDefaultHandler = PolicyMapClassModeTapAgg.delStripHdrBytes

PolicyMapClassModeTapAgg.addCommandClass( RemoveDot1QOuterCmd )

#--------------------------------------------------------------------------------
# set aggregation-group ( { group GROUP } | GROUP ) [ id-tag PORTID
#                                                     [ inner INNERPORTID ] ]
# ( no | default ) set aggregation-group [ ( { group GROUP } | GROUP )
#                                          [ id-tag ... ] ]
#--------------------------------------------------------------------------------
class SetAggregationGroupIdTagCmd( CliCommand.CliCommandClass ):
   syntax = """set aggregation-group ( GROUP | { group GROUP } )
               [ id-tag PORTID [ inner INNERPORTID ] ]"""
   noOrDefaultSyntax = """set aggregation-group [ ( { group GROUP } | GROUP )
                          [ id-tag ... ] ]"""
   data = {
      'set' : matcherSet,
      'aggregation-group' : 'List of groups to aggregate flow',
      'group' : 'Set tap group for the interface',
      'GROUP' : CliMatcher.DynamicNameMatcher(
         TapAggIntfCli.getGroupList, helpdesc='Group name' ),
      'id-tag' : matcherIdTag,
      'PORTID' : matcherPortId,
      'inner' : nodeInner,
      'INNERPORTID' : matcherInnerPortId,
   }

   @staticmethod
   def handler( mode, args ):
      aggGroupAdd = args[ 'GROUP' ]
      if not isinstance( aggGroupAdd, list ):
         aggGroupAdd = [ aggGroupAdd ]
      mode.setAggGroup( aggGroupAdd, delete=False )
      if 'PORTID' in args:
         mode.setIdentityTag( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      aggGroup = args.get( 'GROUP' )
      if aggGroup is None:
         mode.delAggGroup()
      else:
         mode.setAggGroup( aggGroup, delete=True )

      if 'id-tag' in args:
         mode.delIdentityTag( args )

PolicyMapClassModeTapAgg.addCommandClass( SetAggregationGroupIdTagCmd )

#--------------------------------------------------------------------------------
# set id-tag PORTID [ inner INNERPORTID ]
# ( no | default ) set id-tag ...
#--------------------------------------------------------------------------------
class SetIdTagCmd( CliCommand.CliCommandClass ):
   syntax = 'set id-tag PORTID [ inner INNERPORTID ]'
   noOrDefaultSyntax = 'set id-tag ...'
   data = {
      'set' : matcherSet,
      'id-tag' : matcherIdTag,
      'PORTID' : matcherPortId,
      'inner' : nodeInner,
      'INNERPORTID' : matcherInnerPortId,
   }

   handler = PolicyMapClassModeTapAgg.setIdentityTag
   noOrDefaultHandler = PolicyMapClassModeTapAgg.delIdentityTag

PolicyMapClassModeTapAgg.addCommandClass( SetIdTagCmd )

#--------------------------------------------------------------------------------
# set interface INTFS [ id-tag PORTID [ inner INNERPORTID ] ]
# ( no | default ) set interface [ INTFS [ id-tag ... ] ]
#--------------------------------------------------------------------------------
class SetInterfaceIdTagCmd( CliCommand.CliCommandClass ):
   syntax = """set interface INTFS [ id-tag PORTID [ inner INNERPORTID ] ]"""
   noOrDefaultSyntax = 'set interface [ INTFS [ id-tag ... ] ]'
   data = {
      'set' : matcherSet,
      'interface' : 'List of interfaces to aggregate flow',
      'INTFS' : CliCommand.Node(
         matcher=Intf.IntfRange.IntfRangeMatcher(
            explicitIntfTypes=( EthPhyAutoIntfType, LagAutoIntfType ) ) ),
      'id-tag' : matcherIdTag,
      'PORTID' : matcherPortId,
      'inner' : nodeInner,
      'INNERPORTID' : matcherInnerPortId,
   }

   @staticmethod
   def handler( mode, args ):
      allowedRawIntfOp = args[ 'INTFS' ]
      mode.setAggIntf( ( 'add', allowedRawIntfOp ) )
      if 'PORTID' in args:
         mode.setIdentityTag( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      allowedRawIntf = args.get( 'INTFS' )
      if allowedRawIntf is None:
         mode.delAggIntf()
      else:
         allowedRawIntfOpAndList = getIntfOpAndList( ( 'remove', allowedRawIntf ) )
         mode.setAggIntf( allowedRawIntfOpAndList )

      if 'id-tag' in args:
         mode.delIdentityTag( args )
 
PolicyMapClassModeTapAgg.addCommandClass( SetInterfaceIdTagCmd )

def Plugin( entityManager ):
   gv.tapAggHwStatus = LazyMount.mount( entityManager, 'tapagg/hwstatus',
                                        'TapAgg::HwStatus', 'r' )
   gv.tapAggPmapConfig = ConfigMount.mount( entityManager, 'tapagg/pmapconfig',
                                            'TapAgg::PmapConfig', 'w' )
