
#!/usr/bin/env python
# Copyright (c) 2008-2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
from AclCli import showIpAccessLists, showIp6AccessLists, showMacAccessLists
import Arnet
import EthIntfCli
from MirroringLib import metadataElementTokenDict
import MirroringModels
import Tac
from TypeFuture import TacLazyType

AclFeature = TacLazyType( "Acl::AclFeature" )
EthType = Tac.Type( "Arnet::EthType" ) # enum
GrePayloadType = TacLazyType( 'Mirroring::GrePayloadType' )
SampleRateConstants = TacLazyType( 'Mirroring::SampleRateConstants' )
TruncationSize = TacLazyType( 'Mirroring::TruncationSize' )
Constants = TacLazyType( 'Mirroring::Constants' )

def doShowMirrorSessions( gv, mode, args ):
   # show mirror session should only look at mirror status for showing sessions
   # and not mirror config because if the mirror agent is dead then we do have the
   # session in mirroring config but not in mirroring status, so status will be
   # displayed
   # gv should provides:
   #  values:
   #    defaultGreHdrProto
   #    directionBoth
   #    directionRx
   #    directionTx
   #    headerRemovalDefaultVal
   #  mounts:
   #    aclStatusDp
   #    mirroringConfig
   #    mirroringHwCapability
   #    mirroringHwConfig
   #    mirroringHwStatus
   #    mirroringStatus

   name = args.get( 'SESSNAME' )
   detail = 'detail' in args
   detailStr = args.get( 'access-list', '' )

   summaryStr = ''
   if 'summary' in args:
      for sessionFilter in ( 'source', 'destination' ):
         if sessionFilter in args:
            summaryStr = sessionFilter.capitalize()
            break
      else:
         summaryStr = 'True'

   model = MirroringModels.MirrorSessions( _summary=summaryStr, _detail=detailStr )

   def setPayloadTypeForModel( sessionModel, hwConfigSession,
                               isEgressAclSession=None ):
      grePayloadType = hwConfigSession.grePayloadType
      if isEgressAclSession:
         sessionModel.grePayloadType = GrePayloadType.payloadTypeFullPacket
      elif grePayloadType == GrePayloadType.payloadTypeHwDefault:
         sessionModel.grePayloadType = gv.mirroringHwCapability.defaultGrePayloadType
      else:
         sessionModel.grePayloadType = grePayloadType

   def setGreMetadataForModel( sessionModel, hwConfigSession, isEgressSession ):
      if isEgressSession:
         if not hwConfigSession.txGreMetadataEnabled:
            sessionModel.greMetadata = [ 'disabled' ]
            return
         else:
            sessionModel.egressAclMirroredTruncation = \
                  Constants.egressMirroringTruncationSize
      greMetadata = hwConfigSession.greMetadata
      if not greMetadata:
         greMetadata = gv.mirroringHwCapability.defaultGreMetadata
      # Find corresponding metadata set in greMetadataSupported
      # to ensure correct element order is retained
      for metadata in gv.mirroringHwCapability.greMetadataSupported.values():
         hwConfigElements = set( greMetadata )
         defaultElements = set( metadata.metadataElement.values() )
         if hwConfigElements == defaultElements:
            sessionModel.greMetadata = [ metadataElementTokenDict[ elem ]
               for elem in metadata.metadataElement.values() ]

   def getUnsupportedQualifiers( intfName, aclName, aclType ):
      linecardName = None
      if EthIntfCli.isModular():
         result = re.search( r'(\d+)/', intfName )
         # Look for the linecard from the interface name. If
         # it isn't found, leave linecardName as None to check
         # all linecards for warnings on the ACL.
         if result:
            linecardName = 'Linecard%s' % result.group( 1 )
      else:
         linecardName = 'FixedSystem'
      qualSet = set()
      for aclSDp in gv.aclStatusDp.values():
         uComponent = aclSDp.unsupportedComponent.get( AclFeature.featureMirroring )
         if not uComponent:
            continue
         if linecardName:
            lComponent = uComponent.linecardUComponent.get( linecardName )
            uLComponents = [ lComponent ] if lComponent else []
         else:
            uLComponents = uComponent.linecardUComponent.values()
         for uLComponent in uLComponents:
            uAclTypeComponent = uLComponent.aclTypeUComponent.get( aclType )
            if uAclTypeComponent:
               uAclComponent = (
                  uAclTypeComponent.unsupportedComponent.get( aclName ) )
               if uAclComponent:
                  qualSet.update( uAclComponent.qual.values() )
      return list( qualSet )

   def operStateAndReason( sessionStatus, sessName, intfName ):
      operState = 'active'
      operStateReason = ''

      if sessionStatus:
         targetStatus = ( gv.mirroringStatus.sessionStatus[ sessName ]
               .targetOperStatus.get( intfName ) )
         if targetStatus:
            reason = None
            ifHwStatus = gv.mirroringHwStatus.intfHwStatus.get( sessName )
            if ( ifHwStatus is not None
                 and ifHwStatus.hwConfigStatus.get( intfName ) is not None
                 and ifHwStatus.hwConfigStatus[ intfName ].status is False ):
               reason = ifHwStatus.hwConfigStatus[ intfName ].reason
            if targetStatus.state == "operStateInactive":
               operState = "inactive"
               operStateReason = targetStatus.reason
            elif reason is not None:
               operState = "inactive"
               operStateReason = reason
            elif targetStatus.state == "operStateUnknown":
               operState = "unknown"
               operStateReason = targetStatus.reason
         else:
            operState = "unknown"
            operStateReason = "unknown"
      else:
         operState = "unknown"
         operStateReason = "unknown"

      return ( operState, operStateReason )

   def createSessionAclDetailsModel( mode, aclName, aclType ):
      def noShowCounterInfo( aclList ):
         if aclList:
            for acl in aclList.aclList:
               acl.countersEnabled = False
               acl.staleCounters = False
               acl.noMatchCounter = None

               for rule in acl.sequence:
                  rule.counterData = None

      model = MirroringModels.MirrorSessions.MirrorSession.SessionAclDetails()
      if not aclName:
         return model
      parameters = ( aclName, False, False, False )
      if aclType == 'ip':
         model.ipAclList = showIpAccessLists( mode, parameters )
         noShowCounterInfo( model.ipAclList )
      elif aclType == 'ipv6':
         model.ipv6AclList = showIp6AccessLists( mode, parameters )
         noShowCounterInfo( model.ipv6AclList )
      elif aclType == 'mac':
         model.macAclList = showMacAccessLists( mode, parameters )
         noShowCounterInfo( model.macAclList )

      return model

   def createAclListWithMirrorActionModel( mode, name, addRules ):
      egressAclSessions = gv.mirroringHwStatus.egressAclSessions
      model = MirroringModels.MirrorSessions.MirrorSession.AclsWithMirrorAction()
      for aclInfo in egressAclSessions[ name ].aclInfo.iterkeys():
         parameters = ( aclInfo.name, False, False, False )
         if aclInfo.aclType == 'ip':
            acl = MirroringModels.Ipv4AclWithInfo()
            aclRules = None if not addRules else \
                  showIpAccessLists( mode, parameters )
            acl.setInfoAndAcl( aclInfo, aclRules )
            model.ipAcls.append( acl )
         elif aclInfo.aclType == 'ipv6':
            acl = MirroringModels.Ipv6AclWithInfo()
            aclRules = None if not addRules else \
                  showIp6AccessLists( mode, parameters )
            acl.setInfoAndAcl( aclInfo, aclRules )
            model.ipv6Acls.append( acl )
         elif aclInfo.aclType == 'mac':
            acl = MirroringModels.MacAclWithInfo()
            aclRules = None if not addRules else \
                  showMacAccessLists( mode, parameters )
            acl.setInfoAndAcl( aclInfo, aclRules )
            model.macAcls.append( acl )

      return model

   def createSessionModel( mode, name ):
      sessionConfig = gv.mirroringConfig.session.get( name )
      if ( not sessionConfig or
           sessionConfig.secureMonitor != mode.session.secureMonitor() ):
         return None
      sessionModel = MirroringModels.MirrorSessions.MirrorSession(
            _displayInSummary=True )
      sessionStatus = gv.mirroringStatus.sessionStatus.get( name )
      intfNames = Arnet.sortIntf( sessionConfig.srcIntf )
      for intfName in intfNames:
         srcIntf = gv.mirroringConfig.session[ name ].srcIntf[ intfName ]
         if srcIntf.srcAcl:
            sessionModel.aclType = srcIntf.aclType
            sessionModel.sourceAcls[ intfName ] = model.MirrorSession.Acl(
                  aclName=srcIntf.srcAcl, aclPriority=srcIntf.aclPriority,
                  aclType=srcIntf.aclType,
                  aclDetails=detail,
                  aclUnsupportedQualifiers=getUnsupportedQualifiers(
                     intfName, srcIntf.srcAcl, srcIntf.aclType ) )

         if gv.mirroringHwConfig.session.get( name ):
            hwSrcIntf = gv.mirroringHwConfig.session[ name ].srcIntf.get( intfName )
            if hwSrcIntf and hwSrcIntf.srcAcl:
               sessionModel.sourceAcls[ intfName ] = model.MirrorSession.Acl(
                     aclName=hwSrcIntf.srcAcl,
                     aclPriority=hwSrcIntf.aclPriority,
                     aclType=hwSrcIntf.aclType,
                     aclDetails=detail,
                     aclUnsupportedQualifiers=getUnsupportedQualifiers(
                        intfName, srcIntf.srcAcl, srcIntf.aclType ) )

         if sessionStatus:
            srcStatus = sessionStatus.srcOperStatus.get( intfName )
            if srcStatus is None:
               sessionModel.sourceUnknownInterfaces.append( intfName )
               continue
            else:
               if srcStatus.state == "operStateInactive":
                  sessionModel.sourceInactiveInterfaces.append(
                     model.MirrorSession.Interface(
                        name=intfName,
                        operState='inactive',
                        operStateReason=srcStatus.reason ) )
                  sessionModel.sourceInterfaces.append( intfName )
                  continue
         else:
            # session status not present
            sessionModel.sourceUnknownInterfaces.append( intfName )
            continue
         reason = None
         ifHwStatus = gv.mirroringHwStatus.intfHwStatus.get( name )
         if ( ifHwStatus is not None
              and ifHwStatus.hwConfigStatus.get( intfName ) is not None
              and ifHwStatus.hwConfigStatus[ intfName ].status is False ):
            reason = ifHwStatus.hwConfigStatus[ intfName ].reason
         if reason is not None:
            sessionModel.sourceInactiveInterfaces.append(
               model.MirrorSession.Interface(
                  name=intfName,
                  operState='inactive',
                  operStateReason=reason ) )
            continue
         if srcIntf.direction == gv.directionRx:
            sessionModel.sourceRxInterfaces.append( intfName )
         elif srcIntf.direction == gv.directionTx:
            sessionModel.sourceTxInterfaces.append( intfName )
         elif srcIntf.direction == gv.directionBoth:
            sessionModel.sourceBothInterfaces.append( intfName )
         else:
            assert False, "Invalid direction %s" % srcIntf.direction
         sessionModel.sourceInterfaces.append( intfName )

      dstIp = srcIp = ttl = dscp = forwardingDrop = greHdrProto = vrf = None
      for intfName in (
            Arnet.sortIntf( gv.mirroringConfig.session[ name ].targetIntf ) ):
         if ( intfName.startswith( "Gre" )
              and gv.mirroringHwConfig.session.get( name ) ):
            continue
         operState, operStateReason = operStateAndReason( sessionStatus,
                                                          name, intfName )

         interface = model.MirrorSession.Interface(
            name=intfName, operState=operState, operStateReason=operStateReason,
            srcIpGenAddr=srcIp, dstIpGenAddr=dstIp, ttl=ttl, dscp=dscp,
            greHdrProto=greHdrProto, forwardingDrop=forwardingDrop, vrf=vrf )
         sessionModel.targetInterfaces.append( interface )

      for sessName in ( name, name + '__fwdingDrop_' ):
         hwConfigSession = gv.mirroringHwConfig.session.get( sessName )
         if hwConfigSession:
            targetIntfs = hwConfigSession.targetIntf
            if not targetIntfs:
               continue
            intfName = targetIntfs.keys()[ 0 ]
            operState, operStateReason = operStateAndReason( sessionStatus,
                                                             name, intfName )
            if intfName.startswith( "Gre" ):
               greTunnelParams = gv.mirroringStatus.greTunnelKey.get( intfName )
               if greTunnelParams:
                  dstIpGen = greTunnelParams.dstIpGenAddr
                  srcIpGen = greTunnelParams.srcIpGenAddr
                  ttl = greTunnelParams.ttl
                  dscp = greTunnelParams.dscp

                  if ( hwConfigSession.greTimestampingEnabled
                       and greTunnelParams.greHdrProto == gv.defaultGreHdrProto ):
                     greHdrProto = Tac.enumValue( EthType, "ethTypeUnknown" )
                  else:
                     greHdrProto = greTunnelParams.greHdrProto

                  forwardingDrop = greTunnelParams.forwardingDrop
                  vrf = greTunnelParams.vrf

                  interface = model.MirrorSession.Interface(
                     name=intfName, operState=operState,
                     operStateReason=operStateReason,
                     srcIpGenAddr=srcIpGen, dstIpGenAddr=dstIpGen, ttl=ttl,
                     dscp=dscp, greHdrProto=greHdrProto,
                     forwardingDrop=forwardingDrop, vrf=vrf )
                  sessionModel.targetInterfaces.append( interface )

               # show monitor session detail
               sessionModel.grePayloadType = GrePayloadType.payloadTypeNone
               sessionModel.greMetadata = []
               isEgressAclSession = \
                     gv.mirroringHwStatus.egressAclSessions.has_key( sessName )
               if detail:
                  if gv.mirroringHwCapability.supportedGrePayloadTypes:
                     setPayloadTypeForModel( sessionModel, hwConfigSession,
                                             isEgressAclSession=isEgressAclSession )
                  if gv.mirroringHwCapability.greMetadataSupported:
                     setGreMetadataForModel( sessionModel, hwConfigSession,
                                             isEgressAclSession )
                  if gv.mirroringHwCapability.greTimestampingSupported:
                     sessionModel.greTimestampingEnabled = (
                        hwConfigSession.greTimestampingEnabled )

               if gv.mirroringHwStatus.greNextHopIntfSet.get( intfName ) is None:
                  continue
               nextHopIntfSet = gv.mirroringHwStatus.greNextHopIntfSet[ intfName ]
               sortedNextHopIntfs = sorted( nextHopIntfSet.nextHopIntf )
               sortedTunnelIntfs = model.MirrorSession.TunnelIntfs()
               for key in sortedNextHopIntfs:
                  if ( len( nextHopIntfSet.nextHopIntf[ key ].intf ) == 1
                       and key in nextHopIntfSet.nextHopIntf[ key ].intf ):
                     sortedMemberIntfs = list()
                  else:
                     sortedMemberIntfs = sorted(
                        nextHopIntfSet.nextHopIntf[ key ].intf )
                  memberIntfs = model.MirrorSession.TunnelIntfs.MemberIntfs(
                     memberIntfs=sortedMemberIntfs )
                  sortedTunnelIntfs.tunnelIntfs[ key ] = memberIntfs

               sessionModel.greTunnels[ intfName ] = sortedTunnelIntfs

      if gv.mirroringConfig.session[ name ].sessionAcl != "":
         sessionModel.aclType = gv.mirroringConfig.session[ name ].aclType
      sessionModel.sessionAcl = gv.mirroringConfig.session[ name ].sessionAcl
      isEgressAclSession = gv.mirroringHwStatus.egressAclSessions.has_key( name )
      if 'access-list' in args:
         sessionModel.sessionAclDetails = createSessionAclDetailsModel( mode,
               sessionModel.sessionAcl, sessionModel.aclType )
         if isEgressAclSession:
            sessionModel.aclsWithMirrorAction = \
                  createAclListWithMirrorActionModel( mode, name, True )
      elif isEgressAclSession and detail:
         sessionModel.aclsWithMirrorAction = \
               createAclListWithMirrorActionModel( mode, name, False )
      sessionModel.truncate = gv.mirroringConfig.session[ name ].truncate
      truncateStatus = ''
      if gv.mirroringConfig.session[ name ].truncate:
         truncateStatus = 'unknown'
         if sessionStatus:
            if sessionStatus.truncateOperStatus.state == 'operStateActive':
               truncateStatus = 'active'
            elif sessionStatus.truncateOperStatus.state == 'operStateInactive':
               truncateStatus = 'inactive (%s)' % (
                     sessionStatus.truncateOperStatus.reason )
         if ( gv.mirroringHwCapability.truncationSizesSupported
              and gv.mirroringHwCapability.truncationSizesSupported.options ):
            if ( gv.mirroringConfig.session[ name ].truncationSize !=
                  TruncationSize.null ):
               sessionModel.truncationSize = (
                     gv.mirroringConfig.session[ name ].truncationSize )
            else:
               sessionModel.truncationSize = (
                     gv.mirroringHwCapability.truncationSizesSupported.defaultVal )

      sessionModel.truncateStatus = truncateStatus

      if ( gv.mirroringHwCapability.headerRemovalSupported
            and gv.mirroringConfig.session[ name ].headerRemovalSize ):
         sessionModel.headerRemovalSize = (
               gv.mirroringConfig.session[ name ].headerRemovalSize )
      else:
         sessionModel.headerRemovalSize = gv.headerRemovalDefaultVal

      sessionModel.mirrorDeviceName = None
      if sessionStatus and sessionStatus.mirrorDeviceName != '':
         sessionModel.mirrorDeviceName = sessionStatus.mirrorDeviceName

      sessionModel.hwProgrammingStatusSupported = (
         gv.mirroringHwCapability.hwProgrammingStatusSupported )
      sessionModel.programmedInHw = False
      for sessName in ( name, name + '__fwdingDrop_' ):
         programmedSession = gv.mirroringHwStatus.programmedSession.get( sessName )
         if programmedSession is not None:
            sessionModel.programmedInHw |= programmedSession

      sessionModel.rxSampleRate = None
      if ( gv.mirroringConfig.session[ name ].rxSampleRate !=
            SampleRateConstants.sampleRateUndefined ):
         sessionModel.rxSampleRate = gv.mirroringConfig.session[ name ].rxSampleRate

      sn = gv.mirroringConfig.session[ name ]
      sessionModel.mirrorRateLimitChip = sn.mirrorRateLimitChip
      sessionModel.mirrorRateLimit = sn.mirrorRateLimitInBps

      sessionModel.greTimestampingSupported = (
         gv.mirroringHwCapability.greTimestampingSupported )

      return sessionModel

   # This info is not needed if mirror header removal is not supported
   model.multiDestSessions = []
   if gv.mirroringHwCapability.headerRemovalSupported:
      model.multiDestSessions = [ key for key, status in
         gv.mirroringStatus.sessionStatus.iteritems() if status.multiDestActive ]

   if name is not None:
      model.name = name
      sessionModel = createSessionModel( mode, name )
      if sessionModel:
         model.sessions[ name ] = sessionModel
      return model

   for sessionName in gv.mirroringConfig.session:
      sessionModel = createSessionModel( mode, sessionName )
      if not sessionModel:
         continue
      model.sessions[ sessionName ] = sessionModel
      # If a summary filter was applied, mark session if it doesn't match
      if 'source' in args:
         if set( sessionModel.sourceInterfaces ).isdisjoint(
                  args[ 'SRCINTF' ].intfNames() ):
            sessionModel.doNotDisplayInSummary()
      elif 'destination' in args:
         if 'cpu' in args:
            intfs = [ 'Cpu' ]
         else:
            intfs = None
            for intfOpt in [ 'ETHINTF', 'LAGINTF', 'GREINTF' ]:
               if intfOpt in args:
                  intfs = args[ intfOpt ].intfNames()
                  break

            assert intfs
         dests = [ dest.name.stringValue for dest in sessionModel.targetInterfaces ]
         if set( dests ).isdisjoint( intfs ):
            sessionModel.doNotDisplayInSummary()
   return model
