#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the configuration commands for port mirroring.
# All commands defined here begin by 'monitor session'
#-------------------------------------------------------------------------------

from __future__ import absolute_import, division, print_function

import Arnet
import BasicCli
import Cell
import CliCommand
import CliGlobal
import CliMatcher
from CliCommon import GuardError
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.MaintenanceCliLib import isSubIntf
import CliPlugin.MirroringMonitorShow as MirroringMonitorShow
from CliPlugin.RecircIntfCli import RecircAutoIntfType
import CliPlugin.TechSupportCli as TechSupportCli
from CliPlugin.VlanIntfCli import VlanIntf
import CliPlugin.VrfCli as VrfCli
from CliToken.Ip import ipMatcherForConfigIf
from CliToken.Ipv6 import ipv6MatcherForConfigIf
from CliToken.Mac import macMatcherForConfigIf
import ConfigMount
import inspect
from Intf.IntfRange import(
   IntfRangeMatcher, IntfList, GenericRangeIntfType, MultiSubRangeIntfType )
from IntfRangePlugin.EthIntf import EthPhyRangeIntfType
from IntfRangePlugin.LagIntf import(
   LagRangeInfo, PortChannelNumMin, PortChannelNumMax )
from IpLibConsts import DEFAULT_VRF
import LazyMount
import MirroringCliLib
import MirroringLib
import Plugins
import Tac
import Toggles.IntfToggleLib
from TypeFuture import TacLazyType

# Tac types
Direction = TacLazyType( 'Mirroring::Direction' )
GrePayloadType = TacLazyType( 'Mirroring::GrePayloadType' )
SampleRateConstants = TacLazyType( 'Mirroring::SampleRateConstants' )
TruncationSize = TacLazyType( 'Mirroring::TruncationSize' )

# Global variables
gv = CliGlobal.CliGlobal( dict(
   # Sysdb mounts
   aclConfig=None,
   aclStatusDp=None,
   hwCapability=None,
   mirroringConfig=None,
   mirroringHwCapability=None,
   mirroringHwConfig=None,
   mirroringHwStatus=None,
   mirroringStatus=None,
   # Global values
   directionTx=Direction.directionTx,
   directionRx=Direction.directionRx,
   directionBoth=Direction.directionBoth,
   defaultGreHdrProto=TacLazyType( "Mirroring::GreTunnelKey" ).defaultGreHdrProto,
   headerRemovalDefaultVal=0,
) )

# Constants
strToDirection = {
   'rx': gv.directionRx,
   'tx': gv.directionTx,
   'both': gv.directionBoth,
}
LANZ_MIRROR_SESSION = '_InternalLanz'
# Header removal size >= 95 appears to add an extra L2 header. Requires
# further investigation, but for now set 90 as a safe upper limit.
headerRemovalMin = 1
headerRemovalMax = 90
# truncation sizes supported on any platform (for token generation, no duplicates)
# Jericho/Jericho+: 128, 192, QumranAx: 256
truncationSizeList = ( 128, 192, 256 )

# The problem that IntfRangeConfigRule can't parse interface type which is not
# created at startup-config time has been fixed by change @368045,
# we should use IntfRangeConfigRule here to avoid using separate rules for
# singleton phy interface
def mirroringSubIntfSupportedGuard( mode, token ):
   if( gv.mirroringHwCapability.subIfAsSrcSupported
       or not mode.session_.isInteractive()
       or not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

mirSubIntfSupported = [ mirroringSubIntfSupportedGuard ]

if Toggles.IntfToggleLib.toggleDynamicIfIndexEnabled():
   SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
   SubExtendedMin = SubIntfRangeDetails.subIntfExtendedIdMin
   SubExtendedMax = SubIntfRangeDetails.subIntfExtendedIdMax
   ethPhyRangeIntfType = EthPhyRangeIntfType( intfSubLowerBound=SubExtendedMin,
         intfSubUpperBound=SubExtendedMax )
else:
   ethPhyRangeIntfType = EthPhyRangeIntfType( )
ethPhyRangeIntfType.subIntfGuard_ = mirSubIntfSupported
lagAutoIntfType = MultiSubRangeIntfType(
   lambda: ( PortChannelNumMin, PortChannelNumMax ),
   "Port-Channel", "Po", "Lag interface",
   LagRangeInfo,
   subSupported=True,
   subIntfGuard=mirSubIntfSupported )

intfTypes = ( ethPhyRangeIntfType,
              EthIntfCli.UnconnectedEthPhyAutoIntfType,
              lagAutoIntfType, RecircAutoIntfType )

GreAutoIntfType = GenericRangeIntfType(
   lambda: ( 0, gv.hwCapability.maxTunnelIntfNum - 1 ),
   "Gre", "gre", "GRE tunnel interface" )

#----------------------------------------------------------------------------
# Helper functions
#----------------------------------------------------------------------------

def isGreIntf( intfName ):
   return intfName.startswith( "Gre" )

def isCpuIntf( intfName ):
   return intfName.startswith( "Cpu" )

def sessionContainsGreDest( sessionConfig ):
   return any( isGreIntf( intf ) for intf in sessionConfig.targetIntf )

def isRateLimitEgressChipSession( session ):
   return session and session.mirrorRateLimitInBps > 0 and \
          session.mirrorRateLimitChip == 'per-egress-chip'

def rateLimitEgressChipSessionNum( cfg ):
   rateLimitEgressChipSessions = 0
   for session in cfg.session.values():
      if isRateLimitEgressChipSession( session ):
         rateLimitEgressChipSessions += 1
   return rateLimitEgressChipSessions

#----------------------------------------------------------------------------
# Matchers
#----------------------------------------------------------------------------

excludeNames = '^(?!detail$|default$|summary$)([A-Za-z0-9_:{}\\[\\]-]+)'
matcherSessionName = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.mirroringConfig.session,
      helpdesc='Sessions name', pattern=excludeNames )

matcherDestination = CliMatcher.KeywordMatcher( 'destination',
      helpdesc='Mirroring destination configuration commands' )
matcherGre = CliMatcher.KeywordMatcher( 'gre',
      helpdesc='GRE keyword' )
matcherMonitor = CliMatcher.KeywordMatcher( 'monitor',
      helpdesc='Monitor configuration commands' )
matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Mirroring source configuration commands' )
matcherSrcIntfTypes = IntfRangeMatcher( explicitIntfTypes=intfTypes )

#----------------------------------------------------------------------------
# Guards
#----------------------------------------------------------------------------

def mirroringSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.maxSupportedSessions > 0:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGrePayloadSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.supportedGrePayloadTypes:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGrePayloadTypeSupportedGuard( mode, token ):
   grePayloadType = None
   if token == 'inner-packet':
      grePayloadType = GrePayloadType.payloadTypeInnerPacket
   elif token == 'full-packet':
      grePayloadType = GrePayloadType.payloadTypeFullPacket
   if gv.mirroringHwCapability.supportedGrePayloadTypes.get( grePayloadType ):
      return None
   return CliParser.guardNotThisPlatform

def mirroringGreMetadataSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.greMetadataSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGreMetadataElementSupportedGuard( mode, token ):
   for metadataColl in gv.mirroringHwCapability.greMetadataSupported.itervalues():
      if ( MirroringLib.tokenMetadataElementDict[ token ]
            in metadataColl.metadataElement.values() ):
         return None
   return CliParser.guardNotThisPlatform

def mirroringGreTimestampingSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.greTimestampingSupported:
      return None
   return CliParser.guardNotThisPlatform

def disablePortMirroringSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.disablePortMirroringSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringDefaultsSupportedGuard( mode, token ):
   # Add additional defaults to the following if statement as needed
   if ( mirroringGrePayloadSupportedGuard( mode, token ) is None
        or mirroringGreMetadataSupportedGuard( mode, token ) is None
        or mirroringGreTimestampingSupportedGuard( mode, token ) is None ):
      return None
   return CliParser.guardNotThisPlatform

def mirroringTruncationSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.truncationSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringTruncationSizeGuard( mode, token ):
   if ( gv.mirroringHwCapability.truncationSizesSupported
        and gv.mirroringHwCapability.truncationSizesSupported.options ):
      return None
   return CliParser.guardNotThisPlatform

def mirroringTruncationSizeValueGuard( mode, token ):
   size = int( token )
   if ( gv.mirroringHwCapability.truncationSizesSupported and size in
         gv.mirroringHwCapability.truncationSizesSupported.options ):
      return None
   return CliParser.guardNotThisPlatform

def mirroringHeaderRemovalSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.headerRemovalSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringDestLagSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.lagDestSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringMultiDestSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.multiDestSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringAclSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.aclSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringIp4AclSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.ip4AclSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringAclPrioritiesSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.aclPrioritiesSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringMacAclSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.macAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringIp6AclSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.ip6AclSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringCpuSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.cpuDestSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringDestinationGreSupported( mode, token ):
   if gv.mirroringHwCapability.destinationGreSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGreProtocolSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.greProtocolSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringForwardingDropSupported( mode, token ):
   if gv.mirroringHwCapability.forwardingDropSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringRxSamplingSupported( mode, token ):
   if gv.mirroringHwCapability.rxSamplingSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringRateLimitIngressGuard( mode, token ):
   if gv.mirroringHwCapability.rateLimitIngressChipSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringRateLimitEgressGuard( mode, token ):
   if gv.mirroringHwCapability.rateLimitEgressChipSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringTxGreMetadataGuard( mode, token ):
   if gv.mirroringHwCapability.txGreMetadataSupported:
      return None
   return CliParser.guardNotThisPlatform

#----------------------------------------------------------------------------
# Nodes
#----------------------------------------------------------------------------

nodeAccessGroup = CliCommand.Node( matcher=AclCli.accessGroupKwMatcher,
      guard=mirroringAclSupportedGuard )
nodeDefault = CliCommand.guardedKeyword( 'default',
      helpdesc='Default mirroring configuration commands',
      guard=mirroringDefaultsSupportedGuard )
nodeEncapsulation = CliCommand.guardedKeyword( 'encapsulation',
      helpdesc='Mirroring encapsulation configuration commands',
      guard=mirroringDestinationGreSupported )
nodeForwardingDrop = CliCommand.guardedKeyword( 'forwarding-drop',
      helpdesc='Mirror forwarding dropped packets',
      guard=mirroringForwardingDropSupported )
nodeHeader = CliCommand.guardedKeyword( 'header',
      helpdesc='Mirroring header configuration commands',
      guard=mirroringHeaderRemovalSupportedGuard )
nodeIp = CliCommand.Node( matcher=ipMatcherForConfigIf,
      alias='ACLTYPE',
      guard=mirroringIp4AclSupportedGuard )
nodeIpv6 = CliCommand.Node( matcher=ipv6MatcherForConfigIf,
      alias='ACLTYPE',
      guard=mirroringIp6AclSupportedGuard )
nodeMac = CliCommand.Node( matcher=macMatcherForConfigIf,
      alias='ACLTYPE',
      guard=mirroringMacAclSupportedGuard )
nodeMetadata = CliCommand.guardedKeyword( 'metadata',
      helpdesc='Mirroring GRE metadata format configuration',
      guard=mirroringGreMetadataSupportedGuard )
nodePayload = CliCommand.guardedKeyword( 'payload',
      helpdesc='Mirroring GRE payload type configuration commands',
      guard=mirroringGrePayloadSupportedGuard )
nodePriority = CliCommand.guardedKeyword( 'priority',
      helpdesc='ACL priority',
      guard=mirroringAclPrioritiesSupportedGuard )
nodeRemove = CliCommand.guardedKeyword( 'remove',
      helpdesc='Remove leading bytes from packet header',
      guard=mirroringHeaderRemovalSupportedGuard )
nodeSample = CliCommand.guardedKeyword( 'sample',
      helpdesc='Sampling Rate configuration commands',
      guard=mirroringRxSamplingSupported )
nodeSession = CliCommand.guardedKeyword( 'session',
      helpdesc='Monitor configuration commands',
      guard=mirroringSupportedGuard )
nodeTruncate = CliCommand.guardedKeyword( 'truncate',
      helpdesc='Mirroring truncate configuration commands',
      guard=mirroringTruncationSupportedGuard )
nodeTunnel = CliCommand.guardedKeyword( 'tunnel',
      helpdesc='Tunnel destination configuration',
      guard=mirroringDestinationGreSupported )

nodeBoth = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'both',
         helpdesc='Configure mirroring in both transmit and receive directions' ),
      alias="DIRECTION" )
nodeRx = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'rx',
         helpdesc='Configure mirroring only in receive direction' ),
      alias="DIRECTION" )
nodeTx = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'tx',
         helpdesc='Configure mirroring only in transmit direction' ),
      alias="DIRECTION" )

# Match 'ACLTYPE access-group ACLNAME'
class AclTypeNameExpression( CliCommand.CliExpression ):
   expression = """( ip access-group IPACL )
                   | ( ipv6 access-group IPV6ACL )
                   | ( mac access-group MACACL )"""
   data = {
      'access-group': nodeAccessGroup,
      'ip': nodeIp,
      'IPACL': CliCommand.Node(
         matcher=AclCli.userIpAclNameMatcher, alias='ACLNAME' ),
      'ipv6': nodeIpv6,
      'IPV6ACL': CliCommand.Node(
         matcher=AclCli.userIp6AclNameMatcher, alias='ACLNAME' ),
      'mac': nodeMac,
      'MACACL': CliCommand.Node(
         matcher=AclCli.userMacAclNameMatcher, alias='ACLNAME' ),
   }

class MirroringModelet( CliParser.SingletonModelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      self.mode_ = mode
      CliParser.SingletonModelet.__init__( self )

BasicCli.GlobalConfigMode.addModelet( MirroringModelet )

#---------------------------------------------------------------------------
# Helper functions
#----------------------------------------------------------------------------

def isMaxSessionsReached( mode, sessionToAdd, src=None, forwardingDrop=False ):
   return MirroringCliLib.isMaxSessionsReached( mode, sessionToAdd,
      gv.mirroringHwCapability, gv.mirroringConfig, src, forwardingDrop )

def hasValidSessionConfig( sessionCfg ):
   # Check if the session is in use or not
   return ( sessionCfg.srcIntf or sessionCfg.targetIntf
            or sessionCfg.truncate or sessionCfg.sessionAcl
            or sessionCfg.headerRemovalSize != gv.headerRemovalDefaultVal )

def isValidSource( mode, sessionName, srcIntf, direction, deleteAcl=False ):
   # Source is invalid, if it is present as a source or a destination in any
   # other session or is present as a destination in the current session
   # Note that a single source can be shared between two sessions
   # as long as the direction being monitored does not overlap
   cfg = gv.mirroringConfig
   mySession = cfg.session.get( sessionName )
   mySessionAcl = mySession and mySession.sessionAcl
   for ( name, session ) in cfg.session.items():
      sessionSrcIntf = session.srcIntf.get( srcIntf )
      sessionValid = (
            name != sessionName
            and not mode.session_.startupConfig()
            and sessionSrcIntf
            and ( sessionSrcIntf.direction == direction
                  or gv.directionBoth in ( direction, sessionSrcIntf.direction ) ) )
      if ( sessionValid
           and not gv.mirroringHwCapability.multiSessionForSourceSupported ):
         return 'Duplicate'

      if srcIntf in session.targetIntf:
         if gv.mirroringHwCapability.twoWayDestinationSupported:
            if name == sessionName or direction != gv.directionRx:
               return 'Duplicate'
            else:
               pass
         else:
            return 'Duplicate'

      if ( sessionValid and gv.mirroringHwCapability.aclPrioritiesSupported ):
         if ( mySessionAcl
              and ( session.aclType != mySession.aclType
                    or ( sessionSrcIntf.srcAcl
                         and sessionSrcIntf.aclType != mySession.aclType ) ) ):
            # If my session ACL is configured, we may support same source in
            # multi-session in two cases:
            # 1) The session ACLs are of different types; or
            # 2) The mySession session ACL type is different from the source
            #    ACL type of the session being checked
            pass
         else:
            emitDuplicateSourceNoAclMsg( mode, srcIntf, name, deleteAcl )
            return 'Duplicate'

      if name == sessionName:
         for targetIntf in session.targetIntf.iterkeys():
            if direction == gv.directionRx:
               continue
            if ( targetIntf.startswith( "Cpu" )
                 and not gv.mirroringHwCapability.txCpuDestSupported ):
               return "CPU destination is only supported in RX direction"
            elif ( targetIntf.startswith( "Gre" ) and not 
                   gv.mirroringHwCapability.txGreDestSupported ):
               return "Port mirroring to GRE destination is only supported in RX"\
                     " direction"

   # Verify number of TX monitor sessions is within platform limits.
   isNewTxSession = (
         direction in ( gv.directionTx, gv.directionBoth )
         and ( not mySession or not mySession.srcIntf
               or not any( intf.direction in ( gv.directionTx, gv.directionBoth )
                           for intf in mySession.srcIntf.values() ) ) )
   if isNewTxSession and gv.mirroringHwCapability.txMirrorSessionLimit:
      txMirrorSessions = 0
      for session in cfg.session.values():
         for intf in session.srcIntf.values():
            if intf.direction in ( gv.directionTx, gv.directionBoth ):
               txMirrorSessions += 1
               break
      if txMirrorSessions >= gv.mirroringHwCapability.maxSupportedTxMirrorSessions:
         return "maximum number of TX monitor sessions exceeded."

   return None

def isValidDestination( mode, sessionName, destIntf ):
   if not gv.mirroringHwCapability.subIfAsDestSupported and isSubIntf( destIntf ):
      return 'Sub-interface destination not supported'

   if isGreIntf( destIntf ):
      return None

   # Destination is invalid, if it is present as a source or a destination in any
   # other session, or is present as a source interface in the current session
   cfg = gv.mirroringConfig
   # cpu port can serve as destinations of multiple mirror sessions.
   if isCpuIntf( destIntf ):
      if sessionName not in cfg.session:
         return None
      sessionConfig = cfg.session[ sessionName ]

      if not gv.mirroringHwCapability.txCpuDestSupported:
         for intf in cfg.session[ sessionName ].srcIntf.values():
            if intf.direction in ( gv.directionTx, gv.directionBoth ):
               return 'CPU destination only supported in RX direction'

      if sessionContainsGreDest( sessionConfig ):
         return 'Interface type conflict'

      return None

   multiSessionForDestSupported = (
         mode.session_.startupConfig()
         or gv.mirroringHwCapability.multiSessionForDestSupported )
   # For ports other than cpu port, it cannot serve as the mirror destination
   # if it is already a source interface in a session or a destination in another
   # session
   for ( name, session ) in cfg.session.items():
      if destIntf in session.srcIntf:
         if gv.mirroringHwCapability.twoWayDestinationSupported:
            if ( name == sessionName
                 or session.srcIntf[ destIntf ].direction != gv.directionRx ):
               return 'Duplicate'
            else:
               pass
         else:
            return 'Duplicate'
      if ( not multiSessionForDestSupported
           and name != sessionName and destIntf in session.targetIntf ):
         return 'Duplicate'

      if name == sessionName and sessionContainsGreDest( session ):
         return 'Interface type conflict'
   return None

def isIngressOnlySession( sessionName ):
   sessionCfg = gv.mirroringConfig.session.get( sessionName )
   return ( not sessionCfg
            or all( x.direction == gv.directionRx
                    for x in sessionCfg.srcIntf.values() ) )

def isSamplingEnabled( sessionName ):
   sessionCfg = gv.mirroringConfig.session.get( sessionName )
   return ( sessionCfg
            and sessionCfg.rxSampleRate not in
             [ SampleRateConstants.sampleRateUndefined,
               SampleRateConstants.sampleRateMirrorAll ] )

#----------------------------------------------------------------------------
# Mode message emit helpers
#----------------------------------------------------------------------------

def emitNonExistentSessionMsg( mode, name ):
   errMsg = "Monitor session " + name + " does not exist."
   mode.addError( errMsg )

def emitNonExistentSrcIntfMsg( mode, srcIntf, sessionName ):
   errMsg = srcIntf + " is not a source interface in Session " + sessionName
   mode.addError( errMsg )

def emitAclNotMatchMsg( mode, aclName, origAcl ):
   if aclName:
      errMsg = " ACL " + aclName + " cannot be disapplied."
      if origAcl:
         errMsg += " The applied ACL is " + origAcl
      else:
         errMsg += " No ACL is currently applied."
   else:
      errMsg = "Please specify the acl to be disapplied.\
            The currently applied ACL is " + origAcl
   mode.addWarning( errMsg )

def emitInternalLanzSessionMsg( mode, name ):
   errMsg = "Monitor session " + name + " is reserved."
   mode.addError( errMsg )

def emitDirectionNotSupportedMsg( mode ):
   errMsg = "Port mirroring to GRE destination is supported in RX direction only"
   mode.addError( errMsg )

def emitGreWithOtherIntfNotSupportedMsg( mode, intfName ):
   errMsg = "destination cannot be GRE tunnel and " + intfName
   mode.addError( errMsg )

def emitDuplicatePriorityNotSupportedMsg( mode, intfName, sessName, priority ):
   errMsg = ( "Source interface " + intfName + " with priority " +
              str( priority ) + " is already configured in monitor session " +
              sessName + ". Please specify a different priority." )
   mode.addError( errMsg )

def emitDuplicateAclNotSupportedMsg( mode, intfName, sessName, aclName ):
   errMsg = ( "ACL " + aclName + " is already applied to source interface " +
              intfName + " in monitor session " + sessName +
              ". Please apply a different ACL." )
   mode.addError( errMsg )

def emitDuplicateSourceWithSessionAcl( mode ):
   errMsg = ( "Use of single source interface in multiple monitor sessions "
              "does not support interaction with session ACLs." )
   mode.addError( errMsg )

def emitDuplicateSourceNoAclMsg( mode, intfName, sessName, delete=False ):
   errMsg = ( "Interface " + intfName + " is already configured as a source " +
              "interface in monitor session " + sessName + ". " )
   if delete:
      errMsg += ( "Please remove the source interface to disapply the ACL and "
                  "priority from this session." )
   else:
      errMsg += ( "Please apply an ACL and priority to all monitor sessions "
                  "that share a source interface." )
   mode.addError( errMsg )

def emitSessionAclAppliedMsg( aclName, emitFunc ):
   errMsg = ( "The session is applied with ACL %s. "
              "Please remove the ACL from the session." % aclName )
   if inspect.isroutine( emitFunc ):
      emitFunc( errMsg )

def emitDuplicateSourceNoPriorityMsg( mode, intfName, sessName, aclName ):
   errMsg = ( "Use of a single source interface in multiple monitor sessions "
              "requires a priority set on each ACL." )
   mode.addError( errMsg )

def emitDuplicateSourceAclDestinationMsg( mode, intfName, sessName ):
   errMsg = ( "Interface " + intfName + " is already configured as a " +
              "destination interface in monitor session " + sessName + "." )
   mode.addError( errMsg )

def emitMaxSessionsReachedMsg( mode, name ):
   errMsg = ( "Could not create session '" + name +
              "'. Maximum mirroring sessions limit reached" )
   mode.addError( errMsg )

def emitNoMultiForwardingDropSessionMsg( mode ):
   errMsg = "forwarding-drop cannot be configured in multiple sessions"
   mode.addError( errMsg )

def emitFwdDropSubInterfaceSrcNotSupportedMsg( mode ):
   errMsg = "A forwarding-drop session cannot have a sub-interface as a source"
   mode.addError( errMsg )

#----------------------------------------------------------------------------
# Handler functions
#----------------------------------------------------------------------------

def getSessionConfig( mode, name, delete ):
   # returns: the sessionCfg, or raise AlreadyHandledError
   sessionCfg = gv.mirroringConfig.session.get( name )
   if delete and not sessionCfg:
      emitNonExistentSessionMsg( mode, name )
      raise CliParser.AlreadyHandledError()
   if sessionCfg and sessionCfg.secureMonitor != mode.session.secureMonitor():
      mode.addError( str( CliParser.guardNotPermitted ) )
      raise CliParser.AlreadyHandledError()
   return sessionCfg

def setMirrorDestination( mode, name, intf, delete=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      targetIntf = sessionCfg.targetIntf.get( intf )
      if targetIntf:
         del sessionCfg.targetIntf[ intf ]
         if not hasValidSessionConfig( sessionCfg ):
            del cfg.session[ name ]
   else:
      # display error if creating more than max sessions
      if isMaxSessionsReached( mode, name ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      reason = isValidDestination( mode, name, intf )
      if reason:
         mode.addError( 'Mirror destination %s was rejected '
                        'with reason %s' % ( intf, reason ) )
         return

      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()

      # If guards are not enabled or multi-dest is
      # supported add this to the target-intf coll.
      if ( gv.mirroringHwCapability.multiDestSupported
           or not mode.session_.guardsEnabled() ):
         if sessionCfg.targetIntf.get( intf ):
            return
         if ( gv.mirroringHwCapability.destinationsPerSessionLimit
              and len( sessionCfg.targetIntf ) >=
               gv.mirroringHwCapability.maxDestinationsPerSession ):
            mode.addError( 'Mirror destination %s rejected. '
               'Max destinations per session (%d) reached.' % ( intf,
                  gv.mirroringHwCapability.maxDestinationsPerSession ) )
            return

         if ( gv.mirroringHwCapability.multiDestinationSessionsLimit
              and len( sessionCfg.targetIntf ) >= 1 ):
            multiDestsSessions = 0
            for session in cfg.session.values():
               if session.name != name and len( session.targetIntf ) > 1:
                  multiDestsSessions += 1
            if ( multiDestsSessions >=
                 gv.mirroringHwCapability.maxMultiDestinationSessions ):
               mode.addError(
                  'Mirror destination %s rejected. Only %d session(s) '
                  'allowed to have multiple destinations.' % (
                     intf,
                     gv.mirroringHwCapability.maxMultiDestinationSessions ) )
               return

         # Display warning if there is already an interface configured and
         # there is a session with header removal configured
         hdrRemoval = any( sess.headerRemovalSize
               for name, sess in cfg.session.iteritems() )
         if sessionCfg.targetIntf and hdrRemoval:
            mode.addWarning( 'Header removal may stop working as this '
               'monitor session has multiple destinations configured.' )
         sessionCfg.targetIntf.newMember( intf )
      else:
         for oldIntf in sessionCfg.targetIntf:
            del sessionCfg.targetIntf[ oldIntf ]
         sessionCfg.targetIntf.newMember( intf )

def setMirrorDestinations( mode, name, intfList, delete=False ):
   if isinstance( intfList, VlanIntf ):
      raise CliParser.InvalidInputError()
   if name == LANZ_MIRROR_SESSION:
      emitInternalLanzSessionMsg( mode, name )
      return
   if intfList == 'cpu':
      setMirrorDestination( mode, name, 'Cpu', delete )
      return
   if isinstance( intfList, EthIntfCli.EthPhyIntf ):
      if intfList.name.startswith( 'Management' ):
         raise CliParser.InvalidInputError()
      # Its a singleton object, not a list of intfs
      setMirrorDestination( mode, name, intfList.name, delete )
   elif isinstance( intfList, IntfList ):
      if len( set( intfList.intfNames() ) ) > 1:
         guardCode = mirroringMultiDestSupportedGuard( mode, None )
         if guardCode:
            raise GuardError( guardCode )
         if gv.mirroringHwCapability.destinationsPerSessionLimit and not delete:
            sessionCfg = gv.mirroringConfig.session.get( name )
            desiredDests = 0
            if sessionCfg:
               desiredDests = sum( 1 for i in intfList.intfNames()
                                   if i not in sessionCfg.targetIntf )
               desiredDests += len( sessionCfg.targetIntf )
            else:
               desiredDests = len( set( intfList.intfNames() ) )
            if desiredDests > gv.mirroringHwCapability.maxDestinationsPerSession:
               mode.addError( 'Mirror destinations rejected. '
                  'Max destinations per session is %d.' % (
                     gv.mirroringHwCapability.maxDestinationsPerSession ) )
               return

         if gv.mirroringHwCapability.multiDestinationSessionsLimit and not delete:
            multiDestsSessions = 0
            for session in gv.mirroringConfig.session.values():
               if session.name != name and len( session.targetIntf ) > 1:
                  multiDestsSessions += 1
            if ( multiDestsSessions >=
                 gv.mirroringHwCapability.maxMultiDestinationSessions ):
               mode.addError(
                  'Mirror destination rejected. Only %d session(s) '
                  'allowed to have multiple destinations.' % (
                     gv.mirroringHwCapability.maxMultiDestinationSessions ) )
               return

      for intfName in intfList.intfNames():
         setMirrorDestination( mode, name, intfName, delete )
   else:
      raise CliParser.InvalidInputError()

def setMirrorDestinationGre( mode, name, greSrcAddr, greDstAddr,
                             forwardingDrop, ttl, dscp, greHdrProto,
                             vrf, delete=False ):
   forwardingDrop = forwardingDrop is not None
   sessionCfg = getSessionConfig( mode, name, delete )

   # We use a dummy Gre9998-9 for targetIntf for GRE tunnel dest.
   # When EOS supports GRE intf, cli would change to
   # monitor session <name> destination Gre<num>
   # and user will enter the greIntf<num>. For now, make Mirroring
   # agent happy by setting a dummy GreIntfId to targetIntf.
   # Mirroring agent allocated a correct GRE intfId and publishes
   # it to mirroring/hwconfig/session/<name>/targetIntf
   dummyTargetIntf = "Gre9999" if forwardingDrop else "Gre9998"
   if delete:
      greTunnelNone = MirroringLib.createGreTunnelKey( '0.0.0.0', '0.0.0.0', 128, 0 )
      if forwardingDrop:
         if sessionCfg.dropGreTunnelKey is None:
            return
         greTunnelKey = sessionCfg.dropGreTunnelKey
         sessionCfg.dropGreTunnelKey = greTunnelNone
      else:
         if sessionCfg.greTunnelKey is None:
            return
         greTunnelKey = sessionCfg.greTunnelKey
         sessionCfg.greTunnelKey = greTunnelNone
      setMirrorDestination( mode, name, dummyTargetIntf, delete=True )
   else:
      # only 1 forwarding drop session is allowed
      for sessCfg in gv.mirroringConfig.session.itervalues():
         if forwardingDrop:
            if ( sessCfg.name != name
                 and sessCfg.dropGreTunnelKey
                 and sessCfg.dropGreTunnelKey.forwardingDrop ):
               emitNoMultiForwardingDropSessionMsg( mode )
               return
      if isMaxSessionsReached( mode, name, forwardingDrop=forwardingDrop ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      if not sessionCfg:
         sessionCfg = gv.mirroringConfig.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      for srcIntf in sessionCfg.srcIntf.values():
         if ( srcIntf.direction != gv.directionRx and not 
              gv.mirroringHwCapability.txGreDestSupported ):
            emitDirectionNotSupportedMsg( mode, )
            return
         if forwardingDrop:
            if isSubIntf( srcIntf.name ):
               emitFwdDropSubInterfaceSrcNotSupportedMsg( mode )
               return

      existingTargetIntfs = sessionCfg.targetIntf.keys()
      if existingTargetIntfs and not existingTargetIntfs[ 0 ].startswith( "Gre" ):
         emitGreWithOtherIntfNotSupportedMsg( mode, existingTargetIntfs[ 0 ] )
         return

      greTunnelKey = MirroringLib.createGreTunnelKey( greSrcAddr, greDstAddr,
                                         ttl, dscp,
                                         greHdrProto=greHdrProto,
                                         forwardingDrop=forwardingDrop,
                                         vrf=vrf )

      if forwardingDrop:
         greIntfId = gv.mirroringStatus.greTunnelKeyToIntfId.get(
            sessionCfg.dropGreTunnelKey )
         sessionCfg.dropGreTunnelKey = greTunnelKey
      else:
         greIntfId = gv.mirroringStatus.greTunnelKeyToIntfId.get(
            sessionCfg.greTunnelKey )
         sessionCfg.greTunnelKey = greTunnelKey

      if greIntfId != None:
         del sessionCfg.targetIntf[ greIntfId ]
      sessionCfg.targetIntf.newMember( dummyTargetIntf )

def setMirrorSource( mode, name, intfName, direction=gv.directionBoth,
                     delete=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      srcIntf = sessionCfg.srcIntf.get( intfName )
      if srcIntf:
         if direction == gv.directionBoth or srcIntf.direction == direction:
            del sessionCfg.srcIntf[ intfName ]
            if not hasValidSessionConfig( sessionCfg ):
               del cfg.session[ name ]
         else:
            if direction == gv.directionTx and srcIntf.direction == gv.directionBoth:
               setMirrorSource( mode, name, intfName, gv.directionRx )
            if direction == gv.directionRx and srcIntf.direction == gv.directionBoth:
               setMirrorSource( mode, name, intfName, gv.directionTx )
   else:
      # display error if creating more than max sessions
      if isMaxSessionsReached( mode, name, src=( intfName, direction ) ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      reason = isValidSource( mode, name, intfName, direction )
      if reason:
         mode.addWarning( 'Mirror source ' + intfName +
                          ' was rejected with reason ' + reason )
         return
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      if direction != gv.directionRx and isSubIntf( intfName ):
         mode.addError( 'Subintf source can only be used'
                        + ' with ingress mirroring (RX direction) ' + direction )
         raise CliParser.InvalidInputError()

      if ( sessionCfg.dropGreTunnelKey and sessionCfg.dropGreTunnelKey.forwardingDrop
           and isSubIntf( intfName ) ):
         emitFwdDropSubInterfaceSrcNotSupportedMsg( mode, )
         return

      if ( direction != gv.directionRx
           and isSamplingEnabled( name )
           and isIngressOnlySession( name ) ):
         mode.addWarning( 'Sampling disabled due to non-RX source(s).' )
      src = sessionCfg.srcIntf.get( intfName )
      if not src:
         src = sessionCfg.srcIntf.newMember( intfName, direction )
         if sessionCfg.sessionAcl:
            src.aclType = sessionCfg.aclType
            src.srcAcl = sessionCfg.sessionAcl
            # TBD: consider adding an aclPriority to the session, which could
            # be inherited here.
            src.aclPriority = 0
      else:
         # we only support mirroring ACL on rx direction.
         if direction == gv.directionTx:
            # if there's a session sourceAcl, throw an error and return
            if sessionCfg.sessionAcl != '':
               emitSessionAclAppliedMsg( sessionCfg.sessionAcl, mode.addError )
               return
            # if a rx sourceAcl is configured, we must remove it first.
            if src.srcAcl != '':
               # reset aclType to default.
               src.resetAcl()
         src.direction = direction

def setMirrorSources( mode, name, intfList, direction=gv.directionBoth,
                      delete=False ):
   if name == LANZ_MIRROR_SESSION:
      emitInternalLanzSessionMsg( mode, name )
      return
   if isinstance( intfList, VlanIntf ):
      raise CliParser.InvalidInputError()
   if isinstance( intfList, EthIntfCli.EthPhyIntf ):
      if intfList.name.startswith( 'Management' ):
         raise CliParser.InvalidInputError()
      # Its a singleton object, not a list of intfs
      setMirrorSource( mode, name, intfList.name, direction, delete )
   elif isinstance( intfList, IntfList ):
      for intfName in intfList.intfNames():
         setMirrorSource( mode, name, intfName, direction, delete )
   else:
      raise CliParser.InvalidInputError()

def setDefaultGrePayloadType( payloadType ):
   cfg = gv.mirroringConfig
   defaultSessionConfig = cfg.defaultSessionConfig
   defaultSessionConfig.grePayloadType = payloadType

def setDefaultGreMetadata( greMetadata ):
   cfg = gv.mirroringConfig
   defaultSessionConfig = cfg.defaultSessionConfig
   defaultSessionConfig.greMetadata.clear()
   for metadata, value in greMetadata.iteritems():
      defaultSessionConfig.greMetadata[ metadata ] = value

def setDefaultGreTimestamping( timestampingEnabled ):
   cfg = gv.mirroringConfig
   defaultSessionConfig = cfg.defaultSessionConfig
   defaultSessionConfig.greTimestampingEnabled = timestampingEnabled

def setDefaultAction( disablePortMirroring, defaultAction ):
   if defaultAction == 'no-mirror':
      cfg = gv.mirroringConfig
      defaultSessionConfig = cfg.defaultSessionConfig
      defaultSessionConfig.portMirroringDisabled = disablePortMirroring

def isValidSourceAcl( mode, sessName, intfName, direction,
                      aclType, aclName, aclPriority ):
   cfg = gv.mirroringConfig
   for ( name, session ) in cfg.session.items():
      if intfName in session.targetIntf:
         if gv.mirroringHwCapability.twoWayDestinationSupported:
            if name == sessName or direction != gv.directionRx:
               emitDuplicateSourceAclDestinationMsg( mode, intfName, name )
               return False
            else:
               pass
         else:
            emitDuplicateSourceAclDestinationMsg( mode, intfName, name )
            return False

      if ( name == sessName
           or intfName not in session.srcIntf
           or mode.session_.startupConfig() ):
         continue

      # Check for duplicate source with session acl
      # we support same source in different session with different ACL
      # type.
      if session.sessionAcl and session.aclType == aclType:
         emitDuplicateSourceWithSessionAcl( mode )
         return False

      srcToCheck = session.srcIntf.get( intfName )
      if ( ( direction == gv.directionRx
             and srcToCheck.direction == gv.directionTx )
           or ( direction == gv.directionTx
                and srcToCheck.direction == gv.directionRx ) ):
         # If the duplicate source has strictly nonoverlapping direction
         # then we can still apply an ACL
         continue

      # Check for duplicate source without acl
      if not srcToCheck.srcAcl:
         emitDuplicateSourceNoAclMsg( mode, intfName, name )
         return False

      # Check for duplicate source without priority
      if ( ( not aclPriority or not srcToCheck.aclPriority )
           and aclType == srcToCheck.aclType ):
         emitDuplicateSourceNoPriorityMsg( mode, intfName, name, aclName )
         return False

      # Check for duplicate Acl
      if aclName == srcToCheck.srcAcl and aclType == srcToCheck.aclType:
         emitDuplicateAclNotSupportedMsg( mode, intfName, name, aclName )
         return False

      # Check for duplicate priority
      if ( aclPriority == srcToCheck.aclPriority
           and aclType == srcToCheck.aclType ):
         emitDuplicatePriorityNotSupportedMsg( mode, intfName, name, aclPriority )
         return False
   return True

def setSourceAcl( mode, name, intfName, direction,
                  aclType='ip', aclName='', aclPriority=0,
                  delete=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      srcIntf = sessionCfg.srcIntf.get( intfName )
      if not srcIntf:
         emitNonExistentSrcIntfMsg( mode, intfName, name )
         return
      if sessionCfg.sessionAcl != '':
         assert sessionCfg.sessionAcl == srcIntf.srcAcl
         emitSessionAclAppliedMsg( sessionCfg.sessionAcl, mode.addWarning )
         return
      if srcIntf.srcAcl == aclName:
         # we should use actual configured srcIntf direction to validate
         cfgDirection = srcIntf.direction
         reason = isValidSource( mode, name, intfName, cfgDirection, delete )
      else:
         emitAclNotMatchMsg( mode, aclName, srcIntf.srcAcl )
         reason = 'ACL does not match'

      if reason:
         mode.addWarning( 'Removing mirror source ACL on '
                          + intfName + ' was rejected with reason ' + reason )
      else:
         # Reset aclType to default
         srcIntf.resetAcl()
   else:
      if aclName == '':
         return

      if isMaxSessionsReached( mode, name, src=( intfName, direction ) ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      if gv.mirroringHwCapability.aclPrioritiesSupported:
         status = isValidSourceAcl( mode, name, intfName, direction,
                                    aclType, aclName, aclPriority )
         if status is False:
            return
      # This won't catch everything but just add a bit more check
      aclConfig = gv.aclConfig.config[ aclType ].acl.get( aclName )
      if aclConfig and aclConfig.secureMonitor != mode.session.secureMonitor():
         mode.addError( str( CliParser.guardNotPermitted ) )
         return
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      src = sessionCfg.srcIntf.get( intfName )
      if not src:
         src = sessionCfg.srcIntf.newMember( intfName, direction )
         if sessionCfg.sessionAcl:
            src.aclType = sessionCfg.aclType
            src.srcAcl = sessionCfg.sessionAcl
            # TBD: consider adding an aclPriority to the session, which could
            # be inherited here.
            src.aclPriority = 0
      if src.direction == gv.directionTx:
         mode.addWarning(
               "ACL cannot be applied to ports whose egress traffic are"
               " mirrored." )
         return

      if src.srcAcl != '' and src.srcAcl != aclName:
         mode.addWarning(
               "Source %s in Session %s was applied with ACL %s, "
               "please remove this ACL first before applying another ACL."
               % ( intfName, name, src.srcAcl ) )
         return

      if not ( mode.session_.startupConfig()
               or gv.mirroringHwCapability.multipleAclTypesPerSessionSupported ):
         for srcIntfName in sessionCfg.srcIntf:
            srcIntf = sessionCfg.srcIntf[ srcIntfName ]
            if srcIntf.srcAcl != "" and srcIntf.aclType != aclType:
               mode.addError(
                  "Source %s in Session %s was applied with Acl type %s. "
                  % ( srcIntfName, name, srcIntf.aclType ) +
                  "Only one Acl type per session is supported." )
               return

      if aclName not in gv.aclConfig.config[ aclType ].acl:
         mode.addWarning(
            "Acl %s not configured. Assigning anyway." % aclName )

      src.aclType = aclType
      src.srcAcl = aclName
      src.aclPriority = aclPriority

def setSourcesAcl( mode, name, intfList, aclName, direction, aclType='ip',
                   aclPriority=0, delete=False ):
   if name == LANZ_MIRROR_SESSION:
      emitInternalLanzSessionMsg( mode, name )
      return
   if isinstance( intfList, EthIntfCli.EthPhyIntf ):
      # Its a singleton object, not a list of intfs
      setSourceAcl( mode, name, intfList.name, direction,
                         aclType, aclName, aclPriority,
                         delete )
   else:
      assert isinstance( intfList, IntfList )
      for intfName in intfList.intfNames():
         setSourceAcl( mode, name, intfName, direction,
                            aclType, aclName, aclPriority,
                            delete )

def setMirrorTruncate( mode, name, delete=False, size=None ):
   if name == LANZ_MIRROR_SESSION:
      emitInternalLanzSessionMsg( mode, name )
      return
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      sessionCfg.truncate = False
      sessionCfg.truncationSize = TruncationSize.null
      if not hasValidSessionConfig( sessionCfg ):
         del cfg.session[ name ]
   else:
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      sessionCfg.truncate = True
      if size:
         sessionCfg.truncationSize = size
      else:
         sessionCfg.truncationSize = TruncationSize.null

def setMirrorHeaderRemoval( mode, name, size=gv.headerRemovalDefaultVal,
                            delete=False ):
   if name == LANZ_MIRROR_SESSION:
      emitInternalLanzSessionMsg( mode, name )
      return
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      sessionCfg.headerRemovalSize = gv.headerRemovalDefaultVal
      if not hasValidSessionConfig( sessionCfg ):
         del cfg.session[ name ]
   else:
      multiDestSessions = [ key for key, status in
         gv.mirroringStatus.sessionStatus.iteritems() if status.multiDestActive ]

      if multiDestSessions:
         mode.addWarning( 'Header removal will not work as there '
            'are monitor sessions with multiple active destinations: %s.' %
            ( ', '.join( multiDestSessions ) ) )
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      sessionCfg.headerRemovalSize = size

def setMirrorSampleRate( mode, name, rate, delete=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if not sessionCfg:
      sessionCfg = cfg.session.newMember( name )
      sessionCfg.secureMonitor = mode.session.secureMonitor()
   sessionCfg.rxSampleRate = rate
   if ( rate != SampleRateConstants.sampleRateMirrorAll
        and not isIngressOnlySession( name ) ):
      mode.addWarning( 'Sampling disabled due to non-RX source(s).' )

def setSessionAcl( mode, name, aclName="", aclType="ip", delete=False ):
   if name == LANZ_MIRROR_SESSION:
      emitInternalLanzSessionMsg( mode, name )
      return
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      if sessionCfg.sessionAcl != aclName:
         emitAclNotMatchMsg( mode, aclName, sessionCfg.sessionAcl )
         return
      intfNames = Arnet.sortIntf( sessionCfg.srcIntf )
      for intfName in intfNames:
         srcIntf = sessionCfg.srcIntf[ intfName ]
         assert srcIntf.srcAcl == sessionCfg.sessionAcl
         # Reset aclType to default
         srcIntf.resetAcl()
      sessionCfg.sessionAcl = ''
      if not hasValidSessionConfig( sessionCfg ):
         del cfg.session[ name ]
      return

   if not aclName:
      return
   if isMaxSessionsReached( mode, name ):
      emitMaxSessionsReachedMsg( mode, name )
      return
   # This won't catch everything but just add a bit more check
   aclConfig = gv.aclConfig.config[ aclType ].acl.get( aclName )
   if aclConfig and aclConfig.secureMonitor != mode.session.secureMonitor():
      mode.addError( str( CliParser.guardNotPermitted ) )
      return
   if not sessionCfg:
      sessionCfg = cfg.session.newMember( name )
      sessionCfg.secureMonitor = mode.session.secureMonitor()
      sessionCfg.sessionAcl = ''
   else:
      # if the sessio is already applied with the same acl, we do not
      # do anything.
      if sessionCfg.sessionAcl == aclName:
         return

      if sessionCfg.secureMonitor != mode.session.secureMonitor():
         mode.addError( str( CliParser.guardNotPermitted ) )
         return

      # if the sessio is already applied with another acl, the user
      # need to remove the acl first
      if sessionCfg.sessionAcl != '' and sessionCfg.sessionAcl != aclName:
         mode.addWarning(
            "Session %s was applied with ACL %s, "
            "please remove this ACL first before applying another ACL."
            % ( name, sessionCfg.sessionAcl ) )
         return

      # if any source in the sessio is already applied with an acl, the user
      # need to remove the acl first
      intfNames = Arnet.sortIntf( sessionCfg.srcIntf )
      for intfName in intfNames:
         srcIntf = sessionCfg.srcIntf[ intfName ]
         if srcIntf.srcAcl != '':
            mode.addWarning(
               "Source %s in the session was applied with ACL %s, "
               "please remove the ACL first before applying an ACL to the session."
               % ( intfName, srcIntf.srcAcl ) )
            return
         if srcIntf.direction == gv.directionTx:
            mode.addWarning(
                  "ACL cannot be applied because at least one source port has"
                  " egress traffic mirroring." )
            return

      # Right now, we support ip acl only in mirroring. Other acls will be
      # supported in the future.
      if aclName not in gv.aclConfig.config[ aclType ].acl:
         mode.addWarning(
            "Acl %s not configured. Assigning anyway." % aclName )

   sessionCfg.aclType = aclType
   sessionCfg.sessionAcl = aclName
   # Update the srcAcl in each src in the mirror session
   intfNames = Arnet.sortIntf( sessionCfg.srcIntf )
   for intfName in intfNames:
      srcIntf = sessionCfg.srcIntf[ intfName ]
      srcIntf.aclType = aclType
      srcIntf.srcAcl = aclName

def doShowMirrorSessions( mode, args ):
   return MirroringMonitorShow.doShowMirrorSessions( gv, mode, args )

#----------------------------------------------------------------------------
# CLI hander functions
#---------------------------------------------------------------------------

def destinationGreCommand( mode, name, greSrcAddr, greDstAddr,
                           forwardingDrop=None, ttlDscpProtoVrf=None ):
   ttl = 128
   dscp = 0
   greHdrProto = gv.defaultGreHdrProto
   vrf = DEFAULT_VRF
   for param in ttlDscpProtoVrf:
      if param[ 0 ] == 'ttl':
         ttl = param[ 1 ]
      elif param[ 0 ] == 'dscp':
         dscp = param[ 1 ]
      elif param[ 0 ] == 'protocol':
         greHdrProto = param[ 1 ]
      elif param[ 0 ] == 'VRF':
         vrf = param[ 1 ]

   if greSrcAddr == greDstAddr:
      errMsg = 'source and destination cannot be same'
      mode.addError( errMsg )
      return
   if ( not mode.session_.startupConfig() and
        ( not gv.mirroringHwCapability.mirrorToIp6GreSupported
          and ( greSrcAddr.af == 'ipv6' or greDstAddr.af == 'ipv6' ) ) ):
      raise GuardError( 'IPv6 is not supported on this platform' )

   def isInvalidGreAddress( ip ):
      return ( ip.isMulticast or ip.isReserved or ip.isLinkLocal
         or ip.isThisNetwork or ip.isBroadcast )

   if isInvalidGreAddress( greSrcAddr ):
      errMsg = 'Invalid source IP address'
      mode.addError( errMsg )
      return

   if isInvalidGreAddress( greDstAddr ):
      errMsg = 'Invalid destination IP address'
      mode.addError( errMsg )
      return

   setMirrorDestinationGre( mode, name, greSrcAddr.stringValue,
                            greDstAddr.stringValue,
                            forwardingDrop, ttl, dscp,
                            greHdrProto, vrf )

def noDestinationGreCommand( mode, name, greSrcAddr=None, greDstAddr=None,
                             forwardingDrop=None, ttlDscpProtoVrf=None ):
   srcAddr = None if greSrcAddr is None else greSrcAddr.stringValue
   dstAddr = None if greDstAddr is None else greDstAddr.stringValue
   setMirrorDestinationGre( mode, name, srcAddr, dstAddr,
                            forwardingDrop, ttl=128, dscp=0,
                            greHdrProto=gv.defaultGreHdrProto,
                            vrf=DEFAULT_VRF, delete=True )

def defaultGreMetadataCommand( mode, args ):
   upper2Byte = args[ 'UPPER2B' ]
   lower2Byte = args[ 'LOWER2B' ]
   greMetadata = { MirroringLib.tokenMetadataElementDict[ upper2Byte ]: True,
                   MirroringLib.tokenMetadataElementDict[ lower2Byte ]: True }
   setDefaultGreMetadata( greMetadata )

def noDefaultGreMetadataCommand( mode, args ):
   setDefaultGreMetadata( {} )

def rateLimitCommand( mode, name, rateLimitChip, rateLimit, rateLimitUnit,
                      delete=False ):
   mirrorRateLimitInBps = 0
   if rateLimitUnit == 'bps':
      mirrorRateLimitInBps = rateLimit
   elif rateLimitUnit == 'kbps':
      mirrorRateLimitInBps = rateLimit * 1000
   elif rateLimitUnit == 'mbps':
      mirrorRateLimitInBps = rateLimit * 1000 * 1000
   else:
      mode.addWarning( 'Invalid rate-limit configuration.' )
      return
   # The Cli will limit the input range to 18 kpbs and 300 gbps
   # These values are from AradPolicerConstants minRateKbps and maxRateKbps
   if rateLimit > 0:
      if mirrorRateLimitInBps < 18000 or mirrorRateLimitInBps > 300 * 1e9:
         msg = 'rate-limit value should be between 18kbps and 300gbps.'
         mode.addWarning( msg )
         return

   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete=delete )

   if not sessionCfg:
      sessionCfg = cfg.session.newMember( name )
      sessionCfg.secureMonitor = mode.session.secureMonitor()
   sessionCfg.mirrorRateLimitChip = rateLimitChip
   sessionCfg.mirrorRateLimitInBps = mirrorRateLimitInBps

   if not delete and rateLimitChip == 'per-egress-chip' and \
      gv.mirroringHwCapability.rateLimitEgressChipSupported:
      numLimit = gv.mirroringHwCapability.maxRateLimitEgressChipSessions
      if rateLimitEgressChipSessionNum( cfg ) > numLimit:
         msg = ( 'rate-limit per-egress-chip function can only work on maximum %d '
                 'mirroring sessions due to hardware limitations.' ) % numLimit
         mode.addWarning( msg )

def noRateLimitCommand( mode, name, rateLimitChip ):
   rateLimitCommand( mode, name, rateLimitChip, 0, 'bps',
                     delete=True )

def noSessionCommand( mode, args ):
   name = args[ 'NAME' ]
   if name == LANZ_MIRROR_SESSION:
      emitInternalLanzSessionMsg( mode, name )
      return
   cfg = getSessionConfig( mode, name, True )
   if cfg:
      del gv.mirroringConfig.session[ name ]

#---------------------------------------------------------------------------
# CLI commands classes
#---------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# ( no | default ) monitor session NAME
#--------------------------------------------------------------------------------
class NoSessionCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'monitor session NAME'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
   }

   noOrDefaultHandler = noSessionCommand

MirroringModelet.addCommandClass( NoSessionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitor session NAME ( mac | ip | ipv6 ) access-group ACLNAME
#--------------------------------------------------------------------------------
def sessionAclCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   aclName = args[ 'ACLNAME' ]
   aclType = args[ 'ACLTYPE' ]
   setSessionAcl( mode, name, aclName, aclType=aclType, delete=delete )

class NameAclCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME ACL'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'ACL': AclTypeNameExpression,
   }

   handler = sessionAclCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameAclCmd )

#--------------------------------------------------------------------------------
# monitor session NAME [ forwarding-drop ] destination tunnel mode gre
#     source GRESRCADDR destination GREDSTADDR
#     [ { ( ( ttl TTLRANGE ) | ( dscp DSCPRANGE )
#           | ( protocol PROTORANGE ) | VRF ) } ]
# ( no | default ) monitor session NAME [ forwarding-drop ] destination tunnel
#     mode gre ...
#--------------------------------------------------------------------------------

class NameDestinationTunnelModeGreCmd( CliCommand.CliCommandClass ):
   syntax = """monitor session NAME [ forwarding-drop ] destination tunnel mode gre
               source GRESRCADDR GRE_DEST GREDSTADDR
               [ { ( ttl TTLRANGE ) | ( dscp DSCPRANGE )
                   | ( protocol PROTORANGE ) | VRF } ]"""
   noOrDefaultSyntax = """
      monitor session NAME [ forwarding-drop ] destination tunnel mode gre ..."""
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'forwarding-drop': nodeForwardingDrop,
      'destination': matcherDestination,
      'tunnel': nodeTunnel,
      'mode': 'Tunnel mode selection',
      'gre': 'Generic Routing Encapsulation',
      'source': 'GRE source configuration',
      'GRESRCADDR': IpGenAddrMatcher(
         helpdesc='Source IP address of GRE tunnel' ),
      'GRE_DEST': CliMatcher.KeywordMatcher(
         'destination', helpdesc='GRE destination configuration' ),
      'GREDSTADDR': IpGenAddrMatcher(
         helpdesc='Destination IP address of GRE tunnel' ),

      # SetRule: ttl, dscp, protocol and vrf can be matched in any order
      # but each only once
      # i.e. ( ttl, vrf, dscp ) valid
      #      ( ttl, vrf, ttl ) invalid
      'ttl': CliCommand.singleKeyword( 'ttl', helpdesc='TTL of the GRE tunnel' ),
      'TTLRANGE': CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL range' ),
      'dscp': CliCommand.singleKeyword( 'dscp', helpdesc='DSCP of the GRE tunnel' ),
      'DSCPRANGE': CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP range' ),
      'protocol': CliCommand.singleKeyword(
         'protocol', helpdesc='Protocol type in GRE header',
         guard=mirroringGreProtocolSupportedGuard ),
      'PROTORANGE': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='Protocol range', helpname='0x0000-0xFFFF' ),
      'VRF': VrfCli.VrfExprFactory( helpdesc='VRF name of the GRE tunnel',
                                    inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      forwardingDrop = args.get( 'forwarding-drop' )
      greSrcAddr = args[ 'GRESRCADDR' ]
      greDstAddr = args[ 'GREDSTADDR' ]
      ttlDscpProtoVrf = []
      params = {
            'ttl': 'TTLRANGE',
            'dscp': 'DSCPRANGE',
            'protocol': 'PROTORANGE',
            'VRF': 'VRF',
      }
      for param, valName in params.iteritems():
         if param in args:
            ttlDscpProtoVrf.append( ( param, args[ valName ][ 0 ] ) )
      destinationGreCommand(
            mode, name, greSrcAddr, greDstAddr, forwardingDrop, ttlDscpProtoVrf )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'NAME' ]
      forwardingDrop = args.get( 'forwarding-drop' )
      noDestinationGreCommand( mode, name, forwardingDrop=forwardingDrop )

MirroringModelet.addCommandClass( NameDestinationTunnelModeGreCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitor session NAME destination ( CPU | ETH | LAG )
#--------------------------------------------------------------------------------

def destinationCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   intf = args[ 'INTF' ]
   setMirrorDestinations( mode, name, intf, delete=delete )

class NameDestinationCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME destination ( cpu | ETH | LAG )'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'destination': matcherDestination,
      'cpu': CliCommand.guardedKeyword(
         'cpu', helpdesc='CPU port(s)',
         guard=mirroringCpuSupportedGuard,
         alias="INTF" ),
      # Unfortunately we can't use IntfRangeConfigRule here since Lag intf has
      # its own guardian function
      'ETH': CliCommand.Node( matcher=IntfRangeMatcher(
         explicitIntfTypes=set( ( EthIntfCli.EthPhyAutoIntfType, ) ) ),
         alias="INTF" ),
      'LAG': CliCommand.Node( matcher=IntfRangeMatcher(
         explicitIntfTypes=set( ( LagAutoIntfType, ) ) ),
         guard=mirroringDestLagSupportedGuard,
         alias="INTF" ),
   }

   handler = destinationCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameDestinationCmd )

#--------------------------------------------------------------------------------
# monitor session NAME header remove size SIZE
# ( no | default ) monitor session NAME header remove ...
#--------------------------------------------------------------------------------

def headerRemovalCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   size = args.get( 'SIZE', gv.headerRemovalDefaultVal )
   setMirrorHeaderRemoval( mode, name, size=size, delete=delete )

class NameHeaderRemoveCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME header remove size SIZE'
   noOrDefaultSyntax = 'monitor session NAME header remove ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'header': nodeHeader,
      'remove': nodeRemove,
      'size': 'Set the size of header to be removed',
      'SIZE': CliMatcher.IntegerMatcher( 1, 90,
         helpdesc='Mirroring header removal size in bytes' ),
   }

   handler = headerRemovalCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameHeaderRemoveCmd )

#--------------------------------------------------------------------------------
# monitor session NAME rate-limit ( per-ingress-chip | per-egress-chip )
#     LIMIT ( bps | kbps | mbps )
# ( no | default ) monitor session NAME rate-limit
#     ( per-ingress-chip | per-egress-chip ) ...
#--------------------------------------------------------------------------------

class NameRateLimitCmd( CliCommand.CliCommandClass ):
   syntax = """monitor session NAME rate-limit
               ( per-ingress-chip | per-egress-chip ) LIMIT UNIT"""
   noOrDefaultSyntax = """monitor session NAME rate-limit
                          ( per-ingress-chip | per-egress-chip ) ..."""
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'rate-limit': 'Mirroring rate limit commands',
      'per-ingress-chip': CliCommand.guardedKeyword(
         'per-ingress-chip', helpdesc='Rate limit at ingress chip',
         guard=mirroringRateLimitIngressGuard ),
      'per-egress-chip': CliCommand.guardedKeyword(
          'per-egress-chip', helpdesc='Rate limit at egress chip',
          guard=mirroringRateLimitEgressGuard ),
      'LIMIT': CliMatcher.IntegerMatcher( 1, 10000000000,
         helpdesc='Mirroring rate limit' ),
      'UNIT': CliMatcher.EnumMatcher( {
         'bps': 'Rate unit bps',
         'kbps': 'Rate unit kbps',
         'mbps': 'Rate unit mbps',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      rateLimitChip = (
            'per-ingress-chip' if 'per-ingress-chip' in args else 'per-egress-chip' )
      rateLimit = args[ 'LIMIT' ]
      rateLimitUnit = args[ 'UNIT' ]
      rateLimitCommand(
            mode, name, rateLimitChip, rateLimit, rateLimitUnit )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'NAME' ]
      rateLimitChip = (
            'per-ingress-chip' if 'per-ingress-chip' in args else 'per-egress-chip' )
      noRateLimitCommand( mode, name, rateLimitChip )

MirroringModelet.addCommandClass( NameRateLimitCmd )

#--------------------------------------------------------------------------------
# monitor session NAME sample SAMPLERATE
# ( no | default ) monitor session NAME sample ...
#--------------------------------------------------------------------------------

def sampleRateCommand( mode, args ):
   name = args[ 'NAME' ]
   rate = args.get( 'SAMPLERATE', SampleRateConstants.sampleRateMirrorAll )
   setMirrorSampleRate( mode, name, rate=rate,
                        delete=CliCommand.isNoOrDefaultCmd( args ) )

class NameSampleSampleCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME sample SAMPLERATE'
   noOrDefaultSyntax = 'monitor session NAME sample ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'sample': nodeSample,
      'SAMPLERATE': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFE,
         helpdesc='1/<rate> is the packet sampling probability' ),
   }

   handler = sampleRateCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameSampleSampleCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitor session NAME source INTFS [ rx | tx | both ]
#--------------------------------------------------------------------------------

def setMirrorSourcesCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   intfList = args[ 'INTFS' ]
   direction = args.get( 'DIRECTION', 'both' )
   direction = strToDirection[ direction ]
   setMirrorSources(
         mode, name, intfList, direction=direction, delete=delete )

class NameSourceIntfsRxTxBothCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME source INTFS [ rx | tx | both ]'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'source': 'Mirroring source configuration commands',
      'INTFS': matcherSrcIntfTypes,
      'rx': nodeRx,
      'tx': nodeTx,
      'both': nodeBoth,
   }

   handler = setMirrorSourcesCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameSourceIntfsRxTxBothCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitor session NAME source INTFS [ rx | both ]
#     ( ip | ipv6 | mac ) access-group ACLNAME [ priority PRIORITYVALUE ] ]
#--------------------------------------------------------------------------------

def setSourceAclCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   intfList = args[ 'INTFS' ]
   aclName = args[ 'ACLNAME' ]
   aclType = args[ 'ACLTYPE' ]
   priority = args.get( 'PRIORITYVALUE', 0 )
   direction = args.get( 'DIRECTION', 'both' )
   direction = strToDirection[ direction ]
   setSourcesAcl( mode, name, intfList, aclName, direction,
                               aclType=aclType, aclPriority=priority,
                               delete=delete )

class NameSourceIntfsAclCmd( CliCommand.CliCommandClass ):
   syntax = """monitor session NAME source INTFS [ rx | both ]
               ACL [ priority PRIORITYVALUE ]"""
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'source': matcherSource,
      'INTFS': matcherSrcIntfTypes,
      'rx': nodeRx,
      'both': nodeBoth,
      'ACL': AclTypeNameExpression,
      'priority': nodePriority,
      'PRIORITYVALUE': CliMatcher.IntegerMatcher( 0, 100,
         helpdesc='ACL priority value' ),
   }

   handler = setSourceAclCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameSourceIntfsAclCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitor session NAME truncate [ size SIZE ]
#--------------------------------------------------------------------------------

def truncateCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   size = args.get( 'SIZE' )
   if size is not None:
      size = int( size )
   setMirrorTruncate( mode, name, delete=delete, size=size )

class NameTruncateCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME truncate [ size SIZE ]'
   noOrDefaultSyntax = 'monitor session NAME truncate ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'truncate': nodeTruncate,
      'size': CliCommand.guardedKeyword( 'size',
         helpdesc='Mirroring truncation size configuration commands',
         guard=mirroringTruncationSizeGuard ),
      'SIZE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            str( size ): 'Nominal mirroring truncation size in bytes'
            for size in truncationSizeList
         } ), guard=mirroringTruncationSizeValueGuard )
   }

   handler = truncateCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameTruncateCmd )

#--------------------------------------------------------------------------------
# monitor session default encapsulation gre metadata
#     ( ingress-vlan | egress-port ) ingress-port
# ( no | default ) monitor session default encapsulation gre metadata ...
#--------------------------------------------------------------------------------

# BUG433081: The current implementation is specific to Sand platform. Ideally, the
# syntax should be derived from mirroringHwCapability.greMetadataSupported
class DefaultEncapsulationGreMetadataCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session default encapsulation gre metadata UPPER2B LOWER2B'
   noOrDefaultSyntax = 'monitor session default encapsulation gre metadata ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'default': nodeDefault,
      'encapsulation': nodeEncapsulation,
      'gre': matcherGre,
      'metadata': nodeMetadata,
      'UPPER2B': CliCommand.Node( matcher=CliMatcher.EnumMatcher( {
            'ingress-vlan': 'Ingress port VLAN ID',
            'egress-port': 'Egress port ID',
         } ), guard=mirroringGreMetadataElementSupportedGuard ),
      'LOWER2B': CliCommand.Node( matcher=CliMatcher.EnumMatcher( {
            'ingress-port': 'Ingress port ID',
         } ), guard=mirroringGreMetadataElementSupportedGuard ),
   }

   handler = defaultGreMetadataCommand
   noOrDefaultHandler = noDefaultGreMetadataCommand

MirroringModelet.addCommandClass( DefaultEncapsulationGreMetadataCmd )

#--------------------------------------------------------------------------------
# monitor session default encapsulation gre payload ( full-packet | inner-packet )
# ( no | default ) monitor session default encapsulation gre payload ...
#--------------------------------------------------------------------------------

class DefaultEncapsulationGrePayloadCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session default encapsulation gre payload TYPE'
   noOrDefaultSyntax = 'monitor session default encapsulation gre payload ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'default': nodeDefault,
      'encapsulation': nodeEncapsulation,
      'gre': matcherGre,
      'payload': nodePayload,
      'TYPE': CliCommand.Node( matcher=CliMatcher.EnumMatcher( {
            'full-packet': 'Mirror full packet',
            'inner-packet': 'Mirror packet without terminated headers',
         } ), guard=mirroringGrePayloadTypeSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      payloadType = {
            'full-packet': GrePayloadType.payloadTypeFullPacket,
            'inner-packet': GrePayloadType.payloadTypeInnerPacket,
      }[ args[ 'TYPE' ] ]
      setDefaultGrePayloadType( payloadType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setDefaultGrePayloadType( GrePayloadType.payloadTypeHwDefault )

MirroringModelet.addCommandClass( DefaultEncapsulationGrePayloadCmd )

#--------------------------------------------------------------------------------
# monitor session default encapsulation gre timestamp
# ( no | default ) monitor session default encapsulation gre timestamp ...
#--------------------------------------------------------------------------------

class DefaultEncapsulationGreTimestampCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session default encapsulation gre timestamp'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'default': nodeDefault,
      'encapsulation': nodeEncapsulation,
      'gre': matcherGre,
      'timestamp': CliCommand.guardedKeyword( 'timestamp',
         helpdesc='Add timestamps in the GRE payload of mirrored packets',
         guard=mirroringGreTimestampingSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      setDefaultGreTimestamping( timestampingEnabled=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setDefaultGreTimestamping( timestampingEnabled=False )

MirroringModelet.addCommandClass( DefaultEncapsulationGreTimestampCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitor session default-action ACTION
#--------------------------------------------------------------------------------

def defaultActionCommand( mode, args ):
   disablePortMirroring = not CliCommand.isNoOrDefaultCmd( args )
   defaultAction = args[ 'ACTION' ]
   setDefaultAction( disablePortMirroring, defaultAction )

class DefaultActionNoMirrorCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session default-action ( no-mirror )'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'default-action': CliCommand.guardedKeyword( 'default-action',
         helpdesc='Set the default action when no acls are programmed',
         guard=disablePortMirroringSupportedGuard ),
      # Actions
      'no-mirror': CliCommand.guardedKeyword(
         'no-mirror', helpdesc='No mirroring when no ACL is configured',
         guard=disablePortMirroringSupportedGuard,
         alias="ACTION" ),
   }

   handler = defaultActionCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( DefaultActionNoMirrorCmd )

#--------------------------------------------------------------------------------
# monitor session NAME encapsulation gre metadata tx
# ( no | default ) monitor session NAME encapsulation gre metadata tx
#--------------------------------------------------------------------------------

def sessionGreMetadataCommand( mode, args ):
   disableMetadata = not CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   sessionCfg = getSessionConfig( mode, name, True )
   sessionCfg.txGreMetadataEnabled = disableMetadata

class SessionEncapsulationMetadataCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME encapsulation gre metadata tx'
   noOrDefaultSyntax = syntax + "..."
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'encapsulation': nodeEncapsulation,
      'gre': matcherGre,
      'metadata': nodeMetadata,
      'tx': CliCommand.guardedKeyword(
         'tx', helpdesc='Add metadata in tx mirrored packet. The packet will be'
         ' truncated to 128 bytes', guard=mirroringTxGreMetadataGuard )
   }

   handler = sessionGreMetadataCommand
   noOrDefaultHandler = sessionGreMetadataCommand

MirroringModelet.addCommandClass( SessionEncapsulationMetadataCmd )

#----------------------------------------------------------
# Register Mirroring show commands into "show tech-support".
#----------------------------------------------------------

def _showTechCommands():
   return [
      'show monitor session',
      'show tap aggregation groups',
      ]

# Timestamps are made up to maintain historical order within show tech-support
TechSupportCli.registerShowTechSupportCmdCallback(
   '2010-01-01 00:12:00', _showTechCommands )

def mirroringQuietDataLinkExplanation( intfName, mode ):
   for sessionStatus in gv.mirroringStatus.sessionStatus.values():
      operStatus = sessionStatus.targetOperStatus.get( intfName )
      if operStatus and operStatus.state == 'operStateActive':
         return ( 'monitoring', 'Mirroring destination', 'mirroring configuration' )
   return ( None, None, None )

IntfCli.quietDataLinkExplanationHook.addExtension(
      mirroringQuietDataLinkExplanation )

@Plugins.plugin( provides=( 'mirroring/config', ) )
def Plugin( em ):
   gv.mirroringConfig = ConfigMount.mount( em, "mirroring/config",
                                               "Mirroring::Config", "w" )
   gv.mirroringStatus = LazyMount.mount( em, "mirroring/status",
                                             "Mirroring::Status", "r" )
   gv.mirroringHwStatus = LazyMount.mount( em, "mirroring/hwstatus",
                                               "Mirroring::HwStatus", "r" )
   gv.mirroringHwCapability = LazyMount.mount( em, "mirroring/hwcapability",
                                                   "Mirroring::HwCapability", "r" )
   gv.hwCapability = LazyMount.mount( em, "routing/hardware/status",
                                          "Routing::Hardware::Status", "r" )
   gv.aclStatusDp = LazyMount.mount( em,
                                     "cell/%d/acl/status/dp" % Cell.cellId(),
                                     "Tac::Dir", "ri" )
   gv.aclConfig = LazyMount.mount( em, "acl/config/cli",
                                       "Acl::Input::Config", "r" )
   gv.mirroringHwConfig = LazyMount.mount( em, "mirroring/hwconfig",
                                               "Mirroring::Config", "r" )

   def mirrorNameFn( mode ):
      return gv.mirroringConfig.session

   def validateMirror( mode, sessionName ):
      config = gv.mirroringConfig.session.get( sessionName )
      if config and mode.session.secureMonitor() != config.secureMonitor:
         return str( CliParser.guardNotPermitted )
      return None

   AclCli.registerMirroringCallback( mirrorNameFn,
                                     validateMirror )
