#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re, Tac

from CliModel import Bool
from CliModel import Int
from CliModel import Dict
from CliModel import Enum
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from ArnetModel import IpGenericAddress
import IntfModels
import collections
from TableOutput import TableFormatter, Headings, Format
from AclLib import getAclTypeDisplayName
from AclCliModel import IpAclList, Ipv6AclList, MacAclList

class AclWithInfo( Model ):
   name = Str( help="ACL name" )
   aclType = Enum( values=Tac.Type( 'Acl::AclType' ).attributes,
                help="ACL type", optional=True )
   direction = Enum( values=Tac.Type( 'Acl::AclDirection' ).attributes,
                     help="ACL list direction", optional=True )

   def setInfo( self, aclInfo ):
      self.name = aclInfo.name
      self.aclType= aclInfo.aclType
      self.direction = aclInfo.direction
   def setInfoAndAcl( self, aclInfo, acl ):
      pass


class Ipv4AclWithInfo( AclWithInfo ):
   acl = Submodel( valueType=IpAclList, help='IPv4 access lists' )
   def setInfoAndAcl( self, aclInfo, acl ):
      self.setInfo( aclInfo )
      self.acl = acl

class Ipv6AclWithInfo( AclWithInfo ):
   acl = Submodel( valueType=Ipv6AclList, help='IPv6 access lists' )
   def setInfoAndAcl( self, aclInfo, acl ):
      self.setInfo( aclInfo )
      self.acl = acl

class MacAclWithInfo( AclWithInfo ):
   acl = Submodel( valueType=MacAclList, help='MAC access lists' )
   def setInfoAndAcl( self, aclInfo, acl ):
      self.setInfo( aclInfo )
      self.acl = acl

class MirrorSessions( Model ):
   __revision__ = 3

   class MirrorSession( Model ):
      """ Represents a MirroringSession: """
      __revision__ = 3

      class Interface( Model ):
         # Used only by the sourceInactiveInterfaces and
         # destinationInterface
         name = IntfModels.Interface( help="Interface name" )
         operState = Enum( values=( 'unknown', 'active', 'inactive' ),
                           help="The interface's current operating status." )
         srcIpGenAddr = IpGenericAddress( help="GRE Tunnel source address",
                                          optional=True ) 
         dstIpGenAddr = IpGenericAddress( help="GRE Tunnel destination address",
                                          optional=True ) 
         ttl = Int( help="TTL of GRE Tunnel", optional=True ) 
         dscp = Int( help="DSCP of GRE Tunnel", optional=True )
         greHdrProto = Int( help="protocol in GRE header", optional=True )
         vrf = Str( help="VRF of GRE Tunnel", optional=True )
         forwardingDrop = Bool( help="mirror drop packets", optional=True )
         operStateReason = Str( help="Reason for the current operating status. "
                                "Only set if the operState is inactive" )

      sourceRxInterfaces = List( valueType=str,
                                 help="Interfaces mirroring their ingress traffic" )
      sourceTxInterfaces = List( valueType=str,
                                 help="Interfaces mirroring their egress traffic" )
      sourceBothInterfaces = List( valueType=str,
                                   help="Interfaces mirroring both their ingress "
                                   "and egress traffic" )
      sourceInactiveInterfaces = List( valueType=Interface,
                                       help="Inactive interfaces configured "
                                       "to mirror traffic in this session" )
      sourceUnknownInterfaces = List( valueType=str,
                                      help="Interfaces in an unknown state" )
      sourceInterfaces = List( valueType=str,
                               help="All Interfaces configured as mirroring "
                               "sources in this session" )

      targetInterfaces = List( valueType=Interface,
                               help="Interfaces to which traffic is forwarded",
                               optional=True )

      class TunnelIntfs( Model ):
         class MemberIntfs( Model ):
            memberIntfs = List( valueType=IntfModels.Interface,
                                help="Port-Channel member interfaces" )
         tunnelIntfs = Dict( keyType=IntfModels.Interface,
                             valueType=MemberIntfs,
                             help="tunnel next hop interface" )
      greTunnels = Dict( keyType=IntfModels.Interface,
                         valueType=TunnelIntfs,
                         help="Next hop interfaces selected for GRE "
                         "tunnel destination", optional=True )
      aclType = Enum( values=Tac.Type( 'Acl::AclType' ).attributes,
                      help="ACL type", optional=True )

      class Acl( Model ):
         aclName = Str( help="ACL name" )
         aclPriority = Int( help="ACL priority", optional=True )
         aclType = Enum( values=Tac.Type( 'Acl::AclType' ).attributes,
                         help="ACL type", optional=True )
         aclDetails = Bool( help='Display the unsupported TCAM qualifiers' )
         aclUnsupportedQualifiers = List( valueType=str,
                                          help='Unsupported qualifiers',
                                          optional=True )

      class SessionAclDetails( Model ):
         ipAclList = Submodel( valueType=IpAclList, help='IPv4 access-lists',
                               optional=True )
         ipv6AclList = Submodel( valueType=Ipv6AclList, help='IPv6 access-lists',
                                 optional=True )
         macAclList = Submodel( valueType=MacAclList, help='MAC access-lists',
                                optional=True )

         def getList( self, aclType ):
            if aclType  == 'ip':
               return self.ipAclList
            elif aclType == 'ipv6':
               return self.ipv6AclList
            elif aclType == 'mac':
               return self.macAclList
            else:
               return None


      class AclsWithMirrorAction( Model ):
         ipAcls = List( valueType=Ipv4AclWithInfo,
                           help='IPv4 access lists with mirror action',
                           optional=True )
         ipv6Acls = List( valueType=Ipv6AclWithInfo,
                             help='IPv6 access lists with mirror action',
                             optional=True )
         macAcls = List( valueType=MacAclWithInfo,
                            help='MAC access lists with mirror action',
                            optional=True )

      sourceAcls = Dict( keyType=str, valueType=Acl,
                      help="The ACL applied to the source interface",
                      optional=True )
      sessionAcl = Str( help="ACL applied to the session." )
      sessionAclDetails = Submodel( valueType=SessionAclDetails,
                                    help="Session access-list details", 
                                    optional=True )
      aclsWithMirrorAction = Submodel( valueType=AclsWithMirrorAction,
                                       help="Access lists with mirror action"
                                            " rules",
                                       optional=True )
      truncate = Bool( help="Packet truncation enabled in the session." )
      truncationSize = Int( help="Packet truncation size in the session.",
                            optional=True )
      egressAclMirroredTruncation = Int( help="Set to 128 if metadata is enabled "
                                              "for the TX mirroring to GRE",
                                         optional=True )
      truncateStatus = Str( help="Status and reason for truncation in the session." )
      headerRemovalSize = Int( help="Header removal size in the session." )
      mirrorDeviceName = Str( help="The mirror-device name to use for "
                           "receiving packets mirrored to cpu.", optional=True )
      hwProgrammingStatusSupported = Bool( help="Whether platform code supports"
         " outputting hardware programming status. Can be used by CLI to indicate"
         " whether monitor sessions are programmed in hardware" )
      programmedInHw = Bool( help="Whether session is programmed in HW" )
      rxSampleRate = Int( help="Mirroring sampling rate.", optional=True )
      grePayloadType = Enum( values=( "payloadTypeNone",
                                      "payloadTypeFullPacket",
                                      "payloadTypeInnerPacket" ),
                              help="Payload type of mirrored packets in the "
                              "session.", optional=True )
      greMetadata = List( valueType=str,
                          help="Metadata elements in the GRE header." )

      greTimestampingSupported = Bool( help="Whether the hardware platform supports "
                                       "GRE timestamping. Can be used by CLI to "
                                       "determine whether or not to report "
                                       "timestamping enabled state." )
      greTimestampingEnabled = Bool( help="Whether timestamping is enabled for GRE "
                                     "mirror sessions", optional=True )

      mirrorRateLimit = Int( help="Mirroring rate limit in bps.", optional=True )
      mirrorRateLimitChip = Enum(
            values=( "per-ingress-chip", "per-egress-chip" ),
            help="Rate limit on multi-chip.",
            optional=True )
      _displayInSummary = Bool( help="Show session in summary output" )

      def doNotDisplayInSummary( self ):
         self._displayInSummary = False

      def displayInSummary( self ):
         return self._displayInSummary

      def degrade( self, dictRepr, revision ):
         if revision == 1 and 'sourceAcls' in dictRepr:
            dictRepr[ 'sourceAcls' ] = { k : v[ 'aclName' ] for ( k, v ) 
                                         in dictRepr[ 'sourceAcls' ].iteritems() }
         if ( revision == 1 or revision == 2 ) and 'greTunnels' in dictRepr:
            tmp = collections.defaultdict( lambda: collections.defaultdict( list ) )
            for k, v in dictRepr[ 'greTunnels' ].iteritems() :
               for nh, intfs in v[ 'tunnelIntfs' ].iteritems() :
                  if intfs[ 'memberIntfs' ]:
                     tmp[ k ][ 'nextHopIntfs' ].extend( intfs[ 'memberIntfs' ] )
                  else:
                     tmp[ k ][ 'nextHopIntfs' ].append( nh )
            def default_to_normal( d ):
               if isinstance( d, collections.defaultdict ):
                  d = { k : default_to_normal( v ) for k, v in d.iteritems() }
               return d

            dictRepr[ 'nextHopForTunnels' ] = default_to_normal( tmp )
         return dictRepr

   # In the following collection, the key (session name) can
   # correspond to a None value if that session doesn't exist. This
   # happens if a user asks for a non-existant session, i.e. "show
   # monitor session bad_session_name"
   sessions = Dict( valueType=MirrorSession,
                    help="Map from session name to the session details " )

   name = Str( help="The name of the specified session in the CLI command, "
               "if it was set", optional=True )
   _summary = Str( help="Summary of monitor session interfaces, unfiltered or"
                   "filtered by source or destination" )
   _detail = Str( help="Details of monitor sessions, show all details or only "
                  "session access-lists", optional=True )
   multiDestSessions = List( valueType=str,
                              help="List of sessions that have multiple active "
                              "destinations configured." )

   def render( self ):
      def shortenIntfName( intf ):
         match = re.match( 'Cpu', intf )
         if match:
            return 'Cpu'
         match = re.match( 'Ethernet(\d.*)$', intf )
         if match:
            return 'Et%s' % match.group( 1 )
   
         match = re.match( 'Port-Channel(\d.*)$', intf )
         if match:
            return 'Po%s' % match.group( 1 )

         if intf == 'Gre9998':
            return 'Gre'

         return intf

      def sessionHeader( sessionName ):
         sessHeader = ''
         sessHeader += '\n'
         sessHeader += 'Session %s \n' % sessionName
         sessHeader += '------------------------'
         sessHeader += '\n'
         return sessHeader 

      def printList( intfList, sourceAcls, sourceDirBoth=False ):
         # This function is hideous, but it existed in this form
         # originally, so that's what we'll use.
         intfLine = ''
         firstLine = True
         if intfList == []:
            print
         else:
            for intf in intfList:
               intfLine += ", "
               if isinstance( intf, MirrorSessions.MirrorSession.Interface ):
                  intfLine += shortenIntfName( intf.name.stringValue ) + \
                        ' ( ' + intf.operStateReason + ' ) '
               else:
                  intfLine += shortenIntfName( intf )
                  if intf in sourceAcls:
                     aclType = getAclTypeDisplayName( sourceAcls[ intf ].aclType )
                     intfLine += '(%s ACL: %s' % (
                           aclType,
                           sourceAcls[ intf ].aclName )
                     if sourceDirBoth:
                        intfLine += ' ingress'
                     if sourceAcls[ intf ].aclPriority > 0:
                        intfLine += ' [ prio %d ]' % (
                              sourceAcls[ intf ].aclPriority )
                     if sourceAcls[ intf ].aclUnsupportedQualifiers:
                        if sourceAcls[ intf ].aclDetails:
                           intfLine += ' unsupported qualifiers: %s)' % \
                                       ', '.join( sourceAcls[ intf ].
                                                aclUnsupportedQualifiers )
                        else:
                           intfLine += ' with unsupported qualifiers)'
                     else:
                        intfLine += ')'
               if len( intfLine ) > 31 :
                  if not firstLine:
                     print ' ' * 13,
                  intfLine = intfLine.lstrip( ',' )
                  print intfLine
                  firstLine = False
                  intfLine = ''
            if intfLine:
               if not firstLine:
                  print ' ' * 13,
               intfLine = intfLine.lstrip( ',' )
               print intfLine

      def showSession( name, sessModel, multiDestSessions ):
         print sessionHeader( name )
         if sessModel.hwProgrammingStatusSupported:
            status = 'No'
            if sessModel.programmedInHw:
               status = 'Yes'
            print "Programmed in HW: %s" % status
            print
         print 'Source Ports:'
         print
         if len( sessModel.sourceRxInterfaces ) > 0:
            print '%10s %2s' % ( 'Rx Only:', ' ' ),
            printList( sessModel.sourceRxInterfaces, sessModel.sourceAcls )
         if len( sessModel.sourceTxInterfaces ) > 0:
            print '%10s %2s' % ( 'Tx Only:', ' ' ),
            printList( sessModel.sourceTxInterfaces, sessModel.sourceAcls )
         if len( sessModel.sourceBothInterfaces ) > 0:
            print '%7s %5s' %  ( 'Both:', ' ' ),
            printList( sessModel.sourceBothInterfaces, sessModel.sourceAcls, True )
         if len( sessModel.sourceInactiveInterfaces ) > 0:
            print '%11s %1s' % ( 'Inactive:', ' ' ),
            printList( sessModel.sourceInactiveInterfaces, sessModel.sourceAcls )
         if len( sessModel.sourceUnknownInterfaces ) > 0:
            print '%11s %1s' % ( 'Unknown:', ' ' ),
            printList( sessModel.sourceUnknownInterfaces, sessModel.sourceAcls )
   
         print
   
         if not sessModel.targetInterfaces:
            print 'No destination configured'
         else:
            print 'Destination Ports:'
            print
         firstTime = True
         for targetIntf in sessModel.targetInterfaces:
            shortName = shortenIntfName( targetIntf.name.stringValue )
            if shortName.startswith( 'Gre' ) and firstTime and \
                  targetIntf.operState == "active":
               print '\t    status\t    source\t      dest TTL DSCP' + \
                     '  proto\t  VRF fwd-drop'
               firstTime = False
            print '    %s : ' % shortName,
            if targetIntf.operState == "inactive":
               print 'inactive (%s)' % targetIntf.operStateReason
            elif targetIntf.operState == "active":
               print 'active',
               if shortName == 'Cpu' and \
                     sessModel.mirrorDeviceName is not None:
                  print '(%s)' % sessModel.mirrorDeviceName
               elif shortName.startswith( 'Gre' ):
                  if targetIntf.forwardingDrop:
                     fwdDrop = "     yes"
                  else:
                     fwdDrop = "      no"
                  print "%15s %15s %3d %4d 0x%04x %10s %3s" % \
                      ( targetIntf.srcIpGenAddr, targetIntf.dstIpGenAddr,
                        targetIntf.ttl, targetIntf.dscp, targetIntf.greHdrProto,
                        targetIntf.vrf, fwdDrop )
                  # render tunnel next hop interface
                  greTunnels = sessModel.greTunnels
                  if greTunnels and shortName in greTunnels:
                     nextHopIntfStr = "      next hop interfaces: "
                     count = 0
                     tunnelIntfs = greTunnels[ shortName ].tunnelIntfs
                     sortedTunnelIntfs = sorted(
                        greTunnels[ shortName ].tunnelIntfs.keys() )
                     numNHs = len( tunnelIntfs )
                     for tunnelIntf in sortedTunnelIntfs:
                        numNHs -= 1
                        if not tunnelIntfs[ tunnelIntf ].memberIntfs:
                           nextHopIntfStr += "%s" % shortenIntfName( tunnelIntf )
                           if numNHs != 0:
                              nextHopIntfStr += ","
                           if count % 8 == 7:
                              print nextHopIntfStr
                              nextHopIntfStr = "\t\t\t   "
                           else:
                              nextHopIntfStr += " "
                           count += 1
                        else:
                           nextHopIntfStr += "%s( " % shortenIntfName( tunnelIntf )
                           members = len( tunnelIntfs[ tunnelIntf ].memberIntfs )
                           for intf in tunnelIntfs[ tunnelIntf ].memberIntfs:
                              members -= 1
                              nextHopIntfStr += "%s" % shortenIntfName( intf )
                              if members == 0:
                                 nextHopIntfStr += " )"
                                 if numNHs != 0:
                                    nextHopIntfStr += ","
                              else:
                                 nextHopIntfStr += ","
                              if count % 8 == 7:
                                 print nextHopIntfStr
                                 nextHopIntfStr = "\t\t\t   "
                              else:
                                 nextHopIntfStr += " "
                              count += 1
                     if nextHopIntfStr != "":
                        print nextHopIntfStr

               else:
                  print
            else:
               print 'unknown'
      
         sessionAcl = sessModel.sessionAcl
         if sessionAcl != '':
            print
            print ( '%s access list: %s' % ( 
               getAclTypeDisplayName( sessModel.aclType ), sessionAcl ) )
   
         if sessModel.truncate:
            print
            print 'Truncation: %s' % sessModel.truncateStatus
            if sessModel.truncationSize:
               msg = 'Nominal truncation size in bytes: %d' % \
                     sessModel.truncationSize
               print
               print msg

         if sessModel.egressAclMirroredTruncation:
            msg = 'TX Mirrored packet truncation size in bytes: %d' % \
                  sessModel.egressAclMirroredTruncation
            print
            print msg

         if sessModel.headerRemovalSize:
            msg = 'Header removal size in bytes: %d' % \
                  sessModel.headerRemovalSize
            if multiDestSessions:
               msg += '. Not Active*'
            print
            print msg

         if sessModel.rxSampleRate:
            msg = 'Sampling Rate: %d' % sessModel.rxSampleRate
            if sessModel.sourceTxInterfaces or \
               sessModel.sourceBothInterfaces:
               msg += ' (Disabled because session has non-Rx sources)'
            print
            print msg

         if sessModel.grePayloadType and \
            sessModel.grePayloadType != "payloadTypeNone":
            if sessModel.grePayloadType == 'payloadTypeFullPacket':
               grePayloadTypeCliName = 'full-packet'
            elif sessModel.grePayloadType == 'payloadTypeInnerPacket':
               grePayloadTypeCliName = 'inner-packet'
            print
            print 'GRE Payload Type: %s' % grePayloadTypeCliName

         if sessModel.greMetadata:
            msg = 'GRE Metadata: %s' % ( " ".join( sessModel.greMetadata ) )
            print
            print msg

         if sessModel.aclsWithMirrorAction:
            if sessModel.aclsWithMirrorAction.ipAcls:
               aclName = ""
               for ipAcl in sessModel.aclsWithMirrorAction.ipAcls:
                  aclName += " " + ipAcl.name
               msg = 'IPv4 access lists with mirror action:%s' % aclName
               print
               print msg

            if sessModel.aclsWithMirrorAction.ipv6Acls:
               aclName = ""
               for ipv6Acl in sessModel.aclsWithMirrorAction.ipv6Acls:
                  aclName += " " + ipv6Acl.name
               msg = 'IPv6 access lists with mirror action:%s' % aclName
               print
               print msg

            if sessModel.aclsWithMirrorAction.macAcls:
               aclName = ""
               for macAcl in sessModel.aclsWithMirrorAction.macAcls:
                  aclName += " " + macAcl.name
               msg = 'MAC access lists with mirror action:%s' % aclName
               print
               print msg

         if sessModel.greTimestampingSupported and \
            sessModel.greTimestampingEnabled is not None:
            greTimestampingModeCliName = \
               "enabled" if sessModel.greTimestampingEnabled else "disabled"
            msg = 'GRE Timestamp Mode: %s' % greTimestampingModeCliName
            print
            print msg

         if sessModel.mirrorRateLimit:
            msg = 'Rate Limit %s : %d kbps' % ( sessModel.mirrorRateLimitChip,
                     sessModel.mirrorRateLimit / 1000 )
            print
            print msg

      # Displays a summary table of the sessions. Output can be filtered to only
      # show sessions that contain certain sources or destinations. Sessions not
      # to be displayed will have displayInSummary == False
      def showSummary( self ):
         if not [ sess for sess in self.sessions
                  if self.sessions[ sess ].displayInSummary() ]:
            if self._summary == 'Source':
               print 'No sessions with any of the given sources'
               return
            elif self._summary == 'Destination':
               print 'No sessions with any of the given destinations'
               return
            else:
               print 'No sessions created'
               return
         hwProgrammingStatusSupported = False
         for session in self.sessions.itervalues():
            if session.hwProgrammingStatusSupported:
               hwProgrammingStatusSupported = True
               break

         tf = TableFormatter()
         sessionFormat = Format( justify="left", wrap=True,
                                 maxWidth=16, minWidth=1 )
         hwStatusFormat = Format( justify="left", wrap=True,
                                  maxWidth=10, minWidth=1 )
         destinationFormat = Format( justify="left", wrap=True,
                                     maxWidth=16, minWidth=1 )
         sourceFormat = Format( justify="left", wrap=True,
                                maxWidth=24, minWidth=1 )
         formats = [ sessionFormat, hwStatusFormat, destinationFormat, sourceFormat ]
         heading = [ 'Session', 'HW Status', 'Destinations', 'Sources' ]
         if not hwProgrammingStatusSupported:
            del formats[ 1:2 ]
            del heading[ 1:2 ]

         header = Headings( tuple( heading ) )
         header.doApplyHeaders( tf )
         tf.formatColumns( *formats )
         sharedFound = False

         for name in sorted( self.sessions ):
            if not self.sessions[ name ].displayInSummary():
               continue
            # Look through the other sessions to see if any of this session's
            # sources or destinations are shared. Mark the with a * if so
            otherSessions = [ self.sessions[ sess ] for sess in 
                  self.sessions if sess != name ]
            otherTargets = []
            otherSources = []
            sessionSources = []
            sessionTargets = []
            for session in otherSessions:
               otherTargets.extend( session.targetInterfaces )
               otherSources.extend( session.sourceInterfaces )

            for srcIntf in self.sessions[ name ].sourceInterfaces:
               intfName = shortenIntfName( srcIntf )
               if srcIntf in otherSources:
                  sessionSources.append( intfName + '*' )
                  sharedFound = True
               else:
                  sessionSources.append( intfName )
            for targetIntf in self.sessions[ name ].targetInterfaces:
               intfName = shortenIntfName( targetIntf.name.stringValue )
               if targetIntf in otherTargets:
                  sessionTargets.append( intfName + '*' )
                  sharedFound = True
               else:
                  sessionTargets.append( intfName )

            srcString = ', '.join( sessionSources )
            destString = ', '.join( sessionTargets )
            if hwProgrammingStatusSupported:
               status = 'Inactive'
               if self.sessions[ name ].programmedInHw:
                  status = 'Active'
               tf.newRow( name,
                          status,
                          destString,
                          srcString )
            else:
               tf.newRow( name,
                          destString,
                          srcString )
         print tf.output()
         if sharedFound:
            print '* Shared between multiple sessions'

      def showAccessListDetails( self ):
         for name, sessModel in self.sessions.items():
            typedAclList = None
            printSessionHeader = True
            if sessModel.sessionAcl and \
               sessModel.aclType and \
               sessModel.sessionAclDetails:
               typedAclList = sessModel.sessionAclDetails.getList(
                     sessModel.aclType )

            if typedAclList and typedAclList.aclList:
               print sessionHeader( name )
               printSessionHeader = False
               for acl in typedAclList.aclList:
                  print 'Session access list %s:' % sessModel.sessionAcl
                  for rule in acl.sequence:
                     rule.render()
                  print

            if sessModel.aclsWithMirrorAction:
               if printSessionHeader:
                  print sessionHeader( name )
               for ipAcl in sessModel.aclsWithMirrorAction.ipAcls:
                  print 'IPv4 access lists with mirror action: %s' % ipAcl.name
                  for acl in ipAcl.acl.aclList:
                     for rule in acl.sequence:
                        rule.render()
                  print

               for ipv6Acl in sessModel.aclsWithMirrorAction.ipv6Acls:
                  print 'IPv6 access lists with mirror action: %s' % ipv6Acl.name
                  for acl in ipv6Acl.acl.aclList:
                     for rule in acl.sequence:
                        rule.render()
                  print

               for macAcl in sessModel.aclsWithMirrorAction.macAcls:
                  print 'MAC access lists with mirror action: %s' % macAcl.name
                  for acl in macAcl.acl.aclList:
                     for rule in acl.sequence:
                        rule.render()
                  print

      if self.sessions:
         if self._summary:
            showSummary( self )
         elif self._detail == "access-list":
            showAccessListDetails( self )
         else:
            for name in sorted( self.sessions ):
               showSession( name, self.sessions[ name ], self.multiDestSessions )
               print

            hdrRemoval = any( sess.headerRemovalSize
                  for name, sess in self.sessions.iteritems() )
            if self.multiDestSessions and hdrRemoval:
               msg = '* Header removal is not active due to monitor sessions '
               msg += 'having multiple destinations: %s' % (
                        ', '.join( self.multiDestSessions ) )
               print msg
               print

      elif self.name is None:
         print 'No sessions created'
      else:
         # They asked for a specific session
         print 'Session not created'
      print

