#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.TapAggIntfCli as TapAggIntfCli
import CliPlugin.TapAggPmapCli as TapAggPmapCli
import CliPlugin.VlanCli as VlanCli
from CliPlugin.EthIntfCli import EthPhyIntf, EthPhyAutoIntfType
from CliPlugin.MirroringTapAggConfigMode import matcherType, matcherTapagg
from CliPlugin.LagIntfCli import LagAutoIntfType

from Intf.IntfRange import IntfRangeMatcher

nodeServicePolicy = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'service-policy',
      helpdesc='Configure Service Policy' ),
      canMerge=False )
matcherDefaultInterface = CliMatcher.KeywordMatcher( 'default',
      helpdesc='Configure default tap interface' )
matcherTruncationTool = CliMatcher.KeywordMatcher( 'truncation',
      helpdesc='Configure if egress packet should be truncated' )
matcherTruncationTap = CliMatcher.KeywordMatcher( 'truncation',
      helpdesc='Configure if ingress packet should be truncated' )
matcherGroupName = CliMatcher.DynamicNameMatcher( TapAggIntfCli.getGroupList,
      'Group name' )
nodeDot1Q = CliCommand.guardedKeyword( 'dot1q',
      helpdesc='Remove dot1q tag1',
      guard=TapAggIntfCli.dot1qRemoveGuard )
nodeTap = CliCommand.guardedKeyword( 'tap',
      helpdesc='Set tap mode characteristics of the interface',
      guard=TapAggIntfCli.tapaggGuard )
nodeTool = CliCommand.guardedKeyword( 'tool',
      helpdesc='Set tool mode characteristics of the interface',
      guard=TapAggIntfCli.tapaggGuard )
nodeTapagg = CliCommand.Node( matcher=matcherTapagg,
      guard=TapAggPmapCli.guardPMapTapAgg )

#--------------------------------------------------------------------------------
# [ no | default ] service-policy type tapagg input PMAPNAME
#--------------------------------------------------------------------------------
class ServicePolicyTypeTapaggInputPmapCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy type tapagg input PMAP_NAME'
   noOrDefaultSyntax = 'service-policy type tapagg input ...'
   data = {
      'service-policy' : nodeServicePolicy,
      'type' : matcherType,
      'tapagg' : nodeTapagg,
      'input' : 'Apply the policy map to ingress packets',
      'PMAP_NAME' : CliMatcher.DynamicNameMatcher(
         TapAggPmapCli.getPMapNameRuleTapAgg, 'TapAgg Policy Map Name' ),
   }

   handler = TapAggPmapCli.handleServicePolicy
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( ServicePolicyTypeTapaggInputPmapCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap default { group GROUP }
#--------------------------------------------------------------------------------
class SwitchportTapDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap default { group GROUP_NAME }'
   noOrDefaultSyntax = ( 'switchport tap default '
                         ' ( group | { group GROUP_NAME } )' )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'default' : 'Configure default tap group',
      'group': 'Set tap group for the interface',
      'GROUP_NAME' : matcherGroupName,
   }
   handler = TapAggIntfCli.handleTapGroups
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool ( group set { GROUP } | { group GROUP } )
#--------------------------------------------------------------------------------

# The 'set' keyword is obsolete as such, but supported for backward
# compatibility reasons.
# We also support new syntax of group <name> group <name> group <name>
# and old syntax of 'space' separated list but for downgrade compatibility
# reason, we generate the old syntax in 'show running-config'

class SwitchportToolCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport tool ( ( group set { GROUP_NAME } ) | '
              '( { group GROUP_NAME } ) )' )
   noOrDefaultSyntax = ( 'switchport tool ( ( group [ set ] ... ) | '
                         '( { group GROUP_NAME } ) )' )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'group' : 'Set tool group for the interface',
      'set' : 'Set tool groups for the interface',
      'GROUP_NAME' : matcherGroupName,
   }

   handler = TapAggIntfCli.handleToolGroups
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap native vlan VLAN_ID
#--------------------------------------------------------------------------------
class SwitchportTapNativeVlanVlanIdCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap native vlan VLAN_ID'
   noOrDefaultSyntax = 'switchport tap native vlan ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'native' : 'Set native VLAN when interface is in tap mode',
      'vlan' : 'Configure VLAN parameters',
      'VLAN_ID' : VlanCli.vlanIdMatcher,
   }

   handler = TapAggIntfCli.handleTapNativeVlan
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapNativeVlanVlanIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap mpls pop [ all ]
#--------------------------------------------------------------------------------
class SwitchportTapMplsPopAllCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap mpls pop all'
   noOrDefaultSyntax = 'switchport tap mpls pop [ all ]'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'mpls' : CliCommand.guardedKeyword( 'mpls',
         helpdesc='Configure MPLS parameters',
         guard=TapAggIntfCli.mplsPopGuard ),
      'pop' : 'Pop MPLS labels',
      'all' : 'Pop all MPLS labels',
   }

   handler = TapAggIntfCli.handleTapMplsPop
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapMplsPopAllCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool mpls pop [ all ]
#--------------------------------------------------------------------------------
class SwitchportToolMplsPopAllCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool mpls pop all'
   noOrDefaultSyntax = 'switchport tool mpls pop [ all ]'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'mpls' : CliCommand.guardedKeyword( 'mpls',
         helpdesc='Configure MPLS parameters',
         guard=TapAggIntfCli.mplsPopToolGuard ),
      'pop' : 'Pop MPLS labels',
      'all' : 'Pop all MPLS labels',
   }

   handler = TapAggIntfCli.handleToolMplsPop
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolMplsPopAllCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool dot1q remove outer DOT_1Q_TAG
#--------------------------------------------------------------------------------
class SwitchportToolDot1QRemoveOuterCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool dot1q remove outer DOT_1Q_TAG'
   noOrDefaultSyntax = 'switchport tool dot1q remove outer ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'dot1q' : nodeDot1Q,
      'remove' : 'Remove tag',
      'outer' : 'Remove outer tag',
      'DOT_1Q_TAG' : TapAggIntfCli.matcherDot1QTag,
   }

   handler = TapAggIntfCli.handleDot1qRemoveVlan
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolDot1QRemoveOuterCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool identity ( dot1q | qinq )
#--------------------------------------------------------------------------------
class SwitchportToolIdentityCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool identity ( dot1q | qinq )'
   noOrDefaultSyntax = 'switchport tool identity ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'identity' : 'Configure ID tagging',
      'dot1q' : 'Single dot1q tag',
      'qinq' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'qinq',
         helpdesc='Double dot1q tags' ),
         guard=TapAggIntfCli.qinqIdentityTaggingGuard ),
   }

   handler = TapAggIntfCli.handleIdentityTagging
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolIdentityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap encapsulation vxlan strip
#--------------------------------------------------------------------------------
class SwitchportTapEncapVxlanStripCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap encapsulation vxlan strip'
   noOrDefaultSyntax = syntax
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'encapsulation' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'encapsulation',
         helpdesc='Configure encapsulation parameters' ),
         guard=TapAggIntfCli.vxlanStripGuard ),
      'vxlan' : 'Configure VXLAN parameters',
      'strip' : 'Strip encapsulation header',
   }

   handler = TapAggIntfCli.handleTapVxlanStrip
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapEncapVxlanStripCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap DEFAULT_INTF interface
#                  ( ETH_INTF | PHY_AUTO_INTF | LAG_INTF )
#--------------------------------------------------------------------------------
class SwitchportTapDefaultInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport tap DEFAULT_INTF interface '
              '( ETH_INTF | PHY_AUTO_INTF | LAG_INTF )' )
   noOrDefaultSyntax = ( 'switchport tap DEFAULT_INTF interface '
                         '[ ETH_INTF | PHY_AUTO_INTF | LAG_INTF ]' )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'DEFAULT_INTF' : matcherDefaultInterface,
      'interface' : 'Set interfaces for the tap port',
      'ETH_INTF' : EthPhyIntf.ethMatcher,
      'PHY_AUTO_INTF' : IntfRangeMatcher( noSingletons=True,
         explicitIntfTypes=( EthPhyAutoIntfType, ) ),
      'LAG_INTF' : IntfRangeMatcher( noSingletons=False,
         explicitIntfTypes=( LagAutoIntfType, ) ),
   }

   handler = TapAggIntfCli.handleTapRawIntf
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapDefaultInterfaceCmd )

#--------------------------------------------------------------------------------
# switchport mode ( tap | tool | tap-tool )
#--------------------------------------------------------------------------------
class SwitchportModeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport mode ( tap | tool | tap-tool )'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'mode' : VlanCli.modeNode,
      'tap' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'tap',
         helpdesc='Set switchport mode to TAP unconditionally' ),
         guard=TapAggIntfCli.tapaggGuard ),
      'tool' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'tool',
         helpdesc='Set switchport mode to TOOL unconditionally' ),
         guard=TapAggIntfCli.tapaggGuard ),
      'tap-tool' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'tap-tool',
         helpdesc='Set switchport mode to both Tap and Tool' ),
         guard=TapAggIntfCli.tapToolPortGuard ),
   }
   handler = TapAggIntfCli.setSwitchportTapToolMode

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportModeCmd )

#--------------------------------------------------------------------------------
# switchport tap truncation
#--------------------------------------------------------------------------------
class SwitchportTapTruncationCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap truncation'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tap': nodeTap,
      'truncation': CliCommand.Node(
         matcher=matcherTruncationTap,
         guard=TapAggIntfCli.truncationNoSizePerIngressPortGuard ),
   }
   handler = TapAggIntfCli.handleTapTruncation
TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapTruncationCmd )

#--------------------------------------------------------------------------------
# switchport tap truncation SIZE
# [ no | default ] switchport tap truncation ...
#--------------------------------------------------------------------------------
class SwitchportTapTruncationSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap truncation SIZE'
   noOrDefaultSyntax = 'switchport tap truncation ...'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tap': nodeTap,
      'truncation': CliCommand.Node(
         matcher=matcherTruncationTap,
         guard=TapAggIntfCli.ingressTruncationGuard,
         canMerge=False ),
      'SIZE': CliCommand.Node(
         matcher=TapAggIntfCli.matcherTapTruncationSize,
         guard=TapAggIntfCli.truncationSizePerIngressPortGuard ),
   }
   handler = TapAggIntfCli.handleTapTruncation
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapTruncationSizeCmd )

#--------------------------------------------------------------------------------
# switchport tool truncation
#--------------------------------------------------------------------------------
class SwitchportToolTruncationCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool truncation'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tool': nodeTool,
      'truncation': CliCommand.Node(
         matcher=matcherTruncationTool,
         guard=TapAggIntfCli.truncationNoSizePerEgressPortGuard ),
   }
   handler = TapAggIntfCli.handleToolTruncation
TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolTruncationCmd )

#--------------------------------------------------------------------------------
# switchport tool truncation SIZE
# [ no | default ] switchport tool truncation ...
#--------------------------------------------------------------------------------
class SwitchportToolTruncationSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool truncation SIZE'
   noOrDefaultSyntax = 'switchport tool truncation ...'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tool': nodeTool,
      'truncation': CliCommand.Node(
         matcher=matcherTruncationTool,
         guard=TapAggIntfCli.egressTruncationGuard,
         canMerge=False ),
      'SIZE': CliCommand.Node(
         matcher=TapAggIntfCli.matcherToolTruncationSize,
         guard=TapAggIntfCli.truncationSizePerEgressPortGuard ),
   }
   handler = TapAggIntfCli.handleToolTruncation
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolTruncationSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap identity PORT_ID [ inner INNER_PORT_ID ]
#--------------------------------------------------------------------------------
class SwitchportTapIdentityPortIdCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap identity PORT_ID [ inner INNER_PORT_ID ]'
   noOrDefaultSyntax = 'switchport tap identity ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'identity' : 'Configure tap port ID tag',
      'PORT_ID' : CliMatcher.IntegerMatcher( 1, 4094, helpdesc='Tap port ID tag' ),
      'inner' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'inner',
         helpdesc='Configure inner tap port ID tag' ),
         guard=TapAggIntfCli.qinqIdentityTaggingGuard ),
      'INNER_PORT_ID' : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc='Inner tap port ID tag' ),
   }

   handler = TapAggIntfCli.handleTapIdentificationVlan
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapIdentityPortIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport ( tap | tool ) allowed vlan ( all | none |
#                                                 [ add | remove | except ] VLANS )
#--------------------------------------------------------------------------------
class SwitchportTapToolAllowedVlanCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport ( tap | tool ) allowed vlan '
              '( all | none | ( [ OP ] VLANS ) ) ' )
   noOrDefaultSyntax = 'switchport ( tap | tool ) allowed vlan ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'tool' : nodeTool,
      'allowed' : 'Configure allowed VLANs when interface is in tap or tool mode',
      'vlan' : 'Configure VLAN parameters',
      'all' : 'All VLANs',
      'none' : 'No VLANs',
      'OP' : VlanCli.allowedVlanOpMatcher,
      'VLANS' : VlanCli.tapToolAllowedVlanIdRangeMatcher,
   }

   handler = VlanCli.handleSwitchportModeAllowedVlan
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapToolAllowedVlanCmd )
