#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Cell
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliRules as AclCliRules
from CliPlugin.PolicyMapModelImpl import ClassMapModelContainer
from CliPlugin.PolicyMapModelImpl import registerCmapStatusFilter
import CliPlugin.PolicyMapCliLib as PolicyMapCliLib
import CliPlugin.PolicyMapModel as PolicyMapModel
from CliPlugin.TapAggIntfCli import guardMacAcls
import ConfigMount
import LazyMount
import PolicyMap
import Tracing

t0 = Tracing.trace0

gv = CliGlobal.CliGlobal( dict(
   aclStatus=None,
   tapAggHwStatus=None,
   tapAggPmapConfig=None,
   tapAggPmapStatus=None,
   tapAggStatusRequestDir=None,
) )

def guardIp6Acls( mode, token ):
   if not gv.aclStatus.dpIp6AclSupported:
      return CliParser.guardNotThisPlatform
   elif not gv.tapAggHwStatus.pmapIpv6Supported:
      return CliParser.guardNotThisPlatform
   else:
      return None

matcherAccessGroup = CliMatcher.KeywordMatcher( 'access-group',
      helpdesc='Match with given access group' )
matcherMatch = CliMatcher.KeywordMatcher( 'match',
      helpdesc='Match the access rule specified' )

matcherMacAccessGroupType = CliCommand.guardedKeyword(
   'mac', helpdesc='Specify MAC Access-groups',
   guard=guardMacAcls,
   alias='TYPE',
)
matcherIpv6AccessGroupType = CliCommand.guardedKeyword(
   'ipv6', helpdesc='Specify IPv6 Access-groups',
   guard=guardIp6Acls,
   alias='TYPE',
)

#-------------------------------------------------------------------------------
# class-map mode
# Command to enter the mode:
# [no] class-map type tapagg match-any <name>
#-------------------------------------------------------------------------------
class ClassMapModeTapAgg( PolicyMapCliLib.ClassMapMode ):
   name = "Class Map TapAgg Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      PolicyMapCliLib.ClassMapMode.__init__( self, parent, session,
                                             context, 'tapagg' )

def showClassMap( mode, args ):
   mapType = 'tapagg'
   cmapName = args.get( 'CMAPNAME' )
   cmapAllModel = PolicyMapModel.ClassMapAllModel()
   emapType = PolicyMap.mapTypeToEnum( mapType )
   cmapsContainer = ClassMapModelContainer( gv.tapAggPmapConfig,
                                            gv.tapAggPmapStatus, emapType,
                                            cmapAllModel )
   if cmapName is None:
      cmapsContainer.populateAll()
   else:
      cmapsContainer.populateClassMap( cmapName )
   return cmapAllModel

def gotoClassMapMode( mode, args ):
   cmapName = args.get( 'CMAPNAME' )
   mapType = 'tapagg'
   matchType = 'match-any'
   context = None
   t0( 'gotoClassMapMode', cmapName )
   emapType = PolicyMap.mapTypeToEnum( mapType )
   if context is None:
      context = PolicyMapCliLib.ClassMapContext( gv.tapAggPmapConfig,
                                                 gv.tapAggStatusRequestDir,
                                                 gv.tapAggPmapStatus, emapType,
                                                 cmapName, matchType )
   if cmapName in gv.tapAggPmapConfig.cmapType.cmap:
      entry = gv.tapAggPmapConfig.cmapType.cmap[ cmapName ]
      context.copyEditCmap( entry )
   else:
      context.newEditCmap()

   mode.cmapContext = context
   childMode = mode.childMode( ClassMapModeTapAgg, context=context )
   mode.session_.gotoChildMode( childMode )

def deleteClassMap( mode, args ):
   cmapName = args.get( 'CMAPNAME' )
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, PolicyMapCliLib.ClassMapMode ) and
        lastMode.cmapContext and
        lastMode.cmapContext.cmapName() == cmapName ):
      # deleting self inside the config mode, so discard the config
      # since we'll succeed
      lastMode.cmapContext = None

   PolicyMapCliLib.deleteClassMap( mode, cmapName, gv.tapAggPmapConfig,
                                   gv.tapAggStatusRequestDir, gv.tapAggPmapStatus )

#--------------------------------------------------------------------------------
# Handlers
#--------------------------------------------------------------------------------

def setMatchValue( mode, args ):
   seqnum = args.get( 'SEQNUM' )
   accessGroupType = args[ 'TYPE' ]
   accessGroupName = args[ 'NAME' ]
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, seqnum, accessGroupType, accessGroupName )

#--------------------------------------------------------------------------------
# [ no | default ] [ SEQNUM ] match ( ( ip access-group IPGROUP )
#                                     | ( ipv6 access-group IPV6GROUP )
#                                     | ( mac access-group MACGROUP ) )
#--------------------------------------------------------------------------------
class MatchIpIpv6MacAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = """[ SEQNUM ] match ( ( ip access-group IP_ACCESSGROUPNAME )
                                  | ( ipv6 access-group IPV6_ACCESSGROUPNAME )
                                  | ( mac access-group MAC_ACCESSGROUPNAME ) )"""
   noOrDefaultSyntax = syntax
   data = {
      'SEQNUM' : AclCli.seqnumMatcher,
      'match' : matcherMatch,
      # TYPE
      'ip' : CliCommand.Node( matcher=PolicyMapCliLib.matcherIpAccessGroupType,
         alias='TYPE' ),
      'ipv6' : matcherIpv6AccessGroupType,
      'mac' : matcherMacAccessGroupType,
      'access-group' : matcherAccessGroup,
      # NAME
      'IP_ACCESSGROUPNAME' : CliCommand.Node(
         matcher=AclCli.notStandardIpAclNameMatcher,
         alias='NAME' ),
      'IPV6_ACCESSGROUPNAME' : CliCommand.Node(
         matcher=AclCli.notStandardIp6AclNameMatcher,
         alias='NAME' ),
      'MAC_ACCESSGROUPNAME' : CliCommand.Node(
         matcher=AclCli.notStandardMacAclNameMatcher,
         alias='NAME' ),
   }

   handler = setMatchValue
   noOrDefaultHandler = handler

ClassMapModeTapAgg.addCommandClass( MatchIpIpv6MacAccessGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] SEQUENCES
#--------------------------------------------------------------------------------
class TapAggNoSequenceCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'SEQUENCES'
   data = {
      'SEQUENCES': AclCliRules.seqRangeMatcher
   }
   noOrDefaultHandler = ClassMapModeTapAgg.handleNoSeq

ClassMapModeTapAgg.addCommandClass( TapAggNoSequenceCmd )

#--------------------------------------------------------------------------------
# resequence [ START [ INC ] ]
#--------------------------------------------------------------------------------
class ResequenceCmd( CliCommand.CliCommandClass ):
   syntax = 'resequence [ START [ INC ] ]'
   data = {
      'resequence' : PolicyMapCliLib.matcherResequence,
      'START' : PolicyMapCliLib.matcherStartSequence,
      'INC' : PolicyMapCliLib.matcherIncSequence,
   }

   @staticmethod
   def handler( mode, args ):
      start = args.get( 'START', PolicyMapCliLib.defaultSeqStart )
      inc = args.get( 'INC', PolicyMapCliLib.defaultSeqInc )
      mode.resequence( start, inc )

ClassMapModeTapAgg.addCommandClass( ResequenceCmd )

def Plugin( entityManager ):
   # Acl change status callback that returns the impact of an Acl change on
   # routing policy.
   def isPolicyHwStatusOk( mode, aclName, aclType, aclUpdateType ):
      rc = True
      err = {}
      err[ 'error' ] = ''
      if not mode.session_.startupConfig() and AclCli.allHwStatusEnabled():
         rc, result = policyOpChkr.verify( 'acl', aclName )
         if not rc:
            err[ 'error' ] = ( result.error if result else
                               'unknown TagAgg policy status' )
         else:
            if result and result.warning:
               err[ 'warning' ] = "Warning: ACL %s %s" % ( aclName, result.warning )
      return rc, err

   gv.aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                   "Acl::Status", "r" )
   gv.tapAggHwStatus = LazyMount.mount( entityManager, 'tapagg/hwstatus',
                                        'TapAgg::HwStatus', 'r' )
   tapAggCellRootNode = 'cell/%d/tapagg' % Cell.cellId()
   tapAggStatusPath = tapAggCellRootNode + '/pmapstatus'
   gv.tapAggPmapStatus = LazyMount.mount( entityManager, tapAggStatusPath,
                                          "Tac::Dir", 'ri' )
   gv.tapAggPmapConfig = ConfigMount.mount( entityManager, 'tapagg/pmapconfig',
                                            'TapAgg::PmapConfig', 'w' )
   gv.tapAggStatusRequestDir = (
         LazyMount.mount( entityManager, 'tapagg/pmapStatusRequestDir',
                          'PolicyMap::PolicyMapStatusRequestDir', 'w') )
   # Intall a callback to verify impact of acl changes on policy
   policyOpChkr = PolicyMapCliLib.PolicyOpChkr( gv.tapAggStatusRequestDir,
                                                gv.tapAggPmapStatus )
   AclCli.registerAclConfigValidCallback( isPolicyHwStatusOk )
   registerCmapStatusFilter( PolicyMap.tapaggMapType, ( lambda x: x ) )
