# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import ConfigMount
import CliMatcher
import LazyMount
from CliMode.McsClient import McsServiceClientMode
from CliMode.McsClient import McsServiceClientSecondaryCvxMode
import CliCommand
import CliParser
from CliPlugin.ControllerClientCvxMgmtConfigMode import HeartbeatIntervalCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import HeartbeatTimeoutCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import ServerHostnameCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import SourceInterfaceCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import SslProfileNameCmd
from CliPlugin.ControllerClientCvxMgmtConfigMode import VrfNameCmd
from CliPlugin import ControllerClient
from CliPlugin.ControllerClientCvxMgmtConfigMode import ShutdownCmd as \
      CvxMgmtShutdownCmd

mcsClientConfig = None
mcsClientStatus = None
mcsClientControllerConfig = None
mcsClientControllerStatus = None
serviceConfigDir = None
secondaryClientConfig = None
secondaryClientStatus = None
secondarySysdbConfig = None
secondarySysdbStatus = None

# pylint: disable=protected-access


class McsServiceClientConfigMode( McsServiceClientMode,
      BasicCli.ConfigModeBase ):
   name = "MCS client configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name=None ):
      McsServiceClientMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def enableService( self, args ):
      enable = CliCommand.isNoCmd( args )
      mcsClientConfig.enabled = enable
      serviceConfigDir.service[ 'Mcs' ].enabled = enable

class McsServiceClientSecondaryCvxConfigMode( McsServiceClientSecondaryCvxMode,
      BasicCli.ConfigModeBase ):
   name = "Secondary Cvx Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, clusterName=None ):
      McsServiceClientSecondaryCvxMode.__init__( self, clusterName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------
# (config)#mcs client
# (config-mcs-client)#
#-----------------------------------------
class McsClientCmd( CliCommand.CliCommandClass ):
   syntax = 'mcs client'
   data = {
      'mcs': 'Configure Media Control Service',
      'client': 'Configure Media Control Service Client',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( McsServiceClientConfigMode )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( McsClientCmd )

#-----------------------------------------
# [ no | default ] shutdown
#-----------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown Media Control Service Client',
   }
   handler = McsServiceClientConfigMode.enableService
   noOrDefaultHandler = handler

McsServiceClientConfigMode.addCommandClass( ShutdownCmd )

#-----------------------------------------
# (config-mcs-client)# cvx secondary [name]
# (config-mcs-client-cvx)#
#-----------------------------------------
class McsClientSecondaryCvxCmd( CliCommand.CliCommandClass ):
   syntax = 'cvx secondary [ NAME ]'
   noOrDefaultSyntax = 'cvx secondary ...'
   data = {
      'cvx': 'CVX Cluster',
      'secondary': 'Configure Secondary CVX Cluster',
      'NAME': CliMatcher.StringMatcher( helpdesc='Name of cvx cluster' )
   }

   @staticmethod
   def handler( mode, args ):
      clusterName = args.get( 'NAME' )
      mcsClientControllerConfig.clusterName = clusterName or \
            ControllerClient.clusterDefaultName
      childMode = mode.childMode( McsServiceClientSecondaryCvxConfigMode,
            clusterName=clusterName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pass


McsServiceClientConfigMode.addCommandClass( McsClientSecondaryCvxCmd )

# Under this (config-mcs-client-cvx)# we will register
# the cvx related cli commands

# Secondary shutdown command
#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class SecondaryCvxShutdownCmd( CliCommand.CliCommandClass ):
   syntax = CvxMgmtShutdownCmd.syntax
   noOrDefaultSyntax = CvxMgmtShutdownCmd.syntax
   data = CvxMgmtShutdownCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      mcsClientControllerConfig.enabled = False

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      mcsClientControllerConfig.enabled = True

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondaryCvxShutdownCmd )

# Secondary heartbeat interval
#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-interval INTERVAL
#--------------------------------------------------------------------------------
class SecondaryCvxHeartbeatIntervalCmd( CliCommand.CliCommandClass ):
   syntax = HeartbeatIntervalCmd.syntax
   noOrDefaultSyntax = HeartbeatIntervalCmd.noOrDefaultSyntax
   data = HeartbeatIntervalCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      return HeartbeatIntervalCmd._setHeartBeatPunchInterval(
            mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondaryCvxHeartbeatIntervalCmd )

# Secondary heartbeat timeout
#--------------------------------------------------------------------------------
# [ no | default ] heartbeat-timeout TIMEOUT
#--------------------------------------------------------------------------------
class SecondaryCvxHeartbeatTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = HeartbeatTimeoutCmd.syntax
   noOrDefaultSyntax = HeartbeatTimeoutCmd.noOrDefaultSyntax
   data = HeartbeatTimeoutCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      return HeartbeatTimeoutCmd._setHeartBeatTimeout(
            mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondaryCvxHeartbeatTimeoutCmd )

# Secondary server host HOSTNAME
#--------------------------------------------------------------------------------
# [ no | default ] server host HOSTNAME [ sysname SYSNAME ]
#--------------------------------------------------------------------------------
class SecondaryServerHostnameCmd( CliCommand.CliCommandClass ):
   syntax = ServerHostnameCmd.syntax
   noOrDefaultSyntax = ServerHostnameCmd.noOrDefaultSyntax
   data = ServerHostnameCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      return ServerHostnameCmd._setServerHostname(
            mode, args, mcsClientControllerConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return ServerHostnameCmd._setNoServerHostname(
            mode, args, mcsClientControllerConfig )

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondaryServerHostnameCmd )

# Secondary source-interface
#--------------------------------------------------------------------------------
# [ no | default ] source-interface INTF ...
#--------------------------------------------------------------------------------
class SecondarySourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = SourceInterfaceCmd.syntax
   noOrDefaultSyntax = SourceInterfaceCmd.noOrDefaultSyntax
   data = SourceInterfaceCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      return SourceInterfaceCmd._setSourceIntf(
            mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondarySourceInterfaceCmd )

# Secondary ssl profile
#--------------------------------------------------------------------------------
# [ no | default ] ssl profile SSLPROFILENAME
#--------------------------------------------------------------------------------
class SecondarySslProfileCmd( CliCommand.CliCommandClass ):
   syntax = SslProfileNameCmd.syntax
   noOrDefaultSyntax = SslProfileNameCmd.noOrDefaultSyntax
   data = SslProfileNameCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      return SslProfileNameCmd._setSslProfileName(
            mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondarySslProfileCmd )

# Secondary vrf
#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class SecondaryVrfNameCmd( CliCommand.CliCommandClass ):
   syntax = VrfNameCmd.syntax
   noOrDefaultSyntax = VrfNameCmd.noOrDefaultSyntax
   data = VrfNameCmd.data.copy()

   @staticmethod
   def handler( mode, args ):
      return VrfNameCmd._setVrfName(
            mode, args, mcsClientControllerConfig )

   noOrDefaultHandler = handler

McsServiceClientSecondaryCvxConfigMode.addCommandClass(
      SecondaryVrfNameCmd )

def Plugin( entityManager ):
   global mcsClientConfig
   global mcsClientStatus
   global mcsClientControllerConfig
   global mcsClientControllerStatus
   global serviceConfigDir
   global secondaryClientConfig
   global secondaryClientStatus
   global secondarySysdbConfig
   global secondarySysdbStatus

   mcsClientConfig = ConfigMount.mount( entityManager,
         "mcs/switch/config", "Mcs::Client::Config", "w" )
   mcsClientStatus = LazyMount.mount( entityManager,
         "mcs/switch/status", "Mcs::Client::Status", "r" )
   mcsClientControllerConfig = ConfigMount.mount( entityManager,
         "mcs/switch/controller/config", "ControllerClient::Config", "w" )
   mcsClientControllerStatus = LazyMount.mount( entityManager,
         "mcs/switch/controller/status", "ControllerClient::Status", "r" )
   serviceConfigDir = ConfigMount.mount( entityManager,
         "mgmt/controller/service/config", "Controller::ServiceConfigDir", "w" )
   secondarySysdbConfig = LazyMount.mount( entityManager,
         "mcs/switch/controller/publish/config",
         "ControllerClient::SysdbConfig", "r" )
   secondarySysdbStatus = LazyMount.mount( entityManager,
         "mcs/switch/controller/publish/status",
         "ControllerClient::SysdbStatus", "r" )
   ControllerClient.enableSecondaryCvxHook(
         mcsClientControllerConfig,
         mcsClientControllerStatus,
         secondarySysdbConfig,
         secondarySysdbStatus )
