# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
import LazyMount
from CliMode.Mcs import McsServiceMode
import BasicCli
import CliCommand
import CliParser
from CliPlugin.ControllerCli import CvxConfigMode
from CliPlugin.ControllerCli import serviceKwMatcher
from ReversibleSecretCli import reversibleSecretCliExpression

mcsCliConfig = None
mcsStatus = None
httpServiceConfig = None
serviceInputConfigDir = None
controllerConfig = None

CVX_SERVICE_NAME = "Mcs"
HTTP_SERVICE_NAME = "mcs"

class McsServiceConfigMode( McsServiceMode, BasicCli.ConfigModeBase ):
   name = "MCS service configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name=None ):
      McsServiceMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def enableService( self, args ):
      enable = CliCommand.isNoCmd( args )
      mcsCliConfig.enabled = enable
      httpServiceConfig.service[ HTTP_SERVICE_NAME ].enabled = enable
      serviceInputConfigDir.service[ CVX_SERVICE_NAME ].enabled = enable
      if enable and not controllerConfig.enabled:
         self.addWarning( "MCS service will not start when CVX is shutdown" )

#-----------------------------------------
# (config-cvx)# service mcs
# (config-cvx-mcs)#
#-----------------------------------------
class ServiceMcsCmd( CliCommand.CliCommandClass ):
   syntax = 'service mcs'
   data = {
      'service': serviceKwMatcher,
      'mcs': 'Configure Media Control Service',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( McsServiceConfigMode )
      mode.session_.gotoChildMode( childMode )

CvxConfigMode.addCommandClass( ServiceMcsCmd )

#-----------------------------------------
# [ no | default ] shutdown
#-----------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown Media Control Service',
   }

   handler = McsServiceConfigMode.enableService
   noOrDefaultHandler = handler

McsServiceConfigMode.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# [no] redis password <secret>
#-------------------------------------------------------------------------------
class RedisPasswordCommand( CliCommand.CliCommandClass ):
   syntax = "redis password SECRET"
   noOrDefaultSyntax = "redis password ..."
   data = {
      "redis": "Configure Redis settings",
      "password": "Configure Redis password",
      "SECRET": reversibleSecretCliExpression( "SECRET" )
      }

   @staticmethod
   def handler( mode, args ):
      mcsCliConfig.redisPassword = args[ "SECRET" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mcsCliConfig.redisPassword = ""

McsServiceConfigMode.addCommandClass( RedisPasswordCommand )

def Plugin( entityManager ):
   global mcsCliConfig
   global mcsStatus
   global httpServiceConfig
   global serviceInputConfigDir
   global controllerConfig

   mcsCliConfig = ConfigMount.mount(
         entityManager, "mcs/config/cli", "Mcs::CliConfig", "w" )
   mcsStatus = LazyMount.mount(
         entityManager, "mcs/status/agent", "Mcs::AgentStatus", "r" )
   httpServiceConfig = ConfigMount.mount(
         entityManager, "mgmt/capi/config", "HttpService::Config", "w" )
   serviceInputConfigDir = ConfigMount.mount(
         entityManager, "controller/service/config",
                              "Controller::ServiceConfigDir", "w" )
   controllerConfig = LazyMount.mount(
         entityManager, "controller/config", "Controllerdb::Config", "r" )
