#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliParser
import CliToken.Ip
import ConfigMount, LazyMount
import IraVrfCli
from IraIpRouteCliLib import preferenceRangeMatcher
import McastCommonCliLib
from RouterMulticastCliLib import legacyCliCallback
import VlanIntfCli
from CliMatcher import KeywordMatcher
from CliCommand import Node, CliCommandClass

_routingHardwareStatus = None
_mcastHostInject4Config = None
_mcastHostInject6Config = None
_ipStatus = None
_ip6Status = None
_ipConfig = None
_ip6Config = None
_routing6HwStatus = None

AddressFamily = Tac.Type( "Arnet::AddressFamily" )

def mcastRoutingSupportedGuard( mode, token ):
   if McastCommonCliLib.mcastRoutingSupported(
         mode.sysdbRoot, _routingHardwareStatus ):
      return None
   return CliParser.guardNotThisPlatform

def mcastv6RoutingSupportedGuard( mode=None, token=None ):
   if getattr( _routing6HwStatus, 'multicastRoutingSupported' ):
      return None
   return CliParser.guardNotThisPlatform

def _getMcastCommonIntfConfigColl( vrfName, af ):
   ''' Used to check if creating a new multicast interface config will exceed the
       maximum allowed VIF limit for each VRF. '''
   allMcastIntfs = {}

   if af == AddressFamily.ipv4:
      for cKey in _mcastHostInject4Config.config:
         intfId = cKey.intfId
         if vrfName == IraVrfCli.getVrfNameFromIntf( _ipConfig, _ipStatus, intfId ):
            allMcastIntfs[ intfId ] = _mcastHostInject4Config.config[ cKey ]
   else:
      for cKey in _mcastHostInject6Config.config:
         intfId = cKey.intfId
         if vrfName == IraVrfCli.getVrfNameFromIntf( _ip6Config, _ip6Status, intfId,
                                                     af=AddressFamily.ipv6 ):
            allMcastIntfs[ intfId ] = _mcastHostInject6Config.config[ cKey ]

   return allMcastIntfs

modelet = VlanIntfCli.VlanIntfModelet

multicastKwMatcher = KeywordMatcher( 'multicast',
   helpdesc='Multicast routing commands' )
multicastLegacyNode = Node( matcher=multicastKwMatcher,
   guard=mcastRoutingSupportedGuard )
multicastNode = Node( matcher=multicastKwMatcher,
   guard=McastCommonCliLib.mcastGenRoutingSupportedGuard )
sourceKwMatcher = KeywordMatcher( 'source', helpdesc='Source of (S,G) traffic' )
sourceDeprecatedNode = Node( matcher=sourceKwMatcher )

routeKwMatcher = KeywordMatcher( 'route', helpdesc='Source /32 route' )
exportKwMatcher = KeywordMatcher( 'export', helpdesc='Export routes' )

def setSourceRouteExport( mode, args, legacy=False ):
   pref = args.get( 'PREF' )
   # The below prefix and length attributes are just to satisfy the ConfigKey
   # requirements. This will not be used in the HostInject processing.
   prefix = Tac.newInstance( "Arnet::IpGenPrefix", "0.0.0.0/0" )
   length = 32
   cKey = Tac.newInstance( "HostInject::ConfigKey", mode.intf.name, prefix )
   if args.get( 'AF' ) == 'ipv6':
      hic = _mcastHostInject6Config.newConfig( cKey, length )
   else:
      hic = _mcastHostInject4Config.newConfig( cKey, length )
   if pref:
      hic.pref = Tac.Value( "Routing::RoutePreference", pref )

def noSourceRouteExport( mode, args, legacy=False ):
   # The below prefix attribute is ust to satisfy the ConfigKey requirements.
   # This will not be used in the HostInject processing.
   prefix = Tac.newInstance( "Arnet::IpGenPrefix", "0.0.0.0/0" )
   cKey = Tac.newInstance( "HostInject::ConfigKey", mode.intf.name, prefix )
   if args.get( 'AF' ) == 'ipv6':
      if cKey in _mcastHostInject6Config.config:
         del _mcastHostInject6Config.config[ cKey ]
   else:
      if cKey in _mcastHostInject4Config.config:
         del _mcastHostInject4Config.config[ cKey ]

noSourceRouteExportLegacy = legacyCliCallback( noSourceRouteExport )
setSourceRouteExportLegacy = legacyCliCallback( setSourceRouteExport )

class LegacySourceExport( CliCommandClass ):
   syntax = 'ip multicast source route export [ PREF ]'
   noOrDefaultSyntax = 'ip multicast source route export ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'multicast': multicastLegacyNode,
      'source': sourceDeprecatedNode,
      'route': routeKwMatcher,
      'export': exportKwMatcher,
      'PREF': preferenceRangeMatcher
   }
   handler = setSourceRouteExportLegacy
   noOrDefaultHandler = noSourceRouteExportLegacy
modelet.addCommandClass( LegacySourceExport )

class SourceExport( CliCommandClass ):
   syntax = 'multicast IP_FAMILY source route export [ PREF ]'
   noOrDefaultSyntax = 'multicast IP_FAMILY source route export ...'
   data = {
      'multicast': multicastNode,
      'IP_FAMILY': McastCommonCliLib.IpFamilyExpression(),
      'source': sourceKwMatcher,
      'route': routeKwMatcher,
      'export': exportKwMatcher,
      'PREF': preferenceRangeMatcher
   }
   handler = setSourceRouteExport
   noOrDefaultHandler = noSourceRouteExport
modelet.addCommandClass( SourceExport )

def Plugin( entityManager ):
   global _ipConfig, _ipStatus, _ip6Config, _ip6Status
   global _mcastHostInject4Config
   global _mcastHostInject6Config
   global _routingHardwareStatus
   global _routing6HwStatus

   McastCommonCliLib.mcastIfCollHook.addExtension( _getMcastCommonIntfConfigColl )

   _routingHardwareStatus = LazyMount.mount( entityManager,
         "routing/hardware/status", "Routing::Hardware::Status", "r" )
   _routing6HwStatus = LazyMount.mount( entityManager,
         "routing6/hardware/status", "Routing6::Hardware::Status", "r" )
   _ipConfig = LazyMount.mount( entityManager, 'ip/config', 'Ip::Config', 'r' )
   _ip6Config = LazyMount.mount( entityManager, 'ip6/config', 'Ip6::Config', 'r' )
   _ipStatus = LazyMount.mount( entityManager, 'ip/status', 'Ip::Status', 'r' )
   _ip6Status = LazyMount.mount( entityManager, 'ip6/status', 'Ip6::Status', 'r' )
   _mcastHostInject4Config = ConfigMount.mount( entityManager,
         "routing/attached-host/mcast/config", "HostInject::HostInjectConfig",
         "w" )
   _mcastHostInject6Config = ConfigMount.mount( entityManager,
         "routing6/attached-host/mcast/config", "HostInject::HostInjectConfig",
         "w" )

