#/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliParser, IraIpIntfCli, IraIpRouteCliLib, IntfCli
import IraIp6RouteCliLib
import Tac
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import IpGenAddrMatcher
import ConfigMount, LazyMount
import CliToken.Clear, CliToken.Ip
from CliMatcher import KeywordMatcher, DynamicNameMatcher
from CliCommand import Node, CliCommandClass
import Arnet
from Arnet import IpGenPrefix
from McastCommonCliLib import AddressFamily, validateMulticastAddress, \
                              mcastGenRoutingSupportedGuard, \
                              mcastRoutingSupportedGuard, \
                              mcast6RoutingSupportedGuard, \
                              mcastRoutingBoundarySupportedGuard, \
                              mcast6RoutingBoundarySupportedGuard

IpAddress = IpAddrMatcher.Arnet.IpAddress

BoundaryPrefix = Tac.Type( 'Routing::McastBoundary::MulticastBoundaryPrefix' )
BoundaryAcl = Tac.Type( 'Routing::McastBoundary::MulticastBoundaryAcl' )

mcastBoundaryConfig = None
mcast6BoundaryConfig = None
aclConfig = None
routingHwStatus = None
routing6HwStatus = None


def applyMask( ipAddr, mask, base=True ):
   """Returns a specific IP Address resulting from applying the
   32-bit mask to the given IP Address ipAddr.
   If base is true, returns the all-zeros address of the mask,
   otherwise returns the all ones address of the mask."""
   if base:
      return IpAddress( 0xFFFFFFFF & ( ipAddr.value & mask ) )
   else:
      return IpAddress( 0xFFFFFFFF & ( ipAddr.value | ~mask ) )

def isMulticast( addr ):
   addrStr = addr if type( addr ) is str else str( addr )
   return validateMulticastAddress( addrStr ) is None

class McastBoundaryIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      name = self.intf_.name
      del mcastBoundaryConfig.intfConfig[ name ]
      del mcast6BoundaryConfig.intfConfig[ name ]

def _getOrCreateMcastBoundaryIntfConfig( mode, af=AddressFamily.ipv4,
                                         printError=True ):
   if af == AddressFamily.ipv6:
      boundaryConfig = mcast6BoundaryConfig
   else:
      boundaryConfig = mcastBoundaryConfig
   intfConfig = boundaryConfig.intfConfig.get( mode.intf.name )
   if not intfConfig:
      intfConfig = boundaryConfig.newIntfConfig( mode.intf.name )
   return intfConfig

def deleteMcastBoundaryIntfConfig( mode, af=AddressFamily.ipv4, printError = True ):
   intfName = mode.intf.name
   if af == AddressFamily.ipv6:
      boundaryConfig = mcast6BoundaryConfig
   else:
      boundaryConfig = mcastBoundaryConfig
   intfConfig = boundaryConfig.intfConfig.get( intfName )
   if not intfConfig:
      return
   if( ( len( intfConfig.boundary ) == 0 ) and
         ( intfConfig.boundaryAcl.acl == '' ) ):
      del boundaryConfig.intfConfig[ intfName ]

modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

def cmdIpMulticastBoundary( mode, args, legacy=False ):
   groupRange = args.get( 'PREFIX' )
   ipFamily = args.get( 'ipv6' )
   out = args.get( 'out' )

   af = AddressFamily.ipv6 if ipFamily else AddressFamily.ipv4

   if groupRange is None:
      mode.addError( "Source range mask must be contiguous" )
      return
   if af == AddressFamily.ipv6:
      if not IraIp6RouteCliLib.isValidIpv6PrefixWithError(
            mode, groupRange.v6AddrWithMask ):
         mode.addError( "Masked bits of source range must be zero" )
         return
   else:
      if not IraIpRouteCliLib.isValidPrefix( str(groupRange) ):
         mode.addError( "Masked bits of source range must be zero" )
         return
   address = Arnet.IpGenAddrWithMask( str(groupRange) ).ipGenAddr
   err = validateMulticastAddress( address )
   if err != None:
      mode.addError( err )
      return
   intfConfig = _getOrCreateMcastBoundaryIntfConfig( mode, af=af )
   if intfConfig:
      intfConfig.enabled = True
      if intfConfig.boundaryAcl.acl != '':
         mode.addError( "Multicast boundary Acl has been configured already." )
         return
      intfConfig.boundary.addMember( 
            BoundaryPrefix( IpGenPrefix( str(groupRange) ), bool(out) ) )

def cmdNoIpMulticastBoundary( mode, args, legacy=False ):
   if 'PREFIX' in args:
      groupRange = args.get( 'PREFIX' )
   else:
      groupRange = False
   ipFamily = args.get( 'ipv6' )
   af = AddressFamily.ipv6 if ipFamily else AddressFamily.ipv4

   if af == AddressFamily.ipv6:
      boundaryConfig = mcast6BoundaryConfig
   else:
      boundaryConfig = mcastBoundaryConfig
   intfConfig = boundaryConfig.intfConfig.get( mode.intf.name )
   if intfConfig:
      intfConfig.enabled = False
      if groupRange is None:
         mode.addError( "Group range mask must be contiguous" )
         return
      if groupRange is not False:
         if af == AddressFamily.ipv6:
            if not IraIp6RouteCliLib.isValidIpv6PrefixWithError(
                  mode, groupRange.v6AddrWithMask ):
               mode.addError( "Masked bits of source range must be zero" )
               return
         else:
            if not IraIpRouteCliLib.isValidPrefix( str(groupRange) ):
               mode.addError( "Masked bits of group range must be zero" )
               return
         del intfConfig.boundary[ IpGenPrefix( str(groupRange) ) ]
      else:
         for i in intfConfig.boundary.keys():
            del intfConfig.boundary[ i ]
      deleteMcastBoundaryIntfConfig( mode, af=af )

def _boundaryGuard( mode, token ):
   if routingHwStatus.multicastBoundarySupported:
      return None
   return CliParser.guardNotThisPlatform

multicastKwMatcher = KeywordMatcher( 'multicast',
                                     helpdesc='Multicast routing commands' )
tokenMulticast = Node( matcher=multicastKwMatcher,
                       guard=mcastGenRoutingSupportedGuard )

#Some platforms don't support ip multicast boundary alone without
#out option.

tokenBoundary = KeywordMatcher(
   'boundary', helpdesc='Multicast Boundary' )

tokenBoundaryNotGuarded = Node( tokenBoundary, canMerge=False )

tokenBoundaryGuarded = Node( tokenBoundary, guard=_boundaryGuard, canMerge=False )

tokenOut = KeywordMatcher(
   'out', helpdesc='only apply to outgoing traffic' )

ipv4KwMatcher = KeywordMatcher( 'ipv4', helpdesc='IPv4 version' )
ipv4Node = Node( matcher=ipv4KwMatcher,
                 guard=mcastRoutingSupportedGuard )
ipv4BoundaryGuardNode = Node( matcher=ipv4KwMatcher,
                 guard=mcastRoutingBoundarySupportedGuard )

ipv6KwMatcher = KeywordMatcher( 'ipv6', helpdesc='IPv6 version' )
ipv6Node = Node( matcher=ipv6KwMatcher,
                 guard=mcast6RoutingSupportedGuard )
ipv6BoundaryGuardNode = Node( matcher=ipv6KwMatcher,
                 guard=mcast6RoutingBoundarySupportedGuard )

ipGenPrefixMatcher = IpGenAddrMatcher.IpGenPrefixMatcher(
      'group address range with prefix', addrWithMask=True )

class LegacyMulticastBoundary( CliCommandClass ):
   syntax = 'ip multicast boundary PREFIX'
   noOrDefaultSyntax = 'ip multicast boundary [ PREFIX ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'multicast': tokenMulticast,
      'boundary': tokenBoundaryGuarded,
      'PREFIX': ipGenPrefixMatcher
   }
   handler = cmdIpMulticastBoundary
   noOrDefaultHandler = cmdNoIpMulticastBoundary
modelet.addCommandClass( LegacyMulticastBoundary )

class LegacyMulticastBoundaryOut( CliCommandClass ):
   #If the command has "out", it is valid on all platforms
   syntax = 'ip multicast boundary PREFIX out'
   noOrDefaultSyntax = 'ip multicast boundary [ PREFIX ] ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'multicast': tokenMulticast,
      'boundary': tokenBoundaryNotGuarded,
      'PREFIX': ipGenPrefixMatcher,
      'out': tokenOut
   }
   handler = cmdIpMulticastBoundary
   noOrDefaultHandler = cmdNoIpMulticastBoundary
modelet.addCommandClass( LegacyMulticastBoundaryOut )

class MulticastBoundary( CliCommandClass ):
   syntax = 'multicast ( ipv4 | ipv6 ) boundary PREFIX'
   noOrDefaultSyntax = 'multicast ( ipv4 | ipv6 ) boundary [ PREFIX ]'
   data = {
      'multicast': tokenMulticast,
      'ipv4': ipv4BoundaryGuardNode,
      'ipv6': ipv6BoundaryGuardNode,
      'boundary': tokenBoundary,
      'PREFIX': ipGenPrefixMatcher
   }
   handler = cmdIpMulticastBoundary
   noOrDefaultHandler = cmdNoIpMulticastBoundary
modelet.addCommandClass( MulticastBoundary )

class MulticastBoundaryOut( CliCommandClass ):
   #If the command has "out", it is valid on all platforms
   syntax = 'multicast ( ipv4 | ipv6 ) boundary PREFIX out'
   noOrDefaultSyntax = 'multicast ( ipv4 | ipv6 ) boundary [ PREFIX ] ...'
   data = {
      'multicast': tokenMulticast,
      'ipv4': ipv4Node,
      'ipv6': ipv6Node,
      'boundary': tokenBoundary,
      'PREFIX': ipGenPrefixMatcher,
      'out': tokenOut
   }
   handler = cmdIpMulticastBoundary
   noOrDefaultHandler = cmdNoIpMulticastBoundary
modelet.addCommandClass( MulticastBoundaryOut )


# Examines the fitness of an acl IpRuleConfig as a boundaryAcl range.
def checkBoundaryAclRule( rule, af=AddressFamily.ipv4 ):
   # Ip addr/prefix len
   if af == AddressFamily.ipv6:
      source = rule.filter.source
      srcAddr = source.address
      srcmask = 0xFFFFFFFF & source.len
      # pylint: disable=E1103
      if srcmask != 0xFFFFFFFF and not source.validAsPrefix:
         return (False, "Group address %s mask must be contiguous" % \
                        ( source.stringValue ) )
      # pylint: enable=E1103
      if not source.subnet.address.isMulticast:
         return ( False, "Group address %s is invalid multicast address" % \
                  source.stringValue )
   else:
      source = rule.filter.source
      srcAddr = IpAddress( source.address )
      srcmask = 0xFFFFFFFF & source.mask
      # pylint: disable=E1103
      if srcmask != 0xFFFFFFFF and \
            not Arnet.AddrWithFullMask( str( srcAddr ), srcmask ).validAsPrefix:
         return (False, "Group address %s mask must be contiguous" % \
                        ( source.stringValue ) )
      # pylint: enable=E1103
      srcmin = applyMask( srcAddr, srcmask )
      srcmax = applyMask( srcAddr, srcmask, base=False ) 
      if not( isMulticast( srcmin ) and isMulticast( srcmax )  ) :
         return ( False, "Group address %s is invalid multicast address" % \
                  source.stringValue )
   return ( True, "Group address %s is valid" % source.stringValue )

def checkAclValidAsMulticastBoundary( aclName, af=AddressFamily.ipv4 ):
   if af == AddressFamily.ipv6:
      acl = aclConfig.config['ipv6'].acl[ aclName ]
   else:
      acl = aclConfig.config['ip'].acl[ aclName ]
   aclCurrCfg = acl.currCfg
   if not aclCurrCfg or not acl.standard:
      return ( False, "not a standard Acl" )
   for seq, uid in aclCurrCfg.ruleBySequence.iteritems():
      if af == AddressFamily.ipv6:
         maybeRet = checkBoundaryAclRule( aclCurrCfg.ip6RuleById[ uid ], af=af )
      else:
         maybeRet = checkBoundaryAclRule( aclCurrCfg.ipRuleById[ uid ], af=af )
      if not maybeRet[0]:
         return ( False, "Seq no %d: %s" % ( seq, maybeRet[1] ) )
   return ( True, None )

def cmdIpMulticastBoundaryAcl( mode, args, legacy=False ):
   name = args.get( 'ACL' )
   ipFamily = args.get( 'ipv6' )
   out = args.get( 'out' )

   af = AddressFamily.ipv6 if ipFamily else AddressFamily.ipv4

   if af == AddressFamily.ipv6:
      aclNames = aclConfig.config['ipv6'].acl
   else:
      aclNames = aclConfig.config['ip'].acl
   if not name in aclNames:
      mode.addWarning( "Acl %s not configured. Assigning anyway." 
                       % name)
      intfConfig = _getOrCreateMcastBoundaryIntfConfig( mode, af=af )
      if intfConfig:
         intfConfig.enabled = True
         intfConfig.boundaryAcl = BoundaryAcl( name, bool( out ) )
      return
   valid = checkAclValidAsMulticastBoundary( name, af=af )
   if not valid[0]:
      mode.addError( "Acl %s contains rules invalid as "
                     "multicast boundary specifications."
                     "\nError: %s " % ( name, valid[1] ) )
      return
   intfConfig = _getOrCreateMcastBoundaryIntfConfig( mode, af=af )
   if intfConfig:
      if len( intfConfig.boundary ) > 0:
         mode.addError( "Multicast boundary specifications "
                        "have been configured already." )
         return
      intfConfig.enabled =  True
      intfConfig.boundaryAcl = BoundaryAcl( name, bool( out ) )


def cmdNoIpMulticastBoundaryAcl( mode, args, legacy=False ):
   name = args.get( 'ACL' )
   ipFamily = args.get( 'ipv6' )

   af = AddressFamily.ipv6 if ipFamily else AddressFamily.ipv4

   intfConfig = _getOrCreateMcastBoundaryIntfConfig( mode, af=af )
   if intfConfig:
      if intfConfig.boundaryAcl.acl != name:
         mode.addError( "Multicast boundary Acl %s"
                        "has not been configured." % name )
         return
      intfConfig.enabled = False
      intfConfig.boundaryAcl = BoundaryAcl( '', False )
      deleteMcastBoundaryIntfConfig( mode, af=af )

standardAclNameMatcher = DynamicNameMatcher( 
   lambda mode: ( ( x for x, v in aclConfig.config['ip'].acl.iteritems()
                       if v.standard ) +
                  ( x for x, v in aclConfig.config['ipv6'].acl.iteritems()
                       if v.standard ) ),
                  helpdesc='Access-list name', helpname='name' )

class LegacyMulticastBoundaryAcl( CliCommandClass ):
   syntax = 'ip multicast boundary ACL'
   noOrDefaultSyntax = 'ip multicast boundary ACL'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'multicast': tokenMulticast,
      'boundary': tokenBoundaryGuarded,
      'ACL': standardAclNameMatcher
   }
   handler = cmdIpMulticastBoundaryAcl
   noOrDefaultHandler = cmdNoIpMulticastBoundaryAcl
modelet.addCommandClass( LegacyMulticastBoundaryAcl )

class LegacyMulticastBoundaryAclOut( CliCommandClass ):
   #If the command has "out", it is valid on all platforms
   syntax = 'ip multicast boundary ACL out'
   noOrDefaultSyntax = 'ip multicast boundary ACL ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'multicast': tokenMulticast,
      'boundary': tokenBoundaryNotGuarded,
      'ACL': standardAclNameMatcher,
      'out': tokenOut
   }
   handler = cmdIpMulticastBoundaryAcl
   noOrDefaultHandler = cmdNoIpMulticastBoundaryAcl
modelet.addCommandClass( LegacyMulticastBoundaryAclOut )

class MulticastBoundaryAcl( CliCommandClass ):
   syntax = 'multicast ( ipv4 | ipv6 ) boundary ACL'
   noOrDefaultSyntax = 'multicast ( ipv4 | ipv6 ) boundary ACL'
   data = {
      'multicast': tokenMulticast,
      'ipv4': ipv4BoundaryGuardNode,
      'ipv6': ipv6BoundaryGuardNode,
      'boundary': tokenBoundary,
      'ACL': standardAclNameMatcher
   }
   handler = cmdIpMulticastBoundaryAcl
   noOrDefaultHandler = cmdNoIpMulticastBoundaryAcl
modelet.addCommandClass( MulticastBoundaryAcl )

class MulticastBoundaryAclOut( CliCommandClass ):
   #If the command has "out", it is valid on all platforms
   syntax = 'multicast ( ipv4 | ipv6 ) boundary ACL out'
   noOrDefaultSyntax = 'multicast ( ipv4 | ipv6 ) boundary ACL ...'
   data = {
      'multicast': tokenMulticast,
      'ipv4': ipv4Node,
      'ipv6': ipv6Node,
      'boundary': tokenBoundary,
      'ACL': standardAclNameMatcher,
      'out': tokenOut
   }
   handler = cmdIpMulticastBoundaryAcl
   noOrDefaultHandler = cmdNoIpMulticastBoundaryAcl
modelet.addCommandClass( MulticastBoundaryAclOut )

#Plugin

def Plugin( entityManager ):
   global mcastBoundaryConfig, aclConfig, routingHwStatus
   global mcast6BoundaryConfig, routing6HwStatus

   mcastBoundaryConfig = ConfigMount.mount( entityManager, 
         'routing/mcastboundary/config', 'Routing::McastBoundary::Config', 'w' )
   mcast6BoundaryConfig = ConfigMount.mount( entityManager,
         'routing6/mcastboundary/config', 'Routing::McastBoundary::Config', 'w' )

   routingHwStatus = LazyMount.mount( entityManager, 
         'routing/hardware/status', 
         'Routing::Hardware::Status', 'r' )
   routing6HwStatus = LazyMount.mount( entityManager,
         'routing6/hardware/status',
         'Routing6::Hardware::Status', 'r' )


   #priority 22 - one higher than pim
   IntfCli.Intf.registerDependentClass( McastBoundaryIntf, priority=22 )

   aclConfig = LazyMount.mount( entityManager, "acl/config/cli", 
                                "Acl::Input::Config", "r" )

