# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpm MatchList-lib

import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
import CliMode.MatchListCliMode as MatchListCliMode
import CliParser
import Arnet
from MultiRangeRule import MultiRangeMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
from MatchListModels import MatchListInfo
import ShowCommand
import Tac
import itertools
import LazyMount

# --------------------------------------------------------
# Global definitions
# --------------------------------------------------------
AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
MatchStringInfo = Tac.Type( "MatchList::MatchStringInfo" )
MatchStringType = Tac.Type( "MatchList::MatchStringType" )
matchListConfig = None
hwCapability = None

# max seqence number (32-bit)
MAX_SEQ = 0xFFFFFFFF

# --------------------------------------------------------
# Utility functions
# --------------------------------------------------------
def stringSupportGuard( mode, token ):
   if hwCapability.matchListStringSupported:
      return None
   return CliParser.guardNotThisPlatform

def ipv6PrefixSupportGuard( mode, token ):
   if hwCapability.ipv6PrefixSupported:
      return None
   return CliParser.guardNotThisPlatform

def getMatchListModelInfo( matchListName, addressFamily ):
   matchlistInfo = MatchListInfo()
   if addressFamily == AddressFamily.ipv4:
      matchLists = matchListConfig.matchIpv4PrefixList
   else:
      matchLists = matchListConfig.matchIpv6PrefixList

   def populateMatchListModel( mlName ):
      matchListModel = matchlistInfo.MatchList()
      matchListModel.prefixes = []
      mlConfig = matchLists[ mlName ]
      for prefix in mlConfig.subnets:
         matchListModel.prefixes.append( prefix.stringValue )
      matchlistInfo.matchLists[ mlName ] = matchListModel

   if not matchListName:
      # If MatchList name is not given, show all configured
      # lists
      for mlName in matchLists:
         populateMatchListModel( mlName )
   elif matchListName in matchLists:
      # If MatchList name is given
      populateMatchListModel( matchListName )

   return matchlistInfo
   
# Don't allow single or double quotes, since the current use case is feeding
# this into rsyslog conf file which would put quotes around the expression,
# and then we would need to enforce proper string escaping for quotes
regexMatcher = CliMatcher.StringMatcher( helpname='REGEXP',
                                         pattern=r'[^"\']+',
                                     helpdesc="Regular expression (POSIX ERE)" )

def matchListNames( mode, context ):
   inputType = context.sharedResult[ '<inputType>' ]
   if inputType == 'string':
      return matchListConfig.matchStringList
   elif inputType == 'prefix-ipv4':
      return matchListConfig.matchIpv4PrefixList
   elif inputType == 'prefix-ipv6':
      return matchListConfig.matchIpv6PrefixList
   return itertools.chain( matchListConfig.matchStringList,
                           matchListConfig.matchIpv4List,
                           matchListConfig.matchIpv6List )

tokenMatchListName = \
      CliMatcher.DynamicNameMatcher( matchListNames, 'Name of match list',
                                     passContext=True )
#----------------------------------------------------------------------------
# Match-list CLI
#----------------------------------------------------------------------------
class MatchListStringCommandClass( MatchListCliMode.MatchListMode,
                                   BasicCli.ConfigModeBase ):
   name = "Match list configuration for strings"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, matchListName ):
      self.matchListName = matchListName
      MatchListCliMode.MatchListMode.__init__( self, self.matchListName,
                                                     "string" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class MatchListIpv4PrefixCommandClass( MatchListCliMode.MatchListMode,
                                     BasicCli.ConfigModeBase ):
   name = "Match list configuration for IPv4 prefixes"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, matchListName ):
      self.matchListName = matchListName
      MatchListCliMode.MatchListMode.__init__( self, self.matchListName,
                                                     "prefix-ipv4" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class MatchListIpv6PrefixCommandClass( MatchListCliMode.MatchListMode,
                                     BasicCli.ConfigModeBase ):
   name = "Match list configuration for IPv6 prefixes"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, matchListName ):
      self.matchListName = matchListName
      MatchListCliMode.MatchListMode.__init__( self, self.matchListName,
                                                     "prefix-ipv6" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterMatchListStringCommandClass( CliCommand.CliCommandClass ):
   syntax = "match-list input <inputType> <matchListName>"
   noOrDefaultSyntax = syntax

   _inputTypes = { 'string' : MatchListStringCommandClass }
   _inputTypeFn = lambda mode: { 'string' : 'String input' }

   data = { "match-list": "Configure a match list to filter data",
            "input": "Data input type",
            "<inputType>": CliCommand.Node(
               CliMatcher.DynamicKeywordMatcher( _inputTypeFn ),
               storeSharedResult=True,
               guard=stringSupportGuard ),
            "<matchListName>": tokenMatchListName }

   @staticmethod
   def handler( mode, args ):
      matchListName = args[ '<matchListName>' ]
      inputType = args[ '<inputType>' ]
      childMode = mode.childMode(
         EnterMatchListStringCommandClass._inputTypes[ inputType ],
         matchListName=matchListName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      matchListName = args[ "<matchListName>" ]
      inputType = args[ '<inputType>' ]
      if inputType == 'string':
         del matchListConfig.matchStringList[ matchListName ]

BasicCli.GlobalConfigMode.addCommandClass( EnterMatchListStringCommandClass )

class EnterMatchListIpv4PrefixCommandClass( CliCommand.CliCommandClass ):
   syntax = "match-list input <inputType> <matchListName>"
   noOrDefaultSyntax = syntax

   _inputTypes = { 'prefix-ipv4' : MatchListIpv4PrefixCommandClass }
   _inputTypeFn = lambda mode: { 'prefix-ipv4' : 'IPv4 prefix' }

   data = { "match-list": "Configure a match list to filter data",
            "input": "Data input type",
            "<inputType>": CliCommand.Node(
               CliMatcher.DynamicKeywordMatcher( _inputTypeFn ),
               storeSharedResult=True ),
            "<matchListName>": tokenMatchListName }

   @staticmethod
   def handler( mode, args ):
      matchListName = args[ '<matchListName>' ]
      inputType = args[ '<inputType>' ]
      childMode = mode.childMode(
         EnterMatchListIpv4PrefixCommandClass._inputTypes[ inputType ],
         matchListName=matchListName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      matchListName = args[ "<matchListName>" ]
      inputType = args[ '<inputType>' ]
      if inputType == 'prefix-ipv4':
         del matchListConfig.matchIpv4PrefixList[ matchListName ]

BasicCli.GlobalConfigMode.addCommandClass( EnterMatchListIpv4PrefixCommandClass ) 

class EnterMatchListIpv6PrefixCommandClass( CliCommand.CliCommandClass ):
   syntax = "match-list input <inputType> <matchListName>"
   noOrDefaultSyntax = syntax

   _inputTypes = { 'prefix-ipv6' : MatchListIpv6PrefixCommandClass }
   _inputTypeFn = lambda mode: { 'prefix-ipv6' : 'IPv6 prefix' }

   data = { "match-list": "Configure a match list to filter data",
            "input": "Data input type",
            "<inputType>": CliCommand.Node(
               CliMatcher.DynamicKeywordMatcher( _inputTypeFn ),
               storeSharedResult=True, guard=ipv6PrefixSupportGuard ),
            "<matchListName>": tokenMatchListName }

   @staticmethod
   def handler( mode, args ):
      matchListName = args[ '<matchListName>' ]
      inputType = args[ '<inputType>' ]
      childMode = mode.childMode(
         EnterMatchListIpv6PrefixCommandClass._inputTypes[ inputType ],
         matchListName=matchListName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      matchListName = args[ "<matchListName>" ]
      inputType = args[ '<inputType>' ]
      if inputType == 'prefix-ipv6':
         del matchListConfig.matchIpv6PrefixList[ matchListName ]

BasicCli.GlobalConfigMode.addCommandClass( EnterMatchListIpv6PrefixCommandClass )

def handleMatchRegex( mode, regex, seqNumber=None ):
   regexValidator = Tac.newInstance( 'MatchList::RegexValidator' )
   if not regexValidator.validatePosixRegex( regex ):
      mode.addError( 'Invalid regex' )
      return
   matchStringList = matchListConfig.newMatchStringList( mode.matchListName )
   matchInfo = MatchStringInfo( MatchStringType.regex, regex )
   # User specified a sequence number. As with ACLs, we accept duplicate rules if
   # the sequence number is unused. If the sequence number is taken, reject if
   # it's different, and ignore if same (in this case the reactor won't trigger 
   # so set it anyway)
   if seqNumber:
      if ( seqNumber in matchStringList.matchInfo and 
           matchStringList.matchInfo[ seqNumber ] != matchInfo ):
         mode.addError( 'Error: Duplicate sequence number' )
         return
      matchStringList.matchInfo[ seqNumber ] = matchInfo
      if matchStringList.lastSeqNo < seqNumber:
         matchStringList.lastSeqNo = seqNumber
   # Generate our own sequence number. Duplicate rules are silently ignored.
   else:
      newSeq = matchStringList.lastSeqNo + 10
      if newSeq > MAX_SEQ:
         mode.addError( 'Error: Sequence number out of range' )
         return
      if matchInfo not in matchStringList.matchInfo.values():
         matchStringList.matchInfo[ newSeq ] = matchInfo
         matchStringList.lastSeqNo = newSeq

#------------------------------------------------------------------------------
# The "[no|default] match regex REGEX command in
# "match-list input string" mode
#------------------------------------------------------------------------------
class MatchRegex( CliCommand.CliCommandClass ):
   syntax = "match regex REGEX"
   noOrDefaultSyntax = syntax

   data = { "match": "Configure matching",
            "regex": "Match using a regular expression",
            "REGEX": regexMatcher }

   @staticmethod
   def handler( mode, args ):
      regex = args[ "REGEX" ]
      handleMatchRegex( mode, regex )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      regex = args[ "REGEX" ]
      matchStringList = matchListConfig.matchStringList.get( mode.matchListName )
      if not matchStringList:
         return
      matchInfo = matchStringList.matchInfo
      prevSeqNo = 0
      for seqNo, match in matchInfo.iteritems():
         if match.string == regex:
            del matchInfo[ seqNo ]
            if seqNo == matchStringList.lastSeqNo:
               matchStringList.lastSeqNo = prevSeqNo
            break # As with ACLs, we just remove the first match
         prevSeqNo = seqNo

MatchListStringCommandClass.addCommandClass( MatchRegex )

#------------------------------------------------------------------------------
# The "[no] SEQNO match regex REGEX command in
# "match-list input string" mode
#------------------------------------------------------------------------------
class MatchRegexSeqNo( CliCommand.CliCommandClass ):
   syntax = "SEQNO match regex REGEX"
   noOrDefaultSyntax = "SEQ_RANGE"

   data = { "match": "Configure matching",
            "regex": "Match using a regular expression",
            "REGEX": regexMatcher,
            "SEQNO": CliMatcher.IntegerMatcher( 1, MAX_SEQ,
                                                helpdesc='Index in the sequence' ),
            "SEQ_RANGE": MultiRangeMatcher( lambda: ( 1, MAX_SEQ ), False,
                                            'Index in the sequence' )
            }

   @staticmethod
   def handler( mode, args ):
      regex = args[ "REGEX" ]
      seqNo = args[ "SEQNO" ]
      handleMatchRegex( mode, regex, seqNo )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      seqRanges = args[ "SEQ_RANGE" ].ranges()
      matchStringList = matchListConfig.matchStringList.get( mode.matchListName )
      if not matchStringList:
         return
      matchInfo = matchStringList.matchInfo
      for start, end in seqRanges:
         for seqNo in matchInfo:
            if seqNo > end:
               break
            if seqNo >= start:
               del matchInfo[ seqNo ]
      if not matchInfo:
         matchStringList.lastSeqNo = 0
      elif matchStringList.lastSeqNo not in matchInfo:
         matchStringList.lastSeqNo = matchInfo.keys()[ -1 ]

MatchListStringCommandClass.addCommandClass( MatchRegexSeqNo )

#------------------------------------------------------------------------------
# The "resequence [starting] [increment]"  command in 
# "match-list input string" mode
#------------------------------------------------------------------------------
class ResequenceSeqNo( CliCommand.CliCommandClass ):
   syntax = "resequence [ START [ INC ] ]"

   data = { "resequence": "Resequence the list",
            "START": CliMatcher.IntegerMatcher( 1, MAX_SEQ,
                       helpdesc='Starting sequence number (default 10)' ),
            "INC": CliMatcher.IntegerMatcher( 1, MAX_SEQ,
                  helpdesc='Step to increment the sequence number (default 10)' )
   }

   @staticmethod
   def handler( mode, args ):
      start = args.get( 'START', 10 )
      inc = args.get( 'INC', 10 )
      matchStringList = matchListConfig.matchStringList.get( mode.matchListName )
      if not matchStringList:
         return
      matchInfo = matchStringList.matchInfo
      if not matchInfo:
         return
      lastSeqNo = start + ( len( matchInfo ) - 1 ) * inc
      if lastSeqNo > MAX_SEQ:
         mode.addError( 'Error: Sequence number out of range' )
         return
      # Store matchInfos in a temp dictionary
      tempMatchInfo = {}
      sequence = start
      for match in matchInfo.values():
         tempMatchInfo[ sequence ] = match
         sequence += inc
      # Clear matchInfo list and put them all back
      matchInfo.clear()
      for seqNo, match in tempMatchInfo.iteritems():
         matchInfo[ seqNo ] = match
      matchStringList.lastSeqNo = lastSeqNo

MatchListStringCommandClass.addCommandClass( ResequenceSeqNo )

#------------------------------------------------------------------------------
# The "[no] match prefix ipv4 <P>/<M> command in
# "match-list input prefix ipv4" mode
#------------------------------------------------------------------------------
class MatchIpv4Subnets( CliCommand.CliCommandClass ):
   syntax = "match prefix-ipv4 { PREFIX_LIST }"
   noOrDefaultSyntax = syntax

   data = { "match": "Configure matching",
            "prefix-ipv4": "Match IPv4 prefix",
            "PREFIX_LIST":  IpAddrMatcher.ipPrefixMatcher }

   @staticmethod
   def handler( mode, args ):
      matchPrefixList = matchListConfig.newMatchIpv4PrefixList( mode.matchListName )
      prefixes = args[ "PREFIX_LIST" ]
      for prefix in prefixes:
         matchPrefixList.subnets[ Arnet.AddrWithMask( prefix ).subnet ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      matchPrefixList = matchListConfig.matchIpv4PrefixList.get(
         mode.matchListName )
      if matchPrefixList:
         prefixes = args[ "PREFIX_LIST" ]
         for prefix in prefixes:
            del matchPrefixList.subnets[ Arnet.AddrWithMask( prefix ).subnet ]

MatchListIpv4PrefixCommandClass.addCommandClass( MatchIpv4Subnets )

#------------------------------------------------------------------------------
# The "[no] match prefix ipv6 <P>/<M> command in
# "match-list input prefix ipv6" mode
#------------------------------------------------------------------------------
class MatchIpv6Subnets( CliCommand.CliCommandClass ):
   syntax = "match prefix-ipv6 { PREFIX_LIST }"
   noOrDefaultSyntax = syntax

   data = { "match": "Configure matching",
            "prefix-ipv6": "Match IPv6 prefix",
            "PREFIX_LIST":  Ip6AddrMatcher.ip6PrefixMatcher }

   @staticmethod
   def handler( mode, args ):
      matchPrefixList = matchListConfig.newMatchIpv6PrefixList( mode.matchListName )
      prefixes = args[ "PREFIX_LIST" ]
      for prefix in prefixes:
         matchPrefixList.subnets[ Arnet.Ip6AddrWithMask( prefix ) ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      matchPrefixList = matchListConfig.matchIpv6PrefixList.get(
         mode.matchListName )
      if matchPrefixList:
         prefixes = args[ "PREFIX_LIST" ]
         for prefix in prefixes:
            del matchPrefixList.subnets[ Arnet.Ip6AddrWithMask( prefix ) ]

MatchListIpv6PrefixCommandClass.addCommandClass( MatchIpv6Subnets )

#------------------------------------------------------------------------------
# show match-list prefix-ipv4 [ name ]
#------------------------------------------------------------------------------
def getAllIpv4MatchListNames( mode ):
   return matchListConfig.matchIpv4PrefixList

ipv4MatchListNameMatcher = CliMatcher.DynamicNameMatcher(
      getAllIpv4MatchListNames,
      helpdesc='Match list name' )

class ShowIpv4MatchList( ShowCommand.ShowCliCommandClass ):
   syntax = 'show match-list prefix-ipv4 [ MATCHLIST_NAME ]'
   data = {
         'match-list': 'Show match list information',
         'prefix-ipv4': 'IPv4 prefixes',
         'MATCHLIST_NAME': ipv4MatchListNameMatcher
   }
   cliModel = MatchListInfo

   @staticmethod
   def handler( mode, args ):
      matchListName = args.get( 'MATCHLIST_NAME' )
      return getMatchListModelInfo( matchListName, AddressFamily.ipv4 )

BasicCli.addShowCommandClass( ShowIpv4MatchList )

#------------------------------------------------------------------------------
# show match-list prefix-ipv6 [ name ]
#------------------------------------------------------------------------------
def getAllIpv6MatchListNames( mode ):
   return matchListConfig.matchIpv6PrefixList

ipv6MatchListNameMatcher = CliMatcher.DynamicNameMatcher(
      getAllIpv6MatchListNames,
      helpdesc='Match list name' )

class ShowIpv6MatchList( ShowCommand.ShowCliCommandClass ):
   syntax = 'show match-list prefix-ipv6 [ MATCHLIST_NAME ]'
   data = {
         'match-list': 'Show match list information',
         'prefix-ipv6': 'IPv6 prefixes',
         'MATCHLIST_NAME': ipv6MatchListNameMatcher
   }
   cliModel = MatchListInfo

   @staticmethod
   def handler( mode, args ):
      matchListName = args.get( 'MATCHLIST_NAME' )
      return getMatchListModelInfo( matchListName, AddressFamily.ipv6 )

BasicCli.addShowCommandClass( ShowIpv6MatchList )


def Plugin( entityManager ):
   global matchListConfig, hwCapability
   matchListConfig = ConfigMount.mount( entityManager, "matchlist/config/cli",
                                        "MatchList::Config", "w" )
   hwCapability = LazyMount.mount( entityManager, "matchlist/hw/capability",
                                   "MatchList::HwCapability", "r" )
