# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from collections import defaultdict
import threading

import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.XcvrAllStatusDir as XcvrAllStatusDir
import LazyMount
import Tac
from TypeFuture import TacLazyType

mrmCache = threading.local()

IntfSwState = TacLazyType( "AlePhy::IntfSwState::IntfSwState" )
IntfHwState = TacLazyType( "AlePhy::IntfHwState::IntfHwState" )
EthIntfId = TacLazyType( "Arnet::EthIntfId" )

ResourceType = TacLazyType( "AlePhy::ResourceType" )
LogicalPortAllocationPriority = TacLazyType(
   "AlePhy::LogicalPortAllocationPriority::LogicalPortAllocationPriority" )

allIntfConfigDir = None
allIntfStatusDir = None
xcvrAllStatusProxy = None
resourceManagerSliceConfig = None

def getMrm( resourceConsumerDir ):
   """
   Parameters
   ----------
   resourceConsumerDir : AlePhy::ResourceConsumerDir

   Returns
   -------
   An instance of AlePhy::MasterResourceManager that computes states for all the
   interfaces in resourceConsumerDir.
   """
   if not resourceConsumerDir:
      return None

   rcdToMrm = getattr( mrmCache, 'rcdToMrm', None )
   if not rcdToMrm:
      mrmCache.rcdToMrm = {}

   mrm = mrmCache.rcdToMrm.get( resourceConsumerDir )

   if not mrm:
      mrm = MasterResourceManager( resourceConsumerDir )
      mrmCache.rcdToMrm[ resourceConsumerDir ] = mrm

   return mrm

def getRmcFromRcd( resourceConsumerDir ):
   """Retrieve AlePhy::ResourceManagerConfig object in Sysdb for the same subslice
   as the one in the given resourceConsumerDir.

   Parameters
   ----------
   resourceConsumerDir : AlePhy::ResourceConsumerDir

   Returns
   -------
   AlePhy::ResourceManagerConfig in Sysdb or None
   """
   for sliceDir in resourceManagerSliceConfig.itervalues():
      for subSliceDir in  sliceDir.itervalues():
         if subSliceDir.name == resourceConsumerDir.name:
            return subSliceDir
   return None

def getRcdForIntf( intf, resourceConsumerSliceDir ):
   """
   Parameters
   ----------
   intf : CliPlugin.EthIntfCli.EthPhyIntf
   resourceConsumerSliceDir : Tac::Dir
      Dir at /ar/Sysdb/interface/resources/consumers/slice

   Returns
   -------
   AlePhy::ResourceConsumerDir that contains intf.
   """
   for sliceDir in resourceConsumerSliceDir.itervalues():
      for subSliceDir in sliceDir.itervalues():
         if intf.name in subSliceDir.serdesResourceConsumer:
            return subSliceDir
   return None

def getRcdToIntfListMap( intfList, resourceConsumerSliceDir ):
   """
   Parameters
   ----------
   intfList : [ list of CliPlugin.EthIntfCli.EthPhyIntf ]
   resourceConsumerSliceDir : Tac::Dir
      Dir at /ar/Sysdb/interface/resources/consumers/slice

   Returns
   -------
   dict { AlePhy::ResourceConsumerDir : [ list of CliPlugin.EthIntfCli.EthPhyIntf ] }
   """
   rcdToIntfListMap = defaultdict( list )

   for intf in intfList:
      rcd = getRcdForIntf( intf, resourceConsumerSliceDir )
      if not rcd:
         continue
      rcdToIntfListMap[ rcd ].append( intf )

   return rcdToIntfListMap


class MasterResourceManager( object ):
   """
   This class is an interface for MasterResourceManager. It takes the necessary
   directories and processes them as MRM would, so that configuration changes
   can be tested before applying.

   Expected usage is get an instance of the class with getMrm, synchronize its input
   directories with synchronize, modify the directories to apply the changes to be
   tested, call processIntfs, and determine what has changed by looking at
   intfStateDir and other directories.

   getHwActiveChangeIntfs and getDisabledIntfs are provided for convenience for speed
   change.
   """
   def __init__( self, resourceConsumerDir ):
      self.resourceManagerConfig = None
      self.resourceManagerStatus = None
      self.resourceConsumerDir = resourceConsumerDir
      self.intfStateDir = None
      self.mockEthIntfModeDir = None
      self.mockAllIntfConfigDir = None
      self.mockIntfConfigDir = None
      self.internalEthIntfModeDir = None
      self.intfStatusDir = None
      self.xcvrStatusDir = None

      self.createDirectories()
      self.mrm = Tac.newInstance( "AlePhy::MasterResourceManager",
                                  self.resourceManagerConfig,
                                  self.resourceManagerStatus,
                                  self.mockEthIntfModeDir,
                                  self.intfStatusDir,
                                  self.xcvrStatusDir )
      self.addResourceManagers()

   def createDirectories( self ):
      # Fetch the resource manager config or create an empty one. An empty resource
      # manager config has no effect.
      self.resourceManagerConfig = getRmcFromRcd( self.resourceConsumerDir )
      if not self.resourceManagerConfig:
         self.resourceManagerConfig = Tac.newInstance(
            "AlePhy::ResourceManagerConfig", "rmc" )

      self.resourceManagerStatus = Tac.newInstance( "AlePhy::ResourceManagerStatus",
                                                    "rms" )
      self.resourceManagerStatus.ethIntfModeDir = ( "leimd", )
      self.resourceManagerStatus.intfStateDir = ( "risd", )

      self.intfStateDir = self.resourceManagerStatus.intfStateDir

      self.mockAllIntfConfigDir = Tac.newInstance(
         "Interface::AllEthPhyIntfConfigDir", "strataepicd" )

      self.mockIntfConfigDir = Tac.newInstance( "Interface::EthPhyIntfConfigDir",
                                                "sandepicd" )

      self.mockEthIntfModeDir = Tac.newInstance( "Interface::EthIntfModeDir",
                                                 "eimd" )

      self.internalEthIntfModeDir = self.resourceManagerStatus.ethIntfModeDir

      # We need to force the mount, otherwise mounting does not occur and
      # newInstance hangs as that is the first access to the proxy.
      LazyMount.force( allIntfStatusDir )
      self.intfStatusDir = allIntfStatusDir

      self.xcvrStatusDir = xcvrAllStatusProxy.force()

   def _updateLogicalPortResourceManager( self ):
      if not self.mrm:
         return

      expManagement = self.resourceConsumerDir.logicalPortAllocationPriority
      if ResourceType.logicalPort not in self.mrm.resourceManager:
         curManagement = LogicalPortAllocationPriority.none
      else:
         manager = self.mrm.resourceManager[ ResourceType.logicalPort ]
         curManagement = manager.priority

      if expManagement == curManagement:
         return

      if expManagement == LogicalPortAllocationPriority.none:
         del self.mrm.resourceManager[ ResourceType.logicalPort ]
      elif expManagement == LogicalPortAllocationPriority.intfId:
         self.mrm.resourceManager.addMember(
               Tac.newInstance( "AlePhy::LogicalPortResourceManager",
                                self.mockEthIntfModeDir,
                                self.resourceConsumerDir,
                                self.mrm.reactorSmRegistry ) )
      elif expManagement == LogicalPortAllocationPriority.intfIdAndIntfEnabled:
         self.mrm.resourceManager.addMember(
               Tac.newInstance( "AlePhy::LogicalPortReleaseOnShutResourceManager",
                                self.mockEthIntfModeDir,
                                self.resourceConsumerDir,
                                self.mrm.reactorSmRegistry,
                                self.mockAllIntfConfigDir ) )
      else:
         assert False, "Unknown logial port allocation priority type."

   def addResourceManagers( self ):
      """
      This method unconditionally adds all resource managers that could be used. It
      is assumed that they have no side effects if their resources do not exist.

      Strata and Sand use different resource managers so this needs to reflect the
      correct resource managers for the product

      Common:
         SerdesResourceManager
         XcvrAdapterResourceManager
         PllResourceManager
      Strata resource managers:
         LogicalPortResourceManager
         LogicalPortReleaseOnShutResourceManager
         IntfCapsResourceManager
         IntfRestrictedResourceManager
         BabbageFallbackManager
      Sand resource managers:
         IntfPermissionsResourceManager
         IntfShutResourceManager
      """


      self.mrm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::SerdesResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.mrm.reactorSmRegistry ) )

      self.mrm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::XcvrAdapterResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.mrm.reactorSmRegistry ) )

      self.mrm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::PllResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.mrm.reactorSmRegistry ) )

      self._updateLogicalPortResourceManager()

      self.mrm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::IntfCapsResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.mrm.reactorSmRegistry ) )

      self.mrm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::IntfRestrictedResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.mrm.reactorSmRegistry ) )

      self.mrm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::BabbageFallbackManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.mrm.reactorSmRegistry ) )

      self.mrm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::IntfPermissionsResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.mrm.reactorSmRegistry,
                             self.mockIntfConfigDir ) )

      # Empty the reactorSmRegistry. We don't need or want reactions.
      self.mrm.reactorSmRegistry.reactorSm.clear()

   def synchronize( self ):
      """
      Synchronizes our mocked MRM input directories to match the systems.
      """
      self.resetEthIntfModeDir()
      self.resetIntfConfigDir()
      self._updateLogicalPortResourceManager()
      # Always reset intfStateDir at last as it depends on other inputs.
      self.resetIntfStateDir()

   def processIntfs( self ):
      # allow MRM to run
      self.mrm.initialized = True

      self.mrm.processIntfState()

      # stop allowing MRM to run
      self.mrm.initialized = False

   def enableInterface( self, intfName ):
      # Set it for the strata object
      if intfName in self.mockAllIntfConfigDir.intfConfig:
         self.mockAllIntfConfigDir.intfConfig[ intfName ].adminEnabled = True

      # Set it for the sand object
      if intfName in self.mockIntfConfigDir.intfConfig:
         self.mockIntfConfigDir.intfConfig[ intfName ].adminEnabled = True

   def resetEthIntfModeDir( self ):
      self.mockEthIntfModeDir.ethIntfMode.clear()
      for ethIntfModeSliceDir in EthIntfCli.ethIntfModeSliceDir.values():
         for ethIntfModeDir in ethIntfModeSliceDir.values():
            for mode in ethIntfModeDir.values():
               self.mockEthIntfModeDir.ethIntfMode.addMember( mode )

   def resetIntfConfigDir( self ):
      # bug483510 it is assumed that a system will have only one of
      # mockAllIntfConfigDir or mockIntfConfigDir. Currently this works
      # because no resource manager using these objects is supported on both
      # strata and sand so we only ever pass in the correct one to the
      # correct RM. In order to pass in both we will need to find a better
      # way to handle, populating these since one will be assumed to be empty.
      self.mockAllIntfConfigDir.intfConfig.clear()
      self.mockIntfConfigDir.intfConfig.clear()
      for intfId, intfConfig in allIntfConfigDir.intfConfig.iteritems():
         epic = Tac.newInstance( "Interface::EthPhyIntfConfig", intfId )
         epic.adminEnabled = intfConfig.adminEnabled
         epic.adminEnabledStateLocal = intfConfig.adminEnabledStateLocal
         self.mockAllIntfConfigDir.intfConfig.addMember( epic )

         self.mockIntfConfigDir.newIntfConfig( intfId )
         self.mockIntfConfigDir[ intfId ].adminEnabled = intfConfig.adminEnabled
         self.mockIntfConfigDir[ intfId ].adminEnabledStateLocal = \
               intfConfig.adminEnabledStateLocal

   def resetIntfStateDir( self ):
      self.intfStateDir.intfState.clear()
      for intf in self.resourceConsumerDir.serdesResourceConsumer:
         # Skip interfaces that are not ready.
         if intf not in self.mockEthIntfModeDir:
            continue
         # BUG483510 - we currently assume these two config collections are equal.
         assert set( self.mockAllIntfConfigDir ) == set( self.mockIntfConfigDir )
         if intf not in self.mockAllIntfConfigDir:
            continue
         if intf not in self.intfStatusDir:
            continue
         module = EthIntfId.module( intf )
         port = EthIntfId.port( intf )
         xcvrName = ( "Ethernet%d/%d" % ( module, port )
                      if module else "Ethernet%d" % port )
         if xcvrName not in self.xcvrStatusDir.xcvrStatus:
            continue
         self.intfStateDir.intfState.newMember( intf )

   def getHwActiveChangeIntfs( self ):
      """
      Returns the set of interfaces whose activeness would change if the current
      state takes affect.
      """
      affected = set()
      for intfId, newState in self.intfStateDir.items():
         epis = self.intfStatusDir.intfStatus.get( intfId )
         # Interface is currently active,
         if epis.hwActive():
            # speed change would make it inactive.
            if newState.state.hwState == IntfHwState.inactive:
               affected.add( intfId )
         # Interface is currently inactive,
         else:
            # speed change would make it active.
            if newState.state.hwState == IntfHwState.active:
               affected.add( intfId )

      return affected

   def getDisabledIntfs( self, reason=ResourceType.logicalPort ):
      """
      Returns the set of interfaces which will become inactive due to the given
      reason if the current state takes affect.
      """
      # interfaces which have become inactive due to logical port
      disabled = set()
      for intfId, newState in self.intfStateDir.items():
         epis = self.intfStatusDir.intfStatus.get( intfId )
         if not epis.active:
            continue
         # Interface is currently active,
         # speed change would make it inactive.
         if ( newState.state.swState == IntfSwState.inactive
              and reason in newState.inactiveReason
              and len( newState.inactiveReason ) == 1 ):
            disabled.add( intfId )

      return disabled

def Plugin( em ):
   global allIntfConfigDir
   global allIntfStatusDir
   global xcvrAllStatusProxy
   global resourceManagerSliceConfig

   allIntfConfigDir = LazyMount.mount( em, "interface/config/eth/phy/all",
                                       "Interface::AllEthPhyIntfConfigDir", "r" )
   allIntfStatusDir = LazyMount.mount( em, "interface/status/eth/phy/all",
                                       "Interface::AllEthPhyIntfStatusDir", "r" )
   xcvrAllStatusProxy = XcvrAllStatusDir.xcvrAllStatusDir( em )
   resourceManagerSliceConfig = LazyMount.mount(
         em, "interface/resources/config/slice", "Tac::Dir", "ri" )
