# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools
import re

import Cell
import CliParser
import ConfigMount
import LazyMount
import Tac
from CliMatcher import Matcher
from CliParserCommon import MatchResult, noMatch

marginConf = None
dpmSource1 = None
dpmSource2 = None
vrmSource1 = None
vrmSource2 = None

#-------------------------------------------------------------------------------
# Helper functions
#-------------------------------------------------------------------------------
def getDPMs( systemM ):
   chipNameRegex = re.compile( '.*?([0-9]+)$' )
   dpms = {}
   for cardName, cardConfig in itertools.chain( dpmSource1.controller.iteritems(),
                                                dpmSource2.controller.iteritems() ):
      if cardName.startswith( systemM.fru ):
         chipNumMatch = chipNameRegex.match( cardName )
         if chipNumMatch:
            dpms[ chipNumMatch.group( 1 ) ] = cardConfig
         elif cardName.endswith( 'switchCardPowerController' ):
            dpms[ '1' ] = cardConfig
         elif cardName.endswith( 'cpuCardPowerController' ):
            dpms[ '2' ] = cardConfig
   return dpms

def getRails( conf, sourceM ):
   rails = []
   for rail, railConf in conf.voltageRail.iteritems():
      if railConf.voltageRailEnvConfig.marginSource == 'UCD' \
         and sourceM.source == 'DPM':
         rails.append( rail )
   return rails

def getVRMs( systemM ):
   vrms = {}
   for vrm, conf in itertools.chain( vrmSource1.vrm.iteritems(),
                                     vrmSource2.vrm.iteritems() ):
      if conf.fruName == systemM.fru:
         vrms[ vrm ] = conf
   return vrms

def getVRMRails( conf ):
   return [ rail for rail in conf.rail ]

#-------------------------------------------------------------------------------
# 'hardware margin' command
#-------------------------------------------------------------------------------
class SystemMatcher( Matcher ):
   """
   Matches all linecards/fabrics/supervisors on modular systems.
   Matches FixedSystem.
   """
   def __init__( self ):
      self.fru = None
      super( SystemMatcher, self ).__init__()
      return

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         return noMatch
      if root.tacType.fullTypeName == 'EntityMib::FixedSystem':
         if token == 'FixedSystem':
            self.fru = 'FixedSystem'
            return MatchResult( self.fru, self.fru )
      elif root.tacType.fullTypeName == "EntityMib::Chassis":
         for slot in root.cardSlot.values():
            if token == '%s%s' % ( slot.tag, slot.label ):
               self.fru = '%s%s' % ( slot.tag, slot.label )
               return MatchResult( self.fru, self.fru )
      return noMatch

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         return []
      if root.tacType.fullTypeName == 'EntityMib::FixedSystem':
         return [ CliParser.Completion( 'FixedSystem', '' ) ]
      elif root.tacType.fullTypeName == "EntityMib::Chassis":
         completions = []
         for slot in root.cardSlot.values():
            completions.append( CliParser.Completion( '%s%s' % ( slot.tag,
                                                                 slot.label ), '' ) )
         return completions
      return []

class SourceMatcher( Matcher ):
   """
   Matches dpm and vrm chips.
   """
   def __init__( self ):
      self.sources = [ 'dpm', 'vrm' ]
      self.source = None
      super( SourceMatcher, self ).__init__()
      return

   def match( self, mode, context, token ):
      for source in self.sources:
         if token == source:
            self.source = token.upper()
            return MatchResult( self.source, self.source )
      return noMatch

   def completions( self, mode, context, token ):
      return [ CliParser.Completion( source, '' ) for source in self.sources ]

class SourceIdMatcher( Matcher ):
   """
   Matches all chip ids.
   """
   def __init__( self, systemM, sourceM ):
      self.systemM = systemM
      self.sourceM = sourceM
      self.sourceId = None
      super( SourceIdMatcher, self ).__init__()
      return

   def match( self, mode, context, token ):
      if self.sourceM.source == 'DPM':
         chipDict = getDPMs( self.systemM )
      else:
         chipDict = getVRMs( self.systemM )
      if token in chipDict:
         self.sourceId = token
         return MatchResult( int( self.sourceId ), self.sourceId )
      return noMatch

   def completions( self, mode, context, token ):
      if self.sourceM.source == 'DPM':
         chipDict = getDPMs( self.systemM )
      else:
         chipDict = getVRMs( self.systemM )
      return [ CliParser.Completion( chip, '' ) for chip in chipDict ]

class RailIdMatcher( Matcher ):
   """
   Matches all rail ids on a chip.
   """
   def __init__( self, systemM, sourceM, sourceIdM ):
      self.systemM = systemM
      self.sourceM = sourceM
      self.sourceIdM = sourceIdM
      super( RailIdMatcher, self ).__init__()
      return

   def match( self, mode, context, token ):
      if self.sourceM.source == 'DPM':
         chipDict = getDPMs( self.systemM )
      else:
         chipDict = getVRMs( self.systemM )
      if self.sourceIdM.sourceId in chipDict:
         if self.sourceM.source == 'DPM':
            rails = getRails( chipDict[ self.sourceIdM.sourceId ], self.sourceM )
         else:
            rails = getVRMRails( chipDict[ self.sourceIdM.sourceId ] )
         if int( token ) in rails:
            return MatchResult( int( token ), token )
      return noMatch

   def completions( self, mode, context, token ):
      if self.sourceM.source == 'DPM':
         chipDict = getDPMs( self.systemM )
      else:
         chipDict = getVRMs( self.systemM )
      if self.sourceIdM.sourceId in chipDict:
         if self.sourceM.source == 'DPM':
            rails = getRails( chipDict[ self.sourceIdM.sourceId ], self.sourceM )
         else:
            rails = getVRMRails( chipDict[ self.sourceIdM.sourceId ] )
         return [ CliParser.Completion( str( rail ), '' ) for rail in rails ]
      return []

systemMatcher = SystemMatcher()
sourceMatcher = SourceMatcher()
sourceIdMatcher = SourceIdMatcher( systemMatcher, sourceMatcher )

def doSetMargin( mode, args ):
   fru = args[ 'FRU' ]
   marginSource = args[ 'MARGIN' ]
   sourceId = args[ 'SOURCEID' ]
   railId = args[ 'RAILID' ]
   marginPercent = None
   if 'nominal' in args:
      marginPercent = 0
   elif 'low' in args:
      marginPercent = 0 - float( args[ 'PERCENT' ] )
   else:
      marginPercent = float( args[ 'PERCENT' ] )

   railKey = Tac.newInstance( "Hardware::Margin::RailKey",
                              fru, marginSource, sourceId, railId )
   if railKey in marginConf.marginConfig:
      conf = marginConf.marginConfig[ railKey ]
      if conf.marginPercent == marginPercent:
         mode.addMessage( "Nothing to do" )
      else:
         conf.marginPercent = marginPercent
   else:
      conf = marginConf.newMarginConfig( railKey, marginPercent )

def noSetMargin( mode, args ):
   if 'FRU' not in args :
      marginConf.marginConfig.clear()
   else:
      railKey = Tac.newInstance( "Hardware::Margin::RailKey",
                                 args[ 'FRU' ],
                                 args[ 'MARGIN' ],
                                 args[ 'SOURCEID' ],
                                 args[ 'RAILID' ] )
      if railKey in marginConf.marginConfig:
         del marginConf.marginConfig[ railKey ]
      else:
         mode.addMessage( "Margin config does not exist." )

def Plugin( entityManager ):
   global marginConf, dpmSource1, dpmSource2, vrmSource1, vrmSource2
   marginConf = ConfigMount.mount( entityManager, "hardware/margin/config",
      "Hardware::Margin::Config", "w" )
   dpmSource1 = LazyMount.mount( entityManager,
      "hardware/powercontroller/config/ucd90120",
      "Hardware::Ucd9012::Config", "r" )
   dpmSource2 = LazyMount.mount( entityManager,
      "hardware/cell/%d/powercontroller/config/ucd90120" % Cell.cellId(),
      "Hardware::Ucd9012::Config", "r" )
   vrmSource1 = LazyMount.mount( entityManager,
      "hardware/cell/%d/vrmIr/config/vrmIr" % Cell.cellId(),
      "Hardware::Vrm::Config", "r" )
   vrmSource2 = LazyMount.mount( entityManager,
       "hardware/vrmIr/config/vrmIr", "Hardware::Vrm::Config", "r" )
