#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import getPlatform
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.EthIntfModel as EthIntfModel
import CliPlugin.MacAddr as MacAddr
import CliPlugin.IntfCli as IntfCli
import CliPlugin.VirtualIntfRule as VirtualIntfRule
import CliPlugin.FruCli as FruCli
import ConfigMount
import Ethernet
import LazyMount
from ManagementActiveIntfModel import MgmtActiveIntfRedundancyStatus
import Tac
import Tracing

t0 = Tracing.trace0

mgmtActiveIntfConfigDir = None
mgmtActiveIntfStatusDir = None
ethPhyConfigDir = None

mgmtActiveIntfName = "Management0"

def isVeos():
   return getPlatform() == 'veos'

def _checkIfAllowed():
   return EthIntfCli.isModular() or isVeos()

def managementActiveGuard( mode, token ):
   if not _checkIfAllowed():
      return CliParser.guardNotThisPlatform
   return None

# pylint: disable-msg=W0223
# Method 'foo' is abstract in class 'Bar' but is not overridden
class ManagementActiveIntf( EthIntfCli.EthIntf ):

   #----------------------------------------------------------------------------
   # Creates a new EthIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      EthIntfCli.EthIntf.__init__( self, name, mode )
      self.mgmtActiveIntfConfigDir = mgmtActiveIntfConfigDir
      self.mgmtActiveIntfStatusDir = mgmtActiveIntfStatusDir

   helpdesc = 'Management interface on the active supervisor'

   matcher = VirtualIntfRule.VirtualIntfMatcher(
      'Management', 0, 0, helpdesc=helpdesc,
      value=lambda mode, intf: ManagementActiveIntf( intf, mode ),
      guard=managementActiveGuard )

   def config( self ):
      return self.mgmtActiveIntfConfigDir.intfConfig

   def status( self ):
      return self.mgmtActiveIntfStatusDir.intfStatus

   def getRealIntf( self ):
      status = self.status()
      if status and status.realIntfId:
         intf  = EthIntfCli.EthPhyIntf( status.realIntfId, self.mode_ )
         realIntf = intf if intf.status() else None
         return realIntf 
      return None

   def getIntfCounterDir( self ):
      assert False, "getIntfCounterDir not implemented for ManagementActive"

   #----------------------------------------------------------------------------
   # Returns a list containing the MgmtActiveIntf if it exists.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      if mgmtActiveIntfStatusDir.intfStatus:
         return [ ManagementActiveIntf( mgmtActiveIntfName, mode ) ]
      return []

   #----------------------------------------------------------------------------
   # Creates the Interface::MgmtActiveIntfConfig object for this interface if it
   # does not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      self.mgmtActiveIntfConfigDir.intfConfig = ( mgmtActiveIntfName, )

   #----------------------------------------------------------------------------
   # Destroys the Interface::MgmtActiveIntfConfig object for this interface if
   # it already exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      self.mgmtActiveIntfConfigDir.intfConfig = None

   def countersSupported( self ):
      return False

   def switchportEligible( self ):
      return False

   def routingSupported( self ):
      return True

   def vrfSupported( self ):
      return True

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def hardware( self ):
      return "ethernet"

   def addrStr( self ):
      return "address is %s (bia %s)" % ( self.addr(), self.burnedInAddr() )

   def burnedInAddr( self ):
      return self.addr()

   def bandwidth( self ):
      realIntf = self.getRealIntf()
      if realIntf:
         return realIntf.bandwidth()
      return 0
   
   def getIntfStatusModel( self ):
      return EthIntfModel.EthPhyInterfaceStatus( name=self.status().intfId )

   def showLinkTypeSpecific( self, intfStatusModel ):
      realIntf = self.getRealIntf()
      if realIntf:
         ( duplex, autoNegotiate, 
           autoNegotiateActive )= realIntf.showLinkTypeSpecific( intfStatusModel )
      else:
         duplex = "duplexUnknown"
         autoNegotiate = "unknown"
         autoNegotiateActive = False
         intfStatusModel.lanes = 0
         
      intfStatusModel.duplex = duplex
      intfStatusModel.autoNegotiate = autoNegotiate 
      intfStatusModel._autoNegotiateActive = autoNegotiateActive

   def showLoopbackMode( self, intfStatusModel ):
      realIntf = self.getRealIntf()
      if realIntf:
         loopbackMode = realIntf.showLoopbackMode( intfStatusModel )
      else:
         loopbackMode = "loopbackNone"

      intfStatusModel.loopbackMode = loopbackMode

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesStatus()"
   #----------------------------------------------------------------------------
   def autonegActive( self ):
      realIntf = self.getRealIntf()
      if realIntf:
         return realIntf.autonegActive()
      return False

   def getIntfDuplexStr( self ):
      realIntf = self.getRealIntf()
      if realIntf:
         return realIntf.getIntfDuplexStr()
      return "duplexUnknown"

   def xcvrTypeStr( self ):
      realIntf = self.getRealIntf()
      if realIntf:
         return realIntf.xcvrTypeStr()
      return "Unknown"

   #----------------------------------------------------------------------------
   # Sets the MAC address of the interface to a specified value or the default
   # value.
   #----------------------------------------------------------------------------
   def setMacAddr( self, addr, routerMacConfigured=None ):
      if addr is None:
         self.config().addr = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
      elif Ethernet.isUnicast( addr ):
         self.config().addr = addr
      else:
         self.mode_.addError( 'Cannot set the interface MAC address to a '
                              'multicast address' )

   #----------------------------------------------------------------------------
   # The "no interface Management 0" command, in "config" mode.
   #----------------------------------------------------------------------------
   def noInterface( self ):
      self.destroy()

   #----------------------------------------------------------------------------
   # This function is called by the "default interface management0" command.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      if isVeos():
         self.setMacAddr( None )
      if EthIntfCli.isModular():
         self.setBackupLinkToStandby( enable=False )
      EthIntfCli.EthIntf.setDefault( self )

   #---------------------------------------------------------------------------
   # This function is called by the "backup-link standby [fallback-delay <t>]"
   #---------------------------------------------------------------------------
   def setBackupLinkToStandby( self, enable=True, fallbackDelay=None ):
      config = self.config()
      config.backupLinkStandby = enable 

      if fallbackDelay is None:
         config.backupLinkFallbackDelay = (
               config.defaultFallbackDelay )
      elif fallbackDelay == "infinity":
         config.backupLinkFallbackDelay = Tac.endOfTime
      else:
         config.backupLinkFallbackDelay = fallbackDelay

IntfCli.Intf.addPhysicalIntfType( ManagementActiveIntf, None,
                                  withIpSupport=True,
                                  withRoutingProtoSupport=False )

def _updateMgmtRangeIntfForModular():
   if _checkIfAllowed():
      EthIntfCli.MgmtAutoIntfType.helpDesc = [
         'Slot number (0 for Active Management)',
         'Port number' ]
      EthIntfCli.MgmtAutoIntfType.registerManagementIntfClass(
         ManagementActiveIntf,
         lambda name, mode: name[ -1 ] == '0',
         [ mgmtActiveIntfName ] )

FruCli.registerModularSystemCallback( _updateMgmtRangeIntfForModular )
FruCli.registerFixedSystemCallback( _updateMgmtRangeIntfForModular )

class ManagementActiveIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, ManagementActiveIntf ) and
               isVeos() )

#--------------------------------------------------------------------------------
# [ no | default ] mac-address MAC_ADDR
#
# Note that "mac-address 0.0.0" will reset the MAC address to the default.
# Remarkably, this is consistent with the industry-standard.
#--------------------------------------------------------------------------------
class MacAddressMacaddrCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-address MAC_ADDR'
   noOrDefaultSyntax = 'mac-address ...'
   data = {
      'mac-address' : 'Set interface MAC address',
      'MAC_ADDR' : MacAddr.macAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.setMacAddr( args[ 'MAC_ADDR' ] )
      t0( 'Set ' + mode.intf.name + ' MAC address to ' + args[ 'MAC_ADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setMacAddr( None )
      t0( 'Set ' + mode.intf.name + ' MAC address to default' )

ManagementActiveIntfModelet.addCommandClass( MacAddressMacaddrCmd )

IntfCli.IntfConfigMode.addModelet( ManagementActiveIntfModelet )

class ManagementActiveIntfHaModlet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, ManagementActiveIntf ) and
             EthIntfCli.isModular()) 

#--------------------------------------------------------------------------------
# [ no | default ] backup-link standby [ fallback-delay ( infinity | DELAY ) ]
#--------------------------------------------------------------------------------
class BackupLinkStandbyCmd( CliCommand.CliCommandClass ):
   syntax = 'backup-link standby [ fallback-delay ( infinity | DELAY ) ]'
   noOrDefaultSyntax = 'backup-link standby ...'
   data = {
      'backup-link' : 'Backup link for the interface',
      'standby' : 'Backup link using standby',
      'fallback-delay' : 'Fallback delay to the active supervisor',
      'infinity' : "Don't ever fallback",
      'DELAY' : CliMatcher.IntegerMatcher( 0, 3600,
         helpdesc='Fallback delay in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      fallbackDelay = args.get( 'DELAY' )
      if 'infinity' in args:
         fallbackDelay = 'infinity'
      mode.intf.setBackupLinkToStandby( enable=True, fallbackDelay=fallbackDelay )
      t0( 'Set', mode.intf.name, 'Backup link to standby with fallback delay:',
            ( '%s' % fallbackDelay ) if fallbackDelay else 'default' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setBackupLinkToStandby( enable=False )

ManagementActiveIntfHaModlet.addCommandClass( BackupLinkStandbyCmd )
IntfCli.IntfConfigMode.addModelet( ManagementActiveIntfHaModlet )

#--------------------------------------------------
# The "show redundancy backup-link" command.
#--------------------------------------------------
def doShowRedBackupLink( mode, args ):
   intfRedStatusModel = MgmtActiveIntfRedundancyStatus()
   if mgmtActiveIntfConfigDir.intfConfig:
      intfRedStatusModel.name = mgmtActiveIntfConfigDir.intfConfig.intfId
      intfRedStatusModel.redundancy = \
         mgmtActiveIntfConfigDir.intfConfig.backupLinkStandby
      if ( mgmtActiveIntfConfigDir.intfConfig.backupLinkStandby and 
         mgmtActiveIntfStatusDir.intfStatus ):
         intfRedStatusModel.redundancyIntf = \
            mgmtActiveIntfStatusDir.intfStatus.realIntfId
         intfRedStatusModel.lastRedundancyLinkChangeTime = \
            mgmtActiveIntfStatusDir.intfStatus.lastRedundancyLinkChangeTime
   return intfRedStatusModel

def Plugin( entityManager ):
   global mgmtActiveIntfConfigDir
   global mgmtActiveIntfStatusDir
   global ethPhyConfigDir

   mgmtActiveIntfConfigDir = ConfigMount.mount(
      entityManager,
      "interface/config/eth/managementactive",
      "Interface::MgmtActiveIntfConfigDir", "w" )
   mgmtActiveIntfStatusDir = LazyMount.mount(
      entityManager,
      "interface/status/eth/managementactive",
      "Interface::MgmtActiveIntfStatusDir", "r" )
   ConfigMount.mount(
      entityManager,
      "interface/config/eth/phy/slice",
      "Tac::Dir", "wi" )
   ethPhyConfigDir = LazyMount.mount(
      entityManager,
      "interface/config/eth/phy/all",
      "Interface::AllEthPhyIntfConfigDir", "r" )

