# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import os

import AgentCommandRequest
import Arnet
import BasicCli
import Cell
import CliCommand
import CliMatcher
from CliPlugin.EbraEthIntfCliModel import interfaceStatusHook
import CliPlugin.EthIntfCli as EthIntfCli
from CliPlugin.IntfModel import interfaceInfoHook
import CliPlugin.IntfMaintenanceModels as IntfMaintenanceModels
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.MaintenanceModeModels import MaintenanceAll, MaintenanceStatus, \
    MaintenanceUnits, MaintenanceUnit
from CliPlugin.MaintenanceModeModels import MaintenanceInterfaceBrief, \
      MaintenanceInterfacesBrief
from CliPlugin.MaintenanceModels import MaintenanceGroups, intfGroupsHook, \
    bgpGroupsHook, intfProfilesHook, MaintenanceProfiles, bgpProfilesHook
import CliPlugin.MaintenanceModels as MaintenanceModels
from CliPlugin.MaintenanceModels import MaintenanceDebugInfo, MaintenanceEventInfo
from CliPlugin.MaintenanceModels import MaintenanceEventInfoEntry, UnitProfile
from CliPlugin.MaintenanceModels import bgpMaintenanceDebugInfoHook, UnitProfiles
from CliPlugin.MaintenanceModeModels import MaintenanceProtocolInfo
from CliPlugin.MaintenanceModeModels import MaintenanceStages, MaintenanceStage
from CliPlugin.MaintenanceModeModels import UnitMaintenanceDefaultProfile
import CliPlugin.MaintenanceCliLib as MaintenanceCliLib
from CliPlugin.MaintenanceCliLib import intfGroupType, bgpGroupType
from CliPlugin.MaintenanceCliLib import intfProfileType, bgpProfileType
from CliPlugin.MaintenanceCliLib import toUtc
from CliPlugin.MaintenanceCliLib import onBootMaintShowFlagStates
from CliPlugin.MaintenanceCliLib import underGoingMaintStates
from CliPlugin.MaintenanceCliLib import getAdminState
from CliPlugin.MaintenanceCliLib import isDynamicUnit, dynamicUnitComponentName
from CliPlugin.MaintenanceCliLib import dynamicUnitVrfName, DynamicComponentRe
from CliPlugin.MaintenanceCliLib import dynamicUnitName
import CliPlugin.TechSupportCli
from CliPlugin.VlanIntfCli import VlanAutoIntfType
import Intf
import LazyMount
import MaintenanceModeAgent
import MaintenanceModeCliLib
import ShowCommand
import Tac

unitConfigDir = None
unitStatusDir = None
intfToGroupDir = None
groupToUnitDir = None
monIntfStatusDir = None
intfDefaultProfileDir = None
bgpDefaultProfileDir = None
unitDefaultProfileDir = None
unitProfileStatusDir = None
errDisableCauseStatus = None
intfProfileStatusDir = None
maintEnterInstanceLog = None
maintExitInstanceLog = None

def _isIntfTrafficAboveThreshold( monIntfStatus ):
   profileName = monIntfStatus.profileName
   if profileName == MaintenanceCliLib.defaultProfile:
      profileName = intfDefaultProfileDir.profileName
   profile = intfProfileStatusDir.status.get( profileName )
   if not profile:
      return False
   threshold = profile.rateMonThresholdInKbps * 1000
   if monIntfStatus.curRates.inBitsRate > threshold or \
         monIntfStatus.curRates.outBitsRate > threshold:
      return True
   return False

# In case of subintfs, we return the state of the parent intf. If the parent intf
# is under maintenance, then it 'implies' that the sub-intf is also under 
# maintenance - i.e. traffic has drained. But traffic threshold violation will
# be reported only for the parent intf, as we dont yet support rate monitoring of
# subintfs. getIntfMaintenanceState handles the case of returning the state of
# parent intf in case of a subintf.
def showInterfaceStatusHook( intf, interfaceStatusEntry ):
   state, _ = getIntfMaintenanceState( intf )
   if state == MaintenanceCliLib.underMaintenance:
      interfaceStatusEntry.underMaintenance = True
      monIntfStatus = monIntfStatusDir.monIntfStatus.get( intf )
      if monIntfStatus and _isIntfTrafficAboveThreshold( monIntfStatus ):
         interfaceStatusEntry.trafficAboveThreshold = True

def showInterfaceHook( intf, model ):
   state, stateTime = getIntfMaintenanceState( intf )
   if state == MaintenanceCliLib.underMaintenance:
      model.maintenanceEnterTime = toUtc( stateTime )

def showUnitProfile( profileName ):
   if profileName is None:
      return None
   profile = unitProfileStatusDir.status.get( profileName )
   if not profile:
      return None
   unitProfile = UnitProfile()
   unitProfile.onBootDuration = profile.onBootDuration
   unitProfile.onBootEnabled = profile.onBootEnabled
   return unitProfile

@MaintenanceCliLib.maintRunningShowCliCheck
def showMaintenance( mode, args ):
   ret = MaintenanceAll()
   for unitStatus in unitStatusDir.status.values():
      timeOfStateChange = toUtc( unitStatus.timeOfStateChange )

      # We show the onBootMaintenance flag only when the unit state is one of the
      # following [ 'underMaintenance', 'maintenanceModeEnter' ].
      # This allows the user to know if the current reason for the unit entering
      # maintenance is onBootMaintenance. 
      isReasonOnBoot = ( unitStatus.state in onBootMaintShowFlagStates and
                         unitStatus.reason == MaintenanceCliLib.srcOnBoot )
      adminState = getAdminState( unitStatus.state )
      if isDynamicUnit( unitStatus ):
         componentName = dynamicUnitComponentName( unitStatus )
         for unitStatusGroup in unitStatus.group:
            if unitStatusGroup.type == intfGroupType:
               thresholdViolation = bool( unitStatus.intfsViolatingTrafficThreshold )
              
               if unitStatus.state in underGoingMaintStates:
                  aggInBpsRate_ = unitStatus.aggTrafficRate.inBitsRate
                  aggOutBpsRate_ = unitStatus.aggTrafficRate.outBitsRate
               else:
                  aggInBpsRate_ = 0
                  aggOutBpsRate_ = 0
               
               maintStatus = MaintenanceStatus( state=unitStatus.state,
                                                adminState=adminState,
                                                stateChangeTime=timeOfStateChange,
                                                onBootMaintenance=isReasonOnBoot,
                                                intfsViolatingTrafficThreshold=
                                                   thresholdViolation,
                                                aggInBpsRate=aggInBpsRate_,
                                                aggOutBpsRate=aggOutBpsRate_ )
               
               ret.interfaces[ componentName ] = maintStatus
            elif unitStatusGroup.type == bgpGroupType:
               maintStatus = MaintenanceStatus(
                  state=unitStatus.state, adminState=adminState,
                  stateChangeTime=timeOfStateChange,
                  onBootMaintenance=isReasonOnBoot )
               vrfName = dynamicUnitVrfName( unitStatus )
               if vrfName not in ret.vrfs:
                  maintProtoInfo = MaintenanceProtocolInfo()
                  maintProtoInfo.bgpPeers[ componentName ] = maintStatus
                  ret.vrfs[ vrfName ] = maintProtoInfo
               ret.vrfs[ vrfName ].bgpPeers[ componentName ] = maintStatus
      else:
         thresholdViolation = bool( unitStatus.intfsViolatingTrafficThreshold )
         if unitStatus.state in underGoingMaintStates:
            aggInBpsRate_ = unitStatus.aggTrafficRate.inBitsRate
            aggOutBpsRate_ = unitStatus.aggTrafficRate.outBitsRate
         else:
            aggInBpsRate_ = 0
            aggOutBpsRate_ = 0

         ret.units[ unitStatus.name ] = \
               MaintenanceStatus( state=unitStatus.state,
                                  adminState=adminState,
                                  stateChangeTime=timeOfStateChange,
                                  onBootMaintenance=isReasonOnBoot,
                                  intfsViolatingTrafficThreshold=thresholdViolation,
                                  aggInBpsRate=aggInBpsRate_,
                                  aggOutBpsRate=aggOutBpsRate_ )
   return ret

#-------------------------------------------------------------------------------
# show maintenance
#-------------------------------------------------------------------------------
class ShowMaintenanceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
   }
   cliModel = MaintenanceAll
   handler = showMaintenance

BasicCli.addShowCommandClass( ShowMaintenanceCmd )

#-------------------------------------------------------------------------------
# show maintenance summary
#-------------------------------------------------------------------------------
class ShowMaintenanceSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance summary'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'summary' : 'Summarized Maintenance information'
   }
   cliModel = MaintenanceModels.MaintenanceSummary

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      model = MaintenanceModels.MaintenanceSummary()
      model.numUnitsConfigured = 0
      model.unitStatistics[ 'active' ] = 0
      model.unitStatistics[ 'underMaintenance' ] = 0
      model.unitStatistics[ 'maintenanceModeEnter' ] = 0
      model.unitStatistics[ 'maintenanceModeExit' ] = 0
      model.intfDirectlyInMaintenance[ 'underMaintenance' ] = 0
      model.intfDirectlyInMaintenance[ 'maintenanceModeEnter' ] = 0
      model.peerDirectlyInMaintenance[ 'underMaintenance' ] = 0
      model.peerDirectlyInMaintenance[ 'maintenanceModeEnter' ] = 0
      # Count the number of units configured by excluding the dynamic units
      for unitConfig in unitConfigDir.config.values():
         if isDynamicUnit( unitConfig ):
            continue
         model.numUnitsConfigured += 1
      # Count the units per state by excluding the dynamic units
      for unitStatus in unitStatusDir.status.values():
         if isDynamicUnit( unitStatus ):
            for group in unitStatus.group.keys():
               if group.type == intfGroupType:
                  if unitStatus.state == 'maintenanceModeEnter':
                     model.intfDirectlyInMaintenance[ unitStatus.state ] += 1
                  if unitStatus.state == 'underMaintenance':
                     model.intfDirectlyInMaintenance[ unitStatus.state ] += 1
               if group.type == bgpGroupType:
                  if unitStatus.state == 'maintenanceModeEnter':
                     model.peerDirectlyInMaintenance[ unitStatus.state ] += 1
                  if unitStatus.state == 'underMaintenance':
                     model.peerDirectlyInMaintenance[ unitStatus.state ] += 1
            continue
         model.unitStatistics[ unitStatus.state ]  = \
            model.unitStatistics.get( unitStatus.state, 0 ) + 1
      for hook in MaintenanceModels.showMaintenanceSummaryHook.extensions():
         hook( mode, model )
      return model

BasicCli.addShowCommandClass( ShowMaintenanceSummaryCmd )

#-------------------------------------------------------------------------------
# show maintenance units [ UNIT_NAME ]
#-------------------------------------------------------------------------------
def populateUnit( unitStatus ):
   '''  Return the populated MaintenanceUnit
        Note: This function assumes that unitStatus is valid.
   '''
   unitDefaultProfile = unitDefaultProfileDir.profileName
   maintenanceUnit = MaintenanceUnit()
   maintenanceUnit.origin = unitStatus.unitType
   maintenanceUnit.state = unitStatus.state
   maintenanceUnit.adminState = getAdminState( unitStatus.state ) 
   maintenanceUnit.timeOfStateChange = toUtc( unitStatus.timeOfStateChange )
   if unitDefaultProfile == MaintenanceCliLib.defaultProfile:
      maintenanceUnit.unitProfile = 'Default'
   else:
      maintenanceUnit.unitProfile = unitDefaultProfile
   if unitStatus.profileName:
      maintenanceUnit.unitProfile = unitStatus.profileName
   
   # We show the onBootMaintenance flag only when the unit state is one of the
   # following [ 'underMaintenance', 'maintenanceModeEnter' ].
   # This allows the user to know if the current reason for the unit entering
   # maintenance is onBootMaintenance. 
   maintenanceUnit.onBootMaintenance = (
         unitStatus.state in onBootMaintShowFlagStates and
         unitStatus.reason == MaintenanceCliLib.srcOnBoot )
   if maintenanceUnit.onBootMaintenance:
      maintenanceUnit.onBootExitTime = toUtc( unitStatus.onBootMaintExitTime )
   for timeStamp in sorted(  unitStatus.history.keys(), reverse=True ):
      # See BUG139695
      eventInfo = unitStatus.history.get( timeStamp )
      if not eventInfo:
         continue
      entry = MaintenanceEventInfoEntry()
      entry.timeStamp = toUtc( timeStamp )
      entry.eventInfo = eventInfo
      maintenanceUnit.history.append( entry )

   for group in unitStatus.group:
      if group.type == bgpGroupType:
         maintenanceUnit.bgpGroups.append( group.name )
      elif group.type == intfGroupType:
         maintenanceUnit.interfaceGroups.append( group.name )

   if unitStatus.intfsViolatingTrafficThreshold:
      for intf in unitStatus.intfsViolatingTrafficThreshold:
         maintenanceUnit.intfsViolatingTrafficThreshold.append( intf )
   maintenanceUnit.totalIntfTrafficViolations = unitStatus.totalIntfTrafficViolations

   return maintenanceUnit

matcherUnits = CliMatcher.KeywordMatcher( 'units',
      helpdesc='Units information' )

class ShowUnitsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance units [ UNIT_NAME ]'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'units' : matcherUnits,
      'UNIT_NAME' : MaintenanceModeCliLib.UnitNameExpr()
   }
   cliModel = MaintenanceUnits

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      unitName = args.get( 'UNIT_NAME' )
      allMaintenanceUnits = {}
      if unitName:
         if MaintenanceModeCliLib.isLinecardBuiltinUnit( unitName ):
            for linecard in unitName:
               # The unitName 'list' specifies a range of the linecards (say 3-6).
               # Here we have to only show those linecards which are present
               linecardUnitStatus = unitStatusDir.status.get( linecard.name )
               if linecardUnitStatus:
                  allMaintenanceUnits[ linecard.name ] = populateUnit( 
                     linecardUnitStatus )
         else:
            unitName = MaintenanceModeCliLib.fixIfSystemUnitName( unitName )
            unitStatus = unitStatusDir.status.get( unitName )
            if unitStatus:
               allMaintenanceUnits[ unitName ] = populateUnit( unitStatus )
         return MaintenanceUnits( maintenanceUnits=allMaintenanceUnits )
      else:
         for unitStatus in unitStatusDir.status.values():
            unitName = unitStatus.unitName
            if isDynamicUnit( unitStatus ):
               continue
            allMaintenanceUnits[ unitName ] = populateUnit( unitStatus ) 
         return MaintenanceUnits( maintenanceUnits=allMaintenanceUnits )

BasicCli.addShowCommandClass( ShowUnitsCmd )

# Utility function to return the actual profile name or the Default name
def _getProfileName( profileObject ):
   if profileObject.type == bgpProfileType:
      defaultProfileName = bgpDefaultProfileDir.profileName
   if profileObject.type == intfProfileType:
      defaultProfileName = intfDefaultProfileDir.profileName

   if profileObject.name == MaintenanceCliLib.defaultProfile:
      if defaultProfileName == MaintenanceCliLib.defaultProfile:
         return 'Default'
      else:
         return defaultProfileName
   else:
      return profileObject.name

# Return the maintenance state of the intf based on the maint state of the unit
# that it belongs to. We resort to this only when the intf is not being monitored
# by ratemon, either because the intf is down or no intf status.
def derivedIntfMaintStatus( intfName ):
   intfId = Tac.Value( 'Arnet::IntfId', intfName )
   underMaint = False
   enterMaint = False
   exitMaint = False
   stateTime = None
   unitStatus = unitStatusDir.status.get( dynamicUnitName( intfName ) )
   if unitStatus:
      if unitStatus.state == 'underMaintenance':
         underMaint = True
   
   # Use the reverse mapping collections to find the units the intf is part of and
   # derive the state from those units. If an interface is part of multiple units in
   # different states then the states take precedence in the following order:
   # underMaintenance > maintenanceModeEnter > maintenanceModeExit. If the interface
   # is not part of any unit in the above states then its state is considered active.
   if not underMaint:
      intfToGroup = intfToGroupDir.intfToGroup.get( intfId )
      if intfToGroup:
         groups = intfToGroup.group.keys()
         for group in groups:
            groupInUnit = groupToUnitDir.groupToUnit.get( group )
            if groupInUnit:
               units = groupInUnit.unitName.keys()
               for unit in units:
                  unitStatus = unitStatusDir.status.get( unit )
                  if unitStatus:
                     if unitStatus.state == 'underMaintenance':
                        underMaint = True
                        break
                     elif unitStatus.state == 'maintenanceModeEnter':
                        enterMaint = True
                     elif unitStatus.state == 'maintenanceModeExit':
                        exitMaint = True
   if underMaint:
      state = 'underMaintenance'
      stateTime = unitStatus.timeOfStateChange
   elif enterMaint:
      state = 'maintenanceModeEnter'
   elif exitMaint:
      state = 'maintenanceModeExit'
   else:
      state = 'active'
   return ( state, stateTime )

# For the specified interface, return the tuple of maintenance state and
# state change time. The state change time will be valid only when the state is
# underMaintenance. This function is used by show commands which display the
# maintenance state of the interface.
def getIntfMaintenanceState( intfName ):
   state_ = ''
   stateTime = None
   selectedProfiles = None
   parentIntfName = intfName

   # Subintfs follow the state of the parent intf only in cases when the parent
   # intf is a part of Dynamic Unit or a Builtin Unit because the intfToGroup mapping
   # is absent in cases where sub-interfaces are a part of user-configured unit.
   # It is present in cases of Dynamic units only.
   # For other cases, we proceed with the state of the Subintf itself.

   if MaintenanceCliLib.isSubIntf( intfName ):
      parentIntfName = MaintenanceCliLib.parentIntf( intfName )
   intfToGroup = intfToGroupDir.intfToGroup.get( intfName )
   if intfToGroup:
      selectedProfiles = intfToGroup.selectedProfile
   if not selectedProfiles:
      state_ = 'active'
   else:
      # We don't have monIntfStatus for a SubIntf, so get the status from parent.
      monIntfStatus = monIntfStatusDir.monIntfStatus.get( parentIntfName )
      if monIntfStatus:
         monIntfState = monIntfStatus.state
         if monIntfState == 'monitorThreshold' or monIntfState == 'monitorInit':
            state_ = 'maintenanceModeEnter'
         elif monIntfState == 'monitorPassive':
            state_ = 'underMaintenance'
            stateTime = monIntfStatus.stateTime
         elif monIntfState == 'monitorStop':
            state_ = 'maintenanceModeExit'
         else:
            assert False, 'monIntfState: %s is invalid' % monIntfState
      else:
         # Absence rateMon status can happen due to 2 reasons:
         # 1. Interface was already shut and then it is put under maintenance.
         # 2. Bgp stage progession is not completed for the unit.
         state_, stateTime = derivedIntfMaintStatus( parentIntfName )
   return ( state_, stateTime )

def doShowMaintenanceIntf( mode, intfName ):
   status = MaintenanceModels.MaintenanceInterfaceStatus()
   intfId = Tac.Value( 'Arnet::IntfId', intfName )
   intfToGroup = intfToGroupDir.intfToGroup.get( intfId )
   if not intfToGroup:
      return status
   groups = intfToGroup.group.keys()
   for group in groups:
      if re.match( DynamicComponentRe, group.name ):
         continue
      status.groups.append( group.name )

   status.state, _ = getIntfMaintenanceState( intfName )
   status.adminState = getAdminState( status.state )

   selectedProfiles = intfToGroup.selectedProfile
   for profile in selectedProfiles.values():
      if profile.type == bgpProfileType:
         status.bgpProfile = _getProfileName( profile )
      if profile.type == intfProfileType:
         status.intfProfile = _getProfileName( profile )
   for hook in MaintenanceModels.showMaintenanceInterfaceHook.extensions():
      hook( mode, intfId, status )
   return status

def doShowMaintenanceIntfBrief( mode, intfName, stateFilter ):
   intfId = Tac.Value( 'Arnet::IntfId', intfName )

   if not intfId in intfToGroupDir.intfToGroup:
      return None

   monIntfStatus = monIntfStatusDir.monIntfStatus.get( intfName )
   intfMaintState, _ = getIntfMaintenanceState( intfName )

   if stateFilter and ( intfMaintState not in stateFilter ):
      return None

   model = MaintenanceInterfaceBrief()
   model.state = intfMaintState
   model.adminState = getAdminState( intfMaintState )
   if monIntfStatus:
      model.lastInBpsRate = monIntfStatus.curRates.inBitsRate
      model.lastOutBpsRate = monIntfStatus.curRates.outBitsRate
      if intfId in errDisableCauseStatus.intfStatus:
         model.maintDownTime = \
               toUtc( errDisableCauseStatus.intfStatus[ intfId ].errdisabledTime )
      if monIntfStatus.state == 'monitorPassive':
         # We worry about violations only after the interface has entered
         # maintenance
         if _isIntfTrafficAboveThreshold( monIntfStatus ):
            model.trafficAboveThreshold = True
   return model

#-------------------------------------------------------------------------------
# show maintenance interface [ INTF ] detail
#-------------------------------------------------------------------------------
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface-specific details')
matcherIntf = Intf.IntfRange.IntfRangeMatcher( noSingletons=False,
      explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType,
                          LagAutoIntfType, VlanAutoIntfType ) )

# Utility function to return a list of interface names based on the argument
# which could contain a range too.
def getIntfNames( intfs ):
   intfNames = []
   if not intfs:
      # No arguments was specified
      intfNames = Arnet.sortIntf( intfToGroupDir.intfToGroup )
   elif isinstance( intfs, Intf.IntfRange.IntfList ):
      # a range was specified
      intfNames = intfs.intfNames()
   else:
      # just a name was specified
      intfNames.append( intfs.name )
   return intfNames

class ShowMaintenanceIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance interface [ INTF ] detail'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'interface' : matcherInterface,
      'INTF' : matcherIntf,
      'detail' : 'Display information in detail'
   }
   cliModel = IntfMaintenanceModels.MaintenanceInterfaces

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      intfs = args.get( 'INTF' )
      model = IntfMaintenanceModels.MaintenanceInterfaces()
      intfNames = getIntfNames( intfs )
      def genIntfMaintenanceStatus():
         for intfName in intfNames:
            intfModel = doShowMaintenanceIntf( mode, intfName )
            yield intfName, intfModel
      model.interfaces = genIntfMaintenanceStatus()
      return model

BasicCli.addShowCommandClass( ShowMaintenanceIntfCmd )

#-------------------------------------------------------------------------------
# show maintenance interface [ INTF ] [ status STATUS ]
#-------------------------------------------------------------------------------
statusToStateMapping = {
   'entering' : 'maintenanceModeEnter',
   'exiting' : 'maintenanceModeExit',
   'quiesced' : 'underMaintenance',
   'active' : 'active',
}

class ShowMaintenanceIntfBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance interface [ INTF ] [ status STATUS ]'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'interface' : matcherInterface,
      'INTF' : matcherIntf,
      'status' : 'Show interfaces in specified state',
      'STATUS' : CliCommand.SetEnumMatcher( {
         'entering' : 'Show only those entering maintenance',
         'exiting' : 'Show only those exiting maintenance',
         'quiesced' : 'Show only those under maintenance',
         'active' : 'Show only those not under maintenance'
      } )
   }
   cliModel = MaintenanceInterfacesBrief

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      model = MaintenanceInterfacesBrief()
      optMaintState = args.get( 'STATUS' )
      intfNames = getIntfNames( args.get( 'INTF' ) )

      stateFilter = []
      if optMaintState:
         stateFilter = [ statusToStateMapping[ i ] for i in optMaintState ]
      for intfName in intfNames:
         intfBriefModel = doShowMaintenanceIntfBrief( mode, intfName, stateFilter )
         if intfBriefModel:
            model.interfaces[ intfName ] = intfBriefModel

      return model

BasicCli.addShowCommandClass( ShowMaintenanceIntfBriefCmd )

#-------------------------------------------------------------------------------
# show maintenance debug [ ( units [ UNIT_NAME ] ) | ( interface [ INTF ] ) ]
#-------------------------------------------------------------------------------
def populateEventInfo( maintEvents, logs ):
   for logEntry in logs.itervalues():
      entry = MaintenanceEventInfoEntry()
      entry.timeStamp = toUtc( logEntry.timestamp )
      entry.eventInfo = logEntry.msg
      maintEvents.append( entry )

def populateDebugInfo( unitName ):
   maintenanceEventInfo = MaintenanceEventInfo()
   unitStatus = unitStatusDir.status.get( unitName )
   if unitStatus:
      instanceLogs = maintEnterInstanceLog.instanceLog.get( 
         unitStatus.maintEnterInstance, None )
      if instanceLogs:
         populateEventInfo( maintenanceEventInfo.maintEnterEvents,
                            instanceLogs.log )
      instanceLogs = maintExitInstanceLog.instanceLog.get( 
         unitStatus.maintExitInstance, None )
      if instanceLogs:
         populateEventInfo( maintenanceEventInfo.maintExitEvents,
                         instanceLogs.log )
   return maintenanceEventInfo

def showMaintenanceUnitsDebugInfo( mode, unitName=None ):
   allDebugInfo = {}
   if unitName:
      if MaintenanceModeCliLib.isLinecardBuiltinUnit( unitName ):
         for linecard in unitName:
            if linecard.name in unitStatusDir.status:
               allDebugInfo[ ( 'Unit %s' % linecard.name ) ] = \
                              populateDebugInfo( linecard.name )
      else:
         unitName = MaintenanceModeCliLib.fixIfSystemUnitName( unitName )
         if unitName in unitStatusDir.status:
            allDebugInfo[ ( 'Unit %s' % unitName ) ] = populateDebugInfo( unitName )
   else:
      for unitStatus in unitStatusDir.status.values():
         unitName = unitStatus.unitName
         if isDynamicUnit( unitStatus ):
            continue
         allDebugInfo[ ( 'Unit %s' %unitName ) ] = populateDebugInfo( unitName )
   return MaintenanceDebugInfo( maintenanceEventInfo=allDebugInfo )

def showMaintenanceInterfaceDebugInfo( mode, intfs=None ):
   allDebugInfo = {}
   if not intfs:
      for unitStatus in unitStatusDir.status.values():
         unitName = unitStatus.unitName
         if isDynamicUnit( unitStatus ): 
            group =  unitStatus.group.keys()[ 0 ]
            if group.type == intfGroupType:
               intfName = dynamicUnitComponentName( unitStatus )
               allDebugInfo[ ( 'Interface %s' %intfName ) ] = \
                   populateDebugInfo( unitName )
   else:
      intfNames = []
      if isinstance( intfs, Intf.IntfRange.IntfList ):
         intfNames = intfs.intfNames()
      else:
         intfNames.append( intfs.name )
      for intfName in intfNames:
         intfUnitName = dynamicUnitName( intfName )
         if intfUnitName in unitStatusDir.status:
            allDebugInfo[ ( 'Interface %s' %str( intfName ) ) ] = \
                populateDebugInfo( intfUnitName )
   return MaintenanceDebugInfo( maintenanceEventInfo=allDebugInfo )

class ShowMaintenanceUnitsDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show maintenance debug [ ( units [ UNIT_NAME ] ) | '
                                       '( interface [ INTF ] ) ]' )
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'debug' : 'Debug information',
      'units' : matcherUnits,
      'UNIT_NAME' : MaintenanceModeCliLib.UnitNameExpr(),
      'interface' : matcherInterface,
      'INTF' : matcherIntf,
   }
   cliModel = MaintenanceDebugInfo

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      if 'units' in args:
         return showMaintenanceUnitsDebugInfo( mode,
               unitName=args.get( 'UNIT_NAME' ) )
      elif 'interface' in args:
         return showMaintenanceInterfaceDebugInfo( mode, intfs=args.get( 'INTF' ) )
      else:
         intfDebugInfo = showMaintenanceInterfaceDebugInfo( mode )
         unitsDebugInfo = showMaintenanceUnitsDebugInfo( mode )
         for hook in bgpMaintenanceDebugInfoHook.extensions():
            bgpDebugInfo = hook( mode, ignoreDisplay=True )
         allDebugInfo = intfDebugInfo.maintenanceEventInfo
         allDebugInfo = dict( allDebugInfo, 
                              **( unitsDebugInfo.maintenanceEventInfo ) )
         allDebugInfo = dict( allDebugInfo, 
                              **( bgpDebugInfo.maintenanceEventInfo ) )
         return MaintenanceDebugInfo( maintenanceEventInfo=allDebugInfo )

BasicCli.addShowCommandClass( ShowMaintenanceUnitsDebugCmd )

#-------------------------------------------------------------------------------
# show maintenance groups
#-------------------------------------------------------------------------------
def getGroupsInfoFromHooks( mode, groups, groupName=None, hookExtension=None ):
   if not hookExtension:
      return
   for hook in hookExtension.extensions():
      hook( mode, groups, groupName )

class ShowGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance groups'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'groups' : 'Groups information'
   }
   cliModel = MaintenanceGroups

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      groups = MaintenanceGroups()
      getGroupsInfoFromHooks( mode, groups, None, intfGroupsHook )
      getGroupsInfoFromHooks( mode, groups, None, bgpGroupsHook )
      return groups

BasicCli.addShowCommandClass( ShowGroupsCmd )

#-------------------------------------------------------------------------------
# show maintenance profiles
#-------------------------------------------------------------------------------
matcherProfile = CliMatcher.KeywordMatcher( 'profiles',
      helpdesc='Profiles information' )

def populateProfiles( mode, profilesModel, profilesHook, profileName ):
   for hook in profilesHook.extensions():
      hook( mode, profilesModel, profileName )

def showUnitProfiles_( mode, unitProfileName=None ):
   allUnitProfiles = {}
   if not unitProfileName:
      for profile in unitProfileStatusDir.status:
         if profile == '#DEFAULTPROFILE':
            continue
         unitProfile = showUnitProfile( profile )
         allUnitProfiles[ profile ] = unitProfile
   elif unitProfileName in unitProfileStatusDir.status:
      unitProfile = showUnitProfile( unitProfileName )
      allUnitProfiles[ unitProfileName ] = unitProfile
   return allUnitProfiles

class ShowProfilesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance profiles'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'profiles' : matcherProfile,
   }
   cliModel = MaintenanceProfiles

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      profileType = None
      profileName = None
      profilesModel = MaintenanceProfiles()
      if profileType == 'interface':
         populateProfiles( mode, profilesModel, intfProfilesHook, profileName )
      elif profileType == 'bgp':
         populateProfiles( mode, profilesModel, bgpProfilesHook, profileName )
      elif profileType is None and profileName is None:
         populateProfiles( mode, profilesModel, intfProfilesHook, profileName )
         populateProfiles( mode, profilesModel, bgpProfilesHook, profileName )
         profilesModel.unitProfiles = showUnitProfiles_( mode, profileName )
      return profilesModel

BasicCli.addShowCommandClass( ShowProfilesCmd )

#-------------------------------------------------------------------------------
# show maintenance profiles unit [ PROFILE ]
#-------------------------------------------------------------------------------
matcherUnit = CliMatcher.KeywordMatcher( 'unit', helpdesc='Unit information' )

class ShowUnitProfilesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance profiles unit [ PROFILE ]'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'profiles' : matcherProfile,
      'unit' : matcherUnit,
      'PROFILE' : MaintenanceCliLib.profileNameMatcher(
         lambda mode: unitProfileStatusDir.status.keys() )
   }
   cliModel = UnitProfiles

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      allUnitProfiles = showUnitProfiles_( mode, args.get( 'PROFILE' ) )
      return UnitProfiles(unitProfiles = allUnitProfiles )

BasicCli.addShowCommandClass( ShowUnitProfilesCmd )

#-------------------------------------------------------------------------------
#  show maintenance profiles unit default
#-------------------------------------------------------------------------------
class ShowMaintenanceProfilesUnitDefaultCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance profiles unit default'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'profiles' : matcherProfile,
      'unit' : matcherUnit,
      'default' : MaintenanceCliLib.defaultMatcher
   }
   cliModel = UnitMaintenanceDefaultProfile

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      unitDefaultProfile = UnitMaintenanceDefaultProfile()
      defaultProfileName = unitDefaultProfileDir.profileName
      if defaultProfileName == MaintenanceCliLib.defaultProfile:
         unitDefaultProfile.profileName = 'Default'
      else:
         unitDefaultProfile.profileName = defaultProfileName
      if defaultProfileName not in unitProfileStatusDir.status:
         return unitDefaultProfile
      unitDefaultProfile.profileAttributes = showUnitProfile( defaultProfileName )
      return unitDefaultProfile

BasicCli.addShowCommandClass( ShowMaintenanceProfilesUnitDefaultCmd )

#-------------------------------------------------------------------------------
# show maintenance stages [ STAGE ]
#-------------------------------------------------------------------------------
class ShowMaintenanceStagesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show maintenance stages [ STAGE ]'
   data = {
      'maintenance' : MaintenanceCliLib.showMaintenanceMatcher,
      'stages' : 'Maintenance Stages',
      'STAGE' : CliMatcher.EnumMatcher( {
         'enter' : 'Stages for Maintenance Enter',
         'exit' : 'Stages for Maintenance Exit'
      } ),
   }
   cliModel = MaintenanceStages

   @staticmethod
   @MaintenanceCliLib.maintRunningShowCliCheck
   def handler( mode, args ):
      maintenanceEnterStages = []
      maintenanceExitStages = []
      maintenanceOp = args.get( 'STAGE' )
      if not maintenanceOp or maintenanceOp == 'enter':
         for stage in MaintenanceCliLib.stages:
            if stage.internal:
               maintenanceEnterStages.append( 
                  MaintenanceStage( stageName=stage.name, 
                                    stageDescription=stage.description )  )
                                                                
      if not maintenanceOp or maintenanceOp == 'exit':
         for stage in reversed( MaintenanceCliLib.stages ):
            if stage.internal:
               maintenanceExitStages.append(
                  MaintenanceStage( stageName=stage.name,
                                    stageDescription=stage.description ) )
      return MaintenanceStages( maintenanceEnterStages=maintenanceEnterStages,
                                maintenanceExitStages=maintenanceExitStages )

BasicCli.addShowCommandClass( ShowMaintenanceStagesCmd )
                              
#-------------------------------------------------------------------------------
# Register Maintenancemode Show Commands Into 'Show Tech-Support'
#-------------------------------------------------------------------------------

def _maintenanceModeTechSupportCmds():
   Cmds = [ 'show maintenance',
            'show maintenance summary',
            'show maintenance stages',
            'show maintenance units',
            'show maintenance debug',
            'show maintenance bgp ip all vrf all',
            'show maintenance bgp ipv6 all vrf all',
            'show maintenance bgp receiver route-map',
            'show maintenance interface',
            'show maintenance groups',
            'show maintenance profiles',
            'show maintenance profiles interface default',
            'show maintenance profiles bgp default',
            'show maintenance profiles unit default' ]
   return Cmds

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2014-10-24 00:00:00',
   _maintenanceModeTechSupportCmds,
   summaryCmdCallback=lambda: [ 'show maintenance summary' ] )

#--------------------------------------------------------------------------------
# show tech-support maintenanceMode
#--------------------------------------------------------------------------------
class TechSupportMaintenancemodeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support maintenanceMode'
   data = {
      'tech-support' : CliPlugin.TechSupportCli.techSupportKwMatcher,
      'maintenanceMode' : 'Show detailed state of MaintenanceMode agent',
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      command = 'DUMP_STATE'
      if MaintenanceCliLib.maintModeAgentIsRunning():
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               MaintenanceModeAgent.name(), # dirName
                                               'maintenanceMode', # commandType
                                               command,
                                               timeout=os.environ.get( 
                                               'MAINTENANCEMODE_BTEST',
                                               120 ) ) # Default timeout is 2 min
      else:
         mode.addWarning( 'Maintenance Mode is disabled.' )

BasicCli.addShowCommandClass( TechSupportMaintenancemodeCmd )

def Plugin( entityManager ):
   global intfToGroupDir, unitConfigDir, unitStatusDir, monIntfStatusDir
   global intfDefaultProfileDir, bgpDefaultProfileDir, unitProfileStatusDir, \
       unitDefaultProfileDir, groupToUnitDir, errDisableCauseStatus, \
       intfProfileStatusDir
   global maintEnterInstanceLog, maintExitInstanceLog

   interfaceStatusHook.addExtension( showInterfaceStatusHook )
   interfaceInfoHook.addExtension( showInterfaceHook )

   intfToGroupDir = LazyMount.mount( entityManager,
                                     'maintenance/mapping/member/interface',
                                     'IntfMaintenance::IntfToGroupDir', 'r' )
   groupToUnitDir = LazyMount.mount( entityManager,
                                     'maintenance/mapping/group',
                                     'Maintenance::GroupToUnitDir', 'r' )
   unitConfigDir = LazyMount.mount( entityManager,
                                    'maintenance/unit/config',
                                    'Maintenance::Unit::ConfigDir', 'r' )
   unitStatusDir = LazyMount.mount( entityManager,
                                    'maintenance/unit/status',
                                    'Maintenance::Unit::StatusDir', 'r' )
   monIntfStatusDir = LazyMount.mount( entityManager,
                                       'maintenance/interface/status/ratemon',
                                       'Maintenance::MonIntfStatusDir', 'r' )
   maintEnterLogPath = Cell.path( 'stage/' +
         MaintenanceCliLib.maintEnterStageClass + '/log' )
   maintExitLogPath = Cell.path( 'stage/' +
         MaintenanceCliLib.maintExitStageClass + '/log' )
   maintEnterInstanceLog = LazyMount.mount( entityManager, maintEnterLogPath,
                                            'Stage::Log', 'r' )
   maintExitInstanceLog = LazyMount.mount( entityManager, maintExitLogPath,
                                           'Stage::Log', 'r' )

   intfDefaultProfileDir = LazyMount.mount(
      entityManager,
      'maintenance/profile/config/default/interface',
      'Maintenance::Profile::DefaultProfile', 'r' )
   bgpDefaultProfileDir = LazyMount.mount(
      entityManager,
      'maintenance/profile/config/default/bgp',
      'Maintenance::Profile::DefaultProfile', 'r' )
   unitProfileStatusDir = LazyMount.mount( entityManager,
                                           'maintenance/profile/status/unit',
                                           'MaintenanceUnitProfile::StatusDir', 'r' )
   unitDefaultProfileDir = LazyMount.mount(
      entityManager,
      'maintenance/profile/config/default/unit',
      'Maintenance::Profile::DefaultProfile', 'r' )
   errDisableCauseStatus = LazyMount.mount( entityManager,
                           'interface/errdisable/cause/maintenance-shutdown',
                           'Errdisable::CauseStatus', 'r' )
   intfProfileStatusDir = LazyMount.mount( entityManager,
                                           'maintenance/profile/status/interface',
                                           'IntfMaintenanceProfile::StatusDir', 'r' )
