# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliMode.IntfMaintenanceMode import IntfMaintenanceMode
import CliParser
from CliPlugin.MaintenanceMode import MaintenanceConfigMode
from CliPlugin.MaintenanceCliLib import maintenanceKwMatcher
from CliPlugin.MaintenanceMode import MaintenanceBuiltinUnitConfigMode
from CliPlugin.IntfGroupCli import addIntfGroupConfig, delIntfGroupConfig
from CliPlugin.IntfGroupCli import matcherGroup, matcherInterface
from CliPlugin.IntfGroupCli import addLinecardGroupToUnit, remLinecardGroupFromUnit 
import CliPlugin.EthIntfCli as EthIntfCli
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.MaintenanceCliLib import dynamicUnitName, dynamicGroupName
from CliPlugin.MaintenanceCliLib import MaintenanceUnit, UnitProfile
from CliPlugin.MaintenanceCliLib import onBootDurationMin, onBootDurationMax
from CliPlugin.MaintenanceCliLib import defaultMatcher, profileMatcher
from CliPlugin.MaintenanceCliLib import intfGroupType, dynamicUnitType, bgpGroupType
from CliPlugin.MaintenanceCliLib import profileNameMatcher, defaultProfile, isSubIntf
from CliPlugin.MaintenanceCliLib import quiesceMatcher, MaintenanceBuiltinUnit
from CliPlugin.MaintenanceCliLib import builtinUnitType
from CliPlugin.MaintenanceCliLib import linecardBuiltinGroupPrefix
from CliPlugin.MaintenanceMode import MaintenanceUnitConfigMode
from CliPlugin.MaintenanceMode import MaintenanceUnitProfileConfigMode
from CliPlugin.MaintenanceModels import profilesCleanupHook
from CliPlugin.MaintenanceModels import bgpMaintenanceGroupCleanupHook
from CliPlugin.MaintenanceModels import intfMaintenanceGroupCleanupHook
from CliPlugin.VlanIntfCli import VlanAutoIntfType
import ConfigMount
import Intf
import LazyMount
from MaintenanceModeCliLib import UnitNameExpr
from MaintenanceModeCliLib import isLinecardBuiltinUnit, fixIfSystemUnitName
from MaintenanceModeCliLib import getLineCardRangeNode
import MultiRangeRule
import ShowCommand
import Tac

nodeLinecard = getLineCardRangeNode( None, helpDesc='Linecard builtin group',
                                     tagLong=linecardBuiltinGroupPrefix )

maintenanceRunnabilityConfigDir = None
maintenanceUnitInputDir = None
maintenanceUnitStatusDir = None
maintenanceUnitConfigDir = None
maintenanceUnitProfileDir = None
intfGroupConfigDir = None
defaultUnitProfile = None

GroupOrigin = Tac.Type( "Group::GroupOrigin" )

# Helper functions to quiesce a unit under built-in or user-configured modes
def enterMaintenanceMode( mode, unitName ):
   if maintenanceUnitConfigDir.config.has_key( unitName ):
      maintenanceUnitInputDir.unit[ unitName ] = True
   else:
      mode.addError( "The unit '%s' is not configured" % unitName )

def exitMaintenanceMode( mode, unitName ):
   if maintenanceUnitConfigDir.config.has_key( unitName ):
      del maintenanceUnitInputDir.unit[ unitName ]
   else:
      mode.addError( "The unit '%s' is not configured" % unitName )

#-------------------------------------------------------------------------------
# [no] quiesce
# in "config-unit-<unitName>"
#-------------------------------------------------------------------------------
class QuiesceMaintenanceModeCmd( CliCommand.CliCommandClass ):
   syntax = 'quiesce'
   noOrDefaultSyntax = syntax
   data = {
      'quiesce' : quiesceMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      enterMaintenanceMode( mode, mode.maintenanceUnit_.name() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      exitMaintenanceMode( mode, mode.maintenanceUnit_.name() )

MaintenanceUnitConfigMode.addCommandClass( QuiesceMaintenanceModeCmd )

def getIntfIdList( intfs ):
   intfIdList = []
   if isinstance( intfs, Intf.IntfRange.IntfList ):
      for intfName in intfs.intfNames():
         intfId = Tac.ValueConst( 'Arnet::IntfId', intfName )
         intfIdList.append( intfId )
   else:
      intfId = Tac.ValueConst( 'Arnet::IntfId', intfs.name )
      intfIdList.append( intfId )
   return intfIdList

def enterMaintenanceModeIntfs( mode, intfs ):
   intfIdList = getIntfIdList( intfs )
   for intfId in sorted( intfIdList ):
      unitName = dynamicUnitName( intfId.stringValue )
      maintenanceUnitInputDir.unit[ unitName ] = True
      
def exitMaintenanceModeIntfs( mode, intfs ):
   intfIdList = getIntfIdList( intfs )
   for intfId in sorted( intfIdList ):
      unitName = dynamicUnitName( intfId.stringValue )
      if maintenanceUnitInputDir.unit.has_key( unitName ):
         del maintenanceUnitInputDir.unit[ unitName ]

   # Dont delete the dynamic group and unit, since this is required by
   # MaintenanceMode Agent. Then when do we delete this? 
   # Everytime a maintenance operation is performed on a dynamic unit
   # flush the stale dynamic units which have finished their maintenance operation

#-------------------------------------------------------------------------------
# [no] interface <intfRange>
# in "config-maintenance" mode
#-------------------------------------------------------------------------------

class IntfMaintenanceConfigMode( IntfMaintenanceMode,
                                 BasicCli.ConfigModeBase ):
   name = "Interface Maintenance configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, intfRange ):
      self.intfRange_ = intfRange
      self.session_ = session
      IntfMaintenanceMode.__init__( self, intfRange, self.intfRange_.getShortname(
            str( intfRange ) ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      
class IntfMaintenanceRangeConfigMode( BasicCli.ConfigModeBase ):
   name = "Interface Range Maintenance configuration"
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True
   showActiveAllCmdRegistered_ = True

   def __init__( self, parent, session, intfRange ):
      self.intfRange_ = intfRange
      self.individualIntfModes = []
      intfType = intfRange.type()
      for iName in intfRange.intfNames():
         intf = intfType.getCliIntf( parent, iName )
         mode = IntfMaintenanceConfigMode( parent, session, intf )
         self.individualIntfModes.append( mode )
      self.modeKey = "maint-if-range"
      self.longModeKey = "maint-if-%s" % str( self.intfRange_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session, multiInstance=True,
                                        multiModes = self.individualIntfModes )

#--------------------------------------------------------------------------------
# show active [ all [ detail ] ]
#--------------------------------------------------------------------------------
class ActiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show active [ all [ detail ] ]'
   data = {
      'active' : 'Show the current running-config for this sub mode',
      'all' : 'Configuration with defaults',
      'detail' : 'Detail configuration with defaults',
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      detail = args.get( 'detail', False )
      for m in mode.individualIntfModes:
         if 'all' in args:
            m.showActiveAll( detail )
         else:
            m.showActive()

IntfMaintenanceRangeConfigMode.addShowCommandClass( ActiveCmd )

#--------------------------------------------------------------------------------
# [ no | default ] interface INTF
#--------------------------------------------------------------------------------
class IntfMaintenanceConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'interface INTF'
   noOrDefaultSyntax = syntax
   data = {
      'interface' : 'Interface',
      'INTF' : Intf.IntfRange.IntfRangeMatcher( noSingletons=False,
         explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType,
                             LagAutoIntfType, VlanAutoIntfType ) ),
   }

   @staticmethod
   def handler( mode, args ):
      intfs = args[ 'INTF' ]
      if isSubIntf( str( intfs ) ):
         mode.addError( "Sub-interface config not supported" )
         return
      intfIdList = getIntfIdList( intfs )
      for intfId in intfIdList:
         unitName = dynamicUnitName( intfId.stringValue )
         groupName = dynamicGroupName( intfId.stringValue )
         dynamicUnit = maintenanceUnitConfigDir.newConfig( unitName )
         dynamicUnit.unitType = dynamicUnitType
         dynamicGroup = intfGroupConfigDir.newConfig( groupName )
         dynamicGroup.origin = GroupOrigin.dynamic
         dynamicGroup.intf[ intfId ] = True
         groupKey = Tac.Value( 'Group::GroupKey', intfGroupType, groupName )
         dynamicUnit.group[ groupKey ] = True

      if isinstance( intfs, Intf.IntfRange.IntfList ):
         childMode = mode.childMode( IntfMaintenanceRangeConfigMode,
                                     intfRange=intfs )
      else:
         childMode = mode.childMode( IntfMaintenanceConfigMode,
                                     intfRange=intfs )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfs = args[ 'INTF' ]
      if isSubIntf( str( intfs ) ):
         mode.addError( "Sub-interface config not supported" )
         return
      intfIdList = getIntfIdList( intfs )
      for intfId in intfIdList:
         unitName = dynamicUnitName( intfId.stringValue )
         groupName = dynamicGroupName( intfId.stringValue )
         del maintenanceUnitConfigDir.config[ unitName ]
         del intfGroupConfigDir.config[ groupName ]
         if maintenanceUnitInputDir.unit.has_key( unitName ):
            del maintenanceUnitInputDir.unit[ unitName ]

MaintenanceConfigMode.addCommandClass( IntfMaintenanceConfigModeCmd )

#-------------------------------------------------------------------------------
# [no] quiesce
# in "config-maint-if-<intfRange>" mode
#-------------------------------------------------------------------------------
class QuiesceIntfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'quiesce'
   noOrDefaultSyntax = syntax
   data = {
      'quiesce' : quiesceMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      enterMaintenanceModeIntfs( mode, mode.intfRange_ )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      exitMaintenanceModeIntfs( mode, mode.intfRange_ )

IntfMaintenanceConfigMode.addCommandClass( QuiesceIntfModeCmd )
IntfMaintenanceRangeConfigMode.addCommandClass( QuiesceIntfModeCmd )

unitProfileNameMatcher = profileNameMatcher( lambda mode:
                                          maintenanceUnitProfileDir.config.keys() )
matcherUnit = CliMatcher.KeywordMatcher( 'unit',
      helpdesc='Configure unit' )

#--------------------------------------------------------------------------------
# [ no | default ] profile unit PROFILE_NAME default
# in "config-maintenance" mode
#--------------------------------------------------------------------------------
class ProfileUnitProfilenameDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'profile unit PROFILE_NAME default'
   noOrDefaultSyntax = syntax
   data = {
      'profile' : profileMatcher,
      'unit' : matcherUnit,
      'PROFILE_NAME' : unitProfileNameMatcher,
      'default' : defaultMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      defaultUnitProfile.profileName = profileName
    
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName == defaultUnitProfile.profileName:
         defaultUnitProfile.profileName = defaultProfile  

MaintenanceConfigMode.addCommandClass( ProfileUnitProfilenameDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] profile unit PROFILE_NAME
# in "config-maintenance" mode
#--------------------------------------------------------------------------------
def _delUnitProfileConfig( mode, profileName ):
   unitProfile = UnitProfile( profileName )
   unitProfile.remProfile( profileName )

class ProfileUnitProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'profile unit PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile' : profileMatcher,
      'unit' : matcherUnit,
      'PROFILE_NAME' : unitProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName.lower() == "default":
         mode.addError( "The profile name '%s' is reserved."  % profileName )
         return
      unitProfile = UnitProfile( profileName )
      unitProfile.addProfile( profileName )
      childMode = mode.childMode( MaintenanceUnitProfileConfigMode,
                                  unitProfile=unitProfile )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _delUnitProfileConfig( mode, args[ 'PROFILE_NAME' ] )

MaintenanceConfigMode.addCommandClass( ProfileUnitProfilenameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] on-boot duration DURATION
#--------------------------------------------------------------------------------
class OnBootCmd( CliCommand.CliCommandClass ):
   syntax = 'on-boot duration DURATION'
   noOrDefaultSyntax = 'on-boot ...'
   data = {
      'on-boot' : 'Configuration to be used while entering maintenance on boot',
      'duration' : 'Time to stay in maintenance mode',
      'DURATION' : CliMatcher.IntegerMatcher( onBootDurationMin, onBootDurationMax,
         helpdesc='Number of seconds for the on-boot duration timer' ),
   }

   @staticmethod
   def handler( mode, args ):
      profileName = mode.unitProfile_.name()
      unitProfile = maintenanceUnitProfileDir.config.get( profileName )
      if unitProfile:
         unitProfile.onBootDuration = args[ 'DURATION' ]
         unitProfile.onBootEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = mode.unitProfile_.name()
      unitProfile = maintenanceUnitProfileDir.config.get( profileName )
      if unitProfile:
         unitProfile.onBootEnabled = False
         unitProfile.onBootDuration = onBootDurationMin

MaintenanceUnitProfileConfigMode.addCommandClass( OnBootCmd )

#-------------------------------------------------------------------------------
# [no] unit <unit_name>
# in "config-maintenance" mode
#-------------------------------------------------------------------------------
def _handleLinecardBuiltinUnit( mode, unitName ):
   linecardList = []
   for linecard in unitName:
      linecardList.append( int( linecard.label ) )
      MaintenanceBuiltinUnit( linecard.name )
   linecardModeName = "Linecard" + MultiRangeRule.multiRangeToCanonicalString(
      linecardList )
   childMode = mode.childMode( MaintenanceBuiltinUnitConfigMode,
                               builtinUnit=linecardModeName, unitObject=unitName )
   mode.session_.gotoChildMode( childMode )

def _handleUserConfiguredUnit( mode, unitName ):
   maintenanceUnit = MaintenanceUnit( unitName )
   childMode = mode.childMode( MaintenanceUnitConfigMode,
                               maintenanceUnit=maintenanceUnit )
   mode.session_.gotoChildMode( childMode )

def _delUnitConfig( mode, unitName ):
   unitConfig = maintenanceUnitConfigDir.config.get( unitName )
   if not unitConfig:
      return
   if unitConfig.unitType == dynamicUnitType:
      for groupKey in unitConfig.group:
         if groupKey.type == intfGroupType:
            for hook in intfMaintenanceGroupCleanupHook.extensions():
               hook( mode, groupKey.name )
         elif groupKey.type == bgpGroupType:
            for hook in bgpMaintenanceGroupCleanupHook.extensions():
               hook( mode, groupKey.name )

   del maintenanceUnitConfigDir.config[ unitName ]
   del maintenanceUnitInputDir.unit[ unitName ]

def _delMaintenanceUnitConfig( mode, unitName ):
   if isLinecardBuiltinUnit( unitName ):
      for linecard in unitName:
         _delUnitConfig( mode, linecard.name )
      return
   unitName = fixIfSystemUnitName( unitName )
   _delUnitConfig( mode, unitName )

class UnitCmd( CliCommand.CliCommandClass ):
   syntax = 'unit UNIT_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'unit' : matcherUnit,
      'UNIT_NAME' : UnitNameExpr()
   }

   @staticmethod
   def handler( mode, args ):
      unitName = args[ 'UNIT_NAME' ]
      childMode = None
      if isLinecardBuiltinUnit( unitName ):
         _handleLinecardBuiltinUnit( mode, unitName )
      else:
         unitName = fixIfSystemUnitName( unitName )
         unitStatus = maintenanceUnitStatusDir.status.get( unitName )
         if unitStatus and unitStatus.unitType == builtinUnitType:
            # This code will only be hit for built-in System unit as of now
            maintenanceBuiltinUnit = MaintenanceBuiltinUnit( unitName )
            childMode = mode.childMode( MaintenanceBuiltinUnitConfigMode,
                                        builtinUnit=maintenanceBuiltinUnit.name() )
            mode.session_.gotoChildMode( childMode )
         else:
            _handleUserConfiguredUnit( mode, unitName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _delMaintenanceUnitConfig( mode, args[ 'UNIT_NAME' ] )

MaintenanceConfigMode.addCommandClass( UnitCmd )

#-------------------------------------------------------------------------------
# [no] group interface Linecard <3-$>
# in global "config" mode
#-------------------------------------------------------------------------------
class IntfGroupConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'group interface GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'group' : matcherGroup,
      'interface' : matcherInterface,
      'GROUP' : nodeLinecard,
   }

   @staticmethod
   def handler( mode, args ):
      addIntfGroupConfig( mode, args[ 'GROUP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      delIntfGroupConfig( mode, args[ 'GROUP' ] )

BasicCli.GlobalConfigMode.addCommandClass( IntfGroupConfigModeCmd )

#-------------------------------------------------------------------------------
# [no] group interface Linecard <3-$>
# in "config-unit-<unitName>" mode
#-------------------------------------------------------------------------------
class LinecardGroupToUnitCmd( CliCommand.CliCommandClass ):
   syntax = 'group interface GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'group' : matcherGroup,
      'interface' : matcherInterface,
      'GROUP' : nodeLinecard,
   }

   @staticmethod
   def handler( mode, args ):
      addLinecardGroupToUnit( mode, args[ 'GROUP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      remLinecardGroupFromUnit( mode, args[ 'GROUP' ] )

MaintenanceUnitConfigMode.addCommandClass( LinecardGroupToUnitCmd )

#-------------------------------------------------------------------------------
# [no] quiesce
# in "config-maint-builtin-unit-<linecardName>"
#-------------------------------------------------------------------------------
class QuiesceUnitConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'quiesce'
   noOrDefaultSyntax = syntax
   data = {
      'quiesce' : quiesceMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if mode.builtinUnit_ == "System":
         enterMaintenanceMode( mode, mode.builtinUnit_ )
      elif mode.builtinUnit_.startswith( "Linecard" ):
         for linecardUnit in mode.unitObject_:
            enterMaintenanceMode( mode, linecardUnit.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.builtinUnit_ == "System":
         exitMaintenanceMode( mode, mode.builtinUnit_ )
      elif mode.builtinUnit_.startswith( "Linecard" ):
         for linecardUnit in mode.unitObject_:
            exitMaintenanceMode( mode, linecardUnit.name )

MaintenanceBuiltinUnitConfigMode.addCommandClass( QuiesceUnitConfigModeCmd )

#-------------------------------------------------------------------------------
# [ no | default ] profile unit PROFILE_NAME
# in "config-unit-<unitName>" mode
#-------------------------------------------------------------------------------
class ProfileUnitProfilenameMaintenanceUnitCmd( CliCommand.CliCommandClass ):
   syntax = 'profile unit PROFILE_NAME'
   noOrDefaultSyntax = 'profile unit [ PROFILE_NAME ]'
   data = {
      'profile' : profileMatcher,
      'unit' : matcherUnit,
      'PROFILE_NAME' : unitProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName.lower() == "default":
         mode.addError( "The profile name '%s' is reserved."  % profileName )
         return
      unitName = mode.maintenanceUnit_.name()
      unitConfig = maintenanceUnitConfigDir.config.get( unitName )
      if unitConfig:
         unitConfig.profileName = profileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args.get( 'PROFILE_NAME' )
      unitName = mode.maintenanceUnit_.name()
      unitConfig = maintenanceUnitConfigDir.config.get( unitName )
      if unitConfig:
         if profileName:
            if unitConfig.profileName == profileName:
               unitConfig.profileName = ''
         else:
            unitConfig.profileName = ''

MaintenanceUnitConfigMode.addCommandClass( ProfileUnitProfilenameMaintenanceUnitCmd )

#-------------------------------------------------------------------------------
# [ no | default ] profile unit [ PROFILE_NAME ]
# in "config-builtin-unit-<unitName>" mode
#-------------------------------------------------------------------------------
def _addUnitProfile( unitName, profileName ):
   unitConfig = maintenanceUnitConfigDir.config.get( unitName )
   if unitConfig:
      unitConfig.profileName = profileName


def _remUnitProfile( unitName, profileName=None ):
   unitConfig = maintenanceUnitConfigDir.config.get( unitName )
   if unitConfig:
      if profileName:
         if unitConfig.profileName == profileName:
            unitConfig.profileName = ''
      else:
         unitConfig.profileName = ''

class ProfileUnitCmd( CliCommand.CliCommandClass ):
   syntax = 'profile unit PROFILE_NAME'
   noOrDefaultSyntax = 'profile unit [ PROFILE_NAME ]'
   data = {
      'profile' : profileMatcher,
      'unit' : matcherUnit,
      'PROFILE_NAME' : unitProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName.lower() == "default":
         mode.addError( "The profile name '%s' is reserved."  % profileName )
         return
      unitName = mode.builtinUnit_
      if unitName.startswith( 'Linecard' ):
         for linecard in mode.unitObject_:
            _addUnitProfile( linecard.name, profileName )
      else:
         _addUnitProfile( unitName, profileName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args.get( 'PROFILE_NAME' )
      unitName = mode.builtinUnit_
      if unitName.startswith( 'Linecard' ):
         for linecard in mode.unitObject_:
            _remUnitProfile( linecard.name, profileName )
      else:
         _remUnitProfile( unitName, profileName )

MaintenanceBuiltinUnitConfigMode.addCommandClass( ProfileUnitCmd )

#-------------------------------------------------------------------------------
# ( no | default ) maintenance
# in "config" mode
# In 'no maintenance' we are not deleting the maintenance groups.
#-------------------------------------------------------------------------------
class MaintenanceCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'maintenance'
   data = {
      'maintenance' : maintenanceKwMatcher,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for unitName in maintenanceUnitConfigDir.config:
         _delMaintenanceUnitConfig( mode, unitName )

      for profileName in maintenanceUnitProfileDir.config:
         if profileName != defaultProfile:
            _delUnitProfileConfig( mode, profileName )
      defaultUnitProfile.profileName = defaultProfile

      for hook in profilesCleanupHook.extensions():
         hook( mode )

      # This should be done after all units and profiles are cleaned up to ensure
      # that MaintenanceMode agent is stopped only after cleanup is complete
      maintenanceRunnabilityConfigDir.deleteEntity( 'enabled' )

BasicCli.GlobalConfigMode.addCommandClass( MaintenanceCmd )

def Plugin( entityManager ):
   global maintenanceRunnabilityConfigDir
   global maintenanceUnitInputDir, maintenanceUnitStatusDir
   global maintenanceUnitConfigDir, intfGroupConfigDir
   global maintenanceUnitProfileDir, defaultUnitProfile
   
   maintenanceRunnabilityConfigDir = ConfigMount.mount(
      entityManager, 'maintenance/runnability/config', 
      'Tac::Dir', 'w' )
   maintenanceUnitInputDir = ConfigMount.mount(
      entityManager, 'maintenance/unit/input/cli',
      'Maintenance::Unit::Input', 'w' )
   maintenanceUnitStatusDir = LazyMount.mount(
      entityManager, 'maintenance/unit/status',
      'Maintenance::Unit::StatusDir', 'r' )
   maintenanceUnitConfigDir = ConfigMount.mount(
      entityManager, 'maintenance/unit/config',
      'Maintenance::Unit::ConfigDir', 'w' )
   maintenanceUnitProfileDir = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/unit',
      'MaintenanceUnitProfile::ConfigDir', 'w' )
   intfGroupConfigDir = ConfigMount.mount(
      entityManager, 'group/config/interface',
      'IntfGroup::ConfigDir', 'w' )
   defaultUnitProfile = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/default/unit',
      'Maintenance::Profile::DefaultProfile', 'w' )
