#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import CliCommand
import CliMatcher
import CliParser
from CliPlugin.FruCli import rangeFn, getCardsFromIdList, modularRprActiveSupeGuard
from CliPlugin.MaintenanceCliLib import systemBuiltinUnitName, matcherUnitName
from CliPlugin.MaintenanceCliLib import linecardBuiltinUnitName
import MultiRangeRule

def supOnlyGuard( mode, token ):
   if Cell.cellType() == 'supervisor':
      return modularRprActiveSupeGuard( mode, token )

   return CliParser.guardNotThisPlatform

def getLineCardRangeNode( name, helpDesc, tagLong, priority=CliParser.PRIO_HIGH ):
   _linecardRangeRule = MultiRangeRule.MultiRangeMatcher(
      noSingletons=False,
      helpdesc=helpDesc,
      tagLong=tagLong,
      value=lambda mode, grList: getCardsFromIdList( mode, grList.values(),
                                                     tagLong ),
      rangeFn=lambda: rangeFn( tagLong ),
      priority=priority )
   return CliCommand.Node( matcher=_linecardRangeRule,
                           guard=supOnlyGuard,
                           alias=name )

matcherSystem = CliMatcher.KeywordMatcher( systemBuiltinUnitName,
      helpdesc='System builtin unit' )

class UnitNameExpr( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      linecardNode = getLineCardRangeNode( name=name,
            helpDesc='Linecard builtin unit',
            tagLong=linecardBuiltinUnitName )

      class Expression( CliCommand.CliExpression ):
         expression = 'System | UNIT | LINECARD'
         data = {
            'System' : CliCommand.Node( matcher=matcherSystem, alias=name ),
            'UNIT' : CliCommand.Node( matcher=matcherUnitName, alias=name ),
            'LINECARD' : linecardNode,
         }

      return Expression

def isLinecardBuiltinUnit( linecardUnitList ):
   """
   Checks whether the given linecardUnitList contains linecards or not
   """
   if isinstance( linecardUnitList, list ):
      for linecard in linecardUnitList:
         assert linecard.name.startswith( linecardBuiltinUnitName )
      return True
   return False

def isSystemBuiltinUnit( unitName ):
   """
   Checks whether the given unitName is System builtin unit ( case insensitive )
   name or not
   """
   if unitName.lower() == systemBuiltinUnitName.lower():
      return True
   return False

def fixIfSystemUnitName( unitName ):
   """
   returns systemBuiltinUnitName for all case combinations of the word system
   """
   if isSystemBuiltinUnit( unitName ):
      return systemBuiltinUnitName
   return unitName
