# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import BasicCli, CliPlugin
from EthIntfCli import EthPhyIntf
import SmashLazyMount, Tac
import TacSigint
import LazyMount
import CliMatcher
import ShowCommand
from CliToken.Mac import macMatcherForShow
import SubIntfCli
import MacsecModel

# Globals
_macsecCounters = None
_macsecStatus = None
_macsecInput = None
_hwStatusSliceDir = None

def warnIfMacsecDisabled( func ):
   ''' Decorator for show commands. If Macsec is shutdown globally,
   throws a warning.'''
   def newFunc( mode, *args, **kwargs ):
      if _macsecInput[ "cli"].shutdown:
         mode.addWarning( "MAC security is disabled globally.")
      return func( mode, *args, **kwargs )
   return newFunc

#----------------------------------------------------------------------------------
# Show tokens
#----------------------------------------------------------------------------------

matcherSecurityForShow = CliMatcher.KeywordMatcher( 'security', 
            helpdesc='MAC security (802.1AE) information' )

matcherDetail = CliMatcher.KeywordMatcher(
          'detail', helpdesc='Display information in detail' )

matcherForProfile = CliMatcher.KeywordMatcher( 'profile',
            helpdesc='MAC security profile name' )

matcherForSource = CliMatcher.KeywordMatcher( 'source',
            helpdesc="Configuration source" )

#----------------------------------------------------------------------------------
# The "show mac security mka counters [intf <intfName>] [ detail ]" command.
#----------------------------------------------------------------------------------
@warnIfMacsecDisabled
def showCounters( mode, args ):
   intf = args.get( 'INTF' ) or args.get( 'SUBINTF' )
   detail = 'detail' in args
   intfId = None
   if intf:
      if not intf.lookup() or not intf.status().deviceName:
         mode.addError( "%s is not operational" % intf.name )
         return None
      intfId = Tac.newInstance( 'Arnet::IntfId', intf.name )

   messageCountersModel = MacsecModel.MacsecMessageCounters()

   if intfId:
      if intfId in _macsecCounters.msgCounter:
         intfCounter = _macsecCounters.get( intfId )
         if intfCounter:
            messageCounterModel = MacsecModel.MacsecMessageCountersInterface()
            messageCounterModel.fromTacc( intfCounter, detail )
            messageCountersModel.interfaces[ intfId ] = messageCounterModel
   else:
      for intfId in _macsecCounters.msgCounter.keys():
         intfCounter = _macsecCounters.get( intfId )
         if intfCounter:
            messageCounterModel = MacsecModel.MacsecMessageCountersInterface()
            messageCounterModel.fromTacc( intfCounter, detail )
            messageCountersModel.interfaces[ intfId ] = messageCounterModel
            TacSigint.check()

   return messageCountersModel

#----------------------------------------------------------------------------------
# The "show mac security interface [ intf <intfName ] [ detail ]" command.
#----------------------------------------------------------------------------------
@warnIfMacsecDisabled
def showInterfaces( mode, args ):
   intf = args.get( 'INTF' ) or args.get( 'SUBINTF' )
   detail = 'detail' in args
   def getHwPostStatus( intfName ):
      sliceId = 'FixedSystem'
      sliceStatus = _hwStatusSliceDir.get( sliceId )
      if sliceStatus is None:
         # Check if a modular system
         cardNum = re.search( r'Ethernet(\d+)/', intfName ).group( 1 )
         sliceId = 'Linecard%s' % cardNum
         sliceStatus = _hwStatusSliceDir.get( sliceId )
      if sliceStatus is None:
         # Check if a slicified Evora
         intfId = Tac.Value( 'Arnet::IntfId', intfName )
         portId = int( Tac.Value( 'Arnet::EthIntfId' ).port( intfId ) )
         fixedSliceId = ( portId - 1 ) / 8
         sliceId = 'FixedSystem-%s' % fixedSliceId
         sliceStatus = _hwStatusSliceDir.get( sliceId )

      if sliceStatus is None:
         return None
      else:
         intfStatus = sliceStatus.get( intfName )
         if intfStatus is None:
            return None
         else:
            return intfStatus.hwPostStatus

   intfId = None
   if intf:
      if not intf.lookup() or not intf.status().deviceName:
         mode.addError( "%s is not operational" % intf.name )
         return None
      intfId = Tac.newInstance( 'Arnet::IntfId', intf.name )

   fipsRestrictions = _macsecStatus.fipsStatus.fipsRestrictions
   macsecInterfaces = MacsecModel.MacsecInterfaces()
   if intfId:
      portStatus = _macsecStatus.portStatus.get( intfId )
      cpStatus = _macsecStatus.cpStatus.get( intfId )
      intfStatus = _macsecStatus.intfStatus.get( intfId )
      if portStatus and cpStatus and intfStatus:
         hwPostStatus = getHwPostStatus( intf.name ) if fipsRestrictions else None
         macsecInterface = MacsecModel.MacsecInterface()
         macsecInterface.fromTacc( portStatus, cpStatus, intfStatus, hwPostStatus,
                                   detail=detail )
         macsecInterfaces.interfaces[ intfId ] = macsecInterface
   else:
      for intfId, portStatus in _macsecStatus.portStatus.items():
         cpStatus = _macsecStatus.cpStatus.get( intfId )
         intfStatus = _macsecStatus.intfStatus.get( intfId )
         if not cpStatus or not intfStatus:
            continue
         hwPostStatus = getHwPostStatus( intfId ) if fipsRestrictions else None

         macsecInterface = MacsecModel.MacsecInterface()
         macsecInterface.fromTacc( portStatus, cpStatus, intfStatus, hwPostStatus,
                                   detail=detail )
         macsecInterfaces.interfaces[ intfId ] = macsecInterface
         TacSigint.check()

   return macsecInterfaces

#----------------------------------------------------------------------------------
# The "show mac security participants [ intf <intfName ] [ detail ]" command.
#----------------------------------------------------------------------------------
@warnIfMacsecDisabled
def showParticipants( mode, args ):
   intf = args.get( 'INTF' ) or args.get( 'SUBINTF' )
   detail = 'detail' in args
   intfId = None
   if intf:
      if not intf.lookup() or not intf.status().deviceName:
         mode.addError( "%s is not operational" % intf.name )
         return None
      intfId = Tac.newInstance( 'Arnet::IntfId', intf.name )

   macsecParticipantsModel = MacsecModel.MacsecParticipants()

   if intfId:
      portStatus = _macsecStatus.portStatus.get( intfId )
      if portStatus and portStatus.actorStatus.values():
         macsecParticipantIntfModel = MacsecModel.MacsecParticipantsInterface()
         for actorStatus in portStatus.actorStatus.values():
            macsecParticipantModel = MacsecModel.MacsecParticipant()
            macsecParticipantModel.fromTacc( actorStatus, detail=detail )
            macsecParticipantIntfModel.participants[ actorStatus.ckn ] = \
                  macsecParticipantModel
            TacSigint.check()
         macsecParticipantsModel.interfaces[ intfId ] = \
               macsecParticipantIntfModel
   else:
      for intfId, portStatus in _macsecStatus.portStatus.items():
         if portStatus.actorStatus.values():
            macsecParticipantIntfModel = MacsecModel.MacsecParticipantsInterface()
            for actorStatus in portStatus.actorStatus.values():
               macsecParticipantModel = MacsecModel.MacsecParticipant()
               macsecParticipantModel.fromTacc( actorStatus, detail=detail )
               macsecParticipantIntfModel.participants[ actorStatus.ckn ] = \
                     macsecParticipantModel
               TacSigint.check()
            macsecParticipantsModel.interfaces[ intfId ] = \
                  macsecParticipantIntfModel

   return macsecParticipantsModel

#----------------------------------------------------------------------------------
# The "show mac security status" command.
#----------------------------------------------------------------------------------
@warnIfMacsecDisabled
def showStatus( mode, args ):
   macsecStatusModel = MacsecModel.MacsecStatus()
   macsecStatusModel.fromTacc( _macsecStatus )
   return macsecStatusModel

class MacSecurityInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security interface [ INTF | SUBINTF ] [ detail ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'interface': 'MAC security interface information',
      'INTF': EthPhyIntf.ethMatcher,
      'SUBINTF': SubIntfCli.subMatcher,
      'detail': matcherDetail,
   }

   handler = showInterfaces
   cliModel = MacsecModel.MacsecInterfaces

BasicCli.addShowCommandClass( MacSecurityInterfaceCmd )

class MacSecurityMkaCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security mka counters [ INTF | SUBINTF ] [ detail ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'mka': 'MAC security key agreement protocol information',
      'counters': 'MAC security counter information',
      'INTF': EthPhyIntf.ethMatcher,
      'SUBINTF': SubIntfCli.subMatcher,
      'detail': matcherDetail,
   }

   handler = showCounters
   cliModel = MacsecModel.MacsecMessageCounters

BasicCli.addShowCommandClass( MacSecurityMkaCountersCmd )

class MacSecurityParticipantsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security participants [ INTF | SUBINTF ] [ detail ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'participants': 'MAC security participants information',
      'INTF': EthPhyIntf.ethMatcher,
      'SUBINTF': SubIntfCli.subMatcher,
      'detail': matcherDetail,
   }

   handler = showParticipants
   cliModel = MacsecModel.MacsecParticipants

BasicCli.addShowCommandClass( MacSecurityParticipantsCmd )

class MacSecurityStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security status'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'status': 'MAC security overall status information',
   }

   handler = showStatus
   cliModel = MacsecModel.MacsecStatus

BasicCli.addShowCommandClass( MacSecurityStatusCmd )

#----------------------------------------------------------------------------------
# "show tech-support" commands
#----------------------------------------------------------------------------------
def _showTechMacsecCmds():
   cmds = []
   if _macsecStatus and _macsecStatus.macsecEnabled:
      cmds += [ "show mac security status",
                "show mac security mka counters",
                "show mac security interface detail",
                "show mac security participants detail" ]
   return cmds

def _showTechSummaryMacsecCmds():
   cmds = []
   if _macsecStatus and _macsecStatus.macsecEnabled:
      cmds += [
         "show mac security interface",
         "show mac security participants",
         ]
   return cmds

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2019-01-21 01:41:05', _showTechMacsecCmds,
      summaryCmdCallback=_showTechSummaryMacsecCmds )

#----------------------------------------------------------------------------------
# The "show mac security profile [ <profile_name> ] [ source <src> ]" command
#----------------------------------------------------------------------------------
@warnIfMacsecDisabled
def showMacsecProfile( mode, args ):

   def getProfileModel( config, source, priority ):
      macsecProfileModel = MacsecModel.MacsecProfile()
      macsecProfileModel.source = source
      macsecProfileModel.priority = priority
      macsecProfileModel.fromTacc( config )
      return macsecProfileModel

   def getProfilesModel( config, src, profileName=None ):
      profileConfig = None
      priority = config.configPriority
      macsecProfilesModel = MacsecModel.MacsecProfiles()
      profiles = None
      if profileName is None:
         profiles = config.profile
      elif profileName in config.profile:
         profiles = [ profileName ]
      else:
         return macsecProfilesModel

      for profile in profiles:
         profileConfig = config.profile[ profile ]
         profileModel = getProfileModel( profileConfig, src, priority )
         macsecProfilesModel.profiles[ profile ] = profileModel
      return macsecProfilesModel

   profileName = args.get( "PROFILE_NAME" )
   srcName = args.get( "SRC_NAME" )
   sources = None
   if srcName is None:
      sources = _macsecInput
   elif srcName in _macsecInput:
      sources = [ srcName ]
   else:
      mode.addErrorAndStop( "source %s doesn't exist." % srcName )
   
   macsecAllProfilesModel = MacsecModel.MacsecAllProfiles()
   for src in sources:
      profilesModel = getProfilesModel( _macsecInput[ src ], src=src,
                                        profileName=profileName )
      macsecAllProfilesModel.sources[ src ] = profilesModel
   return macsecAllProfilesModel

def getSource( mode ):
   return _macsecInput

class MacSecurityProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac security profile [ PROFILE_NAME ] [ source SRC_NAME ]'
   data = {
      'mac': macMatcherForShow,
      'security': matcherSecurityForShow,
      'profile': matcherForProfile,
      'PROFILE_NAME' : CliMatcher.DynamicNameMatcher( lambda mode:dict(),
         'Profile name' ),
      'source': matcherForSource,
      'SRC_NAME': CliMatcher.DynamicNameMatcher( getSource, "Source name" ),
   }

   handler = showMacsecProfile
   cliModel = MacsecModel.MacsecAllProfiles

BasicCli.addShowCommandClass( MacSecurityProfileCmd )

def Plugin( entityManager ):

   global _macsecCounters
   global _macsecStatus
   global _macsecInput
   global _hwStatusSliceDir

   _macsecCounters = SmashLazyMount.mount( entityManager,
         'macsec/counters', "Macsec::Smash::MessageCounters",
         SmashLazyMount.mountInfo( 'reader' ) )

   _macsecStatus = LazyMount.mount( entityManager,
         'macsec/status', "Macsec::Status", 'r' )
  
   _macsecInput = LazyMount.mount( entityManager,
         'macsec/input', "Tac::Dir", 'ri' )

   _hwStatusSliceDir = LazyMount.mount( entityManager,
         'hardware/phy/status/macsec/slice', "Tac::Dir", 'r' )
