# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.Macsec import MacsecProfileBaseMode, MacsecBaseMode
from CliMode.Macsec import MacsecProfileStaticSakMode
import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliPlugin.EthIntfCli import DataplaneIntfModelet
from CliPlugin.SubIntfCli import SubIntfModelet
import CliPlugin.IntfCli as IntfCli
import CliToken
import ConfigMount
import LazyMount
from MacAddr import macAddrMatcher
from MacsecCommon import cliToTacCipherSuite
from ReversibleSecretCli import decodeKey, reversibleSecretCliExpression
import Tac
import Toggles.MacsecToggleLib as macsecToggle
from TypeFuture import TacLazyType

config = None
status = None
hwMacsecStatus = None
hwMacsecProxyCapability = None
trapConfig = None
SubIntfId = Tac.Type( "Arnet::SubIntfId" )
EthIntfId = Tac.Type( "Arnet::EthIntfId" )
MacsecSak = TacLazyType( "Macsec::Sak" )
MacsecSci = TacLazyType( "Macsec::Sci" )
TrapFeatureName = TacLazyType( 'Arnet::TrapFeatureName' )

# Macsec mode
class MacsecMode( MacsecBaseMode, BasicCli.ConfigModeBase ):
   name = "MAC security configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      MacsecBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# Macsec Profile Mode
class MacsecProfileMode( MacsecProfileBaseMode, BasicCli.ConfigModeBase ):
   name = 'MAC security profile configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      MacsecProfileBaseMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.updateInterfaceList()
      
   def profile( self ):
      return config.profile[ self.profileName ]

   def updateInterfaceList( self ):
      # Walk through all interfaces which have this profile enabled and update the
      # intf collection.
      profile = self.profile()
      for intf in config.intfConfig.values():
         if intf.profileName == self.profileName:
            profile.intf[ intf.intfId ] = True

   def validateKeySize( self, cak, cipher, fallback ):
      # Check if the configured key size is consistent with the cipher in use.
      # We only care if the cipher in use is 128bits and the key configured is
      # longer in size.
      keyType = 'fallback' if fallback else 'primary'
      decodedCakLength = len( decodeKey( cak ) )

      if cipher in [ 'gcmAesXpn128', 'gcmAes128' ] and decodedCakLength > 32:
         self.addError( 'Configured %s key is too big for the cipher in use' % \
               keyType )
         return False

      if cipher in [ 'gcmAesXpn256', 'gcmAes256' ] and decodedCakLength > 64:
         self.addError( 'Configured %s key is too big for the cipher in use' % \
               keyType )
         return False

      return True  

   def parseCak( self, ckn, cak ):
      # Verify that CKN is non zero.
      if int( ckn, 16 ) == 0:
         self.addError( 'Key name cannot be zero' )
         return None
      try:
         intCak = int( decodeKey( cak ), 16 )
      except ValueError:
         self.addError( 'Invalid Encrypted Key ' )
         return None 

      if intCak == 0:
         self.addError( 'Key value cannot be zero' )
         return None

      if len( ckn ) % 2 != 0:
         # If the CKN is not even length in size, make it even. It's important
         # for our crypto libraries that the size of ckn be an even number of hex
         # octets.
         ckn = "0" + ckn
      cakVal = Tac.Value( "Macsec::Cak", ckn=ckn.lower(), cak=cak ) 
      return cakVal

   def addKey( self, args ):
      cknString = args[ 'CKN' ]
      cak = args[ 'SECRET' ]
      fallback = 'fallback' in args 

      cakVal = self.parseCak( cknString, cak )
      
      profile = self.profile()

      if not self.validateKeySize( cak, profile.cipherSuite, fallback ):
         return

      # Throw an error if the key matches the default key. The two keys ought to be
      # different.
      conflict = False
      if fallback and profile.key.ckn == cknString:
         conflict = True
      if not fallback and profile.defaultKey.ckn == cknString:
         conflict = True
      if conflict:
         self.addError( 'Primary key should be different from the fallback key' )
         return

      if cakVal:
         if not fallback:
            # If source dot1x was enabled, remove it now.
            if profile.dot1xEnabled:
               self.dot1xDisabled()
            # Add primary key
            profile.key = cakVal
         else:
            # Add the default key
            profile.defaultKey = cakVal

   def noKey( self, args ):
      cknString = args[ 'CKN' ]
      cak = args[ 'SECRET' ]
      fallback = 'fallback' in args

      cakVal = self.parseCak( cknString, cak )

      profile = self.profile()
      # Delete the key from the profile.
      emptyCak = Tac.Value( "Macsec::Cak" )
      if not fallback:
         if profile.key.ckn == cakVal.ckn:
            profile.key = emptyCak
      else:
         if profile.defaultKey.ckn == cakVal.ckn:
            profile.defaultKey = emptyCak

   def addKeyServerPriority( self, args ):
      # Update the key server priority for this profile.
      self.profile().keyServerPriority = args[ 'PRIORITY' ]

   def noKeyServerPriority( self, args ):
      # Restore the key server priority to its default.
      profile = self.profile()
      profile.keyServerPriority = profile.keyServerPriorityDefault

   def addReKeyPeriod( self, args ):
      # Update the session re-key period in this profile.
      self.profile().sessionReKeyPeriod = args[ 'REKEYPERIOD' ]

   def noReKeyPeriod( self, args ):
      # Restore the session re-key period to its default.
      profile = self.profile()
      profile.sessionReKeyPeriod = profile.sessionReKeyPeriodDefault

   def setLpnValueZero( self, args=None ):
      # Enable publishing of LPN values as 0
      self.profile().lpnValueZero = True

   def unsetLpnValueZero( self, args=None ):
      # Enable publishing of real LPN values
      self.profile().lpnValueZero = False

   def dot1xEnabled( self, args ):
      # Delete any configured primary key.
      emptyCak = Tac.Value( "Macsec::Cak" )
      self.profile().key = emptyCak

      # Enable dot1x setting on the profile.
      self.profile().dot1xEnabled = True

   def dot1xDisabled( self, args=None ):
      # Disable dot1x setting on the profile.
      self.profile().dot1xEnabled = False

   def configureGroupCak( self, args ):
      groupCakLifetime = args.get( 'GROUPCAKLIFETIME' )
      lifetime = groupCakLifetime if groupCakLifetime else \
                 self.profile().groupCak.lifetimeDefault
      self.profile().groupCak = Tac.Value( "Macsec::GroupCak", 
                                           enabled=True,
                                           lifetime = lifetime )

   def noGroupCak( self, args ):
      self.profile().groupCak = Tac.Value( "Macsec::GroupCak" )
 
   def configureCipherSuite( self , args ):
      cipher = args.get( 'CIPHER' )
      # Configure the cipher in the profile
      cipherSuite = cliToTacCipherSuite[ cipher ]
    
      primaryKey = self.profile().key.cak 
      defaultKey = self.profile().defaultKey.cak
      if not self.validateKeySize( primaryKey, cipherSuite, False ) or \
            not self.validateKeySize( defaultKey, cipherSuite, True ):
         return

      self.profile().cipherSuite = cipherSuite
   
   def noCipherSuite( self, args ):
      # Configure default ciphersuite
      primaryKey = self.profile().key.cak 
      defaultKey = self.profile().defaultKey.cak
      cipherSuite = self.profile().cipherSuiteDefault # Default

      if not self.validateKeySize( primaryKey, cipherSuite, False ) or \
            not self.validateKeySize( defaultKey, cipherSuite, True ):
         return
   
      self.profile().cipherSuite = cipherSuite

   def includeSci ( self, args ):
      # Enable sci inclusion
      self.profile().includeSci = True

   def noIncludeSci ( self, args ):
      # Disable sci inclusion
      self.profile().includeSci = self.profile().includeSciDefault

   def bypassLldp ( self, args ):
      # Transmit/Receive LLDP frames without protection
      self.profile().bypassLldp = True

   def noBypassLldp ( self, args ):
      # Transmit/Receive encrypted LLDP frames
      self.profile().bypassLldp = self.profile().bypassLldpDefault
   
   def trafficPolicy( self, args ):
      if macsecToggle.toggleMacsecTrafficBlockOnMkaFailureEnabled():
         if 'drop' in args:
            self.profile().trafficPolicyOnNoMka = 'blocked'
         elif 'active-sak' in args:
            self.profile().trafficPolicyOnNoMka = 'useActiveSak'
         else:
            self.profile().trafficPolicyOnNoMka = 'unprotected'
      else:
         self.profile().allowUnprotected = True

   def noOrDefaultTrafficPolicy( self, args ):
      if macsecToggle.toggleMacsecTrafficBlockOnMkaFailureEnabled():
         self.profile().trafficPolicyOnNoMka = \
                                          self.profile().trafficPolicyOnNoMkaDefault
      else:
         self.profile().allowUnprotected = self.profile().allowUnprotectedDefault
   
   def replayProtectionDisabled ( self, args ):
      # Disable Replay Protection
      self.profile().replayProtection = False

   def noReplayProtectionDisabled ( self, args ):
      # Enable Replay Protection
      self.profile().replayProtection = self.profile().replayProtectionDefault

   def replayProtectionWindow ( self, args ):
      replayProtectionWindow = args[ 'REPLAYPROTECTIONWINDOW' ]
      if replayProtectionWindow is None:
         replayProtectionWindow = self.profile().replayProtectionWindowDefault
      self.profile().replayProtectionWindow = replayProtectionWindow

   def noReplayProtectionWindow ( self, args ):
      self.profile().replayProtectionWindow = \
            self.profile().replayProtectionWindowDefault
   
   def keyRetire( self, args ):
      self.profile().keyRetire = True

   def noKeyRetire ( self, args ):
      self.profile().keyRetire = self.profile().keyRetireDefault

   def gotoStaticSakMode( self, args ):
      childMode = self.childMode ( MacsecStaticSakMode, txOrRx= args[ 'TX_RX' ] )
      self.session_.gotoChildMode( childMode )
         
   def noStaticSak( self, args ):
      txOrRx = args[ 'TX_RX' ]
      if txOrRx == 'tx' and self.profile().txStaticSak:
         self.profile().txStaticSak = MacsecSak()
         self.profile().txStaticSci = MacsecSci()
      elif txOrRx == 'rx' and self.profile().rxStaticSak:
         self.profile().rxStaticSak.clear()
         self.profile().rxStaticSci = MacsecSci()

def getProfileNames( mode=None ):
   return sorted( config.profile.members() )

class MacsecStaticSakMode ( MacsecProfileStaticSakMode, BasicCli.ConfigModeBase ):
   name = "MAC security static SAK configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, txOrRx ):
      self.profile = parent.profile()
      self.direction = txOrRx
      MacsecProfileStaticSakMode.__init__( self, ( self.profile.name, txOrRx ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setTxStaticSakParam( self, sak=None, sci=None ):
      if sak:
         self.profile.txStaticSak = sak
      if sci:
         self.profile.txStaticSci = sci

   def setRxStaticSakParam( self, sak=None, sci=None ):
      if sak:
         self.profile.addRxStaticSak( sak )
      if sci:
         self.profile.rxStaticSci = sci

   def addIdentifier( self, args ):
      addrString = args[ 'IDENTIFIER' ]
      portNum = int( addrString.split( ':' )[-1] )
      addr = addrString.split( '::' )[0]
      newSci = Tac.Value( "Macsec::Sci", addr=addr, portNum=portNum  )
      if self.direction == 'tx':
         self.setTxStaticSakParam( sci=newSci )
      elif self.direction == 'rx':
         self.setRxStaticSakParam( sci=newSci )

   def noIdentifier( self, args ):
      if self.direction == 'tx':
         self.profile.txStaticSci = MacsecSci()
      elif self.direction == 'rx':
         self.profile.rxStaticSci = MacsecSci()

   def addKey( self, args ):
      sakString = args[ 'SECRET' ]
      an = args[ 'AN' ]

      try:
         intSak = int( decodeKey( sakString ), 16 )
      except ValueError:
         self.addError( 'Invalid Encrypted Key' )
         return 

      if intSak == 0:
         self.addError( 'Key value cannot be zero' )
         return

      sakLength = len( decodeKey( sakString ) )
      if not sakLength == 32 and not sakLength == 64:
         self.addError( 'Invalid Key length' )
         return

      sakVal = Tac.Value( "Macsec::Sak", key=sakString, an=an )

      if self.direction == 'tx':
         self.setTxStaticSakParam( sak=sakVal )
      elif self.direction == 'rx':
         self.setRxStaticSakParam( sak=sakVal )

   def noKey( self, args ):
      if ( self.direction == 'tx' and
            self.profile.txStaticSak.an == args[ 'AN' ] ):
         self.profile.txStaticSak = MacsecSak()
      elif self.direction == 'rx':
         del self.profile.rxStaticSak[ args[ 'AN' ] ]

def gotoMacsecMode( mode, args ):
   childMode = mode.childMode( MacsecMode )
   mode.session_.gotoChildMode( childMode )

def noMacsecMode( mode, args ):
   # Reset all the commands at mac security mode.
   # Resetting license first will make sure no ConfigSm reactors are fired further.
   noMacsecLicense( mode )
   handleNoDelayProtection( mode )
   handleNoFipsRestrictions( mode )
   noShutDownMacsec( mode )
   # Delete all configured macsec profiles.
   for profileName in config.profile.keys():
      del config.profile[ profileName ]

def gotoMacsecProfileMode( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   # Create a new profile object if it doesnt exist.
   if profileName not in config.profile.keys():
      config.profile.newMember( profileName )
   # Transition into the new mode.
   childMode = mode.childMode( MacsecProfileMode, profileName=profileName )
   mode.session_.gotoChildMode( childMode )

def noMacsecProfileMode( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   if profileName in config.profile:
      del config.profile[ profileName ]

# Macsec profile specific commands

keyKwMatcher = CliMatcher.KeywordMatcher( 'key',
                                           helpdesc='Connectivity association key' )

keyPattern_ = r'(^[a-fA-F0-9][a-fA-F0-9]{1,63}$)'

cknNameMatcher = CliMatcher.PatternMatcher(
   keyPattern_, helpname='CKN',
   helpdesc='Connectivity association key name in hex octets' )

cleartextKeyMatcher = CliMatcher.PatternMatcher(
   keyPattern_, helpname='CAK',
   helpdesc='Connectivity association key in hex octets' )

class KeyCommand( CliCommand.CliCommandClass ):
   syntax = "key CKN SECRET [ fallback ]"
   noOrDefaultSyntax = "key CKN SECRET [ fallback ]"
   data = {
      'key': keyKwMatcher,
      'CKN': cknNameMatcher,
      'SECRET': reversibleSecretCliExpression( 'SECRET',
                cleartextMatcher=cleartextKeyMatcher,
                keyTransformer=lambda k: k.lower() ),
      'fallback': 'Configure the key as a fallback'
      }

   handler = MacsecProfileMode.addKey
   noOrDefaultHandler = MacsecProfileMode.noKey

MacsecProfileMode.addCommandClass( KeyCommand )

#--------------------------------------------------------------------------------
# "[ no | default ] key retirement immediate", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeyRetirementImmediateCmd( CliCommand.CliCommandClass ):
   syntax = 'key retirement immediate'
   noOrDefaultSyntax = syntax
   data = {
      'key': keyKwMatcher,
      'retirement': 'Retire the key',
      'immediate': 'Retire the key immediately',
   }
   
   handler = MacsecProfileMode.keyRetire
   noOrDefaultHandler = MacsecProfileMode.noKeyRetire

MacsecProfileMode.addCommandClass( KeyRetirementImmediateCmd )

matcherSource = CliMatcher.KeywordMatcher( 'source',
               helpdesc='List of sources to derive MAC security keys' )

#--------------------------------------------------------------------------------
# "[ no | default ] key source dot1x", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeySourceDot1XCmd( CliCommand.CliCommandClass ):
   syntax = 'key source dot1x'
   noOrDefaultSyntax = syntax
   data = {
      'key': keyKwMatcher,
      'source': matcherSource,
      'dot1x': 'Derive MAC security keys from IEEE 802.1X based port authentication',
   }

   handler = MacsecProfileMode.dot1xEnabled
   noOrDefaultHandler = MacsecProfileMode.dot1xDisabled

MacsecProfileMode.addCommandClass( KeySourceDot1XCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] key source group-cak", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeySourceGroupCakCmd( CliCommand.CliCommandClass ):
   syntax = 'key source group-cak [ GROUPCAKLIFETIME ]'
   noOrDefaultSyntax = 'key source group-cak ...'
   data = {
      'key': keyKwMatcher,
      'source': matcherSource,
      'group-cak': 'Derive MAC security keys from Group CAK Distribution',
      'GROUPCAKLIFETIME': CliMatcher.IntegerMatcher( 30, 1000,
         helpdesc='Group CAK rekey period in minutes.' ),
   }
   hidden = True

   handler = MacsecProfileMode.configureGroupCak
   noOrDefaultHandler = MacsecProfileMode.noGroupCak

MacsecProfileMode.addCommandClass( KeySourceGroupCakCmd )

matcherMka = CliMatcher.KeywordMatcher( 'mka',
               helpdesc='MAC security key agreement protocol options' )

#--------------------------------------------------------------------------------
# "( no | default ) mka key-server priority ...", in "config-mac-sec-profile" mode 
#--------------------------------------------------------------------------------
class MkaKeyServerPriorityCmd( CliCommand.CliCommandClass ):
   syntax = ' mka key-server priority PRIORITY '
   noOrDefaultSyntax = 'mka key-server priority ...'
   data = {
      'mka': matcherMka,
      'key-server': 'MKA key server options',
      'priority': 'MKA key server priority',
      'PRIORITY': CliMatcher.IntegerMatcher( 0, 255,
                     helpdesc='Key server priority value' ),
   }

   handler = MacsecProfileMode.addKeyServerPriority
   noOrDefaultHandler = MacsecProfileMode.noKeyServerPriority

MacsecProfileMode.addCommandClass( MkaKeyServerPriorityCmd )

#--------------------------------------------------------------------------------
# "( no | default ) mka session rekey-period ...", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class MkaSessionRekeyPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'mka session rekey-period REKEYPERIOD'
   noOrDefaultSyntax = 'mka session rekey-period ...'
   data = {
      'mka': matcherMka,
      'session': 'Set MKA session options',
      'rekey-period': 'Set MKA session re-key period',
      'REKEYPERIOD': CliMatcher.IntegerMatcher( 30, 100000,
               helpdesc='Session re-key period in seconds.' ),
   }
   
   handler = MacsecProfileMode.addReKeyPeriod
   noOrDefaultHandler = MacsecProfileMode.noReKeyPeriod

MacsecProfileMode.addCommandClass( MkaSessionRekeyPeriodCmd )

#-----------------------------------------------------------------
# "( no | default ) mka session lpn advertisement disabled"
# in "config-mac-sec-profile" mode
#-----------------------------------------------------------------
class MkaSessionLpnAdvertisementDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'mka session lpn advertisement disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mka': matcherMka,
      'session': 'Set MKA session options',
      'lpn': 'Local Lowest PN value',
      'advertisement': 'LPN value as advertised in SakUseInfo parameter of MKPDU',
      'disabled': 'Always advertise LPN value for the MKA session as 0',
   }
   
   handler = MacsecProfileMode.setLpnValueZero
   noOrDefaultHandler = MacsecProfileMode.unsetLpnValueZero

MacsecProfileMode.addCommandClass( MkaSessionLpnAdvertisementDisabledCmd )

#---------------------------------------------------------------------
# The "[no | default] cipher" command, in "config-mac-sec-profile" mode.
#---------------------------------------------------------------------
class CipherCmd( CliCommand.CliCommandClass ):
   _cipherMatcher = CliMatcher.EnumMatcher( {
      "aes128-gcm-xpn": "Advanced Encryption Standard (128 bit, "
                        "Galois/Counter mode, Extended Packet Numbering)",
      "aes256-gcm-xpn": "Advanced Encryption Standard (256 bit, "
                        "Galois/Counter mode, Extended Packet Numbering)",
      "aes128-gcm"    : "Advanced Encryption Standard (128 bit, "
                        "Galois/Counter mode)",
      "aes256-gcm"    : "Advanced Encryption Standard (256 bit, "
                        "Galois/Counter mode)"
   } )
   syntax = 'cipher CIPHER'
   noOrDefaultSyntax = 'cipher ...'
   data = {
      'cipher': 'Data encryption algorithm and mode ',
      'CIPHER': _cipherMatcher
   }

   handler = MacsecProfileMode.configureCipherSuite
   noOrDefaultHandler = MacsecProfileMode.noCipherSuite

MacsecProfileMode.addCommandClass( CipherCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] l2-protocol lldp bypass", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class L2ProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'l2-protocol lldp bypass'
   noOrDefaultSyntax = syntax
   data = {
      'l2-protocol': 'Set L2 protocol processing',
      'lldp': 'LLDP frame processing',
      'bypass': 'Transmit/Receive without protection',
   }

   handler = MacsecProfileMode.bypassLldp
   noOrDefaultHandler = MacsecProfileMode.noBypassLldp

MacsecProfileMode.addCommandClass( L2ProtocolCmd )

#--------------------------------------------------------------------------------
# "( no | default ) replay protection disabled", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class ReplayProtectionDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'replay protection disabled'
   noOrDefaultSyntax = syntax
   data = {
      'replay': 'Replay protection',
      'protection': 'Replay protection',
      'disabled': 'Disable replay protection',
   }
   
   handler = MacsecProfileMode.replayProtectionDisabled
   noOrDefaultHandler = MacsecProfileMode.noReplayProtectionDisabled

MacsecProfileMode.addCommandClass( ReplayProtectionDisabledCmd )

#--------------------------------------------------------------------------------
# "( no | default ) replay protection window ...", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class ReplayProtectionWindowCmd( CliCommand.CliCommandClass ):
   syntax = 'replay protection window REPLAYPROTECTIONWINDOW'
   noOrDefaultSyntax = 'replay protection window ...'
   data = {
      'replay': 'Replay protection',
      'protection': 'Replay protection',
      'window': 'Replay protection window',
      'REPLAYPROTECTIONWINDOW': CliMatcher.IntegerMatcher( 0, 2**32-1,
                     helpdesc='Replay protection window size' ),
   }

   handler = MacsecProfileMode.replayProtectionWindow
   noOrDefaultHandler = MacsecProfileMode.noReplayProtectionWindow

MacsecProfileMode.addCommandClass( ReplayProtectionWindowCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] sci", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class SciCmd( CliCommand.CliCommandClass ):
   syntax = 'sci'
   noOrDefaultSyntax = syntax
   data = {
      'sci': 'Include secure channel identifier in data packets',
   }
   
   handler = MacsecProfileMode.includeSci
   noOrDefaultHandler = MacsecProfileMode.noIncludeSci

MacsecProfileMode.addCommandClass( SciCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] traffic unprotected allow", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class TrafficPolicyCmd( CliCommand.CliCommandClass ):
   if macsecToggle.toggleMacsecTrafficBlockOnMkaFailureEnabled():
      syntax = 'traffic unprotected ( ( allow [ active-sak ] ) | drop ) )'
   else:
      syntax = 'traffic unprotected allow'
   noOrDefaultSyntax = 'traffic unprotected ...'
   data = {
      'traffic': 'Traffic Policy',
      'unprotected': 'Traffic without MAC security protection',
      'allow' : 'Allow transmit/receive of unprotected traffic'
   }
   if macsecToggle.toggleMacsecTrafficBlockOnMkaFailureEnabled():
      data[ 'drop' ] = 'Block transmit/receive of unprotected traffic'
      data[ 'active-sak' ] = 'Allow transmit/receive of encrypted traffic ' \
                                       'using operational SAK and block otherwise'

   handler = MacsecProfileMode.trafficPolicy
   noOrDefaultHandler = MacsecProfileMode.noOrDefaultTrafficPolicy

MacsecProfileMode.addCommandClass( TrafficPolicyCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] secure channel [ tx | rx ]", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class StaticSakCmd( CliCommand.CliCommandClass ):
   syntax = 'secure channel TX_RX'
   noOrDefaultSyntax = syntax
   data = {
         'secure': 'Secure relationship',
         'channel': 'Secure channel',
         'TX_RX': CliMatcher.EnumMatcher( {
            'tx': 'Transmit',
            'rx': 'Receive',
         } ),
   }

   handler = MacsecProfileMode.gotoStaticSakMode
   noOrDefaultHandler = MacsecProfileMode.noStaticSak
if macsecToggle.toggleMacsecStaticSakEnabled():
   MacsecProfileMode.addCommandClass( StaticSakCmd )

port = '([0-9]{1,4})'
pair = '([0-9a-fA-F]{1,2})'
idPattern = '%s:%s:%s:%s:%s:%s::%s$' % ( pair, pair, pair, pair, pair, pair, port )

#--------------------------------------------------------------------------------
# "[ no | default ] identifier", in "config-mac-sec-profile-sc" mode
#--------------------------------------------------------------------------------
class StaticSakIdentifierCmd( CliCommand.CliCommandClass ):
   syntax = 'identifier IDENTIFIER'
   noOrDefaultSyntax = 'identifier'
   data = {
         'identifier': 'Secure Channel Identifier',
         'IDENTIFIER': CliMatcher.PatternMatcher( pattern=idPattern,
            helpname='H:H:H:H:H:H::P', helpdesc='MAC Address with port' ),
   }

   handler = MacsecStaticSakMode.addIdentifier
   noOrDefaultHandler = MacsecStaticSakMode.noIdentifier

MacsecStaticSakMode.addCommandClass( StaticSakIdentifierCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] an AN"
# "an AN key SECRET" commands, in "config-mac-sec-profile-sc" mode
#--------------------------------------------------------------------------------
sakMatcher = CliMatcher.PatternMatcher(
      keyPattern_, helpname='SAK',
      helpdesc='Secure association key in hex octets' )

class StaticSakKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'an AN key SECRET'
   noOrDefaultSyntax = 'an AN ...'
   data = {
         'an': 'Association Number',
         'AN': CliMatcher.IntegerMatcher( 0, 3, helpdesc='Association number' ),
         'key': 'Static secure association key',
         'SECRET': reversibleSecretCliExpression( 'SECRET',
                     cleartextMatcher=sakMatcher,
                     keyTransformer=str.lower ),
   }

   handler = MacsecStaticSakMode.addKey
   noOrDefaultHandler = MacsecStaticSakMode.noKey

MacsecStaticSakMode.addCommandClass( StaticSakKeyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac security profile PROFILE_NAME
#--------------------------------------------------------------------------------
def macsecSupportedGuard( mode, token ):
   intfName = mode.intf.name

   def macsecGuard( statusDir, intfName ):
      for hwSlice in statusDir:
         sliceStatus = statusDir[ hwSlice ]
         if sliceStatus and intfName in sliceStatus:
            return None
      return CliParser.guardNotThisPlatform

   def isSubintfSupported( statusDir, intfName ):
      if not macsecToggle.toggleMacsecSubintfSupportEnabled():
         return False
      for hwSlice in statusDir:
         sliceStatus = statusDir[ hwSlice ]
         if sliceStatus and intfName in sliceStatus:
            if sliceStatus[ intfName ].subIntfSupported:
               return True
      return False

   if SubIntfId.isSubIntfId( intfName ) and EthIntfId.isEthIntfId( intfName ):
      intfName = EthIntfId.parentIntfIdFromSubIntf( intfName )
      if isSubintfSupported( hwMacsecStatus, intfName ):
         return None
      return macsecGuard( hwMacsecProxyCapability, intfName )
   elif EthIntfId.isEthIntfId( intfName ) and EthIntfId.lane( intfName ) > 1:
      if macsecGuard( hwMacsecStatus, intfName ):
         # macsec status may not exist if lane is inactive
         # only guard if primary (/1) interface is also unsupported
         primaryIntfName = '/'.join( intfName.split( '/' )[ : -1 ] + [ '1' ] )
         return macsecGuard( hwMacsecStatus, primaryIntfName )
      else:
         return None
   else:
      return macsecGuard( hwMacsecStatus, intfName )


def delMacsecIntf( intfId ):
   # Get the intfConfig object.
   if intfId in config.intfConfig.keys():
      intfConfig = config.intfConfig[ intfId ]
   else:
      # Nothing to do.
      return

   profileName = intfConfig.profileName
   if not profileName:
      # Nothing to do.
      return

   if ( profileName in config.profile.keys() ) and \
         ( intfId in config.profile[ profileName ].intf.keys() ):
      del config.profile[ profileName ].intf[ intfId ]

   # Delete the interface config.
   del config.intfConfig[ intfId ]

class MacsecIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'mac security profile PROFILE_NAME'
   noOrDefaultSyntax = 'mac security profile ...'
   data = {
      'mac' : CliToken.Mac.macMatcherForConfigIf,
      'security' : CliCommand.guardedKeyword( 'security',
         helpdesc='Media Access Control Security (802.1AE) commands',
         guard=macsecSupportedGuard ),
      'profile' : 'MAC security profile name',
      'PROFILE_NAME' : CliMatcher.DynamicNameMatcher( getProfileNames,
         'Profile name' ),
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      # If an intfConfig does not exist yet, create it now.
      intfId = mode.intf.name
      newProfileExists = True

      if intfId not in config.intfConfig.keys():
         intfConfig = config.intfConfig.newMember( intfId )
      else:
         intfConfig = config.intfConfig[ intfId ]

      # Get the associated profile.
      if profileName not in config.profile.keys():
         mode.addWarning( 'MAC security profile %s does not exist.' % profileName )
         newProfileExists = False

      # If profile associated with the interface has not changed then there is
      # nothing to do.
      if intfConfig.profileName != profileName:
         
         # Disassociate the interface from its old profile.
         if intfConfig.profileName in config.profile.keys():
            del config.profile[ intfConfig.profileName ].intf[ intfId ]
         
         # Associate the interface with its new profile.
         if newProfileExists:
            config.profile[ profileName ].intf[ intfId ] = True

         # Update the interface.
         intfConfig.profileName = profileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Disassociate the interface from its current profile.
      intfId = mode.intf.name
      delMacsecIntf( intfId )

SubIntfModelet.addCommandClass( MacsecIntfCmd )
DataplaneIntfModelet.addCommandClass( MacsecIntfCmd )

matcherSecurityForConfig = CliMatcher.KeywordMatcher( 'security',
                        helpdesc='Media Access Control Security (802.1AE) commands' )

# Macsec global commands.
class MacSecurityCmd( CliCommand.CliCommandClass ):
   syntax = 'mac security'
   noOrDefaultSyntax = syntax
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'security': matcherSecurityForConfig,
   }

   handler = gotoMacsecMode
   noOrDefaultHandler = noMacsecMode

BasicCli.GlobalConfigMode.addCommandClass( MacSecurityCmd )

def doClearMkaCounters( mode, args ):
   config.clearMkaCounters += 1 # pylint: disable-msg=E1103

# config command for clearing mka counters
#--------------------------------------------------------------------------------
# clear mac security mka counters
#--------------------------------------------------------------------------------
class ClearMacSecurityMkaCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear mac security mka counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'mac': CliToken.Mac.macMatcherForClear,
      'security': 'MAC security (802.1AE) information',
      'mka': 'MAC security key agreement protocol options',
      'counters': 'MAC security counter information',
   }

   handler = doClearMkaCounters

BasicCli.EnableMode.addCommandClass( ClearMacSecurityMkaCountersCmd )

# Macsec mode commands.

#--------------------------------------------------------------------------------
# "[ no | default ] profile PROFILENAME", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class ProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile': 'MAC security profile commands',
      'PROFILENAME': CliMatcher.DynamicNameMatcher( getProfileNames,
                        'Profile name' ),
   }
   
   handler = gotoMacsecProfileMode
   noOrDefaultHandler = noMacsecProfileMode

MacsecMode.addCommandClass( ProfileCmd )

def handleFipsRestrictions( mode, args ):
   config.fipsRestrictions = True

def handleNoFipsRestrictions( mode, args=None ):
   config.fipsRestrictions = config.fipsRestrictionsDefault

#--------------------------------------------------------------------------------
# "[ no | default ] fips restrictions", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class FipsRestrictionsCmd( CliCommand.CliCommandClass ):
   syntax = 'fips restrictions'
   noOrDefaultSyntax = syntax
   data = {
      'fips': 'FIPS settings',
      'restrictions': 'Use FIPS compliant algorithms',
   }
   
   handler = handleFipsRestrictions
   noOrDefaultHandler = handleNoFipsRestrictions

MacsecMode.addCommandClass( FipsRestrictionsCmd )

def handleDelayProtection( mode, args ):
   config.delayProtection = True 

def handleNoDelayProtection( mode, args=None ):
   config.delayProtection = config.delayProtectionDefault

#-------------------------------------------------------------------------
# The "[no | default] delay protection" command, in "config-mac-sec" mode.
#-------------------------------------------------------------------------
class DelayProtectionCmd( CliCommand.CliCommandClass ):
   syntax = 'delay protection'
   noOrDefaultSyntax = syntax
   data = {
      "delay" : "MAC security data delay configuration",
      "protection" : "Data delay protection",
   }
   hidden = not macsecToggle.toggleMacsecDelayProtectionEnabled()

   handler = handleDelayProtection
   noOrDefaultHandler = handleNoDelayProtection

MacsecMode.addCommandClass( DelayProtectionCmd )

# Macsec license config
def setMacsecLicense( mode, args ):
   authKey = int( args[ 'AUTHKEYSTR' ], 16 )
   config.licenseConfig = Tac.Value( "Macsec::LicenseConfig", 
                                     licenseeName=args[ 'LICENSEENAME'],
                                     authKey=authKey )
def noMacsecLicense( mode, args=None ):
   config.licenseConfig = Tac.Value( "Macsec::LicenseConfig" )

#--------------------------------------------------------------------------------
# "( no | default ) license ..."
# "license LICENSEENAME AUTHKEYSTR", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class LicenseCmd( CliCommand.CliCommandClass ):
   syntax = 'license LICENSEENAME AUTHKEYSTR'
   noOrDefaultSyntax = 'license ...'
   data = {
      'license': 'MAC security legacy license configuration',
      'LICENSEENAME': CliMatcher.PatternMatcher( pattern='[\\w-]+',
                        helpdesc='Licensee name', helpname='WORD' ),
      'AUTHKEYSTR': CliMatcher.PatternMatcher( pattern='[a-fA-F0-9]{1,8}',
                        helpdesc='Key to authorize Mac security',
                        helpname='8-digit hex number' ),
   }

   handler = setMacsecLicense
   noOrDefaultHandler = noMacsecLicense

MacsecMode.addCommandClass( LicenseCmd )

def shutDownMacsec( mode, args ): 
   config.shutdown = True 
   
def noShutDownMacsec( mode, args=None ): 
   config.shutdown = config.shutdownDefault

#--------------------------------------------------------------------------------
# "[ no | default ] shutdown", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable MAC security',
   }

   handler = shutDownMacsec
   noOrDefaultHandler = noShutDownMacsec

MacsecMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] eapol mac destination MAC
#--------------------------------------------------------------------------------
matcherEapol = CliMatcher.KeywordMatcher( 'eapol',
               helpdesc='EAPoL attributes' )
def setEapolAttributes( dmac=None, ethType=None ):
   eapolAttr = Tac.Value( "Macsec::EapolAttributes",
                          destinationMac=config.eapolAttr.destinationMac,
                          etherType=config.eapolAttr.etherType )
   if dmac:
      eapolAttr.destinationMac = dmac
   if ethType:
      eapolAttr.etherType = ethType
   config.eapolAttr = eapolAttr

def setMacsecEapolDestinationMac( mode, args ):
   dmac = args.get( 'MAC', config.eapolAttr.destinationMacDefault )
   if dmac == config.eapolAttr.destinationMacDefault:
      trapConfig.features.remove( TrapFeatureName.eapol )
   else:
      trapConfig.features.add( TrapFeatureName.eapol )
   setEapolAttributes( dmac=dmac )

class EapolDestinationMacCmd( CliCommand.CliCommandClass ):
   syntax = 'eapol mac destination MAC'
   noOrDefaultSyntax = 'eapol mac destination ...'
   data = {
       'eapol': matcherEapol,
       'mac': 'MAC addresses',
       'destination': 'Configure destination MAC',
       'MAC': macAddrMatcher,
   }
   handler = setMacsecEapolDestinationMac
   noOrDefaultHandler = handler

if macsecToggle.toggleMacsecConfigurableEapolDmacEnabled():
   MacsecMode.addCommandClass( EapolDestinationMacCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] eapol ethertype TYPE
#--------------------------------------------------------------------------------
def setMacsecEapolEtherType( mode, args ):
   ethType = \
         int( args.get( 'TYPE', config.eapolAttr.etherTypeDefault ) )
   setEapolAttributes( ethType=ethType )

class EapolEtherTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'eapol ethertype TYPE'
   noOrDefaultSyntax = 'eapol ethertype ...'
   data = {
       'eapol': matcherEapol,
       'ethertype': 'Configure Ethernet type',
       'TYPE': CliMatcher.IntegerMatcher( 0x600, 0xFFFF,
                                          helpdesc='Ethernet type for EAPoL' )
   }
   handler = setMacsecEapolEtherType
   noOrDefaultHandler = handler

if macsecToggle.toggleMacsecConfigurableEapolEtherTypeEnabled():
   MacsecMode.addCommandClass( EapolEtherTypeCmd )

#-------------------------------------------------------------------------------
# Cleanup per-interface configuration
#-------------------------------------------------------------------------------
class IntfMacsecConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing per-interface Macsec config
   when the interface is deleted."""
   def setDefault( self ):
      delMacsecIntf( self.intf_.name )

def Plugin( entityManager ):
   global config, status, hwMacsecStatus, hwMacsecProxyCapability, trapConfig

   config = ConfigMount.mount( entityManager, 'macsec/input/cli',
                               'Macsec::Config', 'w' )
   status = LazyMount.mount( entityManager, 'macsec/status',
                             'Macsec::Status', 'r' )
   hwMacsecStatus = LazyMount.mount( entityManager, 
                                     'hardware/phy/status/macsec/slice',
                                     'Tac::Dir', 'ri' )
   hwMacsecProxyCapability = LazyMount.mount( entityManager,
                                'hardware/phy/status/macsecproxy/capability/slice',
                                'Tac::Dir', 'ri' )
   trapConfig = ConfigMount.mount( entityManager, 'hardware/trap/config/trapConfig',
                                   'Arnet::TrapConfig', 'w' )

   # register interface-deletion handler
   IntfCli.Intf.registerDependentClass( IntfMacsecConfigCleaner, priority=10 )
