#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the following commands:
#-------------------------------------------------------------------------------
# - show  loop-protection [ detail ]
# - show  loop-protection vlan VLANSET
# - show  loop-protection counters
#-------------------------------------------------------------------------------

from __future__ import absolute_import, division, print_function

# pylint: disable=ungrouped-imports

import BasicCli
import CliMatcher
import ShowCommand
import ConfigMount
import LazyMount
import SmashLazyMount
import CliPlugin.VlanCli as VlanCli
from CliPlugin.LoopProtectModels import LoopProtectShow, loopProtectDetail
from CliPlugin.LoopProtectModels import LoopProtectVlan, loopProtectVlanIntfs
from CliPlugin.LoopProtectModels import LoopProtectCounters, loopProtectCounters
from CliPlugin.LoopProtectCli import TRANSMIT_INTERVAL_DEFAULT
from CliPlugin.LoopProtectCli import DISABLED_TIME_DEFAULT
from CliPlugin.LoopProtectCli import RATE_LIMIT_DEFAULT
import Tac

config = None
status = None
bridgingConfig = None
allIntfConfigDir = None # For intf status
packetCounter = None

matcherLoopProtection = CliMatcher.KeywordMatcher( 'loop-protection', 
      helpdesc='Loop Protection status' )

def tacTimeToUtc( time ):
   return time + Tac.utcNow() - Tac.now()

#--------------------------------------------------------------------------------
# show loop-protection [ detail ]
#--------------------------------------------------------------------------------
def generateLoopProtectShowModel( mode, args ): 
   assert config is not None

   disableTime = getattr( config.globalIntfConfig, 'disabledTime',
                          DISABLED_TIME_DEFAULT )
   transmitInterval = getattr( config.globalIntfConfig, 'transmitInterval',
                               TRANSMIT_INTERVAL_DEFAULT )
   rateLimit = getattr( config.transmitConfig, 'rateLimit',
                        RATE_LIMIT_DEFAULT )
   vlansEnabled = [ vlan for vlan, enabled in config.vlanEnabled.items() if enabled ]
   model = LoopProtectShow( enabled=config.globalEnabled,
                            transmitInterval=transmitInterval,
                            disableTime=disableTime,
                            rateLimit=rateLimit,
                            totalVlansEnabled=len( vlansEnabled ) )

   if 'detail' in args and config.globalEnabled:
      vlanList = []
      vlansEnabled = [ vlan for vlan, enabled in config.vlanEnabled.items()
                       if enabled ]
      for vlanId in vlansEnabled:
         # Need to generate the rows for this table:
         # Vlan    Loop     Disabled Intfs Total Latest
         #         Detected                Intfs Disabled Time
         # ------ --------- -------------- ----- -------------
         #      1 Yes       Et1-2          20    1/1/01 18:01

         vlanConfig = bridgingConfig.vlanConfig.get( vlanId )
         if not vlanConfig:
            vlanList.append( [ vlanId, None, [], 0, None ] )
            continue
         
         intfsExcluded = set( intf for ( intf, value ) in
                                   config.intfExclude.items() if value )
         intfsInVlan = set( vlanConfig.intf )
         intfs = [ status.intfStatus[ intf ] for intf in
                      intfsInVlan - intfsExcluded if intf in status.intfStatus ]

         loopDetected = any( intf for intf in intfs if intf.disabled )
         if not loopDetected:
            vlanList.append( [ vlanId, False, [], len( intfs ), None ] )
            continue

         disabledIntfs = [ intf for intf in intfs if intf.disabled ]
         latestDisabledTime =  \
             sorted( intf.disabledAt for intf in disabledIntfs )[ -1 ]

         vlanList.append( [ vlanId, True, [ intf.intfId for intf in disabledIntfs ],
                            len( intfs ),
                            tacTimeToUtc( latestDisabledTime ) ] )

      model.detail = loopProtectDetail(
         'ff:ff:ff:ff:ff:ff',
         '0x88b7',
         'Interface Disable',
         vlanList )
      
   return model

class LoopProtectionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show loop-protection [ detail ]'
   data = {
      'loop-protection': matcherLoopProtection,
      'detail': 'Loop Protection detailed status',
   }
   
   handler = generateLoopProtectShowModel
   cliModel = LoopProtectShow

BasicCli.addShowCommandClass( LoopProtectionCmd )

#--------------------------------------------------------------------------------
# show loop-protection counters
#--------------------------------------------------------------------------------

def generateLoopProtectCountersModel( mode, args ):
   def _generateCounters( counters ): 
      output = {} 
      for instance in sorted( counters ): 
         ci = counters[ instance ]
         if not isinstance( instance, ( int, str ) ):
            instance = str( instance )
         output[ instance ] = [ ci.tx, ci.rx, ci.rxOther ]
      return output
   return loopProtectCounters( config.globalEnabled,
                               _generateCounters( packetCounter.counterVlan ),
                               _generateCounters( packetCounter.counterIntf ) )

class LoopProtectionCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show loop-protection counters'
   data = {
      'loop-protection': matcherLoopProtection,
      'counters': 'Counters for loop protection messages',
   }
   
   handler = generateLoopProtectCountersModel
   cliModel = LoopProtectCounters

BasicCli.addShowCommandClass( LoopProtectionCountersCmd )

#--------------------------------------------------------------------------------
# show loop-protection vlan VLANSET
#--------------------------------------------------------------------------------

def generateLoopProtectVlanModel( mode, args ): # vlanSet=None
   vlans = {}
   vlanSet = args[ 'VLANSET' ]
   for vlanId in vlanSet.ids:
      # Need to generate rows for this table:
      # Vlan Intf LP Enabled State LP Disabled  Disabled at  Bring up at
      # ---- ---- ---------- ----- ----------- ------------ -------------
      # 3    Et1  Yes        Down  Yes         1/1/01 17:21  1/1/01 18:21
      vlans[ vlanId ] = []
      if vlanId not in config.vlanEnabled or not config.vlanEnabled[ vlanId ]:
         vlans[ vlanId ].append( [ None, False, None, None, None, None ] )
         continue
      if vlanId not in bridgingConfig.vlanConfig or \
             not bridgingConfig.vlanConfig[ vlanId ].intf:
         vlans[ vlanId ].append( [ None, True, None, None, None, None ] )
         continue

      for intfId in bridgingConfig.vlanConfig[ vlanId ].intf:
         if config.intfExclude.get( intfId ):
            vlans[ vlanId ].append( [ intfId, False, None, None, None, None ] )
            continue

         intfStatus = status.intfStatus.get( intfId )
         if not intfStatus:
            # If in bridgingConfig but not in status.intfStatus( not in topoLib )
            # then there is some churning going on. Pass for now.
            continue

         state = allIntfConfigDir[ intfId ].enabledStateLocal
         if intfStatus.disabled:
            vlans[ vlanId ].append( [
                  intfId, True, state, True,
                  tacTimeToUtc( intfStatus.disabledAt ),
                  tacTimeToUtc( intfStatus.disabledUntil ) ] )
         else:
            vlans[ vlanId ].append( [ intfId, True, state, False, None, None ] )

   return LoopProtectVlan( enabled=config.globalEnabled,
                           vlanIntfs=loopProtectVlanIntfs( vlans ) )


class LoopProtectionVlanVlansetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show loop-protection vlan VLANSET'
   data = {
      'loop-protection': matcherLoopProtection,
      'vlan': 'VLANs to see loop protection information on',
      'VLANSET': VlanCli.vlanSetMatcher, 
   }
   
   handler = generateLoopProtectVlanModel
   cliModel = LoopProtectVlan

BasicCli.addShowCommandClass( LoopProtectionVlanVlansetCmd )

def Plugin( entityManager ):
   global config, bridgingConfig
   global status
   global allIntfConfigDir
   global packetCounter

   config = ConfigMount.mount( entityManager, 'loopprotect/config',
                               'LoopProtect::Config::Config', 'w' )
   status = LazyMount.mount( entityManager, 'loopprotect/status',
                             'LoopProtect::Status::Status', 'r' )
   bridgingConfig = LazyMount.mount( entityManager, 'bridging/config',
                                     'Bridging::Config', 'r' )
   allIntfConfigDir = LazyMount.mount( entityManager, 'interface/config/all',
                                       'Interface::AllIntfConfigDir', 'r' )
   packetCounter = SmashLazyMount.mount( entityManager, 'loopprotect/counter',
                                         'LoopProtect::Status::PacketCounter',
                                         SmashLazyMount.mountInfo( 'reader' ) )
