#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the following commands:
#-------------------------------------------------------------------------------
# Loop protect mode:
#      -  ( no | default ) disabled-time [ DISABLEDTIME ]
#      -  [ no ] [ default ] protect vlan VLANSET
#      -  ( no | default ) rate-limit [ RATELIMIT ]
#      -  [ no ] [ default ] shutdown
#      -  ( no | default ) transmit-interval [ TRANSMITINTERVAL ]
#--------------------------------------------------------------------------------

from __future__ import absolute_import, division, print_function

# pylint: disable=ungrouped-imports

import CliPlugin
import CliCommand
import CliMatcher
import CliPlugin.LoopProtectCli as LoopProtectCli
import CliPlugin.VlanCli as VlanCli
from CliPlugin.LoopProtectCli import DISABLED_TIME_DEFAULT
from CliPlugin.LoopProtectCli import RATE_LIMIT_DEFAULT

matcherDisabledTime = CliMatcher.KeywordMatcher( 'disabled-time', 
      helpdesc='Set how long looped interfaces are disabled for' )
matcherRateLimit = CliMatcher.KeywordMatcher( 'rate-limit', 
      helpdesc='Set throttling of loop protection messages' )
matcherTransmitInterval = CliMatcher.KeywordMatcher( 'transmit-interval', 
      helpdesc='Set message transmission interval' )

#--------------------------------------------------------------------------------
# ( no | default ) disabled-time [ SECONDS ]
#--------------------------------------------------------------------------------
# Defaults to 604800

class DisabledTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled-time SECONDS'
   noOrDefaultSyntax = 'disabled-time ...'
   data = {
      'disabled-time': matcherDisabledTime,
      'SECONDS': CliMatcher.IntegerMatcher( 0, DISABLED_TIME_DEFAULT, 
         helpdesc='How long to disable looped interfaces for, in seconds' ),
   }
   
   handler = LoopProtectCli.MonitorLoopProtectConfigMode.disabledTime
   noOrDefaultHandler = handler

CliPlugin.LoopProtectCli.MonitorLoopProtectConfigMode.addCommandClass( 
                                                               DisabledTimeCmd )

#--------------------------------------------------------------------------------
# [ no ] [ default ] protect vlan VLANSET
#--------------------------------------------------------------------------------
# Defaults to no vlans protected

class ProtectVlanVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'protect vlan VLANSET'
   noOrDefaultSyntax = syntax
   data = {
      'protect': 'Protect VLAN from loops',
      'vlan': 'VLANs to be involved in loop protection',
      'VLANSET': VlanCli.vlanSetMatcher, 
   }
   
   handler = LoopProtectCli.MonitorLoopProtectConfigMode.protectVlan
   noHandler = LoopProtectCli.MonitorLoopProtectConfigMode.noProtectVlan
   defaultHandler = handler

CliPlugin.LoopProtectCli.MonitorLoopProtectConfigMode.addCommandClass( 
                                                         ProtectVlanVlansetCmd )

#--------------------------------------------------------------------------------
# ( no | default ) rate-limit [ LIMIT ]
#--------------------------------------------------------------------------------
# Defaults to 1000

class RateLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'rate-limit LIMIT'
   noOrDefaultSyntax = 'rate-limit ...'
   data = {
      'rate-limit': matcherRateLimit,
      'LIMIT': CliMatcher.IntegerMatcher( 0, RATE_LIMIT_DEFAULT, 
      helpdesc='Maximum number of loop protection messages generated per second' ),
   }
   
   handler = LoopProtectCli.MonitorLoopProtectConfigMode.rateLimit
   noOrDefaultHandler = handler

CliPlugin.LoopProtectCli.MonitorLoopProtectConfigMode.addCommandClass( RateLimitCmd )

#--------------------------------------------------------------------------------
# [ no ] [ default ] shutdown
#--------------------------------------------------------------------------------
# Defaults to shutdown. Globally disabled initially

class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable loop protection globally',
   }
   
   handler = LoopProtectCli.MonitorLoopProtectConfigMode.shutdown
   noHandler = LoopProtectCli.MonitorLoopProtectConfigMode.noShutdown
   defaultHandler = handler

CliPlugin.LoopProtectCli.MonitorLoopProtectConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# ( no | default ) transmit-interval [ INTERVAL ]
#--------------------------------------------------------------------------------
# Defaults to 5

class TransmitIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'transmit-interval [ INTERVAL ]'
   noOrDefaultSyntax = 'transmit-interval ...'
   data = {
      'transmit-interval': matcherTransmitInterval,
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 10, 
         helpdesc='How often to transmit loop protection messages in seconds' ),
   }
   
   handler = LoopProtectCli.MonitorLoopProtectConfigMode.transmitInterval
   noOrDefaultHandler = handler

CliPlugin.LoopProtectCli.MonitorLoopProtectConfigMode.addCommandClass( 
                                                               TransmitIntervalCmd )
