#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model
from CliModel import Submodel
from CliModel import Int
from CliModel import Bool
from CliModel import List
from CliModel import Enum
from CliModel import Float
from CliModel import Dict
from IntfModels import Interface
from ArnetModel import MacAddress
from Intf.IntfRange import intfListToCanonical
from TableOutput import createTable
from TableOutput import Format
from datetime import datetime

def renderBool( value ):
   if value is None:
      return '-'
   elif value:
      return 'Yes'
   return 'No'

def renderUtc( value ):
   if value is None:
      return '-'
   return '%s' % datetime.fromtimestamp( int( value ) )

# Use a method to generate correct structure and to not expose more classes
# But also makes the code cleaner on the CLI end.
def loopProtectDetail( destAddr, etherType, recvAction, vlanList ):
   vlans = {}

   for vlan in vlanList:
      vlans[ vlan[ 0 ] ] = LoopProtectShowDetailVlan(
         loopDetected=vlan[ 1 ],
         disabledIntfs=vlan[ 2 ],
         totalIntfs=vlan[ 3 ],
         latestDisabledTime=vlan[ 4 ] )

   return LoopProtectShowDetail(
      destinationAddress=destAddr,
      ethernetType=etherType,
      receiveAction=recvAction,
      vlans=vlans )

# Helper classes to generate the 'detail' structure
class LoopProtectShowDetailVlan( Model ):
   loopDetected = Bool( help='Loop detected on Vlan', optional=True )
   disabledIntfs = List( valueType=Interface,
                         help='Interfaces disabled by loop protection' )
   totalIntfs = Int( help='Total number of interfaces in vlan protected '
                        'by loop protection' )
   latestDisabledTime = Float( help='Last time an interface was disabled on '
                               'this vlan by loop protection', optional=True )

   def renderShortName( self ):
      if not self.disabledIntfs:
         return '-'
      return ','.join( intfListToCanonical( self.disabledIntfs ) )

   def renderList( self ):
      return [ renderBool( self.loopDetected ),
               self.renderShortName(), '%d' % self.totalIntfs,
               renderUtc( self.latestDisabledTime ) ]

# Example output:
#
# Destination address: ffff.ffff.ffff
# Ethernet type: 0x88b7
# Receive action: Interface Disable
#
# Vlan    Loop     Disabled Intfs Total Latest
#         Detected                Intfs Disabled Time
# ------ --------- -------------- ----- -------------
#      1 Yes       Et1-2          20    1/1/01 18:01
#      2....
class LoopProtectShowDetail( Model ):
   destinationAddress = MacAddress( help='Destination address for loop '
                                    'protection packets' )
   ethernetType = Enum( values=( '0x88b7', ), help='Ethernet type used by protocol' )
   receiveAction = Enum( values=( 'Interface Disable', ), help='Trigger when a '
                         'loop protection packet is received' )
   vlans = Dict( keyType=int, valueType=LoopProtectShowDetailVlan,
                 help='A dict with the mapping vlan to loop protect '
                 'vlan information' )

   def render( self ):

      print 'Destination address: %s' % self.destinationAddress
      print 'Ethernet type: %s' % self.ethernetType
      print 'Receive action: %s' % self.receiveAction

      if not self.vlans:
         return

      headers = [ 'Vlan', 'Loop Detected', 'Disabled Intfs', 'Total Intfs',
                  'Latest Disabled Time' ]
      f1 = Format( justify='left' )
      f1.padLimitIs( True )
      outputTable = createTable( headers )
      for row in self.vlans.items():
         outputTable.newRow( row[ 0 ], *row[ 1 ].renderList() )
      outputTable.formatColumns( f1, f1, f1, f1, f1 )
      print '\n' + outputTable.output()

# Example output:
#
# Loop protection is enabled
# Transmit Interval: 5
# Disable Time: 604800
# Packets Transmitted rate: 12/second
# Total: 3 Vlans enabled.
# Render detail
class LoopProtectShow( Model ):

   enabled = Bool( help='Loop Protection enabled' )
   transmitInterval = Int( help='Packet transmission interval per second' )
   disableTime = Int( help='How long interface will be disabled'
                          ', if a loop is detected, in seconds' )
   rateLimit = Int( help='Throttling level of loop protection messages per second' )
   totalVlansEnabled = Int( help='Total number of vlans loop protection is '
                            'enabled on' )

   detail = Submodel( valueType=LoopProtectShowDetail,
                      help='Loop protection detailed', optional=True )

   def render( self ):
      if not self.enabled:
         print 'Loop protection is disabled'
         return
      print 'Loop protection is enabled'
      print 'Transmit Interval: %s' % self.transmitInterval
      if not self.disableTime:
         print 'Disable Time: Permanent'
      else:
         print 'Disable Time: %s' % self.disableTime
      if not self.rateLimit:
         print 'Packets Transmitted rate: Unthrottled'
      else:
         print 'Packets Transmitted rate: %s/second' % self.rateLimit
      print 'Total: %s Vlans enabled' % self.totalVlansEnabled

      if self.detail:
         self.detail.render()


def loopProtectVlanIntfs( vlans ):
   vlanDict = {}
   for vlan in vlans.keys():
      vlanDict[ vlan ] = LoopProtectVlanIntfList( intfs=[] )
      for intf in vlans[ vlan ]:
         vlanDict[ vlan ].intfs.append( LoopProtectIntfInfo(
               intf=intf[ 0 ],
               lpEnabled=intf[ 1 ],
               state=intf[ 2 ],
               lpDisabled=intf[ 3 ],
               disabledAt=intf[ 4 ],
               disabledUntil=intf[ 5 ] ) )
   return vlanDict

# Example output:
# Et1  Yes        Down  Yes         1/1/01 17:21  1/1/01 18:21
class LoopProtectIntfInfo( Model ):
   intf = Interface( help='Loop protection information about this interface',
                     optional=True )
   lpEnabled = Bool( help='Whether this interface is protected on loop protection' )
   state = Enum( values=( 'unknownEnabledState', 'enabled', 'shutdown' ),
                 help='Current state of the interface', optional=True )
   lpDisabled = Bool( help='Loop detected on Vlan', optional=True )
   disabledAt = Float( help='When the interface was disabled', optional=True )
   disabledUntil = Float( help='When loop protection is scheduled to bring '
                             'this interface up', optional=True )

   def renderState( self ):
      if self.state is None:
         return '-'
      return '%s' % self.state

   def renderIntf( self ):
      if self.intf is None:
         return '-'
      # intf is wrapped to get intfId which does have a shortname
      return self.intf.shortName #pylint: disable=E1101

   def renderList( self ):
      return [ self.renderIntf(), renderBool( self.lpEnabled ),
               self.renderState(), renderBool( self.lpDisabled ),
               renderUtc( self.disabledAt ), renderUtc( self.disabledUntil ) ]

class LoopProtectVlanIntfList( Model ):
   intfs = List( valueType=LoopProtectIntfInfo,
                 help='Loop protect information for interfaces in this vlan' )

# Example output:
#
# Vlan Intf LP Enabled State LP Disabled  Disabled at  Bring up at
# ---- ---- ---------- ----- ----------- ------------ -------------
# 3    Et1  Yes        Down  Yes         1/1/01 17:21  1/1/01 18:21
# 3    Et2  Yes        Up    No          -             -
# 3    Et3  No         -     -           -             -
# 4    -    No         -     -           -             -
class LoopProtectVlan( Model ):
   enabled = Bool( help='Loop protection enabled' )

   vlanIntfs = Dict( keyType=int, valueType=LoopProtectVlanIntfList,
                     help='Loop protect vlan information' )

   def render( self ):
      if not self.enabled:
         print 'Loop protection is disabled'
         return

      headers = [ 'Vlan', 'Intf', 'LP Enabled', 'State',
                  'LP Disabled', 'Disabled at', 'Bring up at' ]

      f1 = Format( justify='left' )
      f1.padLimitIs( True )
      outputTable = createTable( headers )
      for vlan in self.vlanIntfs.keys():
         for intf in self.vlanIntfs[ vlan ].intfs:
            outputTable.newRow( vlan, *intf.renderList() )
      outputTable.formatColumns( f1, f1, f1, f1, f1 )
      print outputTable.output()

def loopProtectCounters( enabled, vlans, intfs ):
   counters = LoopProtectCounters( enabled=enabled )

   for ( vlanId, value ) in vlans.items():
      counters.vlanCounters[ vlanId ] = LoopProtectCounter(
         tx=value[ 0 ],
         rx=value[ 1 ],
         rxOther=value[ 2 ] )

   for ( intfId, value ) in intfs.items():
      counters.intfCounters[ intfId ] = LoopProtectCounter(
            tx=value[ 0 ],
            rx=value[ 1 ],
            rxOther=value[ 2 ] )

   return counters

class LoopProtectCounter( Model ):
   tx = Int( help='Packets transmitted' )
   rx = Int( help='Packets received' )
   rxOther = Int( help='Packets received in error' )

   def renderList( self ):
      return [ '%s' % self.tx, '%s' % self.rx, '%s' % self.rxOther ]

# Example output:

# Vlan  Tx Rx Rx-Other
# ---- -- -- --------
#    1  5  0        0
#    3  5  1       12

# Intf Tx Rx Rx-Other
# ---- -- -- --------
#  Et1 12  0        0
#  Et3  9  1        1
#  Et5  0  0        1
class LoopProtectCounters( Model ):
   enabled = Bool( help='Loop protection enabled' )

   vlanCounters = Dict( keyType=int, valueType=LoopProtectCounter,
                        help='List of counters per vlan' )
   intfCounters = Dict( keyType=str, valueType=LoopProtectCounter,
                        help='List of counters per interface' )

   def renderCounter( self, counters, counterType ):
      headers = [ counterType, 'Tx', 'Rx', 'Rx-Other' ]

      f1 = Format( justify='left' )
      f1.padLimitIs( True )
      outputTable = createTable( headers )
      for counterKey in counters.keys():
         outputTable.newRow( '%s' % counterKey,
                             *counters[ counterKey ].renderList() )
      outputTable.formatColumns( f1, f1, f1, f1 )
      print outputTable.output()

   def render( self ):
      if not self.enabled:
         print 'Loop protection is disabled'
         return

      self.renderCounter( self.vlanCounters, 'Vlan' )
      self.renderCounter( self.intfCounters, 'Intf' )
