#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli, CliParser
import IntfCli
import ConfigMount, LazyMount
import CliPlugin.TechSupportCli
from CliMode.LoopProtect import MonitorLoopProtectMode

TRANSMIT_INTERVAL_DEFAULT = 5
DISABLED_TIME_DEFAULT = 604800
RATE_LIMIT_DEFAULT = 1000

config = None
bridgingHwCapabilities = None

class MonitorLoopProtectConfigMode( MonitorLoopProtectMode,
                                    BasicCli.ConfigModeBase ):
   name = 'monitor-loop-protect'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      MonitorLoopProtectMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      if not config.globalIntfConfig:
         config.globalIntfConfig = ( 'globalIntfConfig', )
      if not config.transmitConfig:
         config.transmitConfig = ( 'transmitConfig', )

   def noShutdown( self, args ):
      config.globalEnabled = True

   def shutdown( self, args ):
      config.globalEnabled = False

   def protectVlan( self, args ):
      vlanSet = args[ 'VLANSET' ]
      for vlan in vlanSet.ids:
         config.vlanEnabled[ vlan ] = True

   def noProtectVlan( self, args ):
      vlanSet = args[ 'VLANSET' ]
      for vlan in vlanSet.ids:
         del config.vlanEnabled[ vlan ]

   def transmitInterval( self, args ):
      interval = args.get( 'INTERVAL', TRANSMIT_INTERVAL_DEFAULT )
      config.globalIntfConfig.transmitInterval = interval

   def disabledTime( self, args ):
      seconds = args.get( 'SECONDS', DISABLED_TIME_DEFAULT )
      config.globalIntfConfig.disabledTime = seconds

   def rateLimit( self, args ):
      limit = args.get( 'LIMIT', RATE_LIMIT_DEFAULT )
      config.transmitConfig.rateLimit = limit

#-------------------------------------------------------------------------------
# The LoopProtectIntf class is used to remove the LoopProtect IntfConfig object when
# an interface is deleted. The Intf class will create a new instance of DhcpRelayIntf
# and call destroy when the interface is deleted
#-------------------------------------------------------------------------------
class LoopProtectIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # resets the IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      del config.intfExclude[ self.intf_.name ]

#-------------------------------------------------------------------------------
# add show tech support
#-------------------------------------------------------------------------------

def _showTechCmds():
   if bridgingHwCapabilities.loopProtectSupported:
      cmds = [ 'show loop-protection' ]
   else:
      cmds = []
   return cmds

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( '2015-08-03 17:02:43',
                                                             _showTechCmds ) 

def Plugin( entityManager ):
   global config, bridgingHwCapabilities

   config = ConfigMount.mount( entityManager, 'loopprotect/config',
                               'LoopProtect::Config::Config', 'w' )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )

   IntfCli.Intf.registerDependentClass( LoopProtectIntf )
