# Copyright (c) 2009-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This module implement the CLI for LLDP configuration. 
"""
from __future__ import absolute_import, division

import Tac
import CliCommand
import CliParser
import CliPlugin.EthIntfCli
import CliPlugin.IntfCli
import LldpConstants
import ConfigMount
from LldpTypes import ( MgmtAddrTxOption, NetworkPolicyConfig, PortConfig,
                        PowerViaMdiInfo )
from IpLibConsts import DEFAULT_VRF, DEFAULT_VRF_OLD

lldpCliConfig = None
lldpConfig = None
wattsToDeciWatts = PowerViaMdiInfo().wattsToDeciWatts

### General functions.

def maybeDeletePortConfig( mode ):
   """Deletes an Lldp::CliPortConfig object if it is no longer necessary."""
   cliPortConfig = lldpCliConfig.portConfig.get( mode.intf.name )
   if cliPortConfig:
      if cliPortConfig.adminStatus == LldpConstants.defaultAdminStatus and \
         not cliPortConfig.networkPolicyConfig:
         del lldpCliConfig.portConfig[ mode.intf.name ]

### CLI command implementations.

def setLldpTimer( mode, args ):
   seconds = args.get( 'TSECONDS' )
   lldpCliConfig.messageTxInterval = seconds

def noLldpTimer( mode, args ):
   args[ 'TSECONDS' ] = lldpCliConfig.messageTxIntervalDefault
   setLldpTimer( mode, args )

def setLldpHoldtime( mode, args ):
   seconds = args.get( 'SECONDS', lldpCliConfig.messageTxHoldTimeDefault )
   lldpCliConfig.messageTxHoldTime = seconds

def noLldpHoldtime( mode, args ):
   setLldpHoldtime( mode, args )

def setLldpReinitDelay( mode, args ):
   seconds = args.get( 'RSECONDS' )
   lldpCliConfig.reinitDelay = seconds

def noLldpReinitDelay( mode, args ):
   args[ 'RSECONDS' ] = lldpCliConfig.reinitDelayDefault
   setLldpReinitDelay( mode, args )

def enableLldpTlv( mode, tlv, enabled=True ):
   attrName = LldpConstants.tlvKeywordToAttrNameMap[ tlv ]
   setattr( lldpCliConfig, attrName, enabled )

def enableMedLldpTlv( mode, medTlv, enabled=True ):
   attrName = LldpConstants.medTlvKeywordToAttrNameMap[ medTlv ]
   setattr( lldpCliConfig, attrName, enabled )

def enableLldp( mode, enabled ):
   lldpCliConfig.enabled = enabled

def setLldpMgmtAddrTxOptionAll( mode ):
   lldpCliConfig.mgmtAddrTxOption = MgmtAddrTxOption(
         mgmtAddrTxMode='mgmtAddrTxModeAll' )

def setLldpMgmtAddrTxOptionIntfName( mode, intf ):
   lldpCliConfig.mgmtAddrTxOption = MgmtAddrTxOption(
         mgmtAddrTxMode='mgmtAddrTxModeIntfName',
         intfName=intf.name )

def noLldpMgmtAddrTxOption( mode ):
   lldpCliConfig.mgmtAddrTxOption = MgmtAddrTxOption()

def noLldpMgmtAddrTxOptionIntf( mode, intf ):
   noLldpMgmtAddrTxOption( mode )

def setLldpIp6MgmtAddrIntfName( mode, intf ):
   lldpCliConfig.ip6MgmtAddrIntfName = intf.name

def noLldpIp6MgmtAddrIntfName( mode ):
   lldpCliConfig.ip6MgmtAddrIntfName = ""

def setLldpMgmtAddrVrf( mode, vrfName ):
   assert vrfName != ''
   if vrfName == DEFAULT_VRF_OLD:
      vrfName = DEFAULT_VRF
   lldpCliConfig.mgmtAddrVrf = vrfName

def noLldpMgmtAddrVrf( mode, vrfName ):
   if lldpCliConfig.mgmtAddrVrf == vrfName:
      lldpCliConfig.mgmtAddrVrf = DEFAULT_VRF

def setLldpAdminStatus( mode, cliPortConfig, txEnabled, rxEnabled ):
   if txEnabled:
      if rxEnabled:
         adminStatus = 'txAndRx'
      else:
         adminStatus = 'txOnly'
   else:
      if rxEnabled:
         adminStatus = 'rxOnly'
      else:
         adminStatus = 'disabled'
   cliPortConfig.adminStatus = adminStatus
   maybeDeletePortConfig( mode )

def enableLldpTransmit( mode, args ):
   cliPortConfig = lldpCliConfig.newPortConfig( mode.intf.name )
   txEnabled = not CliCommand.isNoCmd( args )
   rxEnabled = cliPortConfig.adminStatus in [ 'rxOnly', 'txAndRx' ]
   setLldpAdminStatus( mode, cliPortConfig, txEnabled, rxEnabled )

def enableLldpReceive( mode, args ):
   cliPortConfig = lldpCliConfig.newPortConfig( mode.intf.name )
   txEnabled = cliPortConfig.adminStatus in [ 'txOnly', 'txAndRx' ]
   rxEnabled = not CliCommand.isNoCmd( args )
   setLldpAdminStatus( mode, cliPortConfig, txEnabled, rxEnabled )

def setPoeAllocatedPowerFallback( mode, args ):
   if CliCommand.isNoOrDefaultCmd( args ):
      deciWatts = PortConfig.poeAllocatedPowerFallbackDisabled
   elif 'max-requested-power' in args:
      deciWatts = PortConfig.poeAllocatedPowerFallbackMax
   else:
      deciWatts = wattsToDeciWatts( args[ 'POWER' ] )
   cliPortConfig = lldpCliConfig.newPortConfig( mode.intf.name )
   cliPortConfig.poeAllocatedPowerFallback = deciWatts

### CLI rule definitons.

class LldpIntf( CliPlugin.IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del lldpCliConfig.portConfig[ self.intf_.name ]
      
class LldpIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, CliPlugin.EthIntfCli.EthPhyIntf )

CliPlugin.IntfCli.IntfConfigMode.addModelet( LldpIntfModelet )

appKeywords = {
   'default' : 0,
   'video-conferencing' : 6,
   'streaming-video' : 7,
   'video-signaling' : 8,
}

def _getApplication( args ):
   if 'APPID' in args:
      app = args[ 'APPID' ]
      return app, app
   if 'APP' in args:
      app = args[ 'APP' ]
      return app, appKeywords.get( app, app )
   return None, None

def configNetworkPolicy( mode, args ):
   app, appValue = _getApplication( args )
   cliPortConfig = lldpCliConfig.newPortConfig( mode.intf.name )
   netpolConfig = NetworkPolicyConfig( appValue )
   netpolConfig.appName = str( app )
   netpolConfig.tagged = 'VLANID' in args
   netpolConfig.vlanId = args.get( 'VLANID', 0 )
   netpolConfig.cosValue = args.get( 'COS', 0 )
   netpolConfig.dscpValue = args.get( 'DSCP', 0 )
   cliPortConfig.addNetworkPolicyConfig( netpolConfig )

def noNetworkPolicy( mode, args ):
   _, appValue = _getApplication( args )
   cliPortConfig = lldpCliConfig.portConfig.get( mode.intf.name )
   if not cliPortConfig:
      return
   if appValue and appValue in cliPortConfig.networkPolicyConfig:
      del cliPortConfig.networkPolicyConfig[ appValue ]
   else:
      for appValue in cliPortConfig.networkPolicyConfig.keys():
         del cliPortConfig.networkPolicyConfig[ appValue ]
   maybeDeletePortConfig( mode )

def setLldpReceivePacketTaggedDrop( mode, args ):
   lldpCliConfig.rxTaggedPacketsDrop = True

def noLldpReceivePacketTaggedDrop( mode, args ):
   lldpCliConfig.rxTaggedPacketsDrop = False

def Plugin( entityManager ):
   global lldpCliConfig
   lldpCliConfig = ConfigMount.mount(
      entityManager, "l2discovery/lldp/cliConfig", "Lldp::CliConfig", "w" )
   CliPlugin.IntfCli.Intf.registerDependentClass( LldpIntf )
