# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module implements the CLI for Link Flap Detection.  Specifically, it provides
the following commands:

In Global Configuration mode:
-  [no] errdisable flap-setting cause link-flap max-flaps <flap-count> time <seconds>

In EXEC mode:
-  show errdisable flap-values


Profile style CLI Design for Link Flap Detection:

A mode to create and maintain "profiles" of link-flap configuration: 
(config-link-flap)

- monitor link-flap policy

Profile creation under link-flap mode: (config-link-flap):
- [no] profile <profile name> max-flaps <F> time <S> [violations <V> intervals <I>]

Changing global configuration via "link-flap" mode:
(for backward compatibility to support downgrade)
- [no] profile default max-flaps <F> time <S>

Selecting set of profiles as a default setting to be used by link-flap algorithm
- default-profiles <p1> [ <p2> <p3> ... ]

Per interface configuration: (Tri-state)
     
Each interface can either enable link flap monitoring, disable link flap
     monitoring or inherit from global setting.

To enable link flap monitoring with configuration parameters taken from
"default-profiles" if specified, or "profile default" in that order.
   
- monitor link-flap

To enable link flap monitoring with configuration parameters from the
chosen profiles
  
- monitor link-flap [ profiles <p1> [ <P2> <p3> .. ] ]

To disable link flap monitoring:

- no monitor link-flap

To inherit from global setting

- default monitor link-flap

"""

import BasicCli
import CliCommand
import CliMatcher
from CliMode.LinkFlapDetection import LinkFlapMode
import CliParser
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli
import CliToken.Monitor
import ConfigMount
import ShowCommand

linkFlapConfig = None

### CLI command implementations.

def setLinkFlapValues( mode, flaps, seconds ):
   linkFlapConfig.maxFlaps = flaps
   linkFlapConfig.interval = seconds

def noLinkFlapValues( mode ):
   setLinkFlapValues( mode, linkFlapConfig.maxFlapsDefault, 
      linkFlapConfig.intervalDefault )

#--------------------------------------------------------------------------------
# errdisable flap-setting cause link-flap max-flaps FLAPS time SECONDS
#--------------------------------------------------------------------------------
class LinkFlapValuesCmd( CliCommand.CliCommandClass ):
   syntax = 'errdisable flap-setting cause link-flap max-flaps FLAPS time SECONDS'
   noOrDefaultSyntax = 'errdisable flap-setting cause link-flap ...'
   data = {
      'errdisable' : 'Configure error disable functionality',
      'flap-setting' : 'Configure the maximum number of flaps that are allowed',
      'cause' : 'Specify the errdisable cause',
      'link-flap' : 'Specify the flap-setting values for link flap errdisable cause',
      'max-flaps' : 'Specify the maximum number of flaps',
      'FLAPS' : CliMatcher.IntegerMatcher( 1, 100, helpdesc='Max-flaps' ),
      'time' : 'Specify the time period that flaps are counted (in seconds)',
      'SECONDS' : CliMatcher.IntegerMatcher( 1, 1800, helpdesc='Time (in seconds)' ),
   }

   @staticmethod
   def handler( mode, args ):
      setLinkFlapValues( mode, args[ 'FLAPS' ], args[ 'SECONDS' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noLinkFlapValues( mode )

BasicCli.GlobalConfigMode.addCommandClass( LinkFlapValuesCmd )

#--------------------------------------------------------------------------------
# show errdisable flap-values
#--------------------------------------------------------------------------------
class ErrdisableFlapValuesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show errdisable flap-values'
   data = {
      'errdisable' : 'Show errdisable information',
      'flap-values' : ( 'Show flap values that cause an error to be recognized for '
                        'a cause' ),
   }

   @staticmethod
   def handler( mode, args ):
      pformat = '%-20.20s  %-10.10s  %-10.10s'
      print pformat % ( 'ErrDisable Reason', 'Flaps', 'Time' )
      print pformat % ( '=================', '=====', '====' )
      print pformat % ( 'link-flap', linkFlapConfig.maxFlaps,
         linkFlapConfig.interval )

BasicCli.addShowCommandClass( ErrdisableFlapValuesCmd )

### Config commands to define link-flap profiles
profileNameMatcher = CliMatcher.DynamicKeywordMatcher(
         lambda mode: linkFlapConfig.profileConfig,
         value=lambda mode, match: linkFlapConfig.profileConfig[ match ],
            emptyTokenCompletion=[ CliParser.Completion( 'WORD',
                                                         'profile name', 
                                                         literal=False ) ] )
# -----------------------------
# The "link-flap" mode.
# -----------------------------
class LinkFlapConfigMode( LinkFlapMode, BasicCli.ConfigModeBase ):

   name = "Link flap configuration"
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a LinkFlapConfigMode 
   #----------------------------------------------------------------------------
   def __init__( self, parent, session ):
      LinkFlapMode.__init__( self, "" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# monitor link-flap policy
#--------------------------------------------------------------------------------
class MonitorLinkFlapPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor link-flap policy'
   data = {
      'monitor' : CliToken.Monitor.monitorMatcher,
      'link-flap' : 'Specify the flap-setting values for link flap errdisable cause',
      'policy' : 'Manage profiles for link flap detection',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( LinkFlapConfigMode )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( MonitorLinkFlapPolicyCmd )

#--------------------------------------------------------------------------------
# [ no ] profile ( PROFILE_NAME | default ) max-flaps FLAPS time SECONDS
#                                       [ violations VIOLATIONS intervals INTERVALS ]
#--------------------------------------------------------------------------------
class ProfileAllParametersCmd( CliCommand.CliCommandClass ):
   syntax = ( 'profile ( PROFILE_NAME | default ) max-flaps FLAPS time SECONDS '
                                    '[ violations VIOLATIONS intervals INTERVALS ]' )
   noSyntax = 'profile ( PROFILE_NAME | default ) ...'
   data = {
      'profile' : 'Configure profiles for link-flap detection',
      'PROFILE_NAME' : CliMatcher.PatternMatcher( '[a-zA-Z0-9]+',
         helpname='profileName', helpdesc='Profile name' ),
      'default' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'default',
            helpdesc='Reflection of global setting' ),
         alias='PROFILE_NAME' ),
      'max-flaps' : 'Specify the maximum number of flaps',
      'FLAPS' : CliMatcher.IntegerMatcher( 1, 100, helpdesc='Max-flaps' ),
      'time' : 'Specify the time period that flaps are counted (in seconds)',
      'SECONDS' : CliMatcher.IntegerMatcher( 1, 1800, helpdesc='Time (in seconds)' ),
      'violations' : 'Specify how many violations to be detected',
      'VIOLATIONS' : CliMatcher.IntegerMatcher( 1, 1000, helpdesc='Violations' ),
      'intervals' : 'Specify the intervals for monitoring violations',
      'INTERVALS' : CliMatcher.IntegerMatcher( 1, 1000, helpdesc='Intervals' ),
   }

   @staticmethod
   def handler( mode, args):
      profileName = args[ 'PROFILE_NAME' ]
      flaps = args[ 'FLAPS' ]
      seconds = args[ 'SECONDS' ]
      if profileName == "default":
         # "default" Name replicates global setting for max-flaps and time
         # violations and intervals are ignored for this profile
         setLinkFlapValues( mode, flaps, seconds )
         if 'violations' in args:
            print 'violations and intervals are ignored for default profile'
      else:
         profileConfig = linkFlapConfig.newProfileConfig( profileName )
         profileConfig.maxFlaps = flaps
         profileConfig.interval = seconds

         if 'VIOLATIONS' in args:
            profileConfig.violations = args[ 'VIOLATIONS' ]
         if 'INTERVALS' in args:
            profileConfig.trackedIntervals = args[ 'INTERVALS' ]

   @staticmethod
   def noHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName == "default":
         # "default" Name replicates global setting for max-flaps and time
         noLinkFlapValues( mode )
      else:
         if profileName in linkFlapConfig.defaultProfileList:
            del linkFlapConfig.defaultProfileList[ profileName ]
         for intfConfig in linkFlapConfig.intfConfig.itervalues():
            if profileName in intfConfig.profileList:
               del intfConfig.profileList[ profileName ]
         if profileName in linkFlapConfig.profileConfig:
            del linkFlapConfig.profileConfig[ profileName ]

LinkFlapConfigMode.addCommandClass( ProfileAllParametersCmd )

#--------------------------------------------------------------------------------
# [ no | default ] default-profiles [ { PROFILES } ]
#--------------------------------------------------------------------------------
class DefaultProfilesCmd( CliCommand.CliCommandClass ):
   syntax = 'default-profiles [ { PROFILES } ]'
   noOrDefaultSyntax = 'default-profiles ...'
   data = {
      'default-profiles' : 'Specify a default profile list',
      'PROFILES' : profileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      profileNameList = args.get( 'PROFILES' )
      if profileNameList:
         for profileName in linkFlapConfig.defaultProfileList:
            if profileName not in profileNameList:
               del linkFlapConfig.defaultProfileList[ profileName ]
         for profileName in profileNameList:
            linkFlapConfig.defaultProfileList[ profileName.profileName ] = True
      else:
         for profileName in linkFlapConfig.defaultProfileList:
            del linkFlapConfig.defaultProfileList[ profileName ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for p in linkFlapConfig.defaultProfileList:
         del linkFlapConfig.defaultProfileList[ p ]

LinkFlapConfigMode.addCommandClass( DefaultProfilesCmd )

# -----------------------------
# The Interface mode.
# -----------------------------
### Setting Interface Mode: 

class LinkFlapIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del linkFlapConfig.intfConfig[ self.intf_.name ]

class LinkFlapIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      # This is a bit hacky, but there's no better way to recognize physical Ethernet
      # and Management interfaces than to look at their name.
      return ( isinstance( mode.intf, EthIntfCli.EthPhyIntf ) and
               ( mode.intf.name.startswith( 'Ethernet' ) or
                 mode.intf.name.startswith( 'Management' ) ) )

IntfCli.IntfConfigMode.addModelet( LinkFlapIntfModelet )

#--------------------------------------------------------------------------------
# [ no | default ] monitor link-flap [ profiles { PROFILES } ]
#--------------------------------------------------------------------------------
class MonitorLinkFlapCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor link-flap [ profiles { PROFILES } ]'
   noOrDefaultSyntax = 'monitor link-flap ...'
   data = {
      'monitor' : 'Configure monitoring',
      'link-flap' : 'Specify the flap-setting values for link flap errdisable cause',
      'profiles' : 'Assign profiles for link-flap detection',
      'PROFILES' : profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = linkFlapConfig.newIntfConfig( mode.intf.name )
      intfConfig.enabled = True

      if 'PROFILES' in args:
         profileNameList = args[ 'PROFILES' ]
         for profileName in intfConfig.profileList:
            if profileName not in profileNameList:
               del intfConfig.profileList[ profileName ]
         for profileName in profileNameList:
            intfConfig.profileList[ profileName.profileName ] = True
      else:
         for profileName in intfConfig.profileList:
            del intfConfig.profileList[ profileName ]

   @staticmethod
   def noHandler( mode, args ):
      intfConfig = linkFlapConfig.newIntfConfig( mode.intf.name )
      intfConfig.enabled = False
      for profileName in intfConfig.profileList:
         del intfConfig.profileList[ profileName ]

   @staticmethod
   def defaultHandler( mode, args ):
      del linkFlapConfig.intfConfig[ mode.intf.name ]

LinkFlapIntfModelet.addCommandClass( MonitorLinkFlapCmd )

def Plugin( entityManager ):
   global linkFlapConfig
   linkFlapConfig = ConfigMount.mount(
      entityManager, "interface/errdisable/linkFlapConfig", "LinkFlap::Config", "w" )
   IntfCli.Intf.registerDependentClass( LinkFlapIntfCleaner )
