# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import datetime
import time
from CliModel import Model
from CliModel import Str
from CliModel import Int
from CliModel import Float
from CliModel import Dict
from CliModel import List

def getHumanTime( timestamp ):
   # Ignore millisecond precision from human representation of time
   # applicable for licenses loaded from Cloud, DPE license loaders
   timestamp = int( timestamp )
   ts = 'NA'
   try:
      ts = str( datetime.datetime.fromtimestamp( timestamp ) )
   except ValueError:
      ts = str( datetime.datetime.fromtimestamp( 0 ) + \
                datetime.timedelta( seconds=timestamp ) )
   return ts

class LicenseInstance( Model ):
   parameter = Str( help='Parameter associated with the license', optional=True )
   startTime = Float( help='POSIX Timestamp of when license will become valid' )
   endTime = Float( help='POSIX Timestamp of when license will expire' )
   count = Int( help='Number of licenses of this type' )
   source = Str( help='Source' )

   def getActiveString( self, now ):
      if now < self.startTime:
         return "in future"
      if now > self.endTime:
         return "expired"
      return "yes"

   def customRender( self, now, interpretThroughput=False ):
      paramName = "License parameter: "
      paramStr = str( self.parameter ) if self.parameter else "None"
      if interpretThroughput:
         # 1 Gb = 1000 Mb per NIST standards
         paramName = "Throughput:        "
         if paramStr == "500":
            paramStr = "500 Mbps"
         elif paramStr == "1000":
            paramStr = "1 Gbps"
         elif paramStr == "10000":
            paramStr = "10 Gbps"
         elif paramStr == "None":
            paramStr = "Not Throttled"
         else:
            try:
               int( self.parameter )
               paramStr += " Mbps"
            except ValueError:
               pass
      print "\t%s" % paramName , paramStr
      template = "\t{:20}{}"
      print template.format( "Count:", self.count )
      print template.format( "Start:", getHumanTime( self.startTime ) )
      print template.format( "Expiration:", getHumanTime( self.endTime ) )
      print template.format( "Active:", self.getActiveString( now ) )
      print template.format( "License source:", self.source )
      print 

class FeatureModel( Model ):
   licenses = List( valueType=LicenseInstance,
                       help='List of licenses for the feature' )
   featureDescription = Str( help='Description of the feature', optional=True )

   def customRender( self, now, isVeos=False ):
      for lic in self.licenses:
         lic.customRender( now, interpretThroughput=isVeos )

class LicenseStatusModel( Model ):
   deviceSerial = Str( help='System Serial number' )
   systemMac = Str( help='System MAC address' )
   domainName = Str( help='Domain name' )
   customerName = Str( help='Customer name' )
   platform = Str( optional=True, help='Platform model' )

   features = Dict( valueType=FeatureModel,
         help="A mapping between feature name and its license information" )
 
   def render( self ):
      # Pass a timestamp to submodels so that license activity status is
      # determined using the same timestamp
      now = time.time()
      print "Customer name:        ", self.customerName
      print "System Serial number: ", self.deviceSerial
      print "System MAC address:   ", self.systemMac
      print "Domain name:          ", self.domainName
      print "Platform:             ", self.platform

      for f in sorted( self.features ):
         featureDesc = self.features[ f ].featureDescription
         isVeos = f == 'CloudEOS'
         featureDescString = " -  %s" % featureDesc if featureDesc else ''
         print "\nLicense feature: ", f, featureDescString
         self.features[ f ].customRender( now, isVeos=isVeos )

class LicenseFeatureModel( Model ):
   missing = List( valueType=str,
                    help="List of missing licenses" )
   active = List( valueType=str, 
                  help="List of active licenses" )
   def render( self ):
      print "License missing for features:"
      for feature in self.missing:
         print "\t%s" % feature
      print "License in use for features:"
      for feature in self.active:
         print "\t%s" % feature

class LicenseFilesModel( Model ):
   licenseText = Dict( valueType=str, 
         help="A mapping between license file name and its contents" )

   def render( self ):
      for f in self.licenseText:
         print
         print "License name: ", f
         print "Contents: "
         print self.licenseText[ f ]
