# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMatcher

# -----------------------------------------------------------------------------
# Helper function to map Led hardware names to User identifiable names.
# For a Fixed system, the Status Led is identified as the Chassis beacon.
# For a Modular system, the Status Led is identified as the supervisor beacon.
# -----------------------------------------------------------------------------
def ledToShowName( led, systemType ):
   if systemType == "Fixed":
      if led.startswith( 'Status' ):
         return "Chassis"
      elif led.startswith( 'Fan' ) and 'FanTray' not in led:
         return led.replace( 'Fan', 'FanTray' )
      else:
         return led
   elif systemType == "Modular":
      if led.startswith( 'Status' ):
         led = led.replace( 'Status', 'Supervisor' )
      elif led.startswith( 'Fan' ):
         led = led.replace( 'Fan', 'FanTray' )
      else:
         return led
   # unknown systemType, just return led name.
   return led

# ---------------------------------------------------------------------------
# Helper function to check for the existence of an LED by checking for different
# naming variants of the LED's name in the led collection.
#
# Return: Tuple ( ledExists, ledName )
#    ledExists: True if a variant of the LED name exists in the LED collection
#    ledName: The resulting LED name that was found in the LED collection
# ---------------------------------------------------------------------------
def checkLedExists( led, hwLedConfig ):
   if led in hwLedConfig.leds:
      return True, led

   ledName = led.replace( "Fan", "FanTray" )
   if ledName and ledName in hwLedConfig.leds:
      return True, ledName

   # Check for MultiIntf LED
   sliceIndex = led.rfind( '/' )
   if sliceIndex != -1:
      ledName = led[:sliceIndex]
      if ledName and ledName in hwLedConfig.leds:
         return True, ledName

   return False, led

ledSets = { 'HardwareLedCliSet' : 95,
            'beacon' : 90,
            'cliLedSet' : 85 }

def getLedSet( ledSetName, ledConfig ):
   ''' Return the specified set entity. If the set does not exists, create it
   first '''

   assert ledConfig != None
   assert ledSetName in ledSets

   ledSet = ledConfig.get( ledSetName )
   if not ledSet:
      ledConfig.newEntity( "Led::Set", ledSetName )
      ledSet = ledConfig[ ledSetName ]
      # Lower priority number is higher priority.
      # Normal priority is 100
      ledSet.priority = ledSets[ ledSetName ]
   return ledSet

colorEnumMatcher = CliMatcher.EnumMatcher( {
      'green' : 'Green LED color',
      'yellow' : 'Yellow LED color',
      'red' : ' Red LED color',
      'blue' : 'Blue LED color',
      'off' : 'Turn LED off',
   } )
