# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli, CliParser, LazyMount, Tac
import CliCommand
import CliPlugin.FruCli as FruCli
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import LedCommon
import Intf.IntfRange
import ShowCommand
import os

verbose = False

ledConfig = None
hwLedConfigInit = None
hwLedConfig = None
# set to either "Fixed" or "Modular"
systemType = None
ledSetName = 'cliLedSet'
possColors = [ 'green', 'yellow', 'red', 'blue' ]
plutoStatus = Tac.Type( "Led::PlutoStatus" )

def _registerSystemTypeFixed():
   global systemType
   systemType = "Fixed"
FruCli.registerFixedSystemCallback( _registerSystemTypeFixed )

def _registerSystemTypeModular():
   global systemType
   systemType = "Modular"
FruCli.registerModularSystemCallback( _registerSystemTypeModular )

# -----------------------------------------------------------------------------
# Helper function to determine all the colors that are configured for a user
# configured led.
# -----------------------------------------------------------------------------

def _ledToShowColor( led ):
   ledSet = LedCommon.getLedSet( ledSetName, ledConfig )
   colors = []
   for color in possColors:
      if getattr( ledSet.led[ led ], color ):
         colors.append( color )
   if not colors:
      return 'off'
   return ', '.join( colors )

# -----------------------------------------------------------------------------
# [no] led { module { Supervisor <#> | Linecard <#> | Fabric <#> | <#> } |
#                    interface <interface range> |
#                    powersupply <powersupply number> |
#                    fan <fan number> |
#                    chassis |
#                    multifan } color
# -----------------------------------------------------------------------------
def _setLed( led, color, disable ):
   assert hwLedConfigInit != None
   assert ledConfig != None

   ledExists, led = LedCommon.checkLedExists( led, hwLedConfigInit )
   if not ledExists:
      print "LED %s not found!" % led
      return

   ledSet = LedCommon.getLedSet( ledSetName, ledConfig )

   if disable:
      if led in ledSet.led:
         del ledSet.led[ led ]
   else:
      if not color:
         print "Color parameter required!"
         return

      ledSetting = Tac.Value( "Led::LightSetting" )
      ledSetting.rampRate = 0.0104
      ledSetting.maxBright = 100
      ledSetting.flashRate = 0
      ledSetting.plutoStatus = plutoStatus.plutoStatusGood
      if color != 'off':
         setattr( ledSetting, color, True )
      else:
         for clr in possColors:
            setattr( ledSetting, clr, False )
      ledSet.led[ led ] = ledSetting

def _targetToLed( target ):
   if verbose:
      print "Target tag: %s" % target.tag
      print "Target label: %s" % target.label

   for prefix in ( 'Supervisor', 'Linecard', 'Fabric', 'PowerSupply', 'Fan' ):
      if target.tag.startswith( prefix ):
         if prefix == 'Supervisor':
            prefix = "Status"
         return "%s%s" % ( prefix, target.label )

   return None

def doLedTarget( mode, color=None, target=None, noOption=None ):
   assert target != None

   if verbose:
      print "NoOption: ", noOption
      print "Target: ", target
      print "Name: ", target.name
      print "Color: ", color

   _setLed( _targetToLed( target ), color, noOption )
   
def guardMultiFanLed( mode, token ):
   hwLed = hwLedConfigInit.leds.get( "MultiFan1" )
   if hwLed:
      return None
   return CliParser.guardNotThisPlatform

def sonicCeosGuard( mode, token ):
   if os.environ.get( "CEOS_MODE" ) == 'sonic':
      return None
   return CliParser.guardNotThisPlatform

nodeLed = CliCommand.guardedKeyword( 'led',
   helpdesc='Set a specific LED to a color',
   guard=sonicCeosGuard,
   hidden=True )

#--------------------------------------------------------------------------------
# [ no ] led ( ( fantray FAN_TRAY ) | ( powersupply POWER_SUPPLY ) ) [ COLOR ]
#--------------------------------------------------------------------------------
class LedCmd( CliCommand.CliCommandClass ):
   syntax = 'led ( ( fantray FAN_TRAY ) | ( powersupply POWER_SUPPLY ) ) [ COLOR ]'
   noSyntax = syntax
   data = {
      'led' : nodeLed,
      'fantray' : 'Fan tray LED',
      'FAN_TRAY' : FruCli.FanTrayMatcher(),
      'powersupply' : 'Power supply LED',
      'POWER_SUPPLY' : FruCli.PowerSupplyMatcher(),
      'COLOR' : LedCommon.colorEnumMatcher
   }

   @staticmethod
   def handler( mode, args ):
      target = args.get( 'FAN_TRAY', args.get( 'POWER_SUPPLY' ) )
      doLedTarget( mode, color=args.get( 'COLOR' ), target=target,
         noOption=CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LedCmd )

#--------------------------------------------------------------------------------
# [ no ] led multifan [ COLOR ]
#--------------------------------------------------------------------------------
class LedMultifanCmd( CliCommand.CliCommandClass ):
   syntax = 'led multifan [ COLOR ]'
   noSyntax = syntax
   data = {
      'led' : nodeLed,
      'multifan' : CliCommand.guardedKeyword( 'multifan', helpdesc='Multi fan LED',
         guard=guardMultiFanLed ),
      'COLOR' : LedCommon.colorEnumMatcher
   }

   @staticmethod
   def handler( mode, args ):
      _setLed( 'MultiFan1', args.get( 'COLOR' ), CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LedMultifanCmd )

#--------------------------------------------------------------------------------
# [ no ] led interface INTF [ COLOR ]
#--------------------------------------------------------------------------------
class LedInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'led interface INTF [ COLOR ]'
   noSyntax = syntax
   data = {
      'led' : nodeLed,
      'interface' : 'Interface LED',
      'INTF' : Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthPhyAutoIntfType, ) ),
      'COLOR' : LedCommon.colorEnumMatcher
   }

   @staticmethod
   def handler( mode, args ):
      for led in args[ 'INTF' ].intfNames():
         _setLed( led, args.get( 'COLOR' ), CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LedInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# [ no ] led module SLOT [ COLOR ]
#--------------------------------------------------------------------------------
class LedModuleCmd( CliCommand.CliCommandClass ):
   syntax = 'led module SLOT [ COLOR ]'
   noSyntax = syntax
   data = {
      'led' : nodeLed,
      'module' : CliCommand.guardedKeyword( 'module', helpdesc='Module LED',
         guard=FruCli.modularSystemGuard ),
      'SLOT' : FruCli.SlotExpressionFactory(),
      'COLOR' : LedCommon.colorEnumMatcher
   }

   @staticmethod
   def handler( mode, args ):
      target = args[ 'SLOT' ].slot
      doLedTarget( mode, args.get( 'COLOR' ), target, CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LedModuleCmd )

# -----------------------------------------------------------------------------
# [ no ] led chassis [ COLOR ]
# 'led chassis' command is only available on FixedSystems with "Status1" led
# -----------------------------------------------------------------------------
def guardChassisLed( mode, token ):
   if FruCli.fixedSystemGuard( mode, token ):
      return CliParser.guardNotThisPlatform

   hwLed = hwLedConfigInit.leds.get( 'Status1' )
   if hwLed:
      return None
   return CliParser.guardNotThisPlatform

class LedChassisCmd( CliCommand.CliCommandClass ):
   syntax = 'led chassis [ COLOR ]'
   noSyntax = syntax
   data = {
      'led' : nodeLed,
      'chassis': CliCommand.guardedKeyword( 'chassis', 'Chassis beacon LED',
         guard=guardChassisLed ),
      'COLOR' : LedCommon.colorEnumMatcher
   }

   @staticmethod
   def handler( mode, args ):
      _setLed( 'Status1', args.get( 'COLOR' ), CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LedChassisCmd )

#--------------------------------------------------------------------------------
# show led configuration
#--------------------------------------------------------------------------------
class LedConfigurationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show led configuration'
   data = {
      'led' : 'Show LED Information',
      'configuration' : CliCommand.guardedKeyword( 'configuration',
         helpdesc='Show User Configured LEDs', guard=sonicCeosGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      ledSet = LedCommon.getLedSet( ledSetName, ledConfig )

      if ledSet.led:
         print "User configured LEDs currently enabled"
         print "--------------------------------------"
         for led in sorted( ledSet.led ):
            print "%s (%s)" % ( LedCommon.ledToShowName( led, systemType ),
                                _ledToShowColor( led ) )
      else:
         print "There are no user configured LEDs enabled"

BasicCli.addShowCommandClass( LedConfigurationCmd )

# -----------------------------------------------------------------------------
# Cli Plugin initialization
# -----------------------------------------------------------------------------
def Plugin( entityManager ):
   global hwLedConfigInit, ledConfig

   hwLedConfigInit = LazyMount.mount( entityManager, "hardware/led/configInit",
                                  "Hardware::Led::LedSystemConfigDir", "r" )
   ledConfig = LazyMount.mount( entityManager, "led/config", "Tac::Dir", "w" )
