# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from ArnetModel import (
      IpGenericAddr,
      IpGenericAddrAndPort,
)
import BasicCli
import CliCommand
import CliMatcher
from CliModel import cliPrinted
from CliPlugin import LdpModel, TunnelModels
from CliPlugin.AclCli import (
      accessListKwMatcherForServiceAcl,
      ipAclNameExpression,
      showServiceAcl,
)
from CliPlugin.AclCliModel import AllAclList
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.Ip6AddrMatcher import (
      Ip6AddrMatcher,
      Ip6PrefixMatcher,
)
import CliPlugin.LdpCli as LdpCli
from CliPlugin.LdpModel import (
      LdpMessageCount,
      LdpNotificationCount,
      LdpSessionStateLogEntryModel,
      LdpTunnelTable,
      MplsLdpAdjacency,
      MplsLdpAdjacencyDetail,
      MplsLdpBindingsModel,
      MplsLdpDetail,
      MplsLdpDiscovery,
      MplsLdpDiscoveryModel,
      MplsLdpDiscoverySource,
      MplsLdpId,
      MplsLdpNeighbor,
      MplsLdpNeighborList,
      MplsLdpNeighborModel,
      MplsLdpStatus,
      MplsLdpStatusModel,
      PasswordInformation,
      getNeighborGrStateValue,
)
from CliPlugin.MplsCli import (
      mplsNodeForShow,
      bindingsKw,
)
from CliPlugin.RouteDistinguisher import (
      RdDistinguisherMatcher,
)
from CliPlugin.TunnelCli import (
      TunnelTableIdentifier,
      getFecFromIntfId,
      getTunnelEntryAndIdFromIndexAndAf,
      getTunnelIdModel,
      getTunnelIndexFromId,
      readMountTunnelTable,
      tokenTunnelMatcher,
      tunnelIndexMatcher,
)
from CliPlugin.TunnelCliLib import (
      getDyTunTidFromIntfId,
      isDyTunIntfId,
)
import ConfigMount
from Intf.IntfRange import IntfRangeMatcher
from IpLibConsts import DEFAULT_VRF
import LazyMount
import natsort
import ShowCommand
import SharedMem
import Smash
import sys
import Tac
import Toggles.LdpToggleLib
import Toggles.MplsToggleLib
from TypeFuture import TacLazyType

# pkgdeps: library Arnet
# pkgdeps: library Ldp
# pkgdeps: library LdpCli
# pkgdeps: library Mpls
# pkgdeps: library MplsSysdbTypes

CommonLibConsumerSm = TacLazyType( 'CommonLibSmash::CommonLibConsumerSm' )
FecType = TacLazyType( 'Mpls::FecType' )
FwdEqvClass = TacLazyType( 'Mpls::FwdEqvClass' )
IpAddrTac = TacLazyType( 'Arnet::IpAddr' )
IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
LabelAdvertisementColl = TacLazyType( 'Ldp::LabelAdvertisementColl' )
LabelBindingTable = TacLazyType( 'Mpls::LabelBindingTable' )
LabelBindingTableColl = TacLazyType( 'Mpls::LabelBindingTableColl' )
LdpAdjId = TacLazyType( 'Ldp::LdpAdjId' )
LdpAdjType = TacLazyType( 'Ldp::LdpAdjType' )
LdpBindingsHelper = TacLazyType( 'Ldp::Cli::LdpBindingsHelper' )
LdpIdentifier = TacLazyType( 'Ldp::LdpIdentifier' )
LdpLibAdvertisementConsumerSm = \
      TacLazyType( 'Ldp::LdpSmash::LdpLibAdvertisementConsumerSm' )
LdpLibConsumerSm = TacLazyType( 'Ldp::LdpSmash::LdpLibConsumerSm' )
LdpSessionId = TacLazyType( 'Ldp::LdpSessionId' )
LoopbackIntfId = TacLazyType( 'Arnet::LoopbackIntfId' )
RouteDistinguisherTac = TacLazyType( 'Arnet::RouteDistinguisher' )
RouterId = TacLazyType( 'Mpls::RouterId' )

FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
FecId = TacLazyType( 'Smash::Fib::FecId' )

aclCheckpoint = None
aclCpConfig = None
aclStatus = None
commonLibSmashConsumerSm = None
commonLibSmashLdpBindingTable = None
commonLibSmashLdpP2mpBindingTable = None
ldpBindingTable = None
ldpConfigColl = None
ldpLibAdsConsumerSm = None
ldpLibConsumerSm = None
ldpP2mpBindingTable = None
ldpP2mpLibAdsConsumerSm = None
commonLibSmashP2mpConsumerSm = None
ldpP2mpLibConsumerSm = None
ldpProtoConfigColl = None
ldpStatusColl = None
ldpTunnelTable = None

detailKw = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Display information in detail' )
summaryKw = CliMatcher.KeywordMatcher( 'summary', helpdesc='Summarized information' )

def showMldpConfigWarnings( mode, brief=False ):
   # Assuming default vrf for now
   vrfName = DEFAULT_VRF

   config = ldpConfigColl.config.get( vrfName )
   if ( config is None or not config.mLdpEnabled ):
      mode.addWarning( "MLDP is not enabled" )

#--------------------------------------------------------------------------------
# show mpls ldp access-list [ ACLNAME ]
#--------------------------------------------------------------------------------
class MplsLdpAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls ldp access-list [ ACLNAME ]'
   data = {
      'mpls' : mplsNodeForShow,
      'ldp' : LdpCli.ldpKw,
      'access-list' : accessListKwMatcherForServiceAcl,
      'ACLNAME' : ipAclNameExpression,
   }
   cliModel = AllAclList

   @staticmethod
   def handler( mode, args ):
      LdpCli.showLdpConfigWarnings( mode )

      aclType = 'ip'
      vrfName = DEFAULT_VRF
      config = ldpConfigColl.config.get( vrfName )
      status = ldpStatusColl.status.get( vrfName )
      if config is None or status is None:
         return AllAclList()
      return showServiceAcl( mode,
                             aclCpConfig,
                             aclStatus,
                             aclCheckpoint,
                             aclType,
                             args[ '<aclNameExpr>' ],
                             serviceName='ldp' )

BasicCli.addShowCommandClass( MplsLdpAccessListCmd )

#--------------------------------------------------------------------------------
# show mpls ldp bindings [ PREFIX ] [ detail | summary ]
#--------------------------------------------------------------------------------
class MplsLdpBindingsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls ldp bindings [ PREFIX ] [ detail | summary ]'
   data = {
      'mpls' : mplsNodeForShow,
      'ldp' : LdpCli.ldpKw,
      'bindings' : bindingsKw,
      'PREFIX' : IpAddrMatcher.ipPrefixExpr(
         'Network address', 'Network mask', 'Address',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'detail' : detailKw,
      'summary' : summaryKw,
   }
   cliModel = MplsLdpBindingsModel

   @staticmethod
   def handler( mode, args ):
      LdpCli.showLdpConfigWarnings( mode )

      fec = args.get( 'PREFIX' )
      summary = 'summary' in args
      detail = summary or 'detail' in args

      commonLibToggle = Toggles.MplsToggleLib.toggleCommonLibSmashEnabled()
      # grab local and peer label binding tables from Smash
      if commonLibToggle:
         global commonLibSmashConsumerSm
         if commonLibSmashConsumerSm is None:
            peerLbtColl = LabelBindingTableColl( "plbtc" )
            localLbt = LabelBindingTable( RouterId() )
            commonLibSmashConsumerSm = \
                  CommonLibConsumerSm( commonLibSmashLdpBindingTable, localLbt,
                                       peerLbtColl )
         localLbt = commonLibSmashConsumerSm.localLbt
         peerLbtColl = commonLibSmashConsumerSm.peerLbtColl
      else:
         global ldpLibConsumerSm
         if ldpLibConsumerSm is None:
            peerLbtColl = LabelBindingTableColl( "plbtc" )
            localLbt = LabelBindingTable( RouterId() )
            ldpLibConsumerSm = LdpLibConsumerSm( ldpBindingTable, localLbt,
                  peerLbtColl )
         localLbt = ldpLibConsumerSm.localLbt
         peerLbtColl = ldpLibConsumerSm.peerLbtColl

      # populate the binding advertisements from the Smash table directly
      global ldpLibAdsConsumerSm
      if ldpLibAdsConsumerSm is None:
         advertisement = LabelAdvertisementColl( "lads" )
         ldpLibAdsConsumerSm = LdpLibAdvertisementConsumerSm( ldpBindingTable,
               commonLibSmashLdpBindingTable,
               advertisement )

      sys.stdout.flush()
      fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      helper = LdpBindingsHelper( None, False, None )
      helper.ldpStatusAds = ldpLibAdsConsumerSm.advertisement

      if fec:
         fwdEqvClass = FwdEqvClass()
         fwdEqvClass.prefix = IpGenPrefix( str( fec ) )
         helper.populateFecBindingsFromTablesFiltered( localLbt,
                                                       peerLbtColl,
                                                       detail,
                                                       fwdEqvClass )
      else:
         helper.populateFecBindingsFromTables( localLbt,
                                               peerLbtColl,
                                               detail )
      helper.populateLocalConflicts()
      if detail:
         helper.populateAdvertisements()

      helper.renderBindings( fd, fmt, summary )
      return cliPrinted( MplsLdpBindingsModel )

BasicCli.addShowCommandClass( MplsLdpBindingsCmd )

#--------------------------------------------------------------------------------
# show mpls ldp bindings mldp ...
#--------------------------------------------------------------------------------
class MplsLdpBindingsMldpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show mpls ldp bindings mldp
               [ ROOT ( ( internal-id INTERNALID ) |
                        ( lsp-id LSPID ) |
                        ( ipv4 ( SOURCEIPV4ADDR | IPV4RPPREFIX ) GROUPIPV4ADDR ) |
                        ( ipv6 ( SOURCEIPV6ADDR | IPV6RPPREFIX ) GROUPIPV6ADDR ) |
                        ( vpn4 ( SOURCEIPV4ADDR | IPV4RPPREFIX ) GROUPIPV4ADDR RD ) |
                        ( vpn6 ( SOURCEIPV6ADDR | IPV6RPPREFIX ) GROUPIPV6ADDR RD )
                      ) ]
               [ detail | summary ]'''
   data = {
      'mpls' : mplsNodeForShow,
      'ldp' : LdpCli.ldpKw,
      'bindings' : bindingsKw,
      'mldp' : 'MLDP configuration',
      'ROOT' : IpAddrMatcher.IpAddrMatcher( 'P2MP root node address' ),
      'internal-id' : 'Internal ID',
      'INTERNALID' : CliMatcher.IntegerMatcher( 0, 2**32-1, helpdesc='Internal ID' ),
      'lsp-id' : 'LSP ID',
      'LSPID' : CliMatcher.IntegerMatcher( 0, 2**32-1, helpdesc='LSP ID' ),
      'ipv4' : 'IPv4',
      'SOURCEIPV4ADDR' : IpAddrMatcher.IpAddrMatcher( 'IPv4 source address' ),
      'IPV4RPPREFIX' : IpAddrMatcher.IpPrefixMatcher( 'IPv4 RP prefix' ),
      'GROUPIPV4ADDR' : IpAddrMatcher.IpAddrMatcher( 'IPv4 group address' ),
      'ipv6' : 'IPv6',
      'SOURCEIPV6ADDR' : Ip6AddrMatcher( 'IPv6 source address' ),
      'IPV6RPPREFIX' : Ip6PrefixMatcher( 'IPv6 RP prefix' ),
      'GROUPIPV6ADDR' : Ip6AddrMatcher( 'IPv6 group address' ),
      'vpn4' : 'IPv4 VPN',
      'RD': RdDistinguisherMatcher( 'BGP route distinguisher' ),
      'vpn6' : 'IPv6 VPN',
      'detail' : detailKw,
      'summary' : summaryKw,
   }
   cliModel = MplsLdpBindingsModel

   @staticmethod
   def handler( mode, args ):
      LdpCli.showLdpConfigWarnings( mode )
      showMldpConfigWarnings( mode )

      rootIpAddr = args.get( 'ROOT' )
      internalId = args.get( 'INTERNALID', 0 )
      lspId = args.get( 'LSPID', 0 )
      summary = 'summary' in args
      detail = summary or 'detail' in args

      sourceIpAddr = ( args.get( 'SOURCEIPV4ADDR' ) or
                       args.get( 'SOURCEIPV6ADDR' ) or
                       IpGenAddr() )
      rpIpPrefix = ( args.get( 'IPV4RPPREFIX' ) or
                     args.get( 'IPV6RPPREFIX' ) or
                     IpGenPrefix() )
      groupIpAddr = ( args.get( 'GROUPIPV4ADDR' ) or
                      args.get( 'GROUPIPV6ADDR' ) or
                      IpGenAddr() )
      rd = args.get( 'RD' ) or RouteDistinguisherTac()

      if 'internal-id' in args:
         filterType = 'mldpReservedOpaqueType'
      elif 'lsp-id' in args:
         filterType = 'mldpGenericLspIdentifier'
      elif 'ipv4' in args:
         if 'SOURCEIPV4ADDR' in args:
            filterType = 'mldpTransitV4Src'
         else:
            filterType = 'mldpTransitV4Bidir'
      elif 'ipv6' in args:
         if 'SOURCEIPV6ADDR' in args:
            filterType = 'mldpTransitV6Src'
         else:
            filterType = 'mldpTransitV6Bidir'
      elif 'vpn4' in args:
         if 'SOURCEIPV4ADDR' in args:
            filterType = 'mldpTransitVpnV4Src'
         else:
            filterType = 'mldpTransitVpnV4Bidir'
      elif 'vpn6' in args:
         if 'SOURCEIPV6ADDR' in args:
            filterType = 'mldpTransitVpnV6Src'
         else:
            filterType = 'mldpTransitVpnV6Bidir'

      vrfName = DEFAULT_VRF
      status = ldpStatusColl.status.get( vrfName )
      if status is None:
         # we would need to return a valid json, so the helper should be
         # called, even if status is None.
         p2mpOpaqueTable = None
      else:
         p2mpOpaqueTable = status.p2mpOpaqueTable

      commonLibToggle = Toggles.MplsToggleLib.toggleCommonLibSmashEnabled()

      # grab local and peer label binding tables from Smash
      if commonLibToggle:
         global commonLibSmashP2mpConsumerSm
         if commonLibSmashP2mpConsumerSm is None:
            peerLbtColl = LabelBindingTableColl( "pplbtc" )
            localLbt = LabelBindingTable( RouterId() )
            commonLibSmashP2mpConsumerSm = \
                  CommonLibConsumerSm( commonLibSmashLdpP2mpBindingTable, localLbt,
                                       peerLbtColl )
         localLbt = commonLibSmashP2mpConsumerSm.localLbt
         peerLbtColl = commonLibSmashP2mpConsumerSm.peerLbtColl
      else:
         global ldpP2mpLibConsumerSm
         if ldpP2mpLibConsumerSm is None:
            peerLbtColl = LabelBindingTableColl( "pplbtc" )
            localLbt = LabelBindingTable( RouterId() )
            ldpP2mpLibConsumerSm = LdpLibConsumerSm( ldpP2mpBindingTable, localLbt,
                                                     peerLbtColl )
         localLbt = ldpP2mpLibConsumerSm.localLbt
         peerLbtColl = ldpP2mpLibConsumerSm.peerLbtColl

      # populate the binding advertisements from the Smash table directly
      global ldpP2mpLibAdsConsumerSm
      if ldpP2mpLibAdsConsumerSm is None:
         advertisement = LabelAdvertisementColl( "plads" )
         ldpP2mpLibAdsConsumerSm = LdpLibAdvertisementConsumerSm(
               ldpP2mpBindingTable, commonLibSmashLdpP2mpBindingTable,
               advertisement )

      sys.stdout.flush()
      fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      helper = LdpBindingsHelper( None, False, p2mpOpaqueTable )
      helper.ldpStatusAds = ldpP2mpLibAdsConsumerSm.advertisement

      if status and rootIpAddr:
         rootIpGenAddr = IpGenAddr( str( rootIpAddr ) )
         fwdEqvClass = FwdEqvClass()
         fwdEqvClass.fecType = FecType.mldpFec
         fwdEqvClass.mldpRootIp = rootIpGenAddr
         # Find the opaque value id
         fwdEqvClass.mldpOpaqueId = helper.getMldpOpaqueValueId(
               filterType, rootIpGenAddr, internalId, lspId,
               IpGenAddr( str( sourceIpAddr ) ), IpGenPrefix( str( rpIpPrefix ) ),
               IpGenAddr( str( groupIpAddr ) ), rd )
         helper.populateFecBindingsFromTablesFiltered( localLbt,
               peerLbtColl, detail, fwdEqvClass )
      else:
         helper.populateFecBindingsFromTables( localLbt,
                                               peerLbtColl,
                                               detail )
      helper.populateLocalConflicts()
      if detail:
         helper.populateAdvertisements()

      helper.renderBindings( fd, fmt, summary )
      return cliPrinted( MplsLdpBindingsModel )

if Toggles.LdpToggleLib.toggleP2mpToggleEnabled():
   BasicCli.addShowCommandClass( MplsLdpBindingsMldpCmd )

#--------------------------------------------------------------------------------
# show mpls ldp [ detail ]
#--------------------------------------------------------------------------------
class MplsLdpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls ldp [ detail ]'
   data = {
      'mpls' : mplsNodeForShow,
      'ldp' : LdpCli.ldpKw,
      'detail' : detailKw,
   }
   cliModel = MplsLdpStatusModel

   @staticmethod
   def handler( mode, args ):
      vrfs = {}
      allVrfs = ldpStatusColl.status or [ DEFAULT_VRF ]
      LdpCli.showLdpConfigWarnings( mode, brief=True )

      for vrfName in allVrfs:
         protoConfig = ldpProtoConfigColl.protoConfig.get( vrfName )
         config = ldpConfigColl.config.get( vrfName )
         status = ldpStatusColl.status.get( vrfName )
     
         if config is None or not config.enabled:
            MplsLdpInfo = MplsLdpStatus( enabled=False,
                                         operational=False,
                                         reason="LDP is not enabled" )
            vrfs[ vrfName ] = MplsLdpInfo
            continue

         if protoConfig is None:
            MplsLdpInfo = MplsLdpStatus( enabled=True,
                                         operational=False,
                                         reason="Internal error" )
            vrfs[ vrfName ] = MplsLdpInfo
            continue
      
         if protoConfig.runningState != 'ldpRunning' or status is None:
            MplsLdpInfo = MplsLdpStatus( enabled=True,
                                         operational=False,
                                         reason=protoConfig.reason )
            vrfs[ vrfName ] = MplsLdpInfo
            continue

         # LDP is enabled and operational and protoConfig, config and status do exist
         param = protoConfig.ldpProtoParam
         linkConfigColl = status.ldpLinkConfigColl
         adjConfigColl = status.ldpAdjConfigColl
         peerStateColl = status.ldpPeerStateColl
         sessionConfigColl = status.ldpSessionConfigColl
         statistics = status.statistics
      
         if ( param is None or
              linkConfigColl is None or
              adjConfigColl is None or
              peerStateColl is None or
              sessionConfigColl is None or
              statistics is None ):
            MplsLdpInfo = MplsLdpStatus( enabled=True,
                                         operational=False,
                                         reason="Internal error" )
            vrfs.update( { vrfName : MplsLdpInfo } )
            continue

         localLdpIdentifier = protoConfig.getRawAttribute( "ldpIdentifier" )
         localLdpId = MplsLdpId( ldpRouterId=IpAddrTac( localLdpIdentifier.lsrId ),
                                 ldpLabelSpace=localLdpIdentifier.lblSpace )
         loopDetectionEnabled = config.ldpLoopDetection
         
         helloHoldTimeParam = param.defaultBasicHelloHoldTime
         helloIntervalParam = param.defaultBasicHelloInterval
         if config.helloHoldTime != 0:
            helloHoldTimeParam = config.helloHoldTime
         if config.helloInterval != 0:
            helloIntervalParam = config.helloInterval

         targetedHelloHoldTimeParam = param.defaultTargetHelloHoldTime
         targetedHelloIntervalParam = param.defaultTargetHelloInterval
         if config.targetedHelloHoldTime != 0:
            targetedHelloHoldTimeParam = config.targetedHelloHoldTime
         if config.targetedHelloInterval != 0:
            targetedHelloIntervalParam = config.targetedHelloInterval
         if 'detail' in args:
            advertisedIntfIps = []
            for intfIp in natsort.natsorted(
                  sessionConfigColl.ldpSessionCommonConfig.localIntfIp ):
               advertisedIntfIps.append( IpGenericAddr( ip=intfIp ) )
            defaultLdpIntfEnabled = not config.onlyLdpEnabledIntfs

            parameters = MplsLdpDetail(
                  protocolVersion=param.ldpProtocolVersion,
                  keepAlive=param.defaultSessionKeepAliveTime,
                  helloHoldTime=helloHoldTimeParam,
                  helloInterval=helloIntervalParam,
                  targetedHelloHoldTime=targetedHelloHoldTimeParam,
                  targetedHelloInterval=targetedHelloIntervalParam,
                  initialBackoff=param.defaultRetryBackoffTime,
                  maximumBackoff=param.defaultMaxRetryBackoffTime,
                  advertisedIntfIps=advertisedIntfIps,
                  defaultLdpIntfEnabled=defaultLdpIntfEnabled,
                  grStateHoldTime=config.grHoldingTimeout,
                  grReconnectTime=config.grReconnectTimeout,
                  grMaxNeighborLivenessTime=config.grNeighborLivenessTimeout,
                  grMaxRecoveryTime=config.grMaxRecoveryTimeout )
         else:
            parameters = None
      
         if ( ( config.ipTransportAddrIntfId and
                not LoopbackIntfId.isLoopbackIntfId(
                   config.ipTransportAddrIntfId ) ) or
              not ( config.ipTransportAddrIntfId or
                 LoopbackIntfId.isLoopbackIntfId( protoConfig.intfId ) ) ):
            mode.addWarning( "Transport IP is not derived from a loopback "
                  "interface" )
         
         if config.fecFilterPrefixListName != '':
            if protoConfig.fecFilter.prefixList is None:
               mode.addWarning( "FEC filter prefix list does not exist: "
                                "no FECs are being filtered" )
            elif not protoConfig.fecFilter.prefixList.prefixEntry:
               mode.addWarning( "FEC filter prefix list is empty: "
                                "all FECs are being filtered" )

         localGrState = None
         if protoConfig.grOperMode == 'grModeSpeaker':
            localGrState = status.localGrStatus.operGrState

         MplsLdpInfo = MplsLdpStatus( 
                              enabled=True,
                              operational=True,
                              localLdpId=localLdpId,
                              localLdpIdIntf=protoConfig.intfId,
                              transportIp=protoConfig.ipTransportAddr,
                              transportIpIntf=config.ipTransportAddrIntfId,
                              labelAdvDspl=config.ldpLabelAdvDspl,
                              labelDistCtrl=config.ldpLabelDistCtrl,
                              labelRetention=config.ldpLabelRetention,
                              loopDetectionEnabled=loopDetectionEnabled,
                              linkCount=len( linkConfigColl.adjLinkConfig ),
                              adjCount=len( adjConfigColl.adjConfig ),
                              peerCount=len( peerStateColl.peerState ),
                              sessionCount=len( sessionConfigColl.sessionConfig ),
                              fecCount=statistics.fecCountUnfiltered,
                              fecFilteredCount=statistics.fecCountFiltered,
                              fecFilterPrefixListName=config.fecFilterPrefixListName,
                              grMode=protoConfig.grOperMode,
                              localGrState=localGrState,
                              labelLocalTerminationMode=
                                                 config.ldpLabelLocalTerminationMode,
                              entropyLabel=config.entropyLabel,
                              parameters=parameters)

         vrfs.update( { vrfName : MplsLdpInfo } )
      return MplsLdpStatusModel( vrfs=vrfs )

BasicCli.addShowCommandClass( MplsLdpCmd )

#--------------------------------------------------------------------------------
# show mpls ldp discovery [ detail | summary ]
#--------------------------------------------------------------------------------
class MplsLdpDiscoveryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls ldp discovery [ detail | summary ]'
   data = {
      'mpls' : mplsNodeForShow,
      'ldp' : LdpCli.ldpKw,
      'discovery' : 'Discovered LDP adjacencies',
      'detail' : detailKw,
      'summary' : summaryKw,
   }
   cliModel = MplsLdpDiscoveryModel

   @staticmethod
   def handler( mode, args ):
      LdpCli.showLdpConfigWarnings( mode )

      summary = 'summary' in args
      detail = 'detail' in args
      vrfs = {}
      vrfName = DEFAULT_VRF
      status = ldpStatusColl.status.get( vrfName )
      protoConfig = ldpProtoConfigColl.protoConfig.get( vrfName )
      if status is None or protoConfig is None:
         return MplsLdpDiscoveryModel( vrfs=vrfs )

      adjConfigColl = status.ldpAdjConfigColl
      adjStatusColl = status.ldpAdjStatusColl
      linkConfigColl = status.ldpLinkConfigColl
      linkStatusColl = status.ldpLinkStatusColl
      targetConfigColl = status.ldpTargetConfigColl
      targetStatusColl = status.ldpTargetStatusColl

      localLdpIdentifier = protoConfig.getRawAttribute("ldpIdentifier")
      localLdpId = MplsLdpId( ldpRouterId=IpAddrTac( localLdpIdentifier.lsrId ),
                              ldpLabelSpace=localLdpIdentifier.lblSpace )

      # These dicts will contain a python list of LDP ids for each interface (di)
      # and each target (dt).
      # To be used to create MplsLdpDiscoverySource objects for each of them.
      # (See discoveryInterfaces and discoveryTargets below).
      di = dict()
      dt = dict()
      for i in natsort.natsorted( adjConfigColl.adjConfig ):
         adjConfig = adjConfigColl.adjConfig.get( i )
         adjStatus = adjStatusColl.adjStatus.get( i )
         if adjConfig is None or adjStatus is None:
            continue
        
         ldpAdjId = adjConfig.getRawAttribute( "ldpAdjId" )
         adjLdpIdentifier = ldpAdjId.getRawAttribute( "ldpIdentifier" )
         adjIntfId = ldpAdjId.getRawAttribute( "intfId" )
         adjTargetIpAddr = ldpAdjId.getRawAttribute( "targetIpAddr" )
         # Add adjInfId if we haven't seen any adjacency on this interface before
         if ldpAdjId.adjType() == LdpAdjType.LdpLinkAdj and adjIntfId not in di:
            di[ adjIntfId ] = list()
         if ( ldpAdjId.adjType() == LdpAdjType.LdpTargetAdj and
              adjTargetIpAddr not in dt ):
            dt[ adjTargetIpAddr ] = list() 
         adjLdpLsrId = IpAddrTac( adjLdpIdentifier.lsrId )

         adjLdpHello = None
         configurationSource = None
         if detail or summary:
            # MplsLdpAdjacenyDetail model requires UTC time
            nowToUtc = Tac.utcNow() - Tac.now()
            adjLdpHello = MplsLdpAdjacencyDetail(
                  adjSrcIp=adjStatus.peerSrcIpAddr,
                  adjTransportIp=adjStatus.peerIpAddr,
                  localHoldTime=adjConfig.holdTime,
                  peerHoldTime=adjStatus.peerHoldTime,
                  holdTime=adjStatus.operHoldTime,
                  adjExpireTime=( adjStatus.adjExpireTime + nowToUtc ) )
            if ( ldpAdjId.adjType() == LdpAdjType.LdpTargetAdj and
                 targetConfigColl.adjTargetConfig.get( adjTargetIpAddr ) ):
               targetConfig = targetConfigColl.adjTargetConfig.get( adjTargetIpAddr )
               configurationSource = targetConfig.targetMembership.keys()

         adjLdpId = MplsLdpAdjacency( ldpRouterId=adjLdpLsrId,
                                      ldpLabelSpace=adjLdpIdentifier.lblSpace,
                                      helloDetail=adjLdpHello,
                                      configurationSource=configurationSource )

         if ldpAdjId.adjType() == LdpAdjType.LdpLinkAdj:
            di[ adjIntfId ].append( adjLdpId )
         else:
            dt[ adjTargetIpAddr ].append( adjLdpId ) 

      discoveryInterfaces = dict()
      for ( intfId, adjacencies ) in di.items():
         intfHelloInterval = None
         intfSrcIp = None
         if detail or summary:
            adjLinkConfig = linkConfigColl.adjLinkConfig.get( intfId )
            adjLinkStatus = linkStatusColl.adjLinkStatus.get( intfId )
            if adjLinkConfig is None or adjLinkStatus is None:
               continue
            intfHelloInterval = adjLinkStatus.interval
            intfSrcIp = IpGenAddr( adjLinkConfig.ipAddr )
         discoveryIntf = MplsLdpDiscoverySource( adjacencies=adjacencies,
                                                 helloInterval=intfHelloInterval,
                                                 srcIp=intfSrcIp)
         discoveryInterfaces[ intfId ] = discoveryIntf
      discoveryTargets = dict()
      for ( targetIpAddr, adjacencies ) in dt.items():
         targetHelloInterval = None
         targetTransportIp = None
         if detail or summary:
            adjTargetStatus = targetStatusColl.adjTargetStatus.get( targetIpAddr )
            if adjTargetStatus is None:
               continue
            targetHelloInterval = adjTargetStatus.interval
            targetTransportIp = \
                  targetConfigColl.ldpTargetCommonConfig.ipTransportAddr
         discoveryTarget = MplsLdpDiscoverySource( adjacencies=adjacencies,
                                                   helloInterval=targetHelloInterval,
                                                   srcIp=targetTransportIp )
         discoveryTargets[ str( targetIpAddr ) ] = discoveryTarget

      passwordSet = protoConfig.md5Password != protoConfig.md5PasswordDefault 
      passwordInfo = PasswordInformation ( passwordSet=passwordSet ) 
      vrfDiscovery = MplsLdpDiscovery( localLdpId=localLdpId,
                                       discoveryInterfaces=discoveryInterfaces,
                                       discoveryTargets=discoveryTargets,
                                       passwordInfo=passwordInfo )
      vrfs.update( { vrfName : vrfDiscovery} )

      return MplsLdpDiscoveryModel( vrfs=vrfs, _summary=summary )

BasicCli.addShowCommandClass( MplsLdpDiscoveryCmd )

#--------------------------------------------------------------------------------
# show mpls ldp neighbor ...
#--------------------------------------------------------------------------------
class MplsLdpNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show mpls ldp neighbor
               [ vrf VRF ] [ IPADDR | INTF ] [ state STATE ] [ detail | summary ]'''
   data = {
      'mpls' : mplsNodeForShow,
      'ldp' : LdpCli.ldpKw,
      'neighbor' : 'Status of LDP sessions',
      'vrf' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'vrf', helpdesc='VRF instance' ), hidden=True ),
      'VRF' : CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9]+', helpdesc='VRF name',
         helpname='WORD' ),
      'IPADDR' : IpAddrMatcher.IpAddrMatcher( helpdesc='IP address' ),
      'INTF' : IntfRangeMatcher( explicitIntfTypes=( EthPhyAutoIntfType, ) ),
      'state' : 'Filter neighbors according to session state',
      'STATE' : CliMatcher.EnumMatcher( {
            'operational' : 'Show neighbors in operational state',
            'non-operational' : 'Show neighbors in non-operational state',
         } ),
      'detail' : detailKw,
      'summary' : summaryKw,
   }
   cliModel = MplsLdpNeighborModel

   @staticmethod
   def handler( mode, args ):
      LdpCli.showLdpConfigWarnings( mode )

      requestedVrf = args.get( 'VRF' )
      
      requestedIntf = None
      requestedAddr = None
      if 'INTF' in args:
         # filter by interface
         requestedIntf = list( args[ 'INTF' ].intfNames() )
      elif 'IPADDR' in args:
         #filter by address
         requestedAddr = IpGenericAddr( ip=args[ 'IPADDR' ] )

      if requestedVrf == 'all':
         allVrfs = ldpStatusColl.status.keys()
      elif requestedVrf:
         allVrfs = [ requestedVrf ]
      else:
         allVrfs = [ DEFAULT_VRF ]

      vrfs = {}
      for vrfName in allVrfs:
         config = ldpConfigColl.config.get( vrfName )
         status = ldpStatusColl.status.get( vrfName )
         if not config or not status:
            continue
         vrfNeighbors = []

         peerStateColl = status.ldpPeerStateColl
         adjStatusColl = status.ldpAdjStatusColl
         sessionStatusColl = status.ldpSessionStatusColl
         sessionStatisticsColl = status.statistics.sessionStatisticsColl
         sessionConfigColl = status.ldpSessionConfigColl

         # Issue a warning if multiple neighbor have the same ips
         intfIpToSessionId = status.ldpIntfIpToSessionIdTable.intfIpToSessionIdTable
         for member in intfIpToSessionId.itervalues():
            if len( member.ldpSessionId ) > 1:
               mode.addWarning( 'Duplicate interface IPs' )

         for ldpIdentifier, peerState in natsort.natsorted(
               peerStateColl.peerState.iteritems() ):
            discoveryInterfaces = []
            discoveryTargets = []
            boundAddresses = []
            boundAddressesDup = []

            ldpSessionId = LdpSessionId( ldpIdentifier )

            sessionStatus = sessionStatusColl.sessionStatus.get( ldpSessionId )
            if sessionStatus is None:
               continue
            sessionStatistics = sessionStatisticsColl.sessionStatistics.get(
                  ldpSessionId )
            if sessionStatistics is None:
               continue

            sessionConfig = sessionConfigColl.sessionConfig.get( ldpSessionId )
            if sessionConfig is None:
               continue

            peerLdpId = LdpIdentifier( stringValue = ldpIdentifier )
            peerIdent = MplsLdpId( ldpRouterId=IpAddrTac( peerLdpId.lsrId ),
                                   ldpLabelSpace=peerLdpId.lblSpace )
            localLdpId = LdpIdentifier(
                  stringValue=sessionStatus.localLdpIdentifier )
            localIdent = MplsLdpId( ldpRouterId=IpAddrTac( localLdpId.lsrId ),
                                    ldpLabelSpace=localLdpId.lblSpace )

            if sessionStatus.peerIpAddr.isAddrZero:
               # Reusing this attribute to show the peer IP in non-op sessions
               tcpPeerIp = IpGenericAddrAndPort( ip=peerState.peerIpAddr,
                                                 port=0 )
            else:
               tcpPeerIp = IpGenericAddrAndPort( ip=sessionStatus.peerIpAddr,
                                              port=sessionStatus.peerPort )

            peerAddress = IpGenericAddr( ip=tcpPeerIp.ip )
            
            if sessionStatus.localIpAddr.isAddrZero:
               tcpLocalIp = None
            else:
               tcpLocalIp = IpGenericAddrAndPort( ip=sessionStatus.localIpAddr,
                                              port=sessionStatus.localPort )

            state = sessionStatus.sessionState
            if sessionStatus.sessionStateReason == "":
               stateCode = None
               stateCodeDescription = None
            else:
               stateCode = sessionStatus.code
               stateCodeDescription = sessionStatus.sessionStateReason

            stream = sessionStatus.operLabelAdvDspl
            sessionUpTimestamp = sessionStatus.sessionUptime
            localKeepAliveHoldTime = sessionConfig.keepAliveTime
            localKeepAliveInterval = sessionConfig.keepAliveTxInterval
            peerKeepAliveHoldTime = sessionStatus.peerKeepAliveTime
            peerKeepAliveInterval = sessionStatus.peerKeepAliveTxInterval
            sessKeepAliveHoldTime = sessionStatus.operKeepAliveTime
            sessKeepAliveInterval = sessionStatus.operKeepAliveTxInterval

            # sessionExpireTime in the CAPI model is in UTC time while
            # TACC model is now time.  Conversion is required.
            nowToUtc = Tac.utcNow() - Tac.now()
            if sessionStatus.sessionState == 'stateOperational':
               sessionExpireTime = sessionStatus.sessionExpireTime + nowToUtc
            else:
               sessionExpireTime = 0.0

            msgTx = sessionStatistics.ldpMsgTxCountTotal
            msgRx = sessionStatistics.ldpMsgRxCountTotal

            for intfIp in natsort.natsorted( sessionStatus.peerIntfIp ):
               boundAddresses.append( IpGenericAddr( ip=intfIp ) )
               intfIpSessionMember = intfIpToSessionId.get( intfIp )
               if ( intfIpSessionMember and
                    len( intfIpSessionMember.ldpSessionId ) > 1 ):
                  boundAddressesDup.append( IpGenericAddr( ip=intfIp ) )

            for ldpAdjId in natsort.natsorted( peerState.ldpAdj ):
               if ldpAdjId not in adjStatusColl.adjStatus:
                  continue
               adjId = LdpAdjId( stringValue = ldpAdjId )
               if adjId.intfId != "":
                  discoveryInterfaces.append( adjId.intfId )
               else:
                  discoveryTargets.append( adjId.targetIpAddr )
            
            if args.get( 'STATE' ) == 'operational' and state != 'stateOperational':
               continue
            if ( args.get( 'STATE' ) == 'non-operational' and
                 state == 'stateOperational' ):
               continue

            if requestedIntf and not set(requestedIntf).\
                                         intersection( discoveryInterfaces ):
               continue
            
            if ( requestedAddr and 
                 requestedAddr not in boundAddresses and 
                 requestedAddr !=  peerAddress ):
               continue

            if sessionStatus.sessionStateLogEntryColl is None:
               continue

            logEntries = None
            if "detail" in args:
               # Create a list from sessionStateLogEntryColl
               logEntries = []
               logQueue = sessionStatus.sessionStateLogEntryColl.sessionStateLogEntry
               for s in logQueue.values():
                  logEntries.append( LdpSessionStateLogEntryModel(
                                        timeStamp=s.timeStamp,
                                        sessionState=s.sessionState,
                                        code=s.code,
                                        reason=s.reason ) )

            rxMsgCount = None
            txMsgCount = None
            rxNotifCount = None
            txNotifCount = None
            capabilities = sessionStatus.peerCapability.keys()

            endOfLib = 'notSupported'
            if ( config.endOfLib and
                 'typedWildcardFecCapabilityTlv' in capabilities and 
                 'unrecognizedNotificationCapabilityTlv' in capabilities ):
               if ( 'eolPrefixIpv4' in sessionStatus.localEndOfLib and
                    'eolPrefixIpv4' in sessionStatus.peerEndOfLib ):
                  endOfLib = 'sentAndReceived'
               elif 'eolPrefixIpv4' in sessionStatus.localEndOfLib:
                  endOfLib = 'sent'
               elif 'eolPrefixIpv4' in sessionStatus.peerEndOfLib:
                  endOfLib = 'received'
               else:
                  endOfLib = 'supported'

            grState = getNeighborGrStateValue(
                  sessionStatus.operGrSupported, sessionStatus.operGrState )
            peerGrReconnectTime = sessionStatus.peerGrReconnectTimeout
            peerGrRecoveryTime = sessionStatus.peerGrRecoveryTimeout

            if "detail" in args:
               # Just some shorthand functions to decrease clutter
               # pylint: disable=cell-var-from-loop
               calcCountRx = lambda s: sessionStatistics.ldpMsgRxCount.get( s, 0 )
               calcCountTx = lambda s: sessionStatistics.ldpMsgTxCount.get( s, 0 )

               # We do not count Hello packets because they are UDP packets, not
               # part of this TCP session that we are interested in.
               rxMsgCount = LdpMessageCount(
                  ldpNotification=calcCountRx( 'LdpNotification' ),
                  ldpInit=calcCountRx( 'LdpInit' ),
                  ldpKeepAlive=calcCountRx( 'LdpKeepAlive' ),
                  ldpAddress=calcCountRx( 'LdpAddress' ),
                  ldpAddressWithdraw=calcCountRx( 'LdpAddressWithdraw' ),
                  ldpLabelMapping=calcCountRx( 'LdpLabelMapping' ),
                  ldpLabelRequest=calcCountRx( 'LdpLabelRequest' ),
                  ldpLabelWithdraw=calcCountRx( 'LdpLabelWithdraw' ),
                  ldpLabelRelease=calcCountRx( 'LdpLabelRelease' ),
                  ldpLabelAbortReq=calcCountRx( 'LdpLabelAbortReq' ) )

               txMsgCount = LdpMessageCount(
                  ldpNotification=calcCountTx( 'LdpNotification' ),
                  ldpInit=calcCountTx( 'LdpInit' ),
                  ldpKeepAlive=calcCountTx( 'LdpKeepAlive' ),
                  ldpAddress=calcCountTx( 'LdpAddress' ),
                  ldpAddressWithdraw=calcCountTx( 'LdpAddressWithdraw' ),
                  ldpLabelMapping=calcCountTx( 'LdpLabelMapping' ),
                  ldpLabelRequest=calcCountTx( 'LdpLabelRequest' ),
                  ldpLabelWithdraw=calcCountTx( 'LdpLabelWithdraw' ),
                  ldpLabelRelease=calcCountTx( 'LdpLabelRelease' ),
                  ldpLabelAbortReq=calcCountTx( 'LdpLabelAbortReq' ) )

               rxNotifCount = LdpNotificationCount()
               txNotifCount = LdpNotificationCount()
               for k, v in sessionStatistics.ldpMsgNotificationRxCount.iteritems():
                  rxNotifCount.attributeIs( k, v )
               for k, v in sessionStatistics.ldpMsgNotificationTxCount.iteritems():
                  txNotifCount.attributeIs( k, v )
               # Sync 'ipChange' legacy model attribute
               rxNotifCount.attributeIs( 'ipChange',
                     rxNotifCount[ 'ipConnectivityChange' ] )
               txNotifCount.attributeIs( 'ipChange',
                     txNotifCount[ 'ipConnectivityChange' ] )

            neighbor = MplsLdpNeighbor(
                  peerIdent=peerIdent, localIdent=localIdent, tcpLocalIp=tcpLocalIp,
                  tcpPeerIp=tcpPeerIp, state=state, stateCode=stateCode,
                  stateCodeDescription=stateCodeDescription,
                  rxMsgCounters=rxMsgCount, txMsgCounters=txMsgCount,
                  rxNotifCounters=rxNotifCount, txNotifCounters=txNotifCount,
                  sessionStateLog=logEntries, msgTx=msgTx, msgRx=msgRx,
                  stream=stream, sessionUpTimestamp=sessionUpTimestamp,
                  localKeepAliveHoldTime=localKeepAliveHoldTime,
                  localKeepAliveInterval=localKeepAliveInterval,
                  peerKeepAliveHoldTime=peerKeepAliveHoldTime,
                  peerKeepAliveInterval=peerKeepAliveInterval,
                  sessionKeepAliveHoldTime=sessKeepAliveHoldTime,
                  sessionKeepAliveInterval=sessKeepAliveInterval,
                  sessionExpireTime=sessionExpireTime,
                  discoveryInterfaces=discoveryInterfaces,
                  discoveryTargets=discoveryTargets, boundAddresses=boundAddresses,
                  boundAddressesDup=boundAddressesDup,
                  capabilities=capabilities, endOfLib=endOfLib, grState=grState,
                  peerGrReconnectTime=peerGrReconnectTime,
                  peerGrRecoveryTime=peerGrRecoveryTime )
            vrfNeighbors.append( neighbor )

         if vrfNeighbors:
            vrfNeighborsList = MplsLdpNeighborList( neighbors=vrfNeighbors )
            vrfs.update( { vrfName : vrfNeighborsList } )

      if 'detail' in args:
         detailLevel = 'detail'
      elif 'summary' in args:
         detailLevel = 'summary'
      else:
         detailLevel = ''
      return MplsLdpNeighborModel( vrfs=vrfs, _detailLevel=detailLevel )

BasicCli.addShowCommandClass( MplsLdpNeighborCmd )

#--------------------------------------------------------------------------------
# show mpls ldp tunnel [ TUNNEL_INDEX ]
#--------------------------------------------------------------------------------
class MplsLdpTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls ldp tunnel [ TUNNEL_INDEX ]'
   data = {
      'mpls': mplsNodeForShow,
      'ldp': LdpCli.ldpKw,
      'tunnel': tokenTunnelMatcher,
      'TUNNEL_INDEX': tunnelIndexMatcher
   }
   cliModel = LdpTunnelTable

   @staticmethod
   def handler( mode, args ):
      tunnelIndex = args.get( 'TUNNEL_INDEX' )
      LdpCli.showLdpConfigWarnings( mode )
      return showLdpTunnelTable( tunnelIndex=tunnelIndex )

def getViaModel( nexthop, intfId, labels ):
   if isDyTunIntfId( intfId ):
      tid = getDyTunTidFromIntfId( intfId )
      return TunnelModels.MplsTunnelVia( tunnelId=getTunnelIdModel( tid ),
                                         type='tunnel', labels=labels )
   else:
      return TunnelModels.MplsVia( nexthop=nexthop, interface=intfId,
                                   type='ip', labels=labels )

def getLdpTunnelTableEntryModel( tunnelId ):
   vias = []
   ldpTunnelTableEntry = ldpTunnelTable.entry.get( tunnelId )

   if ldpTunnelTableEntry:
      for via in ldpTunnelTableEntry.via.itervalues():
         viaModel = None

         labels = []
         for mplsStackIndex in reversed( range( via.labels.stackSize ) ):
            labels.append( str( via.labels.labelStack( mplsStackIndex ) ) )

         if FecIdIntfId.isFecIdIntfId( via.intfId ):
            fec = getFecFromIntfId( via.intfId )
            if fec is None:
               continue

            for fecVia in sorted( fec.via.values() ):
               addr = Tac.Value( "Arnet::IpGenAddr", str( fecVia.hop ) )
               intfId = fecVia.intfId
               viaModel = getViaModel( addr, intfId, labels )
         else:
            viaModel = getViaModel( via.nexthop, via.intfId, labels )

         if viaModel:
            vias.append( viaModel )

      tep = ldpTunnelTableEntry.tep
      # Make via rendering order deterministic
      vias.sort( cmp=TunnelModels.viaCmp )

      return LdpModel.LdpTunnelTableEntry( endpoint=tep, vias=vias )

   return None

def showLdpTunnelTable( tunnelIndex=None ):
   ldpEntries = {}

   if tunnelIndex is None:
      for tunnelId in ldpTunnelTable.entry:
         ldpEntryModel = getLdpTunnelTableEntryModel( tunnelId=tunnelId )
         if ldpEntryModel:
            ldpEntries[ getTunnelIndexFromId( tunnelId ) ] = ldpEntryModel
   else:
      tunnel, tunnelId = getTunnelEntryAndIdFromIndexAndAf( tunnelIndex,
                            ldpTunnelTable, LdpTunnelTable )
      if tunnel:
         ldpEntryModel = getLdpTunnelTableEntryModel( tunnelId=tunnelId )
         if ldpEntryModel:
            ldpEntries[ getTunnelIndexFromId( tunnelId ) ] = ldpEntryModel

   return LdpModel.LdpTunnelTable( entries=ldpEntries )

BasicCli.addShowCommandClass( MplsLdpTunnelCmd )

def Plugin( entityManager ):
   global aclCheckpoint
   global aclCpConfig
   global aclStatus
   global commonLibSmashLdpBindingTable
   global commonLibSmashLdpP2mpBindingTable
   global ldpBindingTable
   global ldpConfigColl
   global ldpP2mpBindingTable
   global ldpProtoConfigColl
   global ldpStatusColl
   global ldpTunnelTable

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )

   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   commonLibSmashLdpBindingTable = shmemEm.doMount( "mpls/labelBindingTables/ldp",
                                      "CommonLibSmash::LabelBindingTable",
                                      Smash.mountInfo( 'keyshadow' ) )
   commonLibSmashLdpP2mpBindingTable = \
         shmemEm.doMount( "mpls/labelBindingTables/mldp",
                          "CommonLibSmash::LabelBindingTable",
                          Smash.mountInfo( 'keyshadow' ) )
   ldpBindingTable = shmemEm.doMount( "mpls/ldp/ldpBindingTables",
                                      "Ldp::LdpSmash::LdpLabelBindingTable",
                                      Smash.mountInfo( 'keyshadow' ) )
   ldpConfigColl = ConfigMount.mount( entityManager, "mpls/ldp/ldpConfigColl",
                                      "Ldp::LdpConfigColl", "w" )
   ldpP2mpBindingTable = shmemEm.doMount( "mpls/ldp/ldpP2mpBindingTables",
                                          "Ldp::LdpSmash::LdpLabelBindingTable",
                                          Smash.mountInfo( 'keyshadow' ) )
   ldpProtoConfigColl = LazyMount.mount( entityManager, 
                                         "mpls/ldp/ldpProtoConfigColl",
                                         "Ldp::LdpProtoConfigColl", "r" )
   ldpStatusColl = LazyMount.mount( entityManager, "mpls/ldp/ldpStatusColl",
                                    "Ldp::LdpStatusColl", "r" )
   ldpTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.ldpTunnelTable, entityManager )
