#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import sys

import BasicCli
import CliMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.IraIpModel import VrfLdpIpRoutes
from CliPlugin.IraIpRouteCliLib import ip
import CliPlugin.IraRouteCommon as IraRouteCommon
from CliPlugin.VrfCli import DEFAULT_VRF
from CliPrint import CliPrint
from CliToken.Ipv4 import ipv4MatcherForShow
import ShowCommand
import TacSigint

printer = CliPrint().lib

ldpRibRouting = IraRouteCommon.LdpRibRouting( ip )

matcherRib = CliMatcher.KeywordMatcher( 'rib',
      helpdesc='Routing Information Base commands' )
matcherRoute = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Routing Information Base routes' )
ipPrefixMatcher = IpAddrMatcher.ipPrefixExpr(
   'Match this IP address', 'Match this subnet mask',
   'Match this IP prefix', partial=True,
   overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )

#--------------------------------------------------------------------------------
# show rib system-tunneling-ldp-rib route ipv4 [ PREFIX ]
#--------------------------------------------------------------------------------
class LdpRibShortcutCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show rib system-tunneling-ldp-rib route ipv4 [ PREFIX ]"

   data = { 'rib': matcherRib,
            'system-tunneling-ldp-rib': "Route details for LDP over RSVP resolution",
            'route': matcherRoute,
            'ipv4': ipv4MatcherForShow,
            'PREFIX': ipPrefixMatcher
   }

   cliModel = VrfLdpIpRoutes

   @staticmethod
   def handler( mode, args ):
      prefix = args.pop( 'PREFIX', None )
      vrfName = DEFAULT_VRF
      outputFormat = printer.stringToOutputFormat( mode.session_.outputFormat_ )
      fd = sys.stdout.fileno()

      p = printer.initPrinter( fd, outputFormat, True )
      printer.start( p )
      printer.startDict( p, "vrfs" )

      # See comment in IraIpCli.showRoute for why its needed to check for SIGINT here
      TacSigint.check()

      printer.startDict( p, vrfName )

      fmt = mode.session_.outputFormat()

      ldpRibRouting.showRoute( prefix, None, None, False, vrfName, fmt=fmt )

      printer.endDict( p, vrfName )
      printer.endDict( p, "vrfs" )
      printer.end( p )
      printer.deinitPrinter( p )
      return VrfLdpIpRoutes

BasicCli.addShowCommandClass( LdpRibShortcutCmd )
