# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliModel import List, Dict, Enum, Int, Bool, Float, Str
from CliModel import Model, Submodel
from CliPlugin import TunnelCliLib
from IntfModel import Interface
from ArnetModel import IpGenericAddr, IpGenericAddrAndPort
from ArnetModel import Ip4Address, IpGenericAddress
from TableOutput import createTable, Format, TableFormatter
from TunnelModels import TunnelTableEntry, Via
from TunnelTypeLib import tunnelTypesReverseStrDict
import TunnelCli
from MplsModel import MplsBindingsModel, MplsPeerRouterId
from HumanReadable import formatTimeInterval
from IpLibConsts import DEFAULT_VRF
import IntfCli
import datetime
import prettytable
import Toggles.LdpToggleLib
from TypeFuture import TacLazyType

LdpSessionState = TacLazyType( "Ldp::SessionState" )
LdpLabelAdvDspl = TacLazyType( "Ldp::LdpLabelAdvDspl" )
LdpTargetMembership = TacLazyType( 'Ldp::LdpTargetMembership' )
targetMemberStatic = LdpTargetMembership.targetMemberStatic
targetMemberPw = LdpTargetMembership.targetMemberPw
targetMemberHr = LdpTargetMembership.targetMemberHr
LdpProtoParam = TacLazyType( "Ldp::LdpProtoParam" )
LdpStatusCode = TacLazyType( "Ldp::LdpStatusCode" )
LdpMsgType = TacLazyType( "Ldp::LdpMsgType" )
LdpTlvType = TacLazyType( "Ldp::LdpTlvType" )
LdpGrMode = TacLazyType( "Ldp::LdpGrOperationalMode" )
LdpGrState = TacLazyType( "Ldp::GrState" )
LdpLabelLocalTerminationMode = TacLazyType( "Ldp::LdpLabelLocalTerminationMode" )

def printt( val, indent=0 ):
   print ' ' * 3 * indent + val

def printt_prettytable( tbl, indent=0 ):
   table_string = tbl.get_string()
   lines = table_string.split( '\n' )
   for line in lines:
      # Strip leading whitespace for correct indentation (prettytable gives one
      # extra space in the front).
      # Strip trailing whitespace for easier debugging.
      printt( line[1:].rstrip(), indent=indent )

def renderBool( value ):
   if value:
      return 'Yes'
   return 'No'

def renderHoldTime( holdTime, includeSec=True ):
   if holdTime == LdpProtoParam.infiniteHelloHoldTime:
      # never include sec after infinite
      return 'infinite'

   renderedHoldTime = str( holdTime )
   if includeSec:
      renderedHoldTime = '%s sec' % renderedHoldTime
   return renderedHoldTime

def renderAddressTable( addrList, indent=0, duplicates=None ):
   # TODO Make it a set once IpGenericAddr from ArnetModel hash if fixed:
   # http://reviewboard/r/247635/
   duplicates = duplicates or []
   groupSize = 4
   indices = [ i for i in xrange( len( addrList ) ) ]

   table = TableFormatter()
   for i in indices[ ::groupSize ]:
      addrListSubSet = addrList[ i : i + groupSize ]
      ipStr = [ '{}{}'.format( ip.formatStr(),
                               ' (duplicate)' if ip in duplicates else '' )
                for ip in addrListSubSet ]
      table.newRow( *ipStr )

   f1 = Format( justify='left' )
   table.formatColumns( f1, f1, f1, f1 )

   for line in table.output().split('\n'):
      if line == '':
         continue
      printt( line.strip(), indent=indent )

neighborGrStateValues = LdpGrState.attributes + ( 'notSupported', )
def getNeighborGrStateValue( grSupported, grState ):
   if grSupported:
      return grState
   else:
      return 'notSupported'

def grStateToStr( grState ):
   d = {
      LdpGrState.grStateActive: 'active',
      LdpGrState.grStateInactive: 'inactive',
      LdpGrState.grStateUnknown: 'unknown',
   }
   return d.get( grState )

class MplsLdpId( Model ):
   ldpRouterId = Ip4Address( help='LDP Router ID' )
   ldpLabelSpace = Int( help='Local label space' )

   # pylint: disable=arguments-differ
   def render( self, indent=0 ):
      printt( self.renderStr(), indent=indent )

   def renderStr( self ):
      return 'LDP ID: %s' % self.renderStrPlain()

   def renderStrPlain( self ):
      # pylint:disable=E1101
      return '%s:%s' % ( self.ldpRouterId.stringValue, self.ldpLabelSpace )


class MplsLdpAdjacencyDetail( Model ):
   adjSrcIp = IpGenericAddress( help='Source IP of adjacency\'s hello packets' )
   adjTransportIp = IpGenericAddress( 
                                help='Transport IP of adjacency\'s hello packets' )
   holdTime = Int( help='Operational hold time in seconds' )
   localHoldTime = Int( help='Local proposed hold time in seconds' )
   peerHoldTime = Int( help='Peer proposed hold time in seconds' )
   adjExpireTime = Float( help='UTC time at which discovery entry will expire' )

   # pylint: disable=arguments-differ
   def render( self, indent=0 ):
      # pylint:disable=E1101
      printt( 'Source IP addr: %s; Transport IP addr: %s' % \
                 ( self.adjSrcIp.stringValue, self.adjTransportIp.stringValue ),
              indent=indent )
      remainingTime = self.adjExpireTime - Tac.utcNow()
      if self.holdTime == LdpProtoParam.infiniteHelloHoldTime:
         remainingTimeMsg = "Never expires"
      elif remainingTime <= 0:
         remainingTimeMsg = "Expired"
      else:
         remainingTimeMsg = "Expires in: %.2f sec" % remainingTime
      printt( 'Hold time: %s; Proposed local/peer: %s/%s; %s' % \
                 ( renderHoldTime( self.holdTime ),
                   renderHoldTime( self.localHoldTime, False ),
                   renderHoldTime( self.peerHoldTime ),
                   remainingTimeMsg ), 
              indent=indent )

class MplsLdpAdjacency( MplsLdpId ):
   helloDetail = Submodel( valueType=MplsLdpAdjacencyDetail, optional=True,
                                 help='Detailed hello packet information' )
   configurationSource = List( valueType=str, optional=True,
                              help='Configuration source of targeted adjacencies' )
   def render( self, indent=0 ):
      MplsLdpId.render( self, indent=indent )
      if self.helloDetail:
         self.helloDetail.render( indent=indent+1 )
      if self.configurationSource:
         stringForEachMembership = { targetMemberStatic : 'Static',
                                     targetMemberPw : 'Pseudowire',
                                     targetMemberHr : 'Hello Redundancy' }
         printt( 'Target configuration source: %s' \
                  % ', '.join( stringForEachMembership[ source ] \
                     for source in sorted( self.configurationSource ) ), 
                 indent=indent+1 )

   def renderSummaryRow( self ):
      if self.helloDetail:
         return ( self.renderStrPlain(),
                  renderHoldTime( self.helloDetail.holdTime ),
                  self.helloDetail.adjSrcIp,
                  self.helloDetail.adjTransportIp )

class MplsLdpDiscoverySource( Model ):
   adjacencies = List( valueType=MplsLdpAdjacency, help='List of adjacencies' )
   helloInterval = Int( optional=True, help='Hello interval in seconds' )
   srcIp = IpGenericAddress( optional=True, help='Hello source IP address' )

   # pylint: disable=arguments-differ
   def render( self, indent=0 ):
      if self.helloInterval and self.srcIp:
         printt( 'Hello interval: %s sec; Source IP addr: %s' % \
               ( self.helloInterval, self.srcIp), indent=indent )
      for adjacency in sorted( self.adjacencies,
                               key=lambda ldpId: ldpId.renderStrPlain() ):
         adjacency.render( indent=indent )

   def renderSummaryRows( self, summaryTable, discoverySource ):
      for adjacency in sorted( self.adjacencies,
                               key=lambda ldpId: ldpId.renderStrPlain() ):
         ( neighbor, holdTime, adjSrcIp, transportIp ) = adjacency.renderSummaryRow()
         summaryTable.newRow( discoverySource, neighbor, holdTime, adjSrcIp,
                              transportIp )

class PasswordInformation( Model ):
   passwordSet = Bool( help='MD5 Password enabled' )

   def render ( self ):
      if self.passwordSet:
         printt( 'LDP MD5 Password Set' )
      else:
         printt( 'LDP MD5 Password Not Set' )

class MplsLdpDiscovery( Model ):
   localLdpId = Submodel( valueType=MplsLdpId, help='Local LDP ID' )
   discoveryInterfaces = Dict( keyType=Interface, valueType=MplsLdpDiscoverySource,
                               help='Link adjacencies discovered on each interface' )
   discoveryTargets = Dict( keyType=str, valueType=MplsLdpDiscoverySource,
                            help='Targeted adjacencies discovered from each target' )
   passwordInfo = Submodel ( valueType=PasswordInformation, optional=True,
                             help='LDP MD5 password information' )

   def render( self ):
      if self.passwordInfo:
         self.passwordInfo.render()
      printt( 'Local LDP Identifier: %s' % ( self.localLdpId.renderStrPlain(), ) )
      printt( 'Discovery Sources:' )
      printt( 'Interfaces:', indent=1 )
      for ( intfId, discoveryIntf ) in sorted( self.discoveryInterfaces.items() ):
         printt( '%s (ldp):' % intfId, indent=2 )
         discoveryIntf.render( indent=3 )
      if len( self.discoveryTargets ) > 0:
         printt( 'Targeted Hellos:', indent=1 )
         for ( targetIpAddr, discoveryTarget ) in \
                                           sorted( self.discoveryTargets.items() ):
            printt( 'Targeted neighbor %s:' % targetIpAddr, indent=2 )
            discoveryTarget.render( indent=3 )

   def renderSummaryRows( self, intfSummaryTable, tgtSummaryTable ):
      if self.passwordInfo:
         self.passwordInfo.render()
      printt( 'Local LDP Identifier: %s' % ( self.localLdpId.renderStrPlain(), ) )
      for ( intfId, discoveryIntf ) in sorted( self.discoveryInterfaces.items() ):
         discoveryIntf.renderSummaryRows( intfSummaryTable,
                                          IntfCli.Intf.getShortname( intfId ) )
      for ( targetIpAddr, discoveryTarget ) in \
                                        sorted( self.discoveryTargets.items() ):
         discoveryTarget.renderSummaryRows( tgtSummaryTable, targetIpAddr )

class MplsLdpDiscoveryModel( Model ):
   vrfs = Dict( keyType=str, valueType=MplsLdpDiscovery,
                help='Discovery information in all vrfs keyed by vrf name' )
   _summary = Bool( help='Summary of LDP discovery' ) 
   
   def render( self ):
      if self._summary:
         self.renderSummary()
         return

      for ( vrfDiscoveryList ) in self.vrfs.itervalues():
         vrfDiscoveryList.render()

   def renderSummary( self ):
      fmt = Format( justify='left' )
      fmt.noPadLeftIs( True )
      fmt.padLimitIs( True )
      intfHeaders = [ 'Interface', 'Peer ID', 'Hold Time',
                      'Source IP', 'Transport IP' ]
      intfSummaryTable = createTable( intfHeaders, tableWidth=100 )
      intfSummaryTable.formatColumns( fmt, fmt, fmt, fmt, fmt )

      tgtHeaders = [ 'Target IP', 'Peer ID', 'Hold Time',
                     'Source IP', 'Transport IP' ]
      tgtSummaryTable = createTable( tgtHeaders, tableWidth=100 )
      tgtSummaryTable .formatColumns( fmt, fmt, fmt, fmt, fmt )
      
      for vrfDiscoveryList in self.vrfs.itervalues():
         vrfDiscoveryList.renderSummaryRows( intfSummaryTable, tgtSummaryTable )
      
      if len( intfSummaryTable.entries_ ) > 1:
         print intfSummaryTable.output()
      else:
         print "No link adjacencies"
      
      if len( tgtSummaryTable.entries_ ) > 1:
         print tgtSummaryTable.output()
      else:
         print "No targeted adjacencies"

class LdpSessionStateLogEntryModel( Model ):
   timeStamp = Float( help='Time of state change in seconds' )
   sessionState = Enum( values=LdpSessionState.attributes, help="Ldp session state" )
   code = Enum( values=LdpStatusCode.attributes,
                       help='In states non-exist or expired, the code used for '
                             'the last state transition' )
   reason = Str( help='In states non-exist or expired, the reason '
                      'describing the last state transition' )

class LdpMessageCount( Model ):
   # Note: we do not keep track of Hello packets here, because they are
   # transmitted through UDP and are therefore not part of the TCP session that
   # we are interested in.
   ldpNotification = Int( help='Number of LDP Notification messages', default=0 )
   ldpInit = Int( help='Number of LDP Init messages', default=0 )
   ldpKeepAlive = Int( help='Number of LDP KeepAlive messages', default=0 )
   ldpAddress = Int( help='Number of LDP Address messages', default=0 )
   ldpAddressWithdraw = \
                     Int( help='Number of LDP AddressWithdraw messages', default=0 )
   ldpLabelMapping = Int( help='Number of LDP LabelMapping messages', default=0 )
   ldpLabelRequest = Int( help='Number of LDP LabelRequest messages', default=0 )
   ldpLabelWithdraw = Int( help='Number of LDP LabelWithdraw messages', default=0 )
   ldpLabelRelease = Int( help='Number of LDP LabelRelease messages', default=0 )
   ldpLabelAbortReq = Int( help='Number of LDP LabelAbortReq messages', default=0 )

   # Makes it easier to write attributes into this class programmatically.
   def attributeIs( self, s, n ):
      if s == "Notification":
         self.ldpNotification = n
      elif s == "Initialization":
         self.ldpInit = n
      elif s == "Keep Alive":
         self.ldpKeepAlive = n
      elif s == "Address":
         self.ldpAddress = n
      elif s == "Address Withdraw":
         self.ldpAddressWithdraw = n
      elif s == "Label Mapping":
         self.ldpLabelMapping = n
      elif s == "Label Request":
         self.ldpLabelRequest = n
      elif s == "Label Withdraw":
         self.ldpLabelWithdraw = n
      elif s == "Label Release":
         self.ldpLabelRelease = n
      elif s == "Label Abort Request":
         self.ldpLabelAbortReq = n

   def orderedKeys( self ):
      """ Returns a list of strings that can be used as keys to the attributIs
      function and the dictionary returned by ldpConvertToDict function.
      
      These strings are ordered in the same way as the enum LdpMsgType is
      defined in LdpHdr.tac.
      """
      return [ 'Notification', 'Initialization', 'Keep Alive',
               'Address', 'Address Withdraw', 'Label Mapping',
               'Label Request', 'Label Withdraw', 'Label Release',
               'Label Abort Request' ]

   # prevent name collision (don't want to overload)
   def ldpConvertToDict( self ):
      """ Returns the data of this class in dictionary.
      e.g. { 'LdpNotification': 7, 'LdpHello': 0, ... }
      """
      ans = {}

      ans[ 'Notification' ] = self.ldpNotification
      ans[ 'Initialization' ] = self.ldpInit
      ans[ 'Keep Alive' ] = self.ldpKeepAlive
      ans[ 'Address' ] = self.ldpAddress
      ans[ 'Address Withdraw' ] = self.ldpAddressWithdraw
      ans[ 'Label Mapping' ] = self.ldpLabelMapping
      ans[ 'Label Request' ] = self.ldpLabelRequest
      ans[ 'Label Withdraw' ] = self.ldpLabelWithdraw
      ans[ 'Label Release' ] = self.ldpLabelRelease
      ans[ 'Label Abort Request' ] = self.ldpLabelAbortReq
      return ans

class LdpNotificationCount( Model ):
   success = Int( help="Number of notification messages with status code"
                  " 'Success'", default=0 )
   badLdpId = Int( help="Number of notification messages with status code"
                  " 'Bad LDP Identifier'", default=0 )
   badVersion = Int( help="Number of notification messages with status code"
                  " 'Bad Protocol Version'", default=0 )
   badPduLength = Int( help="Number of notification messages with status code"
                  " 'Bad PDU Length'", default=0 )
   unknownMessageType = Int( help="Number of notification messages with status code"
                  " 'Unknown Message Type'", default=0 )
   badMessageLength = Int( help="Number of notification messages with status code"
                  " 'Bad Message Length'", default=0 )
   unknownTlv = Int( help="Number of notification messages with status code"
                  " 'Unknown TLV'", default=0 )
   badTlvLength = Int( help="Number of notification messages with status code"
                  " 'Bad TLV Length'", default=0 )
   malformedTlv = Int( help="Number of notification messages with status code"
                  " 'Malformed TLV Value'", default=0 )
   holdTimerExpired = Int( help="Number of notification messages with status code"
                  " 'Hold Timer Expired'", default=0 )
   shutdown = Int( help="Number of notification messages with status code"
                  " 'Shutdown'", default=0 )
   loopDetected = Int( help="Number of notification messages with status code"
                  " 'Loop Detected'", default=0 )
   unknownFec = Int( help="Number of notification messages with status code"
                  " 'Unknown FEC'", default=0 )
   noRoute = Int( help="Number of notification messages with status code"
                  " 'No Route'", default=0 )
   noLabelResources = Int( help="Number of notification messages with status code"
                  " 'No Label Resources'", default=0 )
   labelResourcesAvailable = Int( help="Number of notification messages with status"
                  " code 'Label Resources / Available'", default=0 )
   rejectNoHello = Int( help="Number of notification messages with status code"
                  " 'Session Rejected / No Hello'", default=0 )
   rejectParamAdvMode = Int( help="Number of notification messages with status code"
                  " 'Session Rejected / Parameters Advertisement Mode'", default=0 )
   rejectMaxPduLength = Int( help="Number of notification messages with status code"
                  " 'Session Rejected / Parameters Max PDU Length'", default=0 )
   rejectParamLabelRange = Int( help="Number of notification messages with status"
                  " code 'Session Rejected / Parameters Label Range'", default=0 )
   keepAliveExpired = Int( help="Number of notification messages with status code"
                  " 'KeepAlive Timer Expired'", default=0 )
   labelRequestAborted = Int( help="Number of notification messages with status" 
                  " code 'Label Request Aborted'", default=0 )
   missingMessageParams = Int( help="Number of notification messages with status"
                  " code 'Missing Message Parameters'", default=0 )
   unsupportedAddressFamily = Int( help="Number of notification messages with"
                  " status code 'Unsupported Address Family'", default=0 )
   sessionRejectBadKeepAliveTime = Int( help="Number of notification messages with"
                  " status code 'Session Rejcted / Bad KeepAlive Time'", default=0 )
   internalError = Int( help="Number of notification messages with status code"
                  " 'Internal Error'", default=0 )
   illegalCBit = Int( help="Number of notification messages with status code"
                  " 'Illegal C Bit'", default=0 )
   wrongCBit = Int( help="Number of notification messages with status code"
                  " 'Wrong C Bit'", default=0 )
   pwStatus = Int( help="Number of notification messages with status code"
                  " 'PW Status'", default=0 )
   endOfLib = Int( help="Number of notification messages with status code"
                  " 'End-of-LIB'", default=0 )
   # Internal use only
   unknownMessageTypeIgnore = Int( help="Number of notification messages with "
                  "status code 'Ignored Unknown Message Type'", default=0 )
   establishingTcpConnection = Int( help="Number of notification messages with "
                  "status code 'Establishing Tcp Connection'", default=0 )
   # Internal, more explicit stateExpired codes
   sessionInitFailed = Int( help="Number of notification messages with status code"
                  " 'Session Init Failed'", default=0 )
   sessionRoleChanged = Int( help="Number of notification messages with status code"
                  " 'Session Role Changed'", default=0 )
   tcpConnectionClosed = Int( help="Number of notification messages with status code"
                  " 'Tcp Connection Closed'", default=0 )
   # ipChange attribute is only for legacy reasons. Same as ipConnectivityChange.
   ipChange = Int( help="Number of notification messages with status "
                  "code 'IP Connectivity Change'", default=0 )
   idChange = Int( help="Number of notification messages with status code"
                  " 'ID Change'", default=0 )
   mLdpChange = Int( help="Number of notification messages with status code"
                  " 'mLdp Change'", default=0 )
   grConfigChange = Int( help="Number of notification messages with"
                                    " status code 'Gr Destructive Config Change'",
                                    default=0 )
   ipConnectivityChange = Int( help="Number of notification messages with status "
                  "code 'IP Connectivity Change'", default=0 )
   eolConfigChange = Int( help="Number of notification messages with"
         " status code 'End-of-LIB Config Change'", default=0 )

   # Makes it easier to write attributes into this class programmatically.
   def attributeIs( self, s, n ):
      self.__setattr__( s, n )

   def orderedKeys( self ):
      return [ 'success', 'badLdpId', 'badVersion', 'badPduLength',
             'unknownMessageType', 'badMessageLength', 'unknownTlv',
             'badTlvLength', 'malformedTlv', 'holdTimerExpired', 'shutdown',
             'loopDetected', 'unknownFec', 'noRoute', 'noLabelResources',
             'labelResourcesAvailable', 'rejectNoHello', 'rejectParamAdvMode',
             'rejectMaxPduLength', 'rejectParamLabelRange', 'keepAliveExpired',
             'labelRequestAborted', 'missingMessageParams',
             'unsupportedAddressFamily', 'sessionRejectBadKeepAliveTime',
             'internalError', 'illegalCBit', 'wrongCBit', 'pwStatus', 'endOfLib',
             'unknownMessageTypeIgnore', 'establishingTcpConnection',
             'sessionInitFailed', 'sessionRoleChanged', 'tcpConnectionClosed',
             'ipConnectivityChange', 'idChange', 'mLdpChange', 'grConfigChange',
             'eolConfigChange' ]

   def ldpConvertToDict( self ):
      """ Returns the data of this class in dictionary.
      e.g. { 'success': 7, 'badLdpId': 0, ... }
      """
      ans = self.toDict()
      # ipChange attribute is only for legacy reasons. Same as ipConnectivityChange.
      del ans[ 'ipChange' ]
      return ans

class MplsLdpNeighbor( Model ):
   peerIdent = Submodel( valueType=MplsLdpId, help='Peer LDP ID' )
   localIdent = Submodel( valueType=MplsLdpId, help='Local LDP ID' )
   tcpLocalIp = Submodel( valueType=IpGenericAddrAndPort, 
                      help='Local IP address',
                      optional=True )
   tcpPeerIp = Submodel( valueType=IpGenericAddrAndPort,
                      help='Peer IP address',
                      optional=True )
   state = Enum( values=LdpSessionState.attributes, help="Ldp session state" )
   stateCode = Enum( values=LdpStatusCode.attributes,
                       help='In states non-exist or expired, the code used for '
                             'the last state transition',
                       optional=True )
   stateCodeDescription = Str( help='In states non-exist or expired, the reason '
                                     'describing the last state transition',
                               optional=True )
   rxMsgCounters = Submodel( valueType=LdpMessageCount,
                             help='Number of received messages, '
                                  'categorized by type',
                             optional=True )
   txMsgCounters = Submodel( valueType=LdpMessageCount,
                             help='Number of transmitted messages, '
                                  'categorized by type',
                             optional=True )
   rxNotifCounters = Submodel( valueType=LdpNotificationCount,
                               help='Number of received notifications, '
                               'categorized by their status code.',
                               optional=True )
   txNotifCounters = Submodel( valueType=LdpNotificationCount,
                               help='Number of received notifications, '
                               'categorized by their status code.',
                               optional=True )
   sessionStateLog = List( valueType=LdpSessionStateLogEntryModel,
                           help="The last few session states and their information",
                           optional=True )
   msgTx = Int( help='Messages sent' )
   msgRx = Int( help='Messages received' )
   stream = Enum( values=LdpLabelAdvDspl.attributes,
                  help="Ldp advertisement discipline" )
   sessionUpTimestamp = Float( help='UTC timestamp of when the session went up' )
   localKeepAliveHoldTime = \
      Int( help='Local proposed KeepAlive hold time in seconds' )
   localKeepAliveInterval = \
      Int( help='Local proposed KeepAlive transmit interval in seconds' )
   peerKeepAliveHoldTime = \
      Int( help='Peer proposed KeepAlive hold time in seconds' )
   peerKeepAliveInterval = \
      Int( help='Peer proposed KeepAlive transmit interval in seconds' )
   sessionKeepAliveHoldTime = \
      Int( help='Negotiated session KeepAlive hold time in seconds' )
   sessionKeepAliveInterval = \
      Int( help='Negotiated session KeepAlive transmit interval in seconds' )
   sessionExpireTime = \
              Float( help='Session expiry in UTC, if no further session'
                     ' messages received' )
   discoveryInterfaces = List( valueType=Interface,
                               help='LDP Discovery interface sources' )
   discoveryTargets = List( valueType=str, 
                               help='LDP Discovery target sources' )
   boundAddresses = List( valueType=IpGenericAddr,
                          help='Addresses bound to peer' )
   boundAddressesDup = List( valueType=IpGenericAddr,
                          help='Addresses bound to peer (duplicate)' )
   capabilities = List( valueType=str,
                        help='Capabilities advertised by the peer' )
   endOfLib = Enum( values=( 'notSupported', 'supported', 'sent', 'received',
                             'sentAndReceived', ),
                    help='End-of-LIB operational state' )
   grState = Enum( values=neighborGrStateValues,
                   help='Graceful restart operational state' )
   peerGrReconnectTime = Int( help="Peer graceful restart reconnect timeout "
                                   "in seconds" )
   peerGrRecoveryTime = Int( help="Peer graceful restart recovery timeout in "
                                  "seconds" )

   def render( self, detailed=False, indent=0 ): #pylint:disable=W0221
      """ Set detailed=True for detailed output.
      """
      printt( 'Peer LDP ID: %s; Local LDP ID: %s' % \
                 ( self.peerIdent.renderStrPlain(),
                   self.localIdent.renderStrPlain() ), 
              indent=indent )
      
      if self.tcpLocalIp is not None and self.tcpPeerIp is not None:
         printt( 'TCP Connection: %s - %s' % \
                    ( self.tcpPeerIp.formatStr(), self.tcpLocalIp.formatStr() ),
                 indent=indent+1 )
      else:
         if self.stateCodeDescription:
            printt( 'No TCP Connection (%s)' % self.stateCodeDescription,
                    indent=indent+1 )
         else:
            printt( 'No TCP Connection', indent=indent+1 )
         printt( 'Peer IP addr: %s' % self.tcpPeerIp.ip, indent=indent+1 )

      printt( 'State: %s; Msgs sent/rcvd: %d/%d; %s' % \
                 ( self.renderStateStr( self.state ), self.msgTx,
                   self.msgRx, self.renderStreamStr( self.stream ) ), 
              indent=indent+1 )
      if self.sessionUpTimestamp > 0:
         printt( 'Uptime: %s' % self.renderUptimeStr( self.sessionUpTimestamp ), 
                 indent=indent+1 )
         printt( 'Keepalive tx interval: %d sec; proposed local/peer: %d/%d sec' %
                 ( self.sessionKeepAliveInterval, self.localKeepAliveInterval,
                   self.peerKeepAliveInterval ), indent=indent + 1 )
         if self.sessionExpireTime > Tac.utcNow():
            expireMsg = 'expires in: %.2f sec' % (
               self.sessionExpireTime - Tac.utcNow() )
         else:
            expireMsg = 'expired session'
         printt( 'Keepalive hold time: %d sec; proposed local/peer: %d/%d sec; %s' %
                 ( self.sessionKeepAliveHoldTime, self.localKeepAliveHoldTime,
                   self.peerKeepAliveHoldTime, expireMsg ), indent=indent + 1 )
      if ( Toggles.LdpToggleLib.toggleEndOfLibEnabled() or
           self.endOfLib != 'notSupported' ):
         printt( 'End-of-LIB: %s' % self.renderEndOfLibStr(), indent=indent + 1 )
      printt( 'Graceful restart: %s' % self.renderGrStateStr(), indent=indent + 1 )
      printt( 'Graceful restart rx time: reconnect: %d sec; recovery: %d sec' %
              ( self.peerGrReconnectTime, self.peerGrRecoveryTime ),
              indent=indent + 1 )
      printt( 'LDP discovery sources:', indent=indent+1 )
      
      for intf in self.discoveryInterfaces:
         printt( intf.strip(), indent=indent+2 )
      for target in self.discoveryTargets:
         printt( target.strip() + " (targeted neighbor)", indent=indent+2 )

      printt( 'Addresses bound to peer:', indent=indent+1 )
      renderAddressTable( self.boundAddresses, indent=indent+2,
                          duplicates=self.boundAddressesDup )

      if detailed:
         self.renderAdditionalDetail( indent=indent+1 )

   def renderAdditionalDetail( self, indent=0 ):
      self.renderCapabilities( indent=indent )
      self.renderCounters( indent=indent )
      self.renderSessionStateLog( indent=indent )

   def renderCapabilities( self, indent=0 ):
      """ Render the capabilities advertised by this neighbor.
      """
      printt( 'Capabilities:', indent=indent )
      if LdpTlvType.dynamicCapabilityTlv in self.capabilities:
         printt( 'Accepts Capability messages', indent=indent+1 )
      if LdpTlvType.upstreamLabelAssignmentCapabilityTlv in self.capabilities:
         printt( 'Can assign upstream labels', indent=indent+1 )
      if LdpTlvType.p2mpCapabilityTlv in self.capabilities:
         printt( 'Supports P2MP labels', indent=indent+1 )
      if LdpTlvType.mp2mpCapabilityTlv in self.capabilities:
         printt( 'Supports MP2MP labels', indent=indent+1 )
      if LdpTlvType.mbbCapabilityTlv in self.capabilities:
         printt( 'Supports make-before-break', indent=indent+1 )
      if LdpTlvType.typedWildcardFecCapabilityTlv in self.capabilities:
         printt( 'Accepts typed wildcard FECs', indent=indent+1 )
      if LdpTlvType.multiTopologyCapabilityTlv in self.capabilities:
         printt( 'Supports multiple topologies', indent=indent+1 )
      if LdpTlvType.stateAdvertisementControlCapabilityTlv in self.capabilities:
         printt( 'Supports state advertisement control', indent=indent+1 )
      if LdpTlvType.mrtCapabilityTlv in self.capabilities:
         printt( 'Supports maximally redundant trees', indent=indent+1 )
      if LdpTlvType.targetedApplicationCapabilityTlv in self.capabilities:
         printt( 'Supports targeted LDP applications', indent=indent+1 )
      if LdpTlvType.unrecognizedNotificationCapabilityTlv in self.capabilities:
         printt( 'Supports unrecognized notification status code',
                 indent=indent + 1 )

   def renderCounters( self, indent=0 ):
      """ Render relevant counters for this LDP session.
      """
      printt( "Counters:", indent=indent )

      # Convert the info into something easy to process in Python
      rxMsgs = self.rxMsgCounters.ldpConvertToDict()
      txMsgs = self.txMsgCounters.ldpConvertToDict()
      assert set( rxMsgs.keys() ) == set( txMsgs.keys() )

      rxNotifs = self.rxNotifCounters.ldpConvertToDict()
      txNotifs = self.txNotifCounters.ldpConvertToDict()
      assert set( rxNotifs.keys() ) == set( txNotifs.keys() )

      tbl = prettytable.PrettyTable( [ 'Counter', 'Received', 'Transmitted' ] )
      tbl.border = False
      tbl.align = 'l'

      allCountsAreZero = True
      for label in self.rxMsgCounters.orderedKeys():
         rx = rxMsgs[ label ]
         tx = txMsgs[ label ]

         # If we haven't received or transmitted messages of this type, do not
         # display them.
         if rx == 0 and tx == 0:
            continue

         allCountsAreZero = False
         tbl.add_row( [ label + ' Msg', rx, tx ] )
         # Need to display sub-categories of the notification messages.
         if label == 'Notification':
            for key in self.rxNotifCounters.orderedKeys():
               rxn = rxNotifs[ key ]
               txn = txNotifs[ key ]

               if rxn == 0 and txn == 0:
                  continue

               tbl.add_row( [ "   Status code '{}'".format( key ), rxn, txn ] )

      if allCountsAreZero:
         printt("== ALL ZERO ==", indent=indent+1 )
      else:
         printt_prettytable( tbl, indent=indent+1 )

   def renderSessionStateLog( self, indent=0 ):
      """ Render a table to display the last few (up to 10) session state
      changes for this neighbour.
      """
      # Do not render anything if the table is null.
      if self.sessionStateLog is None:
         return

      numStates = len( self.sessionStateLog )

      # Print the heading 'Last session state(s)'
      plural = 's' if numStates > 1 else ''
      titleString = "Last session state{}:".format( plural )
      printt( titleString, indent=indent )

      if numStates == 0:
         printt( "== NO ENTRY ==", indent=indent+1 )
      else:
         A = prettytable.PrettyTable(
               [ 'Last Changed', 'Session State', 'Description' ] )
         A.border = False
         A.align = 'l'

         for s in self.sessionStateLog:
            last_changed = self.renderUptimeStr( s.timeStamp ) + ' ago'
            ss = s.sessionState.lstrip( 'state' )
            A.add_row( [ last_changed, ss, s.reason ] )

         printt_prettytable( A, indent=indent+1 )

   def renderSummary( self, summaryTable ):

      if self.sessionUpTimestamp > 0:
         uptime =  self.renderUptimeStrShort( self.sessionUpTimestamp )
      else:
         uptime = ''

      summaryTable.newRow(self.peerIdent.renderStrPlain(),
            self.renderStateStr( self.state ), uptime, self.tcpPeerIp.ip,
            self.renderGrStateStrShort(), self.msgRx, self.msgTx )

   def renderStateStr( self, state ):
      stateToStr = {
         'stateUnknown' : 'unknown',
         'stateNonExist' : 'non-exist',
         'stateInitialized' : 'initialized',
         'stateOpenSent' : 'open-sent',
         'stateOpenRecv' : 'open-recv',
         'stateOperational' : 'oper',
         'stateExpired' : 'expired',
      }
      if state in stateToStr:
         return stateToStr[ state ]
      else:
         return "error"

   def renderStreamStr( self, stream ):
      streamToStr = {
         'onDemand' : 'downstream on demand',
         'unSolicited' : 'downstream unsolicited',
         'unknownAdvDspl' : 'unknown',
      }

      if stream in streamToStr:
         return streamToStr[ stream ]
      else:
         return "error"

   def renderUptimeStr( self, uptime ):
      return str( datetime.timedelta( seconds=int( Tac.utcNow() - uptime ) ) )

   def renderUptimeStrShort( self, uptime ):
      return str( formatTimeInterval( Tac.utcNow() - uptime ) )

   def renderEndOfLibStr( self ):
      return { 'supported': 'supported',
               'sent': 'sent',
               'received': 'received',
               'sentAndReceived': 'sent/received',
             }.get( self.endOfLib, 'not supported' )

   def renderGrStateStr( self ):
      return grStateToStr( self.grState ) or 'not supported'

   def renderGrStateStrShort( self ):
      return grStateToStr( self.grState ) or 'no'

class MplsLdpNeighborTarget( Model ):
   interface = List( valueType=Interface,
                     help='Targeted interfaces for LDP neighbors information',
                     optional=True )
   address = Submodel( valueType=IpGenericAddr, optional=True,
                       help='Targeted IP address for LDP neighbors information' )

class MplsLdpNeighborList( Model ):
   neighbors = List( valueType=MplsLdpNeighbor,
                     help='Neighbor sessions in a specific vrf' )

class MplsLdpNeighborModel( Model ):
   vrfs = Dict( keyType=str, valueType=MplsLdpNeighborList,
                help='Neighbor sessions in requested vrfs keyed by vrf name' )
   _detailLevel = Enum( values=( '', 'summary', 'detail' ),
                        help='Detail level of LDP sessions' )

   def render( self ):
      if self._detailLevel == "summary":
         self.renderNeighborSummary()
      else:
         self.renderNeighbors()

   def renderNeighbors( self ):
      indent = 0
      for ( vrf, vrfNeighbors ) in self.vrfs.items():
         indent = 0 
         if len( self.vrfs ) > 1 or vrf != DEFAULT_VRF:
            printt( 'VRF %s:' % vrf )
            indent += 1
         for neighbor in vrfNeighbors.neighbors:
            neighbor.render( self._detailLevel, indent=indent )

   def renderNeighborSummary( self ):
      f1 = Format( justify='left' )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      headers = [ 'Peer ID', 'State', 'Uptime', 'Peer IP',
                  'GR', 'MsgRcvd', 'MsgSent' ]

      for ( vrf, vrfNeighbors ) in self.vrfs.items():
         if len( self.vrfs ) > 1 or vrf != DEFAULT_VRF:
            printt( 'VRF %s:' % vrf )

         if vrfNeighbors.neighbors:
            summaryTable = createTable( headers, tableWidth=100 )
            summaryTable.formatColumns( f1, f1, f1, f1, f1, f1 )
            localId = vrfNeighbors.neighbors[ 0 ].localIdent.renderStrPlain()
            printt( 'Ldp summary information for local Ldp ID %s' % localId )
            for neighbor in vrfNeighbors.neighbors:
               neighbor.renderSummary( summaryTable )

            print summaryTable.output()

      if len( self.vrfs ) == 0:
         summaryTable = createTable( headers, tableWidth=100 )
         summaryTable.formatColumns( f1, f1, f1, f1, f1, f1 )
         print summaryTable.output()

class MplsLdpDetail( Model ):
   protocolVersion = Int( help='Protcol version' )
   keepAlive = Int( help='Session keepalive hold time in seconds' )
   helloHoldTime = Int( help='Discovery hello hold time in seconds,' + \
                           ' 65535 is infinite as per RFC 5036' )
   helloInterval = Int( help='Discovery hello interval in seconds' )
   targetedHelloHoldTime = Int( help='Discovery targeted-hello hold time in ' + \
                                   'seconds, 65535 is infinite as per RFC 5036' )
   targetedHelloInterval = Int( help='Discovery targeted-hello interval in seconds' )
   initialBackoff = Float( help='LDP initial backoff in seconds' )
   maximumBackoff = Float( help='LDP maximum backoff in seconds' )
   advertisedIntfIps = List( valueType=IpGenericAddr,
                               help='Interface addresses advertised to peers' )
   defaultLdpIntfEnabled = Bool( help='Default LDP interface state is enabled' )
   grStateHoldTime = Int( help='Graceful restart state holding timeout in seconds' )
   grReconnectTime = Int( help='Graceful restart reconnect timeout in seconds' )
   grMaxNeighborLivenessTime = Int( help='Graceful restart maximum neighbor '
                                         'liveness timeout in seconds' )
   grMaxRecoveryTime = Int( help='Graceful restart max recovery timeout in seconds' )

   def render( self ):
      printt( 'Protocol version: %s' % self.protocolVersion )
      printt( 'Session keepalive hold time: %s sec' % self.keepAlive )
      printt( 'Discovery hello: hold time: %s; interval: %s sec' % \
                 ( renderHoldTime( self.helloHoldTime ), self.helloInterval ) )
      printt( 'Discovery targeted-hello: hold time: %s; interval: %s sec' % \
                 ( renderHoldTime( self.targetedHelloHoldTime ),
                   self.targetedHelloInterval ) )
      printt( 'LDP initial/maximum backoff: %s/%s sec' % \
                 ( self.initialBackoff, self.maximumBackoff ) )
      printt( 'Default interface state: %s' % (
         "enabled" if self.defaultLdpIntfEnabled else "disabled" ) )
      printt( 'Graceful restart state holding timeout: %d sec' %
               self.grStateHoldTime )
      printt( 'Graceful restart reconnect timeout: %d sec' %
               self.grReconnectTime )
      printt( 'Graceful restart neighbor liveness timeout: %d sec' %
               self.grMaxNeighborLivenessTime )
      printt( 'Graceful restart recovery timeout: %d sec' %
               self.grMaxRecoveryTime )
      printt( 'Addresses advertised to peers:' )
      renderAddressTable( self.advertisedIntfIps, indent=1 )

class MplsLdpStatus( Model ):
   enabled = Bool( help='LDP enabled/disabled' )
   operational = Bool( help='LDP operational state' )
   reason = Str( help='Reason for LDP not to be operational', optional=True )
   localLdpId = Submodel( valueType=MplsLdpId, 
                          help='Local LDP ID', optional=True )
   localLdpIdIntf = Interface( help='Interface used for local LDP ID', 
                               optional=True )
   transportIp = IpGenericAddress(help='Transport IP', optional=True )
   transportIpIntf = Interface( help='Interface used for Transport IP', 
                                optional=True )
   labelAdvDspl = Enum( values=( 'onDemand', 'unSolicited', 'unknownAdvDspl' ),
                        help='Label advertisement discipline', optional=True )
   labelDistCtrl = Enum( values=( 'orderedCtrl', ),
                         help='Label distribution control', optional=True )
   labelRetention = Enum( values=( 'conservativeRetention', 'liberalRetention' ),
                          help='Label retention mode', optional=True ) 
   loopDetectionEnabled = Bool( help='Loop detection status', optional=True )
   linkCount = Int( help='Number of links', optional=True )
   adjCount = Int( help='Number of adjacencies', optional=True )
   peerCount = Int( help='Number of peers', optional=True )
   sessionCount = Int( help='Number of sessions', optional=True )
   fecCount = Int( help='Number of unfiltered FECs', optional=True )
   fecFilteredCount = Int( help='Number of filtered FECs', optional=True )
   fecFilterPrefixListName = Str( help='FEC filter prefix list',
                                  optional=True )
   grMode = Enum( values=LdpGrMode.attributes, help='Graceful restart mode',
                  optional=True )
   localGrState = Enum( values=LdpGrState.attributes,
                        help='Graceful restart speaker operational state',
                        optional=True )
   labelLocalTerminationMode = Enum( values=LdpLabelLocalTerminationMode.attributes,
                                     help='Label local termination mode',
                                     optional=True )

   entropyLabel = Bool( help='Entropy label signaling', optional=True )

   parameters = Submodel( valueType=MplsLdpDetail, 
                          help='Protocol parameters', optional=True )

   def render( self ):
      # pylint:disable=E1101
      printt( 'LDP status: %s, %s' % ( 
                 'enabled' if self.enabled else 'disabled', 
                 'operational' if self.operational else 'operationally down' ) )
      if not self.enabled or not self.operational:
         printt( 'Reason: %s' % ( self.reason, ), indent=1 )
         return
      
      printt( 'Local LDP Identifier: %s' % ( self.localLdpId.renderStrPlain(), ) )
      if self.localLdpIdIntf.stringValue:
         printt( 'Interface: %s' % ( self.localLdpIdIntf.stringValue, ), indent=1 )
      else:
         printt( 'LDP Identifier set explicitly', indent=1 )

      printt( 'Transport IP addr: %s' % ( self.transportIp.stringValue, ) )
      if self.transportIpIntf.stringValue:
         printt( 'Interface: %s' % ( self.transportIpIntf.stringValue, ), indent=1 )
      else:
         printt( 'Transport IP addr derived from LDP Identifier', indent=1 )

      d = { 'unSolicited': 'unsolicited', 'onDemand': 'on-demand' }
      if self.labelAdvDspl in d:
         printt( 'Label advertisement discipline: %s' % ( d[ self.labelAdvDspl ], ) )
     
      d = { 'orderedCtrl': 'ordered' }
      if self.labelDistCtrl in d:
         printt( 'Label distribution control: %s' % ( d[ self.labelDistCtrl ], ) )

      d = { 'liberalRetention': 'liberal', 'conservativeRetention': 'conservative' }
      if self.labelRetention in d:
         printt( 'Label retention mode: %s' % ( d[ self.labelRetention ], ) )

      printt( 'LDP loop detection: %s' % ( 'enabled' if self.loopDetectionEnabled 
                                           else 'disabled', ) )
      printt( 'Number of links: %d' % (self.linkCount, ) )
      printt( 'Number of adjacencies: %d' % (self.adjCount, ) )
      printt( 'Number of peers: %d' % (self.peerCount, ) )
      printt( 'Number of sessions: %d' % (self.sessionCount, ) )

      # Generate the string to display number of FECs, depending on whether
      # there are any filtered FECs. When there are no filtered FECs, the
      # string in the parentheses is not displayed.
      if self.fecFilteredCount == 0:
         filteredFecString = ""
      else:
         filteredFecString = \
            " (not including %d filtered FECs)" % (self.fecFilteredCount, )
      printt( 'Number of FECs: %d%s' % (self.fecCount, filteredFecString, ) )

      if ( self.fecFilterPrefixListName ):
         printt( "FEC filter: prefix list '%s'" % ( self.fecFilterPrefixListName ) )
      else:
         printt( 'FEC filter: not configured' )
      # Keys correspond to LdpGrOperationalMode Enum
      d = {
         'grModeSpeaker': 'speaker',
         'grModeHelper': 'helper',
         'grModeDisabled': 'disabled',
      }
      if self.grMode in d:
         grStr = d[ self.grMode ]
         if self.grMode == 'grModeSpeaker':
            grStr += ' (%s)' % grStateToStr( self.localGrState )
         printt( 'Graceful restart: %s' % grStr )

      d = {
         LdpLabelLocalTerminationMode.explicitNull: 'explicit null',
         LdpLabelLocalTerminationMode.implicitNull: 'implicit null',
      }
      if self.labelLocalTerminationMode in d:
         printt( 'Label type for local termination: %s' %
                 ( d[ self.labelLocalTerminationMode ] ) )

      printt( 'Entropy label: %s' %
               ( 'enabled' if self.entropyLabel else 'disabled', ) )

      if self.parameters is not None:
         self.parameters.render()

class MplsLdpStatusModel( Model ):
   vrfs = Dict( keyType=str, valueType=MplsLdpStatus,
                help='MPLS LDP information for all vrfs keyed by vrf name' )

   def render( self ):
      for ( vrfMplsLdpInfo ) in self.vrfs.itervalues():
         vrfMplsLdpInfo.render()

class MplsLdpBindingAdvertisements( Model ):
   advertisedTo = List( valueType=MplsPeerRouterId,
                        help='Peers this FEC binding has been sent to' )

class LdpTunnelTableEntry( TunnelTableEntry ):
   vias = List( valueType=Via, help="List of nexthops" )

   def renderLdpTunnelTableEntry( self, table, tunnelIndex ):
      nhStr = intfStr = labelsStr = '-'
      if self.vias:
         firstVia = self.vias[ 0 ]
         nhStr, intfStr = TunnelCliLib.getNhAndIntfStrs( firstVia )
         labelsStr = '[ ' + ' '.join( firstVia.labels ) + ' ]'
      table.newRow( tunnelIndex, str( self.endpoint ), nhStr, intfStr, labelsStr )
      for via in self.vias[ 1 : ]:
         nhStr, intfStr = TunnelCliLib.getNhAndIntfStrs( via )
         labelsStr = '[ ' + ' '.join( via.labels ) + ' ]'
         table.newRow( '-', '-', nhStr, intfStr, labelsStr )

class LdpTunnelTable( Model ):
   entries = Dict( keyType=long, valueType=LdpTunnelTableEntry,
                   help="LDP tunnel table entries keyed by tunnel index" )

   def render( self ):
      headings = [ "Index", "Endpoint", "Nexthop/Tunnel Index",
                   "Interface", "Labels" ]
      fl = Format( justify='left' )
      table = createTable( headings )
      table.formatColumns( fl, fl, fl, fl, fl )
      # pylint: disable-msg=E1101
      for tunnelId, ldpTunnelTableEntry in sorted( self.entries.iteritems() ):
         ldpTunnelTableEntry.renderLdpTunnelTableEntry( table, tunnelId )

      print table.output()

   __revision__ = 3
   def degrade( self, dictRepr, revision ):
      for entry in dictRepr[ 'entries' ].itervalues():
         for via in entry[ 'vias' ]:
            if ( revision in [ 1, 2 ] and via[ 'type' ] == 'tunnel' ):
               tunnelType = tunnelTypesReverseStrDict[ via[ 'tunnelId' ][ 'type' ] ]
               tunnelIndex = via[ 'tunnelId' ][ 'index' ]
               tunnelId = TunnelCli.getTunnelIdFromIndex( tunnelType,
                                                          tunnelIndex )
               del via[ 'tunnelId' ]

               dynTidType = Tac.Type( 'Arnet::DynamicTunnelIntfId' )
               intfId = dynTidType.tunnelIdToIntfId( tunnelId )
               via[ 'nexthop' ] = '0.0.0.0'
               via[ 'interface' ] = intfId
               via[ 'type' ] = 'ip'

            # For both 'ip' & 'tunnel' type vias delete the 'type' field
            # in revision 1.
            if revision == 1:
               del via[ 'type' ]

      return dictRepr

   def getTunnelIdFromIndex( self, index, addrFamily=None ):
      return TunnelCli.getTunnelIdFromIndex( 'ldpTunnel', index )

class MplsLdpBindingsModel( MplsBindingsModel ):
   advertisementsSent = Dict( keyType=str,
                              valueType=MplsLdpBindingAdvertisements,
                              optional=True,
                              help='Advertisements sent for FEC keyed by FEC' )
