#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliPlugin.AclCli import (
      accessGroupKwMatcher,
      inKwMatcher,
      ipAclNameMatcher,
      ipKwForServiceAclMatcher,
)
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.LdpCli as LdpCli
import CliPlugin.MplsCli as MplsCli
import CliPlugin.RouteMapCli as RouteMapCli
import ConfigMount
import Intf
from IntfRangePlugin.EthIntf import EthPhyAutoIntfType
from IntfRangePlugin.LagIntf import LagAutoIntfType
from IntfRangePlugin.LoopbackIntf import LoopbackAutoIntfType
from IntfRangePlugin.VlanIntf import VlanAutoIntfType
import LazyMount
import ReversibleSecretCli
import Tac
import Toggles.LdpToggleLib
import Toggles.MplsToggleLib
from TypeFuture import TacLazyType

 # pylint: disable=ungrouped-imports
if Toggles.LdpToggleLib.toggleP2mpToggleEnabled():
   from CliMode.LdpMode import MldpMode

# pkgdeps: library Arnet
# pkgdeps: library Ldp
# pkgdeps: library MplsSysdbTypes

FwdEqvClass = TacLazyType( 'Mpls::FwdEqvClass' )
IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
LdpProtoParam = TacLazyType( 'Ldp::LdpProtoParam' )

matcherHello = CliMatcher.KeywordMatcher( 'hello',
      helpdesc='Ldp link hello parameters' )
matcherHoldTime = CliMatcher.KeywordMatcher( 'hold-time',
      helpdesc='Max time allowed between hello msgs' )
matcherInfinite = CliMatcher.KeywordMatcher( 'infinite',
      helpdesc='Infinite hold time' )
matcherTargetedHello = CliMatcher.KeywordMatcher( 'targeted-hello',
      helpdesc='Ldp targeted hello parameters' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Network interface' )
matcherFec = CliMatcher.KeywordMatcher( 'fec',
      helpdesc='LDP FEC configuration' )
matcherFilter = CliMatcher.KeywordMatcher( 'filter',
      helpdesc='Filter for FECs' )
matcherDiscovery = CliMatcher.KeywordMatcher( 'discovery',
      helpdesc='Discovered LDP adjacencies' )
matcherLdpNeighbor = CliMatcher.KeywordMatcher( 'neighbor',
      helpdesc='LDP Neighbor configuration' )

ldpConfigColl = None
mplsHwCapability = None

intfTypes = ( EthPhyAutoIntfType, LagAutoIntfType,
              LoopbackAutoIntfType, VlanAutoIntfType )

def entropyLabelSupportedGuard( mode, token ):
   if mplsHwCapability.mplsEntropyLabelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def clearLdpPassword ( mode ):
   mode.enableLdpPassword ( "", True )

def clearMldp( mode, no=None ): 
   if Toggles.LdpToggleLib.toggleP2mpToggleEnabled():
      mLdpMode = mode.childMode( MldpConfigMode )
      mLdpMode.shutdownMldpIs( no )
      mLdpMode.lspStaticP2mpFecBindingCollDelAll()

def setLinkReadyTimeout( mode, timeout ):
   mode.ldpLinkReadyTimeout( timeout )

def clearLinkReadyTimeout( mode ):
   mode.ldpLinkReadyTimeout( 0 )

def defaultLinkReadyTimeout( mode ):
   mode.ldpLinkReadyTimeout( -1 )

def setLinkReadyDelay( mode, delay ):
   mode.ldpLinkReadyDelay( delay )

def clearLinkReadyDelay( mode ):
   mode.ldpLinkReadyDelay( 0 )

def defaultLinkReadyDelay( mode ):
   mode.ldpLinkReadyDelay( -1 )

def shutdownLdp( mode, no=None ):
   mode.shutdownIs( no )
   
def entropyLabel( mode, args=None ):
   mode.entropyLabelIs( args )

def protoParam( mode, param=0 ):
   mode.protoParamIs( param )

def protoParamDefault( mode ):
   mode.protoParamIs( 0 )

def routerIdDefault( mode, force ):
   mode.routerIdIs( True, '0.0.0.0', '',
                    force if mode.session.isInteractive() else True )

def routerIdIpAddr( mode, ipAddr, force ):
   mode.routerIdIs( None, ipAddr, '', 
                    force if mode.session.isInteractive() else True )

def routerIdIntfId( mode, intfs, force ):
   intfId = list( intfs.intfNames() )[ 0 ]
   mode.routerIdIs( None, '0.0.0.0', intfId,
                    force if mode.session.isInteractive() else True )

def transportAddrDefault( mode ):
   mode.transportAddrIntfIdIs( True, '' )

def transportAddrIntfId( mode, intfs ):
   intfId = list( intfs.intfNames() )[ 0 ]
   mode.transportAddrIntfIdIs( None, intfId )

def advDsplModeIs( mode, advDsplMode ):
   mode.advDsplModeIs( advDsplMode )

def advDsplModeDefault( mode ):
   mode.advDsplModeIs( 'unsolicited' )

def retentionModeIs( mode, retentionMode ):
   mode.retentionModeIs( retentionMode )

def retentionModeDefault( mode ):
   mode.retentionModeIs( 'liberal' )

def ipPrefixLabelBinding( mode, prefix, binding ):
   mode.fecStaticBindingIs( prefix, binding )

def ipPrefixLabelBindingDefault( mode, prefix ):
   fec = FwdEqvClass( IpGenPrefix( prefix ) )
   mode.fecStaticBindingIs( fec, None )

def ipPrefixLabelBindingDelAll( mode ):
   mode.fecStaticBindingIs( None, None )

def neighborTargetedIp( mode, ipAddr ):
   mode.targetedIs( ipAddr )

def neighborTargetedDefault( mode, ipAddr ):
   mode.targetedDel( ipAddr )

def neighborTargetedAllDefault( mode ):
   mode.targetedDelAll()

def pseudowireAgentDel( mode ):
   mode.pseudowireAgentDel()

def fecFilterPrefixListName( mode, listName ):
   mode.fecFilterPrefixListNameIs( listName )

def fecFilterDefault( mode ):
   mode.fecFilterPrefixListNameDel()

# Link-hello
def helloHoldTime( mode, seconds ):
   mode.helloHoldTimeIs( seconds )

def helloHoldTimeInfinite( mode ):
   mode.helloHoldTimeIs( LdpProtoParam.infiniteHelloHoldTime )

def clearHelloHoldTime( mode ):
   mode.helloHoldTimeIs( LdpProtoParam.defaultBasicHelloHoldTime )

def helloInterval( mode, seconds ):
   mode.helloIntervalIs( seconds )

def clearHelloInterval( mode ):
   mode.helloIntervalIs( LdpProtoParam.defaultBasicHelloInterval )

# Targeted-hello
def targetedHelloHoldTime( mode, seconds ):
   mode.targetedHelloHoldTimeIs( seconds )

def targetedHelloHoldTimeInfinite( mode ):
   mode.targetedHelloHoldTimeIs( LdpProtoParam.infiniteHelloHoldTime )

def clearTargetedHelloHoldTime( mode ):
   mode.targetedHelloHoldTimeIs( LdpProtoParam.defaultTargetHelloHoldTime )

def targetedHelloInterval( mode, seconds ):
   mode.targetedHelloIntervalIs( seconds )

def clearTargetedHelloInterval( mode ):
   mode.targetedHelloIntervalIs( LdpProtoParam.defaultTargetHelloInterval )

def aclNameIs( mode, aclName ):
   mode.aclNameIs( aclName )

def aclNameDel( mode ):
   mode.aclNameDel()

def ldpIntfsDisabled( mode, no=None ):
   mode.ldpIntfsDisabledIs( no )

class LdpClearConfig( object ):
   '''
   The purpose of this class and the map it contains it to allow easier code
   introspection from the tests, map keys are ignored here but used by the tests
   to determine wether all possible "no" completions have an associated cleaning
   handler
   '''
   def __init__( self, clearConfigHandlersMap ):
      # dictionary containing cli configuration to be cleaned
      self.clearConfigHandlersMap = clearConfigHandlersMap
   def __call__( self, mode, *args ):
      '''
      This allows this class to act as a kind of "functor" to be able to treat
      leaves cleanup handler functions the same way as the ones containing maps
      '''
      for cleanupList in self.clearConfigHandlersMap.itervalues():
         if cleanupList is not None:
            clearConfigHandler, args = cleanupList[ 0 ], cleanupList[ 1: ]
            clearConfigHandler( mode, *args )

clearIgpSync = LdpClearConfig( {
                 'delay': [ defaultLinkReadyDelay ],
                 'holddown': [ defaultLinkReadyTimeout ],
               } )

clearIgp = LdpClearConfig( { 'sync': [ clearIgpSync ] } )

clearDiscoveryHello = LdpClearConfig( {
                        'hold-time': [ clearHelloHoldTime ],
                        'interval': [ clearHelloInterval ],
                      } )

clearDiscoveryTargetedHello = LdpClearConfig( {
                                'hold-time': [ clearTargetedHelloHoldTime ],
                                'interval': [ clearTargetedHelloInterval ],
                              } )

clearDiscovery = LdpClearConfig( {
                   'hello': [ clearDiscoveryHello ],
                   'targeted-hello': [ clearDiscoveryTargetedHello ],
                 } )

clearAdvDiscipline = LdpClearConfig( { 'mode': [ advDsplModeDefault ] } )

clearRetention = LdpClearConfig( { 'mode': [ retentionModeDefault ] } )

clearMplsStaticBindingIpv4 = LdpClearConfig( {
                               'all': [ ipPrefixLabelBindingDelAll ]
                             } )

clearMplsStaticBinding = LdpClearConfig( { 'ipv4': [ clearMplsStaticBindingIpv4 ] } )

clearMplsStatic = LdpClearConfig( { 'binding': [ clearMplsStaticBinding ] } )

clearMpls = LdpClearConfig( { 'static': [ clearMplsStatic ] } )

clearPseudowireAgent = LdpClearConfig( { 'pseudowires' : [ pseudowireAgentDel ] } )

clearIp = LdpClearConfig( { 'access-group' : [ aclNameDel ] } )

if Toggles.LdpToggleLib.toggleGrToggleEnabled():
   clearGrRole = LdpClearConfig(
      { 'helper': [ lambda mode : mode.noLdpGrHelperMode() ],
        'speaker': [ lambda mode : mode.noLdpGrSpeakerMode() ] } )
   clearGr = LdpClearConfig( { 'role': [ clearGrRole ] } )
else:
   clearGr = LdpClearConfig( {} )

clearMplsLdp = LdpClearConfig( {
                 'shutdown': None, # executed first below
                 'router-id': [ routerIdDefault, True ],
                 'transport-address': [ transportAddrDefault ],
                 'password': [ clearLdpPassword ],
                 'igp': [ clearIgp ],
                 'fec': [ fecFilterDefault ],
                 'discovery': [ clearDiscovery ],
                 'neighbor': [ neighborTargetedAllDefault ],
                 'pseudowires': [ clearPseudowireAgent ],
                 'adv-discipline': [ clearAdvDiscipline ],
                 'retention': [ clearRetention ],
                 'proto-param': [ protoParamDefault ],
                 'mpls': [ clearMpls ],
                 'ip': [ clearIp ],
                 'mldp' : [ clearMldp ],
                 'graceful-restart': [ clearGr ],
                 'interface' : [ ldpIntfsDisabled, 'default' ],
                 'label': [ LdpCli.LdpConfigMode.ldpLabelLocalTermination,
                             'implicit-null' ],
                 'entropy-label': [ entropyLabel, CliCommand.NAME_DEFAULT ],
               } )

#--------------------------------------------------------------------------------
# [ no | default ] mpls ldp
#--------------------------------------------------------------------------------
class MplsLdpCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': MplsCli.mplsNodeForConfig,
      'ldp': LdpCli.ldpKw,
   }

   @staticmethod
   def handler( mode, args ):
      mplsLdpMode = mode.childMode( LdpCli.LdpConfigMode )
      mode.session_.gotoChildMode( mplsLdpMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsLdpMode = mode.childMode( LdpCli.LdpConfigMode )
      # shutdown should be done first to take down the agent
      shutdownLdp( mplsLdpMode )
      clearMplsLdp( mplsLdpMode )

BasicCli.GlobalConfigMode.addCommandClass( MplsLdpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] adv-discipline mode DSPL_MODE
#--------------------------------------------------------------------------------
class AdvDisciplineModeCmd( CliCommand.CliCommandClass ):
   syntax = 'adv-discipline mode DSPL_MODE'
   noOrDefaultSyntax = 'adv-discipline mode ...'
   data = {
      'adv-discipline': 'Set Label Adv Discipline Type',
      'mode': 'Set Mode',
      'DSPL_MODE': CliMatcher.EnumMatcher( {
         'unsolicited': 'Downstream Unsolicited',
         'on-demand': 'Downstream On-Demand',
      } ),
   }
   hidden = True

   handler = lambda mode, args: advDsplModeIs( mode, args[ 'DSPL_MODE' ] )
   noOrDefaultHandler = lambda mode, args: advDsplModeDefault( mode )

LdpCli.LdpConfigMode.addCommandClass( AdvDisciplineModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] discovery hello hold-time ( HOLD_TIME | infinite )
#--------------------------------------------------------------------------------
class DiscoveryHelloHoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'discovery hello hold-time ( HOLD_TIME | infinite )'
   noOrDefaultSyntax = 'discovery hello hold-time ...'
   data = {
      'discovery': matcherDiscovery,
      'hello': matcherHello,
      'hold-time': matcherHoldTime,
      'HOLD_TIME': CliMatcher.IntegerMatcher( 1, 65534,
         helpdesc='Link hello hold time in seconds' ),
      'infinite': matcherInfinite,
   }

   @staticmethod
   def handler( mode, args ):
      if 'infinite' in args:
         helloHoldTimeInfinite( mode )
      else:
         helloHoldTime( mode, args[ 'HOLD_TIME' ] )

   noOrDefaultHandler = lambda mode, args: clearHelloHoldTime( mode )

LdpCli.LdpConfigMode.addCommandClass( DiscoveryHelloHoldTimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] discovery hello interval INTERVAL
#--------------------------------------------------------------------------------
class DiscoveryHelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'discovery hello interval INTERVAL'
   noOrDefaultSyntax = 'discovery hello interval ...'
   data = {
      'discovery': matcherDiscovery,
      'hello': matcherHello,
      'interval': 'Time between hello transmitions',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Link hello interval in seconds' ),
   }

   handler = lambda mode, args: helloInterval( mode, args[ 'INTERVAL' ] )
   noOrDefaultHandler = lambda mode, args: clearHelloInterval( mode )

LdpCli.LdpConfigMode.addCommandClass( DiscoveryHelloIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] discovery targeted-hello hold-time ( HOLD_TIME | infinite )
#--------------------------------------------------------------------------------
class DiscoveryTargetedHelloHoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'discovery targeted-hello hold-time ( HOLD_TIME | infinite )'
   noOrDefaultSyntax = 'discovery targeted-hello hold-time ...'
   data = {
      'discovery': matcherDiscovery,
      'targeted-hello': 'Ldp targeted hello parameters',
      'hold-time': matcherHoldTime,
      'HOLD_TIME': CliMatcher.IntegerMatcher( 1, 65534,
         helpdesc='Targeted hold time in seconds' ),
      'infinite': matcherInfinite,
   }

   @staticmethod
   def handler( mode, args ):
      if 'infinite' in args:
         targetedHelloHoldTimeInfinite( mode )
      else:
         targetedHelloHoldTime( mode, args[ 'HOLD_TIME' ] )

   noOrDefaultHandler = lambda mode, args: clearTargetedHelloHoldTime( mode )

LdpCli.LdpConfigMode.addCommandClass( DiscoveryTargetedHelloHoldTimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] discovery targeted-hello interval INTERVAL
#--------------------------------------------------------------------------------
class DiscoveryTargetedHelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'discovery targeted-hello interval INTERVAL'
   noOrDefaultSyntax = 'discovery targeted-hello interval ...'
   data = {
      'discovery': matcherDiscovery,
      'targeted-hello': matcherTargetedHello,
      'interval': 'Time between hello transmitions',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Link hello interval in seconds' ),
   }

   handler = lambda mode, args: targetedHelloInterval( mode, args[ 'INTERVAL' ] )
   noOrDefaultHandler = lambda mode, args: clearTargetedHelloInterval( mode )

LdpCli.LdpConfigMode.addCommandClass( DiscoveryTargetedHelloIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] igp sync delay [ DELAY ]
#--------------------------------------------------------------------------------
class IgpSyncDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'igp sync delay [ DELAY ]'
   noOrDefaultSyntax = 'igp sync delay ...'
   data = {
      'igp': 'IGP configuration',
      'sync': LdpCli.syncForShowKw,
      'delay': 'Time to wait before notifying IGP of link readiness',
      'DELAY': CliMatcher.IntegerMatcher( 1, 60, helpdesc='Time in seconds.' ),
   }

   handler = lambda mode, args: setLinkReadyDelay( mode, args.get( 'DELAY' ) )
   noHandler = lambda mode, args: clearLinkReadyDelay( mode )
   defaultHandler = lambda mode, args: defaultLinkReadyDelay( mode )

LdpCli.LdpConfigMode.addCommandClass( IgpSyncDelayCmd )

#--------------------------------------------------------------------------------
# [ no | default ] igp sync holddown [ until-established | TIMEOUT ]
#--------------------------------------------------------------------------------
class IgpSyncHolddownCmd( CliCommand.CliCommandClass ):
   syntax = 'igp sync holddown [ until-established | TIMEOUT ]'
   noOrDefaultSyntax = 'igp sync holddown ...'
   data = {
      'igp': 'IGP configuration',
      'sync': LdpCli.syncForShowKw,
      'holddown': ( 'Time to wait before forcing an interface ready for mpls '
                     'forwarding' ),
      'until-established': 'Wait until session is established',
      'TIMEOUT': CliMatcher.IntegerMatcher( 1, 2**32 - 1,
         helpdesc='Number of seconds' ),
   }

   handler = lambda mode, args: setLinkReadyTimeout( mode,
                                             args.get( 'TIMEOUT', Tac.endOfTime ) )
   noHandler = lambda mode, args: clearLinkReadyTimeout( mode )
   defaultHandler = lambda mode, args: defaultLinkReadyTimeout( mode )

LdpCli.LdpConfigMode.addCommandClass( IgpSyncHolddownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] interface disabled default
#--------------------------------------------------------------------------------
class InterfaceDisabledDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'interface disabled default'
   noOrDefaultSyntax = syntax
   data = {
      'interface': 'LDP interface configuration',
      'disabled': 'Disable LDP on interfaces',
      'default': 'Default LDP setting for interfaces',
   }

   handler = lambda mode, args: ldpIntfsDisabled( mode, no=None )
   noHandler = lambda mode, args: ldpIntfsDisabled( mode, no=True )
   defaultHandler = lambda mode, args: ldpIntfsDisabled( mode, no='default' )

LdpCli.LdpConfigMode.addCommandClass( InterfaceDisabledDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor IPADDR targeted
#--------------------------------------------------------------------------------
class NeighborIpaddrTargetedCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor IPADDR targeted'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': matcherLdpNeighbor,
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
      'targeted': 'Targeted Peer',
   }

   handler = lambda mode, args: neighborTargetedIp( mode, args[ 'IPADDR' ] )
   noOrDefaultHandler = lambda mode, args: neighborTargetedDefault( mode,
                                                                 args[ 'IPADDR' ] )

LdpCli.LdpConfigMode.addCommandClass( NeighborIpaddrTargetedCmd )

#--------------------------------------------------------------------------------
# [ no | default ] proto-param PROTO_PARAM
#--------------------------------------------------------------------------------
class ProtoParamCmd( CliCommand.CliCommandClass ):
   syntax = 'proto-param PROTO_PARAM'
   noOrDefaultSyntax = 'proto-param ...'
   data = {
      'proto-param': 'Protocol Parameter Setting',
      'PROTO_PARAM': CliMatcher.IntegerMatcher( 0, 2**32 - 1,
         helpdesc='Proto Parameter' ),
   }
   hidden = True

   handler = lambda mode, args: protoParam( mode, args[ 'PROTO_PARAM' ] )
   noOrDefaultHandler = lambda mode, args: protoParamDefault( mode )

LdpCli.LdpConfigMode.addCommandClass( ProtoParamCmd )

#--------------------------------------------------------------------------------
# [ no | default ] retention mode MODE
#--------------------------------------------------------------------------------
class RetentionModeCmd( CliCommand.CliCommandClass ):
   syntax = 'retention mode MODE'
   noOrDefaultSyntax = 'retention mode ...'
   data = {
      'retention': 'Set Label Retention Mode',
      'mode': 'Set Mode',
      'MODE': CliMatcher.EnumMatcher( {
         'conservative': 'Conservative Retention',
         'liberal': 'Liberal Retention',
      } )
   }
   hidden = True

   handler = lambda mode, args: retentionModeIs( mode, args[ 'MODE' ] )
   noOrDefaultHandler = lambda mode, args: retentionModeDefault( mode )

LdpCli.LdpConfigMode.addCommandClass( RetentionModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class LdpShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown LDP',
   }

   handler = lambda mode, args: shutdownLdp( mode, no=None )
   noHandler = lambda mode, args: shutdownLdp( mode, no=True )
   defaultHandler = lambda mode, args: shutdownLdp( mode, no='default' )

LdpCli.LdpConfigMode.addCommandClass( LdpShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] transport-address interface INTF
#--------------------------------------------------------------------------------
class TransportAddrIntfId( CliCommand.CliCommandClass ):
   syntax = 'transport-address interface INTF'
   noOrDefaultSyntax = 'transport-address ...'
   data = {
      'transport-address': 'Set LDP Transport Addr',
      'interface': matcherInterface,
      'INTF': Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=intfTypes ),
   }

   handler = lambda mode, args: transportAddrIntfId( mode, args[ 'INTF' ] )
   noOrDefaultHandler = lambda mode, args: transportAddrDefault( mode )

LdpCli.LdpConfigMode.addCommandClass( TransportAddrIntfId )

#--------------------------------------------------------------------------------
# [ no | default ] fec filter prefix-list LISTNAME
#--------------------------------------------------------------------------------
class FecFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'fec filter prefix-list LISTNAME'
   noOrDefaultSyntax = 'fec filter ...'
   data = {
      'fec' : matcherFec,
      'filter' : matcherFilter,
      'prefix-list' : 'Prefix list',
      'LISTNAME' : RouteMapCli.prefixListNameMatcher,
   }

   handler = lambda mode, args: fecFilterPrefixListName( mode, args[ 'LISTNAME' ] )
   noOrDefaultHandler = lambda mode, args: fecFilterDefault( mode )

LdpCli.LdpConfigMode.addCommandClass( FecFilterCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip access-group ACLNAME [ in ]
#--------------------------------------------------------------------------------
class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACLNAME [ in ]'
   noOrDefaultSyntax = 'ip access-group ...'
   data = {
      'ip' : ipKwForServiceAclMatcher,
      'access-group' : accessGroupKwMatcher,
      'ACLNAME' : ipAclNameMatcher,
      'in' : inKwMatcher,
   }

   handler = lambda mode, args: aclNameIs( mode, args[ 'ACLNAME' ] )
   noOrDefaultHandler = lambda mode, args: aclNameDel( mode )

LdpCli.LdpConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mpls static binding ipv4 ...
#--------------------------------------------------------------------------------
class MplsStaticBindingIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = '''mpls static binding ipv4 PREFIX
               [ { ( input INPUT ) |
                   ( output NEXTHOP ( explicit-null |
                                      implicit-null |
                                      OUTPUT ) ) } ]'''
   noOrDefaultSyntax = 'mpls static binding ipv4 ( PREFIX | all ) ...'
   data = {
      'mpls' : 'Set MPLS Param',
      'static' : 'Static Config',
      'binding' : 'Label binding',
      'ipv4' : 'IPv4',
      'PREFIX' : IpAddrMatcher.ipPrefixExpr( 'Network address', 'Network mask',
         'Address', overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'all': '',
      'input' : CliCommand.singleKeyword( 'input', helpdesc='Input' ),
      'INPUT' : CliMatcher.IntegerMatcher( 16, 1048575, helpdesc='MPLS Label' ),
      'output' : CliCommand.singleKeyword( 'output', helpdesc='Output' ),
      'NEXTHOP' : IpAddrMatcher.IpAddrMatcher( helpdesc='IPv4 Address' ),
      'explicit-null' : 'Explicit NULL Label',
      'implicit-null' : 'Implicit NULL Label',
      'OUTPUT' : CliMatcher.IntegerMatcher( 16, 1048575, helpdesc='MPLS Label' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      binding = LdpCli.CliFecLabelBindingObj()
      if 'input' in args:
         binding.inputLabel = args[ 'INPUT' ][ 0 ]
      if 'output' in args:
         binding.nexthop = args[ 'NEXTHOP' ][ 0 ]
         if 'explicit-null' in args:
            binding.outputLabel = 0
         elif 'implicit-null' in args:
            binding.outputLabel = 3
         elif 'OUTPUT' in args:
            binding.outputLabel = args[ 'OUTPUT' ][ 0 ]
      ipPrefixLabelBinding( mode, args[ 'PREFIX' ], [ binding ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'PREFIX' in args:
         ipPrefixLabelBindingDefault( mode, args[ 'PREFIX' ] )
      else:
         ipPrefixLabelBindingDelAll( mode )

LdpCli.LdpConfigMode.addCommandClass( MplsStaticBindingIpv4Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] router-id ( IPADDR | ( interface INTF ) ) [ force ]
#--------------------------------------------------------------------------------
class RouterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ( IPADDR | ( interface INTF ) ) [ force ]'
   noOrDefaultSyntax = 'router-id [ force ] ...'
   data = {
      'router-id' : 'Set LDP Router-Id',
      'IPADDR' : IpAddrMatcher.IpAddrMatcher( helpdesc='IP address' ),
      'interface' : matcherInterface,
      'INTF' : Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=intfTypes ),
      'force' : 'Forcibly change the LDP Router-Id',
   }

   @staticmethod
   def handler( mode, args ):
      force = 'force' in args
      if 'IPADDR' in args:
         routerIdIpAddr( mode, args[ 'IPADDR' ], force )
      elif 'interface' in args:
         routerIdIntfId( mode, args[ 'INTF' ], force )

   noOrDefaultHandler = lambda mode, args: routerIdDefault( mode,
         'force' in args )

LdpCli.LdpConfigMode.addCommandClass( RouterIdCmd )

# password <password>
class PasswordCommand( CliCommand.CliCommandClass ):
   syntax = 'password PASSWORD'
   noOrDefaultSyntax = 'password ...'
   data = {
      'password': 'LDP MD5 password configuration',
      'PASSWORD':
      ReversibleSecretCli.reversibleSecretCliExpression( 'PASSWORD' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.enableLdpPassword( args[ 'PASSWORD' ], False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearLdpPassword( mode )

LdpCli.LdpConfigMode.addCommandClass( PasswordCommand )

# label local-termination { implicit-null | explicit-null }
class LabelLocalTermination( CliCommand.CliCommandClass ):
   syntax = 'label local-termination ( implicit-null | explicit-null )'
   noOrDefaultSyntax = 'label local-termination ...'
   # BUG372208 remove hidden CLI once label local-termination feature is completed
   hidden = True
   data = {
      'label': 'Label to be advertised',
      'local-termination': 'Local termination label to be advertised',
      'implicit-null': 'Advertise implicit null',
      'explicit-null': 'Advertise explicit null'
   }
   handler = LdpCli.LdpConfigMode.ldpLabelLocalTermination
   noOrDefaultHandler = handler

if Toggles.LdpToggleLib.toggleLdpExplicitNullToggleEnabled():
   LdpCli.LdpConfigMode.addCommandClass( LabelLocalTermination )

# entropy-label
class LdpEntropyLabel( CliCommand.CliCommandClass ):
   syntax = 'entropy-label'
   noOrDefaultSyntax = syntax
   data = {
      'entropy-label': CliCommand.guardedKeyword( 'entropy-label',
         helpdesc='Enable entropy label signaling capability',
         guard=entropyLabelSupportedGuard ),
   }

   handler = entropyLabel
   noOrDefaultHandler = handler

if Toggles.MplsToggleLib.toggleEntropyLabelSignalEnabled():
   LdpCli.LdpConfigMode.addCommandClass( LdpEntropyLabel )

# ---------------------------------------------------------------------------------
# [ no | default ] neighbor hello-redundancy [ none |
#                                              ( duration ( SECONDS | infinite ) ) ]
# ---------------------------------------------------------------------------------
class NeighborHelloRedundancy( CliCommand.CliCommandClass ):
   syntax = ( 'neighbor hello-redundancy [ none | ( duration ( SECONDS | '
                                                              'infinite ) ) ]' )
   noOrDefaultSyntax = 'neighbor hello-redundancy'
   data = {
      'neighbor' : matcherLdpNeighbor,
      'hello-redundancy' : 'Redundant targeted adjacencies',
      'none' : 'Disable hello redundancy',
      'duration' : ( 'Duration of redundant adjacencies after losing all link '
                     'adjacencies' ),
      'SECONDS' : CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                               helpdesc='Redundant adjacency duration in seconds' ),
      'infinite' : 'Retain redundant targeted adjacencies indefinitely',
   }

   @staticmethod
   def handler( mode, args ):
      mode.helloRedundancyIs( 'none' not in args )
      if 'infinite' in args:
         mode.helloRedundancyTimeoutIs( Tac.endOfTime )
      else:
         mode.helloRedundancyTimeoutIs( args.get( 'SECONDS' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.helloRedundancyIs( False )
      mode.helloRedundancyTimeoutIs( None )

LdpCli.LdpConfigMode.addCommandClass( NeighborHelloRedundancy )

if Toggles.LdpToggleLib.toggleEndOfLibEnabled():
   # --------------------------------------------------------------------------------
   # [ no | default ] neighbor end-of-lib
   # --------------------------------------------------------------------------------
   class NeighborEndOfLib( CliCommand.CliCommandClass ):
      syntax = 'neighbor end-of-lib'
      noOrDefaultSyntax = syntax
      data = {
         'neighbor' : matcherLdpNeighbor,
         'end-of-lib' : 'Signal label advertisement completion',
      }

      @staticmethod
      def handler( mode, args ):
         mode.endOfLibIs( True )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         mode.endOfLibIs( False )

   LdpCli.LdpConfigMode.addCommandClass( NeighborEndOfLib )

# ------------------------------------------------------------------------------
# mpls ldp configuration mode. Like ldp, 
# mldp is not enabled till "no shutdown" is also configured 
# under mldp.
# ------------------------------------------------------------------------------
if Toggles.LdpToggleLib.toggleP2mpToggleEnabled():
   class MldpConfigMode( MldpMode, BasicCli.ConfigModeBase ):
      name = "Mpls Ldp Configuration"
      modeParseTree = CliParser.ModeParseTree()

      def __init__( self, parent, session, vrfName='default' ):
         self.vrfName = vrfName
         MldpMode.__init__( self )
         BasicCli.ConfigModeBase.__init__( self, parent, session )

      def shutdownMldpIs( self, no ):
         config = ldpConfigColl.config.newMember( self.vrfName )
         config.mLdpEnabled = bool( no )

      def lspRootOpaqueId( self, args ):
         root = args[ 'NODE' ]
         opaqueId = args[ 'OPAQUE_ID' ]
         config = ldpConfigColl.config.newMember( self.vrfName )
         p2mpGenericFecId = Tac.newInstance( 'Ldp::LdpP2mpFecGenericOpaqueId', 
               IpGenAddr( root ), opaqueId ) 
         config.staticP2mpFecBindingColl.newMember( p2mpGenericFecId )

      def lspRootOpaqueIdDefault( self, args ):
         root = args[ 'NODE' ]
         opaqueId = args[ 'OPAQUE_ID' ]
         config = ldpConfigColl.config.newMember( self.vrfName )
         p2mpGenericFecId = Tac.newInstance('Ldp::LdpP2mpFecGenericOpaqueId', 
               IpGenAddr( root ), opaqueId ) 
         del config.staticP2mpFecBindingColl[ p2mpGenericFecId ]

      def lspStaticP2mpFecBindingCollDelAll( self ): 
         config = ldpConfigColl.config.newMember( self.vrfName )
         config.staticP2mpFecBindingColl.clear()

   # mpls ldp mldp, add only if feature toggle is enabled
   #--------------------------------------------------------------------------------
   # [ no | default ] mldp
   #--------------------------------------------------------------------------------
   class MldpCmd( CliCommand.CliCommandClass ):
      syntax = 'mldp'
      noOrDefaultSyntax = syntax
      data = {
         'mldp': 'MLDP configuration',
      }

      @staticmethod
      def handler( mode, args ):
         childMode = mode.childMode( MldpConfigMode )
         mode.session_.gotoChildMode( childMode )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         clearMldp( mode )

   LdpCli.LdpConfigMode.addCommandClass( MldpCmd )

   #--------------------------------------------------------------------------------
   # [ no | default ] shutdown
   #--------------------------------------------------------------------------------
   class MldpShutdownCmd( CliCommand.CliCommandClass ):
      syntax = 'shutdown'
      noOrDefaultSyntax = syntax
      data = {
         'shutdown': 'Shutdown mLDP',
      }

      @staticmethod
      def handler( mode, args ):
         mode.shutdownMldpIs( no=None )

      @staticmethod
      def noHandler( mode, args ):
         mode.shutdownMldpIs( no=True )

      @staticmethod
      def defaultHandler( mode, args ):
         mode.shutdownMldpIs( no='default' )

   MldpConfigMode.addCommandClass( MldpShutdownCmd )

   #--------------------------------------------------------------------------------
   # [ no | default ] p2mp-lsp root NODE opaque-id OPAQUE_ID
   #--------------------------------------------------------------------------------
   class LspRootOpaqueId( CliCommand.CliCommandClass ):
      syntax = 'p2mp-lsp root NODE opaque-id OPAQUE_ID'
      noOrDefaultSyntax = syntax
      data = {
         'p2mp-lsp': 'LSP configuration',
         'root': 'Root node address',
         'NODE': IpAddrMatcher.IpAddrMatcher( helpdesc='P2mp root node address' ),
         'opaque-id': 'Opaque ID',
         'OPAQUE_ID': CliMatcher.IntegerMatcher( 1, 2**32 - 1,
            helpdesc='Opaque-id' ),
      }

      hidden = True
      handler = MldpConfigMode.lspRootOpaqueId
      noOrDefaultHandler = MldpConfigMode.lspRootOpaqueIdDefault

   MldpConfigMode.addCommandClass( LspRootOpaqueId )

if Toggles.LdpToggleLib.toggleGrToggleEnabled():
   matcherGracefulRestart = CliMatcher.KeywordMatcher( 'graceful-restart',
      helpdesc='LDP graceful restart' )
   matcherRole = CliMatcher.KeywordMatcher( 'role',
      helpdesc='LDP graceful restart role' )
   matcherTimer = CliMatcher.KeywordMatcher( 'timer',
      helpdesc='LDP graceful restart timer' )
   grTimeoutMatcher = CliMatcher.IntegerMatcher( 1, 3600,
      helpdesc='Number of seconds' )

   # Gr Helper timers
   #--------------------------------------------------------------------------------
   # [ no | default ] timer neighbor-liveness [ TIMEOUT ]
   #--------------------------------------------------------------------------------
   class TimerNeighborLivenessCmd( CliCommand.CliCommandClass ):
      syntax = 'timer neighbor-liveness [ TIMEOUT ]'
      noOrDefaultSyntax = 'timer neighbor-liveness ...'
      data = {
         'timer': matcherTimer,
         'neighbor-liveness': ( 'Maximum time to preserve the session '
                                 'forwarding-state before the session is '
                                 'reestablished' ),
         'TIMEOUT': grTimeoutMatcher,
      }

      @staticmethod
      def handler( mode, args ):
         return mode.neighborLivenessIs( args.get( 'TIMEOUT' ) )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         return mode.noNeighborLiveness()

   LdpCli.LdpGrHelperConfigMode.addCommandClass( TimerNeighborLivenessCmd )

   #--------------------------------------------------------------------------------
   # [ no | default ] timer recovery [ TIMEOUT ]
   #--------------------------------------------------------------------------------
   class TimerRecoveryCmd( CliCommand.CliCommandClass ):
      syntax = 'timer recovery [ TIMEOUT ]'
      noOrDefaultSyntax = 'timer recovery ...'
      data = {
         'timer': matcherTimer,
         'recovery': ( 'Maximum time to preserve the session forwarding-state '
                        'after the session has reestablished' ),
         'TIMEOUT': grTimeoutMatcher,
      }

      @staticmethod
      def handler( mode, args ):
         return mode.recoveryIs( args.get( 'TIMEOUT' ) )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         return mode.noRecovery()

   LdpCli.LdpGrHelperConfigMode.addCommandClass( TimerRecoveryCmd )

   # Gr Speaker timers
   #--------------------------------------------------------------------------------
   # [ no | default ] timer state-holding [ TIMEOUT ]
   #--------------------------------------------------------------------------------
   class TimerStateHoldingCmd( CliCommand.CliCommandClass ):
      syntax = 'timer state-holding [ TIMEOUT ]'
      noOrDefaultSyntax = syntax
      data = {
         'timer': matcherTimer,
         'state-holding': 'Time to preserve the forwarding-state after restart',
         'TIMEOUT': grTimeoutMatcher,
      }

      @staticmethod
      def handler( mode, args ):
         return mode.stateHoldingIs( args.get( 'TIMEOUT' ) )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         return mode.noStateHolding()

   LdpCli.LdpGrSpeakerConfigMode.addCommandClass( TimerStateHoldingCmd )
   #--------------------------------------------------------------------------------
   # [ no | default ] timer reconnect [ TIMEOUT ]
   #--------------------------------------------------------------------------------

   class TimerReconnectCmd( CliCommand.CliCommandClass ):
      syntax = 'timer reconnect [ TIMEOUT ]'
      noOrDefaultSyntax = syntax
      data = {
         'timer': matcherTimer,
         'reconnect': ( 'Time the peer should preserve the forwarding-state '
                         'after restart' ),
         'TIMEOUT': grTimeoutMatcher,
      }

      @staticmethod
      def handler( mode, args ):
         return mode.reconnectIs( args.get( 'TIMEOUT' ) )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         return mode.noReconnect()

   LdpCli.LdpGrSpeakerConfigMode.addCommandClass( TimerReconnectCmd )

   #--------------------------------------------------------------------------------
   # [ no | default ] graceful-restart role helper
   #--------------------------------------------------------------------------------
   class GracefulRestartRoleHelperCmd( CliCommand.CliCommandClass ):
      syntax = 'graceful-restart role helper'
      noOrDefaultSyntax = syntax
      data = {
         'graceful-restart': matcherGracefulRestart,
         'role': matcherRole,
         'helper': 'Ldp graceful restart helper',
      }

      @staticmethod
      def handler( mode, args ):
         return mode.gotoLdpGrHelperMode()

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         return mode.noLdpGrHelperMode()

   LdpCli.LdpConfigMode.addCommandClass( GracefulRestartRoleHelperCmd )

   #--------------------------------------------------------------------------------
   # [ no | default ] graceful-restart role speaker
   #--------------------------------------------------------------------------------
   class GracefulRestartRoleSpeakerCmd( CliCommand.CliCommandClass ):
      syntax = 'graceful-restart role speaker'
      noOrDefaultSyntax = syntax
      data = {
         'graceful-restart': matcherGracefulRestart,
         'role': matcherRole,
         'speaker': 'LDP graceful restart speaker',
      }

      @staticmethod
      def handler( mode, args ):
         return mode.gotoLdpGrSpeakerMode()

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         return mode.noLdpGrSpeakerMode()

   LdpCli.LdpConfigMode.addCommandClass( GracefulRestartRoleSpeakerCmd )

#------------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ldpConfigColl
   global mplsHwCapability

   ldpConfigColl = ConfigMount.mount( entityManager, "mpls/ldp/ldpConfigColl",
                                      "Ldp::LdpConfigColl", "w" )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
