# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements MPLS configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for MPLS'''

from eunuchs.in_h import IPPROTO_IP
from socket import TCP_MD5SIG_MAXKEYLEN
import AclCli
import AclCliLib
import AgentDirectory
import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
from CliMode.LdpMode import LdpMode
import CliParser
import CliPlugin.IntfCli as IntfCli
import CliPlugin.MplsCli as MplsCli
import CliPlugin.TechSupportCli
import CliToken.Clear
import ConfigMount
import IpAddrMatcher
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ReversibleSecretCli
import Tac
import Toggles.LdpToggleLib
from TypeFuture import TacLazyType

 # pylint: disable=ungrouped-imports
if Toggles.LdpToggleLib.toggleGrToggleEnabled():
   from CliMode.LdpMode import (
      GrHelperMode,
      GrSpeakerMode,
   )

FwdEqvClass = TacLazyType( 'Mpls::FwdEqvClass' )
IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
LdpFecBindingInfo = TacLazyType( 'Ldp::LdpFecBindingInfo' )
LdpIdentifier = TacLazyType( 'Ldp::LdpIdentifier' )
LdpLabelLocalTerminationMode = TacLazyType( 'Ldp::LdpLabelLocalTerminationMode' )
LdpProtoParam = TacLazyType( 'Ldp::LdpProtoParam' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )

class CliFecLabelBindingObj( object ):
   def __init__( self, inputLabel=MplsLabel.null, outputLabel=MplsLabel.null,
                 nexthop='0.0.0.0' ):
      self.inputLabel = inputLabel
      self.outputLabel = outputLabel
      self.nexthop = nexthop

aclCheckpoint = None
aclConfig = None
aclCpConfig = None
aclStatus = None
ldpConfigColl = None
ldpEnabledVrfColl = None
ldpProtoConfigColl = None
ldpStatusColl = None
mplsHwCapability = None
mplsRoutingConfig = None
routingVrfInfoDir = None
sysname = None

ldpKw = CliMatcher.KeywordMatcher( 'ldp', helpdesc='LDP configuration' )
ldpForShowKw = ldpKw # Legacy use only
syncForShowKw = CliMatcher.KeywordMatcher( 'sync',
      helpdesc='IGP sync configuration' )


__ldpConfigPseudowireCleanupHook = CliExtensions.CliHook()
def getLdpConfigPseudowireCleanupHook():
   '''Call this to add extra cleanup routines for when ldp config is cleared.

   It is used to clean up pseudowire config when "no mpls ldp" is run.
   '''

   return __ldpConfigPseudowireCleanupHook

def showLdpConfigWarnings( mode, brief=False ):
   # Assuming default vrf for now
   vrfName = DEFAULT_VRF 

   if not AgentDirectory.agentIsRunning( sysname, 'LdpAgent' ):
      mode.addWarning( "Agent 'LdpAgent' is not running" )
   if not AgentDirectory.agentIsRunning( sysname, 'Mpls' ):
      mode.addWarning( "Agent 'Mpls' is not running" )

   if brief:
      # This is for e.g. the general 'show mpls ldp' command
      return

   config = ldpConfigColl.config.get( vrfName )
   if ( config is None or not config.enabled ):
      mode.addWarning( "LDP is not enabled" )
   else:
      protoConfig = ldpProtoConfigColl.protoConfig.get( vrfName )
      if protoConfig is None:
         mode.addWarning( "LDP is operationally down (internal error)" )
      elif protoConfig.runningState != 'ldpRunning':
         mode.addWarning( "LDP is operationally down: %s" %
                           protoConfig.reason )
   if not mplsRoutingConfig.mplsRouting:
      mode.addWarning( "MPLS routing is not enabled" )
   routingInfo = routingVrfInfoDir.get( DEFAULT_VRF )
   if not ( routingInfo and routingInfo.routing ):
      mode.addWarning( "IP routing is not enabled" )

# ------------------------------------------------------------------------------
# mpls ldp configuration mode
# ------------------------------------------------------------------------------
class LdpConfigMode( LdpMode, BasicCli.ConfigModeBase ):
   name = "Mpls Ldp Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName='default' ):
      self.vrfName = vrfName
      LdpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def enableLdpPassword ( self, passwd, no ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      decodedPasswd = ReversibleSecretCli.decodeKey( passwd )
      if not no:
         if len( decodedPasswd ) > TCP_MD5SIG_MAXKEYLEN:
            self.addError( "Maximum password length exceeded, Max length=80" )
            return
         # Store encoded password
         config.md5Password = passwd
      else:
         config.md5Password = ""

   def ldpLabelLocalTermination( self, args ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if 'explicit-null' in args:
         termMode = LdpLabelLocalTerminationMode.explicitNull
      else:
         termMode = LdpLabelLocalTerminationMode.implicitNull
      config.ldpLabelLocalTerminationMode = termMode

   def ldpLinkReadyTimeout( self, timeout ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if timeout >= 0:
         config.linkReadyTimeout = timeout
      else:
         config.linkReadyTimeout = config.linkReadyTimeoutDefault

   def ldpLinkReadyDelay( self, delay ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if delay >= 0:
         config.linkReadyDelay = delay
      else:
         config.linkReadyDelay = config.linkReadyDelayDefault

   def pseudowireAgentDel( self ):
      getLdpConfigPseudowireCleanupHook().notifyExtensions()

   def shutdownIs( self, no ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      config.enabled = bool( no )
      if config.enabled:
         ldpEnabledVrfColl.enabledVrf[ self.vrfName ] = True
      else:
         del ldpEnabledVrfColl.enabledVrf[ self.vrfName ]

   def entropyLabelIs( self, args ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      enabled = not CliCommand.isNoOrDefaultCmd( args )
      config.entropyLabel = enabled

   def protoParamIs( self, param ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      config.ldpProtoParamSetting = param

   def routerIdIs( self, no, ipAddr, intfId, force ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if no:
         config.routerId = '0.0.0.0'
         config.routerIdIntfId = ''
      else:
         config.routerId = ipAddr
         config.routerIdIntfId = intfId
      if force:
         config.forceRouterId = Tac.now()

   def transportAddrIntfIdIs( self, no, intfId ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if no:
         config.ipTransportAddrIntfId = ''
      else:
         config.ipTransportAddrIntfId = intfId

   def advDsplModeIs( self, advDsplMode ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if advDsplMode == 'unsolicited':
         config.ldpLabelAdvDspl = 'unSolicited'
      elif advDsplMode == 'on-demand':
         config.ldpLabelAdvDspl = 'onDemand'
      else:
         assert False, "Internal Error"

   def retentionModeIs( self, retentionMode ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if retentionMode == 'liberal':
         config.ldpLabelRetention = 'liberalRetention'
      elif retentionMode == 'conservative':
         config.ldpLabelRetention = 'conservativeRetention'
      else:
         assert False, "Internal Error"

   def fecStaticBindingIs( self, ipPrefix, binding ):
      config = ldpConfigColl.config.newMember( self.vrfName )

      if not ipPrefix:
         for key in config.staticFecBindingColl.keys():
            del config.staticFecBindingColl[ key ]
         return
      if not binding:
         del config.staticFecBindingColl[ ipPrefix ]
         return

      # Consolidate all the inputs from CLI
      tempValueObj = CliFecLabelBindingObj()
      for valueObj in binding:
         if MplsLabel( valueObj.inputLabel ).isValid():
            tempValueObj.inputLabel = valueObj.inputLabel
         if MplsLabel( valueObj.outputLabel ).isValid():
            tempValueObj.outputLabel = valueObj.outputLabel
            tempValueObj.nexthop = valueObj.nexthop
      # Update LdpConfig
      fecBindingInfo = LdpFecBindingInfo( tempValueObj.inputLabel,
                                      tempValueObj.outputLabel,
                                      IpGenAddr( tempValueObj.nexthop ) )
      fec = FwdEqvClass( IpGenPrefix( ipPrefix ) )
      fecBindingEntry = config.staticFecBindingColl.newMember( fec )
      fecBindingEntry.fecBindingInfo = fecBindingInfo

   def targetedIs( self, ipAddr ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      config.staticTarget[ IpGenAddr( ipAddr ) ] = True

   def targetedDel( self, ipAddr ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      del config.staticTarget[ IpGenAddr( ipAddr ) ]

   def targetedDelAll( self ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      config.staticTarget.clear()

   def fecFilterPrefixListNameIs( self, prefixListName ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      config.fecFilterPrefixListName = prefixListName

   def fecFilterPrefixListNameDel( self ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      config.fecFilterPrefixListName = ""

   def helloHoldTimeIs( self, seconds ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if config is not None:
         config.helloHoldTime = seconds

   def helloIntervalIs( self, seconds ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if config is not None:
         config.helloInterval = seconds

   def targetedHelloHoldTimeIs( self, seconds ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if config is not None:
         config.targetedHelloHoldTime = seconds

   def targetedHelloIntervalIs( self, seconds ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if config is not None:
         config.targetedHelloInterval = seconds

   def aclNameIs( self, aclName ):
      config = ldpConfigColl.config.get( self.vrfName )
      if config is None:
         config = ldpConfigColl.config.newMember( self.vrfName )
      AclCliLib.setServiceAcl( self, 'ldp', IPPROTO_IP,
                               aclConfig, aclCpConfig, aclName, vrfName=self.vrfName,
                               port=[ LdpProtoParam.ldpTcpPort ], tracked=True )

   def aclNameDel( self ):
      AclCliLib.noServiceAcl( self, 'ldp', aclConfig, aclCpConfig, None,
                              vrfName=self.vrfName )

   def ldpIntfsDisabledIs( self, no ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      config.onlyLdpEnabledIntfs = not no

   if Toggles.LdpToggleLib.toggleGrToggleEnabled():
      def gotoLdpGrHelperMode( self ):
         childMode = self.childMode( LdpGrHelperConfigMode )
         childMode.helperIs()
         self.session_.gotoChildMode( childMode )

      def noLdpGrHelperMode( self ):
         childMode = self.childMode( LdpGrHelperConfigMode )
         childMode.noHelper()
         childMode.noRecovery()
         childMode.noNeighborLiveness()

      def gotoLdpGrSpeakerMode( self ):
         childMode = self.childMode( LdpGrSpeakerConfigMode )
         childMode.speakerIs()
         self.session_.gotoChildMode( childMode )

      def noLdpGrSpeakerMode( self ):
         childMode = self.childMode( LdpGrSpeakerConfigMode )
         childMode.noSpeaker()
         childMode.noStateHolding()
         childMode.noReconnect()

   def helloRedundancyIs( self, enabled ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if config is not None:
         config.helloRedundancy = enabled

   def helloRedundancyTimeoutIs( self, seconds ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if config is not None:
         if seconds is not None:
            config.helloRedundancyTimeout = seconds
         else:
            config.helloRedundancyTimeout = \
                  LdpProtoParam.defaultHelloRedundancyTimeout

   def endOfLibIs( self, enabled ):
      config = ldpConfigColl.config.newMember( self.vrfName )
      if config is not None:
         config.endOfLib = enabled

# ---------------------------------------------------------------------------------
# clear mpls ldp counters access-list
# ---------------------------------------------------------------------------------
matcherLdp = CliMatcher.KeywordMatcher( 'ldp',
      helpdesc='Clear LDP information' )

class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear mpls ldp counters access-list'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'mpls': MplsCli.mplsMatcherForClear,
            'ldp': matcherLdp,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'counters': AclCli.countersKwMatcher }

   @staticmethod
   def handler( mode, args ):
      showLdpConfigWarnings( mode )

      aclType = 'ip'
      vrfName = DEFAULT_VRF
      status = ldpStatusColl.status.get( vrfName )
      if status is None:
         return

      AclCli.clearServiceAclCounters( mode,
                                      aclStatus,
                                      aclCheckpoint,
                                      aclType )

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

#--------------------------------------------------------------------------------
# clear mpls ldp neighbor ( * | NEIGHBOR )
#--------------------------------------------------------------------------------
class ClearMplsLdpNeighborCmd( CliCommand.CliCommandClass ):
   syntax = 'clear mpls ldp neighbor ( * | NEIGHBOR )'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'mpls': MplsCli.mplsMatcherForClear,
      'ldp': matcherLdp,
      'neighbor': 'Neighbor to clear',
      '*': 'Clear all neighbors',
      'NEIGHBOR': IpAddrMatcher.IpAddrMatcher( helpdesc='LDP router ID' ),
   }

   @staticmethod
   def handler( mode, args ):
      showLdpConfigWarnings( mode )
      vrfName = DEFAULT_VRF

      if '*' in args:
         neighborToDelete = LdpIdentifier.allLdpNeighborIdentifier
      else:
         # We only expect lable space to be 0
         neighborToDelete = args[ 'NEIGHBOR' ] + ':0'

      config = ldpConfigColl.config.get( vrfName )
      config.ldpNeighborClearRequest[ neighborToDelete ] = Tac.now()

BasicCli.EnableMode.addCommandClass( ClearMplsLdpNeighborCmd )

# ------------------------------------------------------------------------------
# mpls ldp graceful-restart role (helper|speaker) config mode
# Graceful-restart is disabled by default.
# Enabling speaker implies support for helper as well. The role configurations
# are not mutually exclusive, as the helper-specific timer configurations are under
# the helper submode.
# ------------------------------------------------------------------------------
if Toggles.LdpToggleLib.toggleGrToggleEnabled():
   class LdpGrHelperConfigMode( GrHelperMode, BasicCli.ConfigModeBase ):
      name = "Mpls Ldp Graceful Restart Helper Configuration"
      modeParseTree = CliParser.ModeParseTree()

      def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
         assert vrfName == DEFAULT_VRF
         self.vrfName = vrfName
         GrHelperMode.__init__( self )
         BasicCli.ConfigModeBase.__init__( self, parent, session )

      def helperIs( self ):
         config = ldpConfigColl.config.newMember( self.vrfName )
         flag = Tac.nonConst( config.grOperFlag )
         flag.grFlagHelper = True
         config.grOperFlag = flag

      def noHelper( self ):
         config = ldpConfigColl.config.get( self.vrfName )
         if config:
            flag = Tac.nonConst( config.grOperFlag )
            flag.grFlagHelper = False
            config.grOperFlag = flag

      def recoveryIs( self, timeout ):
         config = ldpConfigColl.config.get( self.vrfName )
         if config:
            config.grMaxRecoveryTimeout = timeout

      def noRecovery( self ):
         self.recoveryIs( LdpProtoParam.defaultGrMaxRecoveryTimeout )

      def neighborLivenessIs( self, timeout ):
         config = ldpConfigColl.config.get( self.vrfName )
         if config:
            config.grNeighborLivenessTimeout = timeout

      def noNeighborLiveness( self ):
         self.neighborLivenessIs( LdpProtoParam.defaultGrNeighborLivenessTimeout )

   class LdpGrSpeakerConfigMode( GrSpeakerMode, BasicCli.ConfigModeBase ):
      name = "Mpls Ldp Graceful Restart Speaker Configuration"
      modeParseTree = CliParser.ModeParseTree()

      def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
         assert vrfName == DEFAULT_VRF
         self.vrfName = vrfName
         GrSpeakerMode.__init__( self )
         BasicCli.ConfigModeBase.__init__( self, parent, session )

      def speakerIs( self ):
         config = ldpConfigColl.config.newMember( self.vrfName )
         flag = Tac.nonConst( config.grOperFlag )
         flag.grFlagSpeaker = True
         config.grOperFlag = flag

      def noSpeaker( self ):
         config = ldpConfigColl.config.get( self.vrfName )
         if config:
            flag = Tac.nonConst( config.grOperFlag )
            flag.grFlagSpeaker = False
            config.grOperFlag = flag

      def stateHoldingIs( self, timeout ):
         config = ldpConfigColl.config[ self.vrfName ]
         if config:
            config.grHoldingTimeout = timeout

      def noStateHolding( self ):
         self.stateHoldingIs( LdpProtoParam.defaultGrHoldingTimeout )

      def reconnectIs( self, timeout ):
         config = ldpConfigColl.config[ self.vrfName ]
         if config:
            config.grReconnectTimeout = timeout

      def noReconnect( self ):
         self.reconnectIs( LdpProtoParam.defaultGrReconnectTimeout )

#------------------------------------------------------------------------------------
class LdpIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      '''Remove interface level ldp igp sync if it was configured'''
      vrfName = DEFAULT_VRF
      ldpConfig = ldpConfigColl.config.get( vrfName )
      if ldpConfig:
         del ldpConfig.intfConfigColl[ self.intf_. name ]

#------------------------------------------------------------------------------------
# Support for show tech-support
#------------------------------------------------------------------------------------

def _showTechCmds():
   if not mplsHwCapability.mplsSupported:
      return []
   cmds = [ 'show mpls ldp detail',
            'show mpls ldp discovery detail',
            'show mpls ldp neighbor detail',
            'show mpls ldp bindings detail',
            'show mpls ldp tunnel',
            'show mpls ldp bindings mldp detail',
          ]
   return cmds

def _showTechSummaryCmds():
   if not mplsHwCapability.mplsSupported:
      return []
   return [ 'show mpls ldp neighbor' ]

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( '2016-02-10 10:13:37',
               _showTechCmds,
               summaryCmdCallback=_showTechSummaryCmds )

#------------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global aclCheckpoint
   global aclConfig
   global aclCpConfig
   global aclStatus
   global ldpConfigColl
   global ldpEnabledVrfColl
   global ldpProtoConfigColl
   global ldpStatusColl
   global mplsHwCapability
   global mplsRoutingConfig
   global routingVrfInfoDir
   global sysname

   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   ldpConfigColl = ConfigMount.mount( entityManager, "mpls/ldp/ldpConfigColl",
                                      "Ldp::LdpConfigColl", "w" )
   ldpEnabledVrfColl = ConfigMount.mount( entityManager, 
                                        "mpls/ldp/ldpEnabledVrfColl", 
                                        "Ldp::LdpEnabledVrfColl", "w" )
   ldpProtoConfigColl = LazyMount.mount( entityManager, 
                                         "mpls/ldp/ldpProtoConfigColl",
                                         "Ldp::LdpProtoConfigColl", "r" )
   ldpStatusColl = LazyMount.mount( entityManager, "mpls/ldp/ldpStatusColl",
                                    "Ldp::LdpStatusColl", "r" )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
   mplsRoutingConfig = LazyMount.mount( entityManager, "routing/mpls/config",
                                 "Mpls::Config", "r" )
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                    "routing/vrf/routingInfo/status",
                                    "Tac::Dir", "ri" )
   sysname = entityManager.sysname()
   IntfCli.Intf.registerDependentClass( LdpIntf )
