#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AaaCliLib
import Ark
import BasicCli
import Cell
import CliCommand
import CliToken.Clear
import ConfigMgmtMode
import Intf.Log
import LazyMount
import ShowCommand
import Tac
from IpLibConsts import DEFAULT_VRF
from LdapModel import ( ShowLdapModel, LdapStatsModel, ServerGroupModel,
                        ServerInfoModel, LdapStatusModel )

aaaConfig = None
ldapCounterConfig = None
ldapConfig = None
ldapStatus = None

def hostStr( spec ):
   name = '%s/%d' % ( spec.hostname, spec.port )
   if spec.vrf != DEFAULT_VRF:
      name += ' vrf %s' % spec.vrf
   return name

def showLdap( mode, args ):
   ldapModel = ShowLdapModel()
   for host in ldapConfig.host.values():
      stat = ldapStatus.counter.get( host.spec )
      if stat:
         s = LdapStatsModel()
         s._index = host.index  # pylint: disable=protected-access
         s.bindRequests = stat.bindRequests
         s.bindFails = stat.bindFails
         s.bindSuccesses = stat.bindSuccesses
         s.bindTimeouts = stat.bindTimeouts
         ldapModel.ldapServers[ hostStr( host.spec ) ] = s
      ldapStatusModel = LdapStatusModel()
      ldapStatusModel._index = host.index  # pylint: disable=protected-access
      ldapStatusModel.fipsStatus = ldapStatus.fips
      ldapModel.ldapStatus[ hostStr( host.spec ) ] = ldapStatusModel

   for k in sorted( aaaConfig.hostgroup.keys() ):
      g = aaaConfig.hostgroup[ k ]
      if g.groupType == 'ldap':
         serverGroupDisplay = AaaCliLib.getCliDisplayFromGroup( g.groupType )
         serverGroupName = g.name
         ldapModel.groups[ serverGroupName ] = ServerGroupModel()
         ldapModel.groups[ serverGroupName ].serverGroup = serverGroupDisplay
         for _, m in g.member.iteritems():
            si = ServerInfoModel()
            si.hostname = m.spec.hostname
            si.port = m.spec.port
            ldapModel.groups[ serverGroupName ].members.append( si )

   ldapModel.lastCounterClearTime = Ark.switchTimeToUtc( ldapStatus.lastClearTime )
   return ldapModel

#-------------------------------------------------------------------------------
# "show management ldap"
#-------------------------------------------------------------------------------
class MgmtLdapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ldap'
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'ldap': 'LDAP server statistics',
   }
   handler = showLdap
   cliModel = ShowLdapModel

BasicCli.addShowCommandClass( MgmtLdapCmd )

#-------------------------------------------------------------------------------
# "clear aaa counters ldap" in enable mode
#-------------------------------------------------------------------------------
def clearCounters( mode, args ):
   Intf.Log.logClearCounters( "ldap" )
   ldapCounterConfig.clearCounterRequestTime = Tac.now()
   try:
      Tac.waitFor( lambda: ldapStatus.lastClearTime >=
                   ldapCounterConfig.clearCounterRequestTime,
                   description='LDAP clear counter request to complete',
                   warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      mode.addWarning(
         "LDAP counters may not have been reset yet" )

class ClearAaaCountersLdapCmd( CliCommand.CliCommandClass ):
   syntax = 'clear aaa counters ldap'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'aaa': 'Clear AAA information',
      'counters': 'Clear AAA counters',
      'ldap': 'Clear LDAP server statistics'
   }
   handler = clearCounters

BasicCli.EnableMode.addCommandClass( ClearAaaCountersLdapCmd )

def Plugin( entityManager ):

   global aaaConfig
   global ldapCounterConfig
   global ldapConfig
   global ldapStatus
   aaaConfig = LazyMount.mount( entityManager, "security/aaa/config",
                                "Aaa::Config", "r" )
   ldapCounterConfig = LazyMount.mount( entityManager,
                                        "security/aaa/ldap/counterConfig",
                                        "AaaPlugin::CounterConfig", "w" )
   ldapConfig = LazyMount.mount( entityManager,
                                 "security/aaa/ldap/config",
                                 "Ldap::Config", "r" )

   ldapStatus = LazyMount.mount( entityManager,
                                 Cell.path( 'security/aaa/ldap/status' ),
                                 'Ldap::Status', 'r' )
