#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
from CliModel import Dict, Float, Int, List, Model, Str, Bool

class ServerInfoModel( Model ):
   hostname = Str( help="IP address or hostname of LDAP server" )
   port = Int( help="LDAP authorization port number" )

class ServerGroupModel( Model ):
   serverGroup = Str( help="Configured server group" )
   members = List( valueType=ServerInfoModel,
                   help="List of LDAP servers in the server group" )

class LdapStatsModel( Model ):
   _index = Int( help="Sort index for preserving display order" )
   bindRequests = Int( help="Number of bind requests to the LDAP server" )
   bindSuccesses = Int( help="Number of successful binds to the LDAP server" )
   bindFails = Int( help="Number of failed binds to the LDAP server" )
   bindTimeouts = Int( help="Number of binds timedout to the LDAP server" )

class LdapStatusModel( Model ):
   _index = Int( help="Sort index for preserving display order" )
   fipsStatus = Bool( help="Status of LDAP FIPS mode" )

class ShowLdapModel( Model ):
   ldapServers = Dict(
      valueType=LdapStatsModel,
      help="Mapping of server IP address or hostname to server statistics" )
   groups = Dict( keyType=str, valueType=ServerGroupModel, optional=True,
                  help="Mapping of group name to group information" )
   lastCounterClearTime = Float(
      help="UTC time when the counters were last cleared" )
   ldapStatus = Dict(
      valueType=LdapStatusModel,
      help="Mapping of server IP address or hostname to server status" )

   def render( self ):
      # pylint: disable=protected-access
      for host, status in sorted( self.ldapStatus.items(),
                                 key=lambda x: x[ 1 ]._index ):
         print "LDAP server: %s" % host
         if host in self.ldapServers:
            stats = self.ldapServers[ host ]
            print "   Binds requested: %d" % stats.bindRequests
            print "   Binds successful: %d" % stats.bindSuccesses
            print "   Binds failed: %d" % stats.bindFails
            print "   Binds timed out: %d" % stats.bindTimeouts
         print "   FIPS is %s" % ( "ON" if status.fipsStatus else "OFF" )
      # pylint: enable=protected-access
      for serverGroupName, ldapGroup in self.groups.iteritems():
         print "\n%s server group: %s" % ( ldapGroup.serverGroup,
                                         serverGroupName )
         for sequence, host in enumerate( ldapGroup.members ):
            print "   %d: %s/%d" % ( sequence, host.hostname, host.port )
         if not ldapGroup.members:
            print "(server group is empty)"

      print "\nLast time counters were cleared:", Ark.utcTimeRelativeToNowStr(
         self.lastCounterClearTime )
