#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AaaCliLib
import BasicCli
import CliCommand
import CliMatcher
import CliMode.Ldap
import CliParser
import ConfigMount
import HostnameCli
import LazyMount
import LdapGroup
import Tac
from AaaDefs import roleNameRe
from CliPlugin.VrfCli import DEFAULT_VRF, VrfExprFactory
from LocalUserLib import usernameRe
from ReversibleSecretCli import reversibleSecretCliExpression
from Tracing import Handle, t0

import ConfigMgmtMode

__defaultTraceHandle__ = Handle( 'LdapConfigCliPlugin' )

DEFAULT_LDAP_TIMEOUT = 5
DEFAULT_ROLE_PRIVLEVEL = 1
ldapConfig = None
sslConfig = None
localConfig = None

ldapKwMatcher = CliMatcher.KeywordMatcher(
   'ldap',
   helpdesc='Configure management LDAP options for the switch' )

# util function to access or create an ldap host entry ( of type AaaPlugin::Host )
def ldapHost( label, hostnameOrIp, port, vrf, create=False ):
   hosts = ldapConfig.host
   assert vrf and vrf != ''
   spec = Tac.Value( "Aaa::HostSpec", hostname=hostnameOrIp, port=port,
                     acctPort=0, vrf=vrf )
   if spec in hosts:
      host = hosts[ spec ]
   elif create:
      t0( "Creating ldap host:", spec.hostname, ":", spec.port )
      host = hosts.newMember( spec )
      if host is None:
         t0( "Unable to create ldap host:", spec.hostname, ":", spec.port )
      else:
         host.index = AaaCliLib.getHostIndex( hosts )
         t0( spec.vrf, spec.hostname, "Host index", host.index )
   else:
      host = None

   if host is not None:
      assert host.hostname == hostnameOrIp
      assert host.port == port
   return host

def ldapConfigure( label, attribute, value ):
   if label == 'defaults':
      setattr( ldapConfig, attribute, value )
   elif label in ldapConfig.host:
      host = ldapConfig.host[ label ]
      setattr( host.serverConfig, attribute, value )
   else:
      t0( "Host not found", label )

def defaultServerDefaults( config ):
   config.sslProfile = ""
   config.baseDn = ""
   config.userRdnAttribute = ""
   config.activeGroupPolicy = ""
   config.searchUsernamePassword = Tac.Value( "Ldap::UsernamePassword", "", "" )

#-------------------------------------------
#  (config)#[ no | default ] management ldap
#-------------------------------------------
class LdapConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = 'Ldap configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, 'ldap' )

   def enterCmd( self ):
      return "management ldap"

class EnterLdapConfigMode( CliCommand.CliCommandClass ):
   syntax = """management ldap"""
   noOrDefaultSyntax = syntax

   data = { "management": ConfigMgmtMode.managementKwMatcher,
            "ldap": 'Configure LDAP options for AAA' }

   @staticmethod
   def handler( mode, args ):
      ldapConfig.defaultConfig = ( 'defaults', )
      childMode = mode.childMode( LdapConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'Clear all ldap hosts and set defaults' )
      ldapConfig.host.clear()
      ldapConfig.groupPolicy.clear()
      if ldapConfig.defaultConfig:
         defaultServerDefaults( ldapConfig.defaultConfig )
         ldapConfig.defaultConfig = None

BasicCli.GlobalConfigMode.addCommandClass( EnterLdapConfigMode )

#---------------------------------------------------------------
# (config-mgmt-ldap)# [ no | default ] group policy <POLICYNAME>
#---------------------------------------------------------------
groupKwMatcher = CliMatcher.KeywordMatcher(
   'group',
   helpdesc='Configure LDAP group options' )
policyKwMatcher = CliMatcher.KeywordMatcher(
   'policy',
   helpdesc='Configure LDAP group policy' )
policyNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( name for name in ldapConfig.groupPolicy ),
   helpname="WORD",
   helpdesc='Policy name',
   priority=CliParser.PRIO_LOW )

# Util class to manipulate sequences of group to rule mappings.
# fromSysdb() reads in config from Sysdb to object local list and dict.
# insert() / delete() modifies object local list and dict.
# toSysdb() writes out object local list and dict to Sysdb.
class GroupPrivilegeSeq( object ):
   def __init__( self, name, sysdbGroupPolicy ):
      self.groupPolicyName = name
      self.groupNamesInSeq = []
      self.groupToRolePrivilege = {}
      self.sysdbGroupPolicy = sysdbGroupPolicy
      # read in from Sysdb
      self.fromSysdb()

   def exists( self, groupName, role, privilege ):
      if self.groupToRolePrivilege.get( groupName ) == ( role, privilege ):
         return True
      return False
   # bgroupName = groupname to insert before
   # agroupName = groupname to insert after
   def insert( self, groupName, role, privilege, bgroupName, agroupName ):
      t0( "insert", groupName, role, privilege, bgroupName, agroupName )
      if bgroupName and bgroupName != groupName:
         try:
            self.groupNamesInSeq.insert( self.groupNamesInSeq.index( bgroupName ),
                                         groupName )
         except ValueError:  # bgroupName not found
            self.groupNamesInSeq.append( groupName )
      elif agroupName and agroupName != groupName:
         try:
            self.groupNamesInSeq.insert( self.groupNamesInSeq.index(
               agroupName ) + 1, groupName )
         except ValueError:  # agroupName not found
            self.groupNamesInSeq.append( groupName )
      else:
         #   ( not bgroupName and not agroupName ) or
         #   ( bgroupName == groupName and not agroupName ) or
         #   ( agroupName == groupName and not bgroupName )
         self.groupNamesInSeq.append( groupName )

      self.groupToRolePrivilege[ groupName ] = (
         role, privilege )
      # write out to Sysdb configuration
      self.toSysdb()

   def delete( self, groupName, role, privilege ):
      if role:
         if ( role, privilege ) != self.groupToRolePrivilege.get( groupName,
                                                                  ( None, None ) ):
            t0( "Given role and privilege, do not match any existing entry" )
            return
      if ( groupName not in self.groupNamesInSeq and
           groupName not in self.groupToRolePrivilege ):
         return
      if groupName in self.groupNamesInSeq:
         del self.groupNamesInSeq[ self.groupNamesInSeq.index( groupName ) ]
      if groupName in self.groupToRolePrivilege:
         del self.groupToRolePrivilege[ groupName ]
      # write out to Sysdb configurtion
      self.toSysdb()

   def fromSysdb( self ):
      for v in self.sysdbGroupPolicy.itervalues():
         self.groupNamesInSeq.append( v.group )
         self.groupToRolePrivilege[ v.group ] = ( v.role, v.privilege )

   def toSysdb( self ):
      self.sysdbGroupPolicy.clear()
      for groupName in self.groupNamesInSeq:
         val = self.groupToRolePrivilege.get( groupName )
         if not val:
            continue
         rP = Tac.Value( "Ldap::GroupRolePrivilege", group=groupName,
                         role=val[ 0 ], privilege=val[ 1 ] )
         self.sysdbGroupPolicy.enq( rP )

class GroupPolicyConfigMode( CliMode.Ldap.GroupPolicyConfigModeBase,
                             BasicCli.ConfigModeBase ):
   name = "Group policy configuration mode"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, pName=None ):
      CliMode.Ldap.GroupPolicyConfigModeBase.__init__( self, pName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.pName = pName
      sysdbGroupPolicy = ldapConfig.groupPolicy.newMember( pName )
      self.groupPolicy_ = GroupPrivilegeSeq( pName,
                                             sysdbGroupPolicy.groupRolePrivilege )

   def onExit( self ):
      sysdbGroupPolicy = ldapConfig.groupPolicy.get( self.pName )
      if ( sysdbGroupPolicy and not sysdbGroupPolicy.groupRolePrivilege and
           sysdbGroupPolicy.searchFilter == Tac.Value( "Ldap::ObjectClassOptions",
                                                       "", "" ) ):
         del ldapConfig.groupPolicy[ self.pName ]

      BasicCli.ConfigModeBase.onExit( self )

class EnterGroupPolicyConfig( CliCommand.CliCommandClass ):
   syntax = """group policy <POLICYNAME>"""
   noOrDefaultSyntax = syntax
   data = { 'group': groupKwMatcher,
            'policy': policyKwMatcher,
            "<POLICYNAME>": policyNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      assert "<POLICYNAME>" in args
      pName = args[ "<POLICYNAME>" ]
      childMode = mode.childMode( GroupPolicyConfigMode,
                                  pName=pName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      assert "<POLICYNAME>" in args
      pName = args[ "<POLICYNAME>" ]
      del ldapConfig.groupPolicy[ pName ]
      mode.groupPolicy_ = None

LdapConfigMode.addCommandClass( EnterGroupPolicyConfig )

#------------------------------------------------------------------------------------
# (config-mgmt-ldap-group-policy)# [ no | default ] group <GROUPNAME> role <ROLENAME>
# [ privilege <PRIVLEVEL> ]
#------------------------------------------------------------------------------------
roleKwMatcher = CliMatcher.KeywordMatcher(
   'role',
   helpdesc='specify role' )

privilegeKwMatcher = CliMatcher.KeywordMatcher(
   'privilege',
   helpdesc='specify privilege' )

beforeKwMatcher = CliMatcher.KeywordMatcher(
   'before',
   helpdesc='Add this rule immediately before <before-rule>' )
afterKwMatcher = CliMatcher.KeywordMatcher(
   'after',
   helpdesc='Add this rule immediately after <after-rule>' )

roleKwMatcher = CliMatcher.KeywordMatcher(
   'role',
   helpdesc='specify role' )

groupNameMatcher = CliMatcher.QuotedStringMatcher(
   helpname="Quoted String",
   helpdesc="Group name" )

roleNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( localConfig.role ), "Role name",
   pattern=roleNameRe, helpname="WORD" )

class GroupRole( CliCommand.CliCommandClass ):
   syntax = 'group <GROUPNAME> role <ROLENAME> [ privilege <PRIVLEVEL> ]' \
            '[ ( before <BGROUPNAME> ) | ( after <AGROUPNAME> ) ]'
   noOrDefaultSyntax = 'group <GROUPNAME> [ role <ROLENAME>' \
                       '[ privilege <PRIVLEVEL> ] ] ...'
   data = { 'group': groupKwMatcher,
            '<GROUPNAME>': groupNameMatcher,
            'role': roleKwMatcher,
            '<ROLENAME>': roleNameMatcher,
            'privilege': privilegeKwMatcher,
            '<PRIVLEVEL>': CliMatcher.IntegerMatcher( 0, 15,
                                                      helpdesc="Privilege level" ),
            'before': 'Add this rule immediately before <before-rule>',
            '<BGROUPNAME>': groupNameMatcher,
            'after': 'Add this rule immediately after <after-rule>',
            '<AGROUPNAME>': groupNameMatcher,
            }

   @staticmethod
   def handler( mode, args ):
      groupName = args[ '<GROUPNAME>' ]
      role = args[ '<ROLENAME>' ]
      privilege = args.get( '<PRIVLEVEL>', DEFAULT_ROLE_PRIVLEVEL )
      groupPolicy = ldapConfig.groupPolicy.get( mode.pName, None )
      bgroupName = args.get( '<BGROUPNAME>', None )
      agroupName = args.get( '<AGROUPNAME>', None )
      if not groupPolicy:
         t0( "Did not find group policy for ", mode.pName )
         return
      else:
         t0( groupName, role, privilege, bgroupName, agroupName )
         if mode.groupPolicy_.exists( groupName, role, privilege ):
            t0( "Matching entry already exists" )
            return
         else:
            mode.groupPolicy_.delete( groupName, None, None )
            mode.groupPolicy_.insert( groupName, role, privilege, bgroupName,
                                      agroupName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      groupName = args[ '<GROUPNAME>' ]
      role = args.get( '<ROLENAME>', None )
      privilege = args.get( '<PRIVLEVEL>', DEFAULT_ROLE_PRIVLEVEL )
      mode.groupPolicy_.delete( groupName, role, privilege )

GroupPolicyConfigMode.addCommandClass( GroupRole )

#------------------------------------------------------------------------------------
# (config-mgmt-ldap-group-policy)# [ no | default ] search filter objectclass
# <GROUPKWD> attribute <MEMBERKWD>
#------------------------------------------------------------------------------------
searchKwMatcher = CliMatcher.KeywordMatcher(
   'search',
   helpdesc='Configure search options' )

class SearchFilterObjectclass( CliCommand.CliCommandClass ):
   syntax = 'search filter objectclass <GROUPKWD> attribute <MEMBERKWD>'
   noOrDefaultSyntax = 'search filter objectclass ...'
   data = {
      "search": searchKwMatcher,
      "filter": "Configure search filter options",
      "objectclass": "Configure objectclass parameters",
      "<GROUPKWD>": CliMatcher.PatternMatcher(
         pattern=r'[A-Za-z0-9_:{}\[\]-]+',
         helpname="WORD",
         helpdesc="objectclass value" ),
      "attribute": CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'attribute',
         helpdesc="Configure search attribute for the objectclass" ) ),
      "<MEMBERKWD>": CliMatcher.PatternMatcher(
         pattern=r'[A-Za-z0-9_:{}\[\]-]+',
         helpname="WORD",
         helpdesc="attribute for objectclass" )
      }

   @staticmethod
   def handler( mode, args ):
      groupPolicy = ldapConfig.groupPolicy.get( mode.pName )
      if groupPolicy:
         groupPolicy.searchFilter = Tac.Value( "Ldap::ObjectClassOptions",
                                               args[ "<GROUPKWD>" ],
                                               args[ "<MEMBERKWD>" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      groupPolicy = ldapConfig.groupPolicy.get( mode.pName )
      if groupPolicy:
         groupPolicy.searchFilter = Tac.Value( "Ldap::ObjectClassOptions", "", "" )

GroupPolicyConfigMode.addCommandClass( SearchFilterObjectclass )

#-----------------------------------------------------------------------------------
# (config-mgmt-ldap)# [ no | default ] server defaults
# and
# (config-mgmt-ldap)# [ no | default ] server host <HOSTNAME> [ ( vrf <VRFNAME> ) ] \
#   [ ( port <PORT> ) ]
#------------------------------------------------------------------------------------
serverKwMatcher = CliMatcher.KeywordMatcher(
   'server',
   helpdesc='Configure LDAP options for the server' )

defaultsKwMatcher = CliMatcher.KeywordMatcher(
   'defaults',
   helpdesc='Configure default LDAP options for all servers' )

class ServerConfigMode( CliMode.Ldap.ServerConfigModeBase,
                        BasicCli.ConfigModeBase ):
   name = "Server configuration mode"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, label=None ):
      if isinstance( label, str ):
         param = label
      else:
         param = '%s-%s-%s' % ( label.hostname, label.vrf, label.port )
      CliMode.Ldap.ServerConfigModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.label = label
   def onExit( self ):
      BasicCli.ConfigModeBase.onExit( self )

class EnterServerConfig( CliCommand.CliCommandClass ):
   syntax = 'server ( defaults | ( host <HOSTNAME> ' \
            '[ { ( VRF ) | ( port <PORT> ) } ] ) )'
   noOrDefaultSyntax = 'server ( defaults | ( host ' \
                       '[ <HOSTNAME> [ VRF ] ' \
                       '[ port <PORT> ] ] ) ) ...'

   data = { 'server': serverKwMatcher,
            'defaults': defaultsKwMatcher,
            'host': 'Specify a LDAP server',
            '<HOSTNAME>': HostnameCli.IpAddrOrHostnameMatcher(
               helpname='WORD',
               helpdesc='Hostname or IP address of LDAP server',
               ipv6=True ),
            'VRF': VrfExprFactory( helpdesc='VRF for this LDAP server',
                                   maxMatches=1 ),
            'port':
            CliCommand.Node( CliMatcher.KeywordMatcher(
               'port',
               helpdesc='port of LDAP server ( default %s )' % (
                  LdapGroup.DEFAULT_LDAP_PORT ) ),
                             maxMatches=1 ),
            '<PORT>': CliMatcher.IntegerMatcher( 0, 65535,
               helpdesc='Number of the port to use' )
          }

   @staticmethod
   def handler( mode, args ):
      hostname = args.get( '<HOSTNAME>' )
      label = 'defaults'
      if hostname:
         vrf = args.get( 'VRF', DEFAULT_VRF )
         port = args.get( '<PORT>', LdapGroup.DEFAULT_LDAP_PORT )
         t0( 'setHost hostname:', hostname, "port:", port, 'vrf:', vrf )
         HostnameCli.resolveHostname( mode, hostname, doWarn=True )
         if isinstance( vrf, list ):
            assert len( vrf ) == 1
            vrf = vrf[ 0 ]
         assert vrf != ''
         if isinstance( port, list ):
            assert len( port ) == 1
            port = port[ 0 ]

         host = ldapHost( mode, hostname, port,
                          Tac.Value( "L3::VrfName", vrf ), create=True )

         label = host.spec
         host.serverConfig = ( str( label ), )
      else:
         ldapConfig.defaultConfig = ( label, )
      childMode = mode.childMode( ServerConfigMode,
                                  label=label )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "Set all attributes to default values" )
      hostname = args.get( '<HOSTNAME>' )
      hosts = ldapConfig.host
      defaults = args.get( 'defaults' )
      if defaults:
         if ldapConfig.defaultConfig:
            defaultServerDefaults( ldapConfig.defaultConfig )
            ldapConfig.defaultConfig = None
      elif hostname:
         vrf = args.get( 'VRF', DEFAULT_VRF )
         port = args.get( '<PORT>', LdapGroup.DEFAULT_LDAP_PORT )
         if isinstance( vrf, list ):
            assert len( vrf ) == 1
            vrf = vrf[ 0 ]
         assert vrf != ''
         if isinstance( port, list ):
            assert len( port ) == 1
            port = port[ 0 ]

         t0( 'noHost hostname:', hostname, "vrf: ", vrf, "port:", port )
         spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                           acctPort=0, vrf=vrf )
         if spec in hosts:
            del hosts[ spec ]
         else:
            if mode.session_.interactive_:
               warningMessage = "LDAP host %s with port %s not found" \
                                % ( hostname, port )
               mode.addWarning( warningMessage )
      else:
         # Delete all hosts since no hostname was specified
         hosts.clear()

LdapConfigMode.addCommandClass( EnterServerConfig )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] ssl-profile <profile-name>
# -------------------------------------------------------------------------------
class SslProfile( CliCommand.CliCommandClass ):
   syntax = """ssl-profile <profile-name>"""
   noOrDefaultSyntax = "ssl-profile ..."
   data = {
      "ssl-profile": "Configure SSL profile to use",
      "<profile-name>": CliMatcher.DynamicNameMatcher(
         lambda mode: ( name for name in sslConfig.profileConfig ),
         helpname="WORD",
         helpdesc="Profile name",
         priority=CliParser.PRIO_LOW )
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ '<profile-name>' ]
      t0( "Configured ssl-profile ", profileName )
      ldapConfigure( mode.label, 'sslProfile', profileName )
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "Unconfigured ssl profile" )
      ldapConfigure( mode.label, 'sslProfile', "" )

ServerConfigMode.addCommandClass( SslProfile )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] base-dn <WORD>
# -------------------------------------------------------------------------------
class BaseDN( CliCommand.CliCommandClass ):
   syntax = """base-dn <base-dn>"""
   noOrDefaultSyntax = "base-dn ..."
   data = {
      "base-dn": "Configure base Distinguished Name to use",
      "<base-dn>": CliMatcher.QuotedStringMatcher(
         helpdesc="base Distinguised Name",
         helpname="WORD" )
   }

   @staticmethod
   def handler( mode, args ):
      baseDn = args[ '<base-dn>' ]
      t0( "Configured base-DN", baseDn )
      ldapConfigure( mode.label, 'baseDn', baseDn )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "Unconfigured base-DN" )
      ldapConfigure( mode.label, 'baseDn', "" )

ServerConfigMode.addCommandClass( BaseDN )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] rdn attribute user <WORD>
# -------------------------------------------------------------------------------
class RdnUser( CliCommand.CliCommandClass ):
   syntax = """rdn attribute user <rdn-user>"""
   noOrDefaultSyntax = "rdn ..."
   data = {
      "rdn": "Configure Relative Distinguished Name to use",
      "attribute": "Configure attribute of RDN",
      "user": "Configure attribute user of RDN",
      "<rdn-user>": CliMatcher.StringMatcher( pattern=r'\w*',
                                              helpdesc="user RDN attribute",
                                              helpname="WORD" )
   }

   @staticmethod
   def handler( mode, args ):
      rdnUser = args[ '<rdn-user>' ]
      t0( "Configured relative-DN user", rdnUser )
      ldapConfigure( mode.label, 'userRdnAttribute', rdnUser )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "Unconfigured relative-DN user" )
      ldapConfigure( mode.label, 'userRdnAttribute', "" )

ServerConfigMode.addCommandClass( RdnUser )

# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] authorization group policy
# <POLICYNAME>
# -------------------------------------------------------------------------------
authzKwMatcher = CliMatcher.KeywordMatcher(
   'authorization',
   helpdesc='Configure LDAP options for user authorization' )

class AuthzGroupPolicy( CliCommand.CliCommandClass ):
   syntax = """authorization group policy <POLICYNAME>"""
   noOrDefaultSyntax = "authorization group policy ..."
   data = {
      "authorization": authzKwMatcher,
      "group": groupKwMatcher,
      "policy": policyKwMatcher,
      "<POLICYNAME>": policyNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      assert "<POLICYNAME>" in args
      pName = args[ "<POLICYNAME>" ]
      ldapConfigure( mode.label, 'defaultGroupPolicy' if mode.label == 'defaults'
                     else 'activeGroupPolicy', pName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "Unconfigured policy" )
      ldapConfigure( mode.label, 'defaultGroupPolicy' if mode.label == 'defaults'
                     else 'activeGroupPolicy', "" )

ServerConfigMode.addCommandClass( AuthzGroupPolicy )
# -------------------------------------------------------------------------------
# (config-mgmt-ldap-server-defaults)# [ no | default ] search username <USERNAME>
# password ( [ 0 <PASSWORD> ] | [ 7 <ENCRPASSWORD> ] | <PASSWORD> )
# -------------------------------------------------------------------------------
usernameMatcher = CliMatcher.QuotedStringMatcher( usernameRe,
                                                  helpname='<USERNAME>',
                                                  helpdesc='LDAP user name' )

class SearchUsername( CliCommand.CliCommandClass ):
   syntax = """search username <USERNAME> password <PASSWORD>"""
   noOrDefaultSyntax = "search username ..."
   data = {
      "search": searchKwMatcher,
      "username": "LDAP user name for search",
      "<USERNAME>": usernameMatcher,
      "password": "Configure password for the user",
      "<PASSWORD>": reversibleSecretCliExpression( "<PASSWORD>" )
   }
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      uname = args[ "<USERNAME>" ]
      assert uname
      pswd = args[ "<PASSWORD>" ]
      ldapConfigure( mode.label, 'searchUsernamePassword',
                     Tac.Value( "Ldap::UsernamePassword", uname, pswd ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldapConfigure( mode.label, 'searchUsernamePassword',
                     Tac.Value( "Ldap::UsernamePassword", "", "" ) )
ServerConfigMode.addCommandClass( SearchUsername )

def Plugin( entityManager ):
   global ldapConfig
   global sslConfig
   global localConfig
   ldapConfig = ConfigMount.mount( entityManager,
                                   'security/aaa/ldap/config',
                                   'Ldap::Config', 'w' )
   sslConfig = LazyMount.mount( entityManager,
                                'mgmt/security/ssl/config',
                                'Mgmt::Security::Ssl::Config', 'r' )
   localConfig = LazyMount.mount( entityManager,
                                  'security/aaa/local/config',
                                  'LocalUser::Config', 'r' )
