# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import BasicCli
import CliCommand
import CliMode.Launcher
import CliParser
import CliToken.Monitor
import CliToken.Reset
import CliToken.System
import ConfigMount
import LauncherModels
import LazyMount
import ShowCommand

lowMemStatus = None
lowMemConfig = None
lowMemRequest = None

#---------------------------------------------------------------
# The "show monitor system" command, in enable mode.
#---------------------------------------------------------------
class ShowMonitorSystem( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor system'
   data = { 'monitor': CliToken.Monitor.monitorMatcherForShow,
            'system': CliToken.System.systemMatcherForShow }
   cliModel = LauncherModels.MonitorSystemShow

   @staticmethod
   def handler( mode, args ):
      showMonitorSysModel = LauncherModels.MonitorSystemShow()
      showMonitorSysModel.featureEnabled = lowMemConfig.enable
      showMonitorSysModel.currLowMemModeStatus = lowMemStatus.status
      showMonitorSysModel.lowMemModeCount = lowMemStatus.lowMemModeCount
      showMonitorSysModel.lastLowMemModeEntered = Ark.switchTimeToUtc(
                                                 lowMemStatus.lastLowMemModeEntered )
      showMonitorSysModel.lastLowMemModeExited = Ark.switchTimeToUtc(
                                                  lowMemStatus.lastLowMemModeExited )
      return showMonitorSysModel

BasicCli.addShowCommandClass( ShowMonitorSystem )

#---------------------------------------------------------------
# The "reset system memory exhaustion" command, in enable mode.
#---------------------------------------------------------------
class ResetMemExhaustionCmd( CliCommand.CliCommandClass ):
   syntax = 'reset system memory exhaustion'
   data = { 'reset': CliToken.Reset.resetKwApi,
            'system': CliToken.System.systemMatcherForResetSystem,
            'memory': 'reset memory specific parameters',
            'exhaustion': 'exits system from hold-down state'
          }

   @staticmethod
   def handler( mode, args ):
      lowMemRequest.reset += 1

BasicCli.EnableMode.addCommandClass( ResetMemExhaustionCmd )

#---------------------------------------------------------------
# Enter config monitor system mode
# (config)# monitor system
# (config-monitor-system)#
#---------------------------------------------------------------
class MonitorSystemMode( CliMode.Launcher.MonitorSystemMode,
                         BasicCli.ConfigModeBase ):
   name = 'Monitor System Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.Launcher.MonitorSystemMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMonitorSystemMode( mode, args ):
   childMode = mode.childMode( MonitorSystemMode )
   mode.session_.gotoChildMode( childMode )

class ConfigureLowMemModeCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor system'
   data = { 'monitor': CliToken.Monitor.monitorMatcher,
            'system': 'system configuration commands'
          }

   handler = gotoMonitorSystemMode

BasicCli.GlobalConfigMode.addCommandClass( ConfigureLowMemModeCmd )

#---------------------------------------------------------------
# Enter config memory exhaustion mode from monitor system mode:
# (config-monitor-system)# memory exhaustion
# (config-monitor-system-memory)#
#---------------------------------------------------------------
class MemoryExhaustionMode( CliMode.Launcher.MemoryExhaustionMode,
                            BasicCli.ConfigModeBase ):
   name = 'Memory Exhaustion Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.Launcher.MemoryExhaustionMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMemoryExhaustionMode( mode, args ):
   childMode = mode.childMode( MemoryExhaustionMode )
   mode.session_.gotoChildMode( childMode )

class ConfigureMemExhaustionModeCmd( CliCommand.CliCommandClass ):
   syntax = 'memory exhaustion'
   noOrDefaultSyntax = 'memory exhaustion'
   data = { 'memory': 'memory management commands',
            'exhaustion': 'memory exhaustion handling commands'
          }

   handler = gotoMemoryExhaustionMode

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # running "no memory exhaustion" will configure everything under this mode to
      # default values
      lowMemConfig.enable = False

MonitorSystemMode.addCommandClass( ConfigureMemExhaustionModeCmd )

#---------------------------------------------------------------
# The "[no | default] action agent non-critical hold-down" command, in config mode.
#---------------------------------------------------------------
class AgentNonCriticalCmd( CliCommand.CliCommandClass ):
   syntax = 'action agent non-critical hold-down'
   noOrDefaultSyntax = 'action agent non-critical hold-down'
   data = { 'action': 'action to be taken',
            'agent': 'select agent type to take action on',
            'non-critical': 'agents considered non-critical to the system',
            'hold-down': 'prevent selected agents from restarting after going down',
          }

   @staticmethod
   def handler( mode, args ):
      lowMemConfig.enable = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      lowMemConfig.enable = False

MemoryExhaustionMode.addCommandClass( AgentNonCriticalCmd )

def Plugin( entityManager ):
   global lowMemStatus, lowMemConfig, lowMemRequest
   lowMemConfig = ConfigMount.mount( entityManager, "sys/config/lowMemConfig",
                                   "ProcMgr::LowMemoryModeConfig", "w" )
   lowMemStatus = LazyMount.mount( entityManager, "sys/status/lowMemStatus",
                                 "ProcMgr::LowMemoryModeStatus", "r" )
   lowMemRequest = LazyMount.mount( entityManager, "sys/config/lowMemRequest",
                                   "ProcMgr::LowMemoryModeRequest", "wf" )
