#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements interface-specific Lanz configuration.
#
#-------------------------------------------------------------------------------
'''Interface configuration commands for Lanz'''
import LanzIntfCli
import CliCommand
import CliToken.Lanz as Lanz

# Only includes the no and default handlers for 'queue-monitor length' command
#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor length
#--------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# queue-monitor length mirror
# [no | default] queue-monitor length mirror
#-------------------------------------------------------------------------------

class QueueMonitorLengthMirrorCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length mirror'
   noOrDefaultSyntax = 'queue-monitor length [ mirror ]'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'mirror' : Lanz.nodeMirror,
   }

   @staticmethod
   def handler( mode, args ):
      LanzIntfCli.qMonitorLengthMirror( mode )

   @staticmethod
   def noHandler( mode, args ):
      if 'mirror' in args:
         LanzIntfCli.noQMonitorLengthMirror( mode )
      else:
         LanzIntfCli.noQMonitor( mode, cpuOrFabric=None )

   @staticmethod
   def defaultHandler( mode, args ):
      if 'mirror' in args:
         LanzIntfCli.qMonitorLengthMirror( mode )
      else:
         LanzIntfCli.defaultQMonitor( mode, cpuOrFabric=None )

LanzIntfCli.modelet.addCommandClass( QueueMonitorLengthMirrorCmd )

#----------------------------------------------------------------------------------
# The "queue-monitor length thresholds 2-max_threshold 1-max_threshold" command
# Platforms which support low threshold as well would have this command
# enabled.
#----------------------------------------------------------------------------------

#----------------------------------------------------------------------------------
# The "queue-monitor length threshold 2-max_threshold command
# Platforms which don't support low threshold would have this command
# enabled.
#----------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# queue-monitor length ( threshold HIGH_THRESHOLD ) |
#                      ( thresholds HIGH_THRESHOLD LOW_THRESHOLD )
# [ no | default ] queue-monitor length ( threshold | thresholds ) ...
#--------------------------------------------------------------------------------

class QueueMonitorLengthThresholdsCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length ( threshold HIGH_THRESHOLD ) |\
                                  ( thresholds HIGH_THRESHOLD LOW_THRESHOLD )'
   noOrDefaultSyntax = 'queue-monitor length ( threshold | thresholds ) ...'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'threshold' : Lanz.nodeThreshold,
      'thresholds' : Lanz.nodeThresholds,
      'HIGH_THRESHOLD' : Lanz.matcherHighThreshold,
      'LOW_THRESHOLD' : Lanz.matcherLowThreshold,
   }

   @staticmethod
   def handler( mode, args ):
      highThreshold = args[ 'HIGH_THRESHOLD' ]
      if 'thresholds' in args:
         lowThreshold = args[ 'LOW_THRESHOLD' ]
         LanzIntfCli.configThresholds(
             mode, highThreshold, lowThreshold, cpuOrFabric=None )
      else:
         LanzIntfCli.configThreshold( mode, highThreshold, cpu=None )

   @staticmethod
   def noHandler( mode, args ):
      LanzIntfCli.noQMonitor( mode, cpuOrFabric=None )

   @staticmethod
   def defaultHandler( mode, args ):
      LanzIntfCli.setDefaultThresholds( mode, cpuOrFabric=None )

LanzIntfCli.modelet.addCommandClass( QueueMonitorLengthThresholdsCmd )
