#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCliModes
import CliCommand
import CliPlugin.LanzIntfCli as LanzIntfCli
import CliPlugin.LanzCli as LanzCli
import CliPlugin.EthIntfCli as EthIntfCli
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.VrfCli import VrfExprFactory
import CliToken.Lanz as Lanz
import CliMatcher
from Intf.IntfRange import IntfRangeMatcher

matcherIntfRange = IntfRangeMatcher( noSingletons=True,
                            explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType, ) )
nodeInterface = CliCommand.Node( matcher=EthIntfCli.EthPhyIntf.ethMatcher,
                                 guard=LanzCli.ethPortMirroringGuard )
nodeIntfRange = CliCommand.Node( matcher=matcherIntfRange,
                                 guard=LanzCli.ethPortMirroringGuard )

# Only the no & default handlers for 'queue-monitor length ( cpu | fabric )' command
#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor length ( cpu | fabric )
#--------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# queue-monitor length ( cpu | fabric ) thresholds HIGH_THRESHOLD LOW_THRESHOLD
# [ no | default ] queue-monitor length ( cpu | fabric ) [ thresholds ] ...
#--------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# queue-monitor length cpu threshold HIGH_THRESHOLD
# [ no | default ] queue-monitor length cpu [ threshold ] ...
#--------------------------------------------------------------------------------

class QueueMonitorLengthCpuOrFabricThresholdsCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length ( cpu ( threshold HIGH_THRESHOLD ) |\
                                       ( thresholds HIGH_THRESHOLD LOW_THRESHOLD ) )\
                              | ( fabric thresholds HIGH_THRESHOLD LOW_THRESHOLD )'
   noOrDefaultSyntax = 'queue-monitor length ( cpu [ threshold | thresholds ] ) |\
                                             ( fabric [ thresholds ] ) ...'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'cpu' : Lanz.nodeCpu,
      'fabric' : Lanz.nodeFabric,
      'threshold' : Lanz.nodeThreshold,
      'thresholds' : Lanz.nodeThresholds,
      'HIGH_THRESHOLD' : Lanz.matcherHighThreshold,
      'LOW_THRESHOLD' : Lanz.matcherLowThreshold,
   }

   @staticmethod
   def handler( mode, args ):
      cpuOrFabric = LanzIntfCli.isCpuOrFabric( args )
      highThreshold = args[ 'HIGH_THRESHOLD' ]
      if 'threshold' in args:
         LanzIntfCli.configThreshold( mode, highThreshold, cpu=cpuOrFabric )
      else:
         lowThreshold = args[ 'LOW_THRESHOLD' ]
         LanzIntfCli.configThresholds( mode,
                                       highThreshold, lowThreshold, cpuOrFabric )

   @staticmethod
   def noHandler( mode, args ):
      cpuOrFabric = LanzIntfCli.isCpuOrFabric( args )
      LanzIntfCli.noQMonitor( mode, cpuOrFabric )

   @staticmethod
   def defaultHandler( mode, args ):
      cpuOrFabric = LanzIntfCli.isCpuOrFabric( args )
      if 'threshold' or 'thresholds' in args:
         LanzIntfCli.setDefaultThresholds( mode, cpuOrFabric )
      else:
         LanzIntfCli.defaultQMonitor( mode, cpuOrFabric )

BasicCliModes.GlobalConfigMode.addCommandClass(
              QueueMonitorLengthCpuOrFabricThresholdsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor length
#--------------------------------------------------------------------------------

class QueueMonitorLengthCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length'
   noOrDefaultSyntax = syntax
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
   }

   @staticmethod
   def handler( mode, args ):
      LanzCli.queueMonitorLength( mode, forceEnabledValue=True )

   @staticmethod
   def noHandler( mode, args ):
      LanzCli.noQueueMonitorLength( mode, forceEnabledValue=False )

   defaultHandler = LanzCli.defaultQueueMonitorLength

BasicCliModes.GlobalConfigMode.addCommandClass( QueueMonitorLengthCmd )

#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor length ( notifying |
#                                         tx-latency |
#                                       ( update-interval UPDATE_INTERVAL ) |
#                                       ( log LOG ) )
#--------------------------------------------------------------------------------
class QueueMonitorLengthNotifyingCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length ( notifying | \
                                    tx-latency | \
                                  ( update-interval UPDATE_INTERVAL ) | \
                                  ( log LOG ) )'
   noOrDefaultSyntax = 'queue-monitor length ( notifying | \
                                               tx-latency | \
                                             ( update-interval ... ) | \
                                               log )'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'notifying' : Lanz.nodeNotifying,
      'tx-latency' : Lanz.nodeEnableTxLatency,
      'update-interval' : Lanz.nodeUpdateInterval,
      'UPDATE_INTERVAL' : Lanz.matcherUPDATEINTERVAL,
      'log' : 'Sys-logging of congestion events',
      'LOG' : CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='Time interval in seconds between two syslogs per interface' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'notifying' in args:
         LanzCli.qMonitorLengthNotifying( mode )
      elif 'tx-latency' in args:
         LanzCli.qMonitorLengthTxLatency( mode )
      elif 'update-interval' in args:
         qLenInterval = args[ 'UPDATE_INTERVAL' ]
         LanzCli.qMonitorLengthUpdateInterval( mode, qLenInterval )
      elif 'log' in args:
         logInterval = args[ 'LOG' ]
         LanzCli.qMonitorLengthLogInterval( mode, logInterval )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'notifying' in args:
         LanzCli.noQMonitorLengthNotifying( mode )
      elif 'tx-latency' in args:
         LanzCli.noQMonitorLengthTxLatency( mode )
      elif 'update-interval' in args:
         LanzCli.noQMonitorLengthUpdateInterval( mode )
      elif 'log' in args:
         LanzCli.noQMonitorLengthLogInterval( mode )

BasicCliModes.GlobalConfigMode.addCommandClass( QueueMonitorLengthNotifyingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor length default threshold THRESHOLD
#--------------------------------------------------------------------------------

class QueueMonitorLengthDefaultThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length DEFAULT (\
                                ( threshold HIGH_THRESHOLD ) | \
                                  thresholds HIGH_THRESHOLD LOW_THRESHOLD ) )'
   noOrDefaultSyntax = 'queue-monitor length DEFAULT ( ( threshold ... ) | \
                                                       ( thresholds ... ) )'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'DEFAULT' : Lanz.nodeDefault,
      'threshold' : Lanz.nodeDefaultThreshold,
      'thresholds' : Lanz.nodeDefaultThresholds,
      'HIGH_THRESHOLD' : Lanz.matcherHighThreshold,
      'LOW_THRESHOLD' : Lanz.matcherLowThreshold,
   }

   @staticmethod
   def handler( mode, args ):
      highThreshold = args[ 'HIGH_THRESHOLD' ]
      if 'threshold' in args:
         LanzCli.setEthernetThreshold( mode, highThreshold )
      elif 'thresholds' in args:
         lowThreshold = args[ 'LOW_THRESHOLD' ]
         LanzCli.setEthernetThresholds( mode, highThreshold, lowThreshold )

   noOrDefaultHandler = LanzCli.setDefaultEthernetThresholds

BasicCliModes.GlobalConfigMode.addCommandClass(
   QueueMonitorLengthDefaultThresholdCmd )

# Temporary disabling for Berlin release
# Beware of conflicts with the implementation of the above command
'''
#--------------------------------------------------------------------------------
# 'queue-monitor length DEFAULT ( \
#             ( threshold PERCENT_HIGH_THRESHOLD percentHighThreshold ) | \
#             ( thresholds PERCENT_HIGH_THRESHOLD percentHighThreshold \
#                          PERCENT_LOW_THRESHOLD percentLowThreshold ) )'
#--------------------------------------------------------------------------------

class QueueMonitorLengthDefaultThresholdPercenthighthresholdPercentCmd(
   CliCommand.CliCommandClass ):

   syntax = 'queue-monitor length DEFAULT ( \
             ( threshold PERCENT_HIGH_THRESHOLD percentHighThreshold ) | \
             ( thresholds PERCENT_HIGH_THRESHOLD percentHighThreshold \
                          PERCENT_LOW_THRESHOLD percentLowThreshold ) )'
   noOrDefaultSyntax = 'queue-monitor length DEFAULT ( ( threshold ... ) | \
                                                       ( thresholds ... ) )'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'DEFAULT' : Lanz.nodeDefault,
      'threshold' : Lanz.nodeDefaultThreshold,
      'thresholds' : Lanz.nodeDefaultThresholds,
      'PERCENT_HIGH_THRESHOLD' : Lanz.matcherPercentHighThreshold,
      'percentHighThreshold' : Lanz.matcherPercent,
      'PERCENT_LOW_THRESHOLD' : Lanz.matcherPercentLowThreshold,
      'percentLowThreshold' : Lanz.matcherPercent,
   }

   @staticmethod
   def handler( mode, args ):
      percentHighThreshold = args[ 'percentHighThreshold' ]
      if 'threshold' in args:
         LanzCli.qMonitorLengthDefaultThresholdPercent( mode, percentHighThreshold )
      elif 'thresholds' in args:
         percentLowThreshold = args[ 'percentLowThreshold' ]
         LanzCli.qMonitorLengthDefaultThresholdsPercent(
            mode, percentHighThreshold, percentLowThreshold )

   noOrDefaultHandler = LanzCli.noQMonitorLengthDefaultThreshold

BasicCliModes.GlobalConfigMode.addCommandClass(
  QueueMonitorLengthDefaultThresholdPercenthighthresholdPercentCmd )

#'''

#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor length global-buffer
#--------------------------------------------------------------------------------

class QueueMonitorLengthGlobalBufferCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length global-buffer'
   noOrDefaultSyntax = syntax

   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'global-buffer' : Lanz.nodeGlobalBuffer,
   }

   handler = LanzCli.qMonitorLengthGlobalBuffer

   noHandler = LanzCli.noQMonitorLengthGlobalBuffer

   defaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass(
   QueueMonitorLengthGlobalBufferCmd )

#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor length global-buffer ( log LOG ) | \
#                                     ( thresholds HIGH_THRESHOLD LOW_THRESHOLD )'
#--------------------------------------------------------------------------------

class QueueMonitorLengthGlobalBufferLogCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length global-buffer ( log LOG ) | \
                                       ( thresholds HIGH_THRESHOLD LOW_THRESHOLD )'
   noOrDefaultSyntax = 'queue-monitor length global-buffer ( log | thresholds ) ...'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'global-buffer' : Lanz.nodeGlobalBuffer,
      'log' : 'Sys-logging of global congestion events',
      'LOG' : CliMatcher.IntegerMatcher( 0, 65535,
       helpdesc='Time interval in seconds between two syslogs per interface' ),
      'thresholds' : Lanz.nodeGlobalBufferThresholds,
      'HIGH_THRESHOLD' : Lanz.matcherGlobalBufferHighThreshold,
      'LOW_THRESHOLD' : Lanz.matcherGlobalBufferLowThreshold,
   }

   @staticmethod
   def handler( mode, args ):
      if 'log' in args:
         logInterval = args[ 'LOG' ]
         LanzCli.qMonitorLengthGlobalLogInterval( mode, logInterval )
      elif 'thresholds' in args:
         highThreshold = args[ 'HIGH_THRESHOLD' ]
         lowThreshold = args[ 'LOW_THRESHOLD' ]
         LanzCli.setThresholds( mode, highThreshold, lowThreshold )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'log' in args:
         LanzCli.setDefaultGlobalLogInterval( mode )
      elif 'thresholds' in args:
         LanzCli.setDefaultThresholds( mode )

BasicCliModes.GlobalConfigMode.addCommandClass(
  QueueMonitorLengthGlobalBufferLogCmd )

#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor length mirror [ destination ( Cpu | INTERFACE |
# INTFS ) ]
#--------------------------------------------------------------------------------

class QueueMonitorLengthMirrorCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor length mirror [ destination \
                                         ( Cpu | INTERFACE | INTFS ) ]'
   noOrDefaultSyntax = 'queue-monitor length mirror'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'mirror' : Lanz.nodeMirror,
      'destination' : Lanz.matcherDestination,
      'Cpu' : Lanz.nodeCpuPorts,
      'INTERFACE' : nodeInterface,
      'INTFS' : nodeIntfRange,
   }

   @staticmethod
   def handler( mode, args ):
      if 'Cpu' in args:
         intfs = 'Cpu'
      elif 'INTERFACE' in args:
         intfs = args[ 'INTERFACE' ]
      elif 'INTFS' in args:
         intfs = args[ 'INTFS' ]
      else:
         intfs = False
      LanzCli.qMonitorLengthFrameMirroring( mode, False, intfs, False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      LanzCli.qMonitorLengthFrameMirroring(
                            mode, no=True, intfs=False, deleteDestAll=False )

BasicCliModes.GlobalConfigMode.addCommandClass( QueueMonitorLengthMirrorCmd )

#--------------------------------------------------------------------------------
# ( no | default ) queue-monitor length mirror destination [ ( Cpu | INTERFACE |
# INTFS ) ]
#--------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# ( no | default ) queue-monitor length mirror destination tunnel mode gre
#--------------------------------------------------------------------------------

class QueueMonitorLengthMirrorDestinationCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'queue-monitor length mirror destination \
                                                [ ( Cpu | INTERFACE | INTFS ) | \
                                                  ( tunnel mode gre ) ]'
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'mirror' : Lanz.nodeMirror,
      'destination' : Lanz.matcherDestination,
      'Cpu' : Lanz.nodeCpuPorts,
      'INTERFACE' : nodeInterface,
      'INTFS' : nodeIntfRange,
      'tunnel' : Lanz.nodeTunnel,
      'mode' : Lanz.matcherMode,
      'gre' : Lanz.matcherGre,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'tunnel' in args:
         LanzCli.noQMonitorLengthMirrorGreDestination( mode )
      else:
         intfs = None
         if 'Cpu' in args:
            intfs = 'Cpu'
         elif 'INTERFACE' in args:
            intfs = args[ 'INTERFACE' ]
         elif 'INTFS' in args:
            intfs = args[ 'INTFS' ]
         LanzCli.qMonitorLengthDeleteMirrorDest( mode, intfs )

BasicCliModes.GlobalConfigMode.addCommandClass(
   QueueMonitorLengthMirrorDestinationCmd )

#--------------------------------------------------------------------------------
# queue-monitor length mirror destination tunnel mode gre \
#                                        source SOURCE \
#                                        dest DESTINATION \
#                                        [ { ( ( ttl TTL ) | \
#                                              ( dscp DSCP ) | \
#                                              ( protocol PROTOCOL ) | \
#                                              ( vrf VRF ) ) } ]'
#--------------------------------------------------------------------------------

class QueueMonitorLengthMirrorDestinationTunnelModeGreSourceDestinationCmd(
      CliCommand.CliCommandClass ):

   syntax = '''queue-monitor length mirror DEST_KW tunnel mode gre
               source SOURCE destination DESTINATION
               [ { ( ttl TTL )
                 | ( dscp DSCP )
                 | ( protocol PROTOCOL )
                 | ( VRF ) } ]'''

   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'length' : Lanz.matcherLength,
      'mirror' : Lanz.nodeMirror,
      'DEST_KW' : Lanz.matcherDestination,
      'tunnel' : Lanz.nodeTunnel,
      'mode' : Lanz.matcherMode,
      'gre' : Lanz.matcherGre,
      'source' : 'Source keyword',
      'SOURCE' : IpAddrMatcher( helpdesc='Source IP address of GRE tunnel' ),
      'destination' : 'Destination keyword',
      'DESTINATION' : IpAddrMatcher(
                           helpdesc='Destination IP address of GRE tunnel' ),
      'ttl' : 'TTL of the GRE tunnel',
      'TTL' : CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL range' ),
      'dscp' : 'DSCP of the GRE tunnel',
      'DSCP' : CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP range' ),
      'protocol' : 'Protocol type in GRE header',
      'PROTOCOL' : CliMatcher.IntegerMatcher( 0, 65535,
                             helpdesc='Protocol range', helpname='0x0000-0xFFFF' ),
      'VRF' : VrfExprFactory( helpdesc='VRF name of the GRE tunnel' ),
   }

   @staticmethod
   def handler( mode, args ):
      greSrcAddr = args[ 'SOURCE' ]
      greDstAddr = args[ 'DESTINATION' ]

      greTunnelOpts = []
      opts = ( 'ttl', 'dscp', 'protocol', 'vrf' )
      for opt in opts:
         value = args.get( opt.upper() )
         if value:
            greTunnelOpts.append( [ opt, value[ 0 ] ] )

      LanzCli.qMonitorLengthMirrorGreDestination(
              mode, greSrcAddr, greDstAddr, greTunnelOpts, no=False )

BasicCliModes.GlobalConfigMode.addCommandClass(
   QueueMonitorLengthMirrorDestinationTunnelModeGreSourceDestinationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] queue-monitor streaming
#--------------------------------------------------------------------------------

class QueueMonitorStreamingCmd( CliCommand.CliCommandClass ):
   syntax = 'queue-monitor streaming'
   noOrDefaultSyntax = syntax
   data = {
      'queue-monitor' : Lanz.nodeQueueMonitor,
      'streaming' : Lanz.nodeStreaming,
   }

   handler = LanzCli.gotoStreamingMode

   noOrDefaultHandler = LanzCli.defaultConfig

BasicCliModes.GlobalConfigMode.addCommandClass( QueueMonitorStreamingCmd )
