#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import LazyMount
import ShowCommand
import Tac
import CliPlugin.LagCli as LagCli
import CliPlugin.RecircCli as RecircCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.VlanCli as VlanCli
from CliPlugin.LagModel import ( 
      LacpSysId, PortChannelLimits, LacpPort, LacpNeighbors, LacpAggregates, 
      LacpInternal, PortChannels, PortChannelsTraffic, RecircChannelLimits, 
      PortChannelsSummary, LacpCounters
)
from CliToken.PortChannel import portChannelMatcherForShow
from MultiRangeRule import MultiRangeMatcher
from Toggles.EbraToggleLib import togglePortChannelIdIncreaseEnabled

lagMemberIntfMatcher = IntfCli.Intf.matcherWOSubIntf
bridgingHwCapabilities = None

PortChannelNum = Tac.Type( "Lag::PortChannelNum" )
def portChannelRange():
   if bridgingHwCapabilities and bridgingHwCapabilities.extendedLagIdSupported and \
      togglePortChannelIdIncreaseEnabled():
      return ( PortChannelNum.min, PortChannelNum.extendedMax )
   return ( PortChannelNum.min, PortChannelNum.max )

def recircChannelRange():
   return ( PortChannelNum.min, PortChannelNum.max )

matcherActivePorts = CliMatcher.KeywordMatcher( 'active-ports', 
      helpdesc='Display info on ports that are active members of the LAG' )
matcherAggregates = CliMatcher.KeywordMatcher( 'aggregates', 
      helpdesc='Display info about LACP aggregates' )
matcherAllPortsUnselected = CliMatcher.KeywordMatcher( 'all-ports', 
      helpdesc='Display LACP info on all ports in LAG, even those not selected '
               ' by LACP to be part of aggregate' )
matcherAllPortsInactive = CliMatcher.KeywordMatcher( 'all-ports', 
      helpdesc='Display info on all ports configured for LAG, '
               'even those not currently active' )
matcherBriefRecirc = CliMatcher.KeywordMatcher( 'brief', 
      helpdesc='Display all recirc-channel status briefly' )
matcherBriefPortChannel = CliMatcher.KeywordMatcher( 'brief', 
      helpdesc='Display all port-channel status briefly' )
matcherBriefLacpStatus = CliMatcher.KeywordMatcher( 'brief', 
      helpdesc='Display relevant Link Aggregation Control Protocol (LACP) '
               'status briefly' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters', 
      helpdesc='Display LACP counters' )
matcherDetailedLacpStatus = CliMatcher.KeywordMatcher( 'detailed', 
      helpdesc='Display detailed Link Aggregation Control Protocol status' )
matcherDetailedPortChannel = CliMatcher.KeywordMatcher( 'detailed', 
      helpdesc='Display detailed port-channel status' )
matcherDetailedRecirc = CliMatcher.KeywordMatcher( 'detailed', 
      helpdesc='Display detailed recirc-channel status' )
matcherEtherchannel = CliMatcher.KeywordMatcher( 'etherchannel', 
      helpdesc='Synonym for show port-channel parameters' )
matcherEthernet = CliMatcher.KeywordMatcher( 'Ethernet', 
      helpdesc='Ethernet interface' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface', 
      helpdesc='Display per-interface LACP info' )
matcherInternal = CliMatcher.KeywordMatcher( 'internal', 
      helpdesc='Display information internal to the '
               'Link Aggregation Control Protocol' )
matcherLacp = CliMatcher.KeywordMatcher( 'lacp', 
      helpdesc='Link Aggregation Control Protocol (LACP) status' )
matcherLoopback = CliMatcher.KeywordMatcher( 'Loopback', 
      helpdesc='Hardware interface used for looping packets' )
matcherManagement = CliMatcher.KeywordMatcher( 'Management', 
      helpdesc='Management interface' )
matcherMatchInterface = CliMatcher.KeywordMatcher( 'match-interface', 
      helpdesc='Filter results by intf' )
matcherPeer = CliMatcher.KeywordMatcher( 'peer', 
      helpdesc='Display identity and info about LACP neighbor(s)' )
matcherPortChannel = CliMatcher.KeywordMatcher( 'Port-Channel', 
      helpdesc='Link Aggregation Group (LAG)' )
matcherRecircChannelIntf = CliMatcher.KeywordMatcher( 'Recirc-Channel', 
      helpdesc='Recirc-Channel Interface' )
matcherRecircChannelParams = CliMatcher.KeywordMatcher( 'recirc-channel', 
      helpdesc='Show recirc-channel parameters' )
matcherUnconnectedethernet = CliMatcher.KeywordMatcher( 'UnconnectedEthernet', 
      helpdesc='UnconnectedEthernet interface' )
matcherVlan = CliMatcher.KeywordMatcher( 'Vlan', 
      helpdesc='Logical interface into a VLAN' )
matcherChannelIds = MultiRangeMatcher(
      portChannelRange, False, 'Channel Group ID(s)' )
matcherRecircChannelIds = MultiRangeMatcher( 
      recircChannelRange, False, 'Recirculation Channel Group ID(s)' )
nodeEtherchannel = CliCommand.Node( matcher=matcherEtherchannel, 
      guard=LagCli.lagSupportedGuard,
      deprecatedByCmd='show port-channel' )
nodeLacp = CliCommand.Node( matcher=matcherLacp, 
      guard=LagCli.lagSupportedGuard )
nodeNeighborDeprecated = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'neighbor', 
      helpdesc='Display identity and info about LACP neighbor(s)' ),
      deprecatedByCmd='show lacp peer' )
nodePortChannel = CliCommand.Node( matcher=portChannelMatcherForShow, 
      guard=LagCli.lagSupportedGuard )
nodeRecircChannelParams = CliCommand.Node( matcher=matcherRecircChannelParams, 
      guard=LagCli.recircGuard )
nodeRecircChannelIntf = CliCommand.Node( matcher=matcherRecircChannelIntf, 
      guard=LagCli.recircGuard )
nodeUnconnectedethernet = CliCommand.Node( matcher=matcherUnconnectedethernet, 
      guard=EthIntfCli.ueGuard )
nodeVlan = CliCommand.Node( matcher=matcherVlan, 
      guard=VlanCli.bridgingSupportedGuard )

def getChannelGroupIdList( intfs ):
   if intfs is None:
      return None
   return map( LagCli.ChannelGroup, intfs.values() )

def getRecircGroupIdList( intfs ):
   if intfs is None:
      return None
   return map( LagCli.RecircChannelGroup, intfs.values() )

class EtherchannelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show etherchannel [ INTFS ] '
              '[ ( ( brief | detailed ) [ active-ports | all-ports ] ) | ' 
              '  ( ( active-ports | all-ports ) [ brief | detailed ] ) ] ' )
   data = {
      'etherchannel': nodeEtherchannel,
      'INTFS': matcherChannelIds, 
      'brief': matcherBriefPortChannel,
      'detailed': matcherDetailedPortChannel,
      'active-ports': matcherActivePorts,
      'all-ports': matcherAllPortsInactive,
   }
   handler = LagCli.doShowPortChannel
   cliModel = PortChannels

BasicCli.addShowCommandClass( EtherchannelCmd )

class LacpAggregatesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] aggregates '
              ' [ ( ( brief | detailed ) [ all-ports ] ) | '
              '   ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'aggregates': matcherAggregates,
      'brief': matcherBriefLacpStatus, 
      'detailed': matcherDetailedLacpStatus, 
      'all-ports': matcherAllPortsUnselected, 
   }
   handler = LagCli.doShowLacpAggregates
   cliModel = LacpAggregates

BasicCli.addShowCommandClass( LacpAggregatesCmd )

class LacpCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] counters '
              ' [ ( ( brief | detailed ) [ all-ports ] ) | '
              '   ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'counters': matcherCounters,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus, 
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpCounters
   cliModel = LacpCounters

BasicCli.addShowCommandClass( LacpCountersCmd )

class LacpInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] ' 
              '[ ( ( brief | detailed ) [ all-ports ] ) | ' 
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpPort
   cliModel = LacpPort
   hidden = True
   privileged = True

BasicCli.addShowCommandClass( LacpInterfaceCmd )

class LacpInterfaceCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] counters ' 
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'counters': matcherCounters,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpCounters
   cliModel = LacpCounters
   hidden = True
   privileged = True

BasicCli.addShowCommandClass( LacpInterfaceCountersCmd )

class LacpInterfaceInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] internal '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'internal': matcherInternal,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpInternal
   cliModel = LacpInternal
   hidden = True
   privileged = True

BasicCli.addShowCommandClass( LacpInterfaceInternalCmd )

class LacpInterfaceNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] neighbor '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'neighbor': nodeNeighborDeprecated,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpNeighbors
   cliModel = LacpNeighbors
   hidden = True
   privileged = True

BasicCli.addShowCommandClass( LacpInterfaceNeighborCmd )

class LacpInterfacePeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] peer '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'peer': matcherPeer,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpNeighbors
   cliModel = LacpNeighbors
   hidden = True
   privileged = True

BasicCli.addShowCommandClass( LacpInterfacePeerCmd )

class LacpInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] internal '  
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'internal': matcherInternal,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpInternal
   cliModel = LacpInternal

BasicCli.addShowCommandClass( LacpInternalCmd )

class LacpNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] neighbor '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'neighbor': nodeNeighborDeprecated,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpNeighbors
   cliModel = LacpNeighbors

BasicCli.addShowCommandClass( LacpNeighborCmd )

class LacpPeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] peer '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'peer': matcherPeer,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus, 
      'all-ports': matcherAllPortsUnselected,
   }
   handler = LagCli.doShowLacpNeighbors
   cliModel = LacpNeighbors

BasicCli.addShowCommandClass( LacpPeerCmd )

class LacpSysIdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show lacp sys-id [ ( brief | detailed ) ]'
   data = {
      'lacp': nodeLacp,
      'sys-id': 'Display System Identifier used by LACP',
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
   }
   handler = LagCli.doShowLacpSysId
   cliModel = LacpSysId

BasicCli.addShowCommandClass( LacpSysIdCmd )

class LagInitializationTimeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show lag initialization-time'
   data = {
      'lag': 'Link Aggregation Agent (LAG)',
      'initialization-time': 'Display time spent in LagAgent initialization',
   }
   handler = LagCli.doShowLagInitializationTime
   hidden = True

BasicCli.addShowCommandClass( LagInitializationTimeCmd )

class PortChannelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show port-channel [ INTFS ] ' 
              '[ ( ( brief | detailed ) [ active-ports | all-ports ] ) | '
              '  ( ( active-ports | all-ports ) [ brief | detailed ] ) ] ' )
   data = {
      'port-channel': nodePortChannel,
      'INTFS': matcherChannelIds,
      'brief': matcherBriefPortChannel,
      'detailed': matcherDetailedPortChannel,
      'active-ports': matcherActivePorts, 
      'all-ports': matcherAllPortsInactive,
   }
   handler = LagCli.doShowPortChannel
   cliModel = PortChannels

BasicCli.addShowCommandClass( PortChannelCmd )

class PortChannelDenseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel dense'
   data = {
      'port-channel': nodePortChannel,
      'dense': 'Display summary of port-channel status',
   }
   handler = LagCli.doShowPortChannelSummary
   cliModel = PortChannelsSummary

BasicCli.addShowCommandClass( PortChannelDenseCmd )

class PortChannelLimitsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel limits'
   data = {
      'port-channel': nodePortChannel,
      'limits': 'Display limits on port-channel membership',
   }
   handler = LagCli.doShowPortChannelLimits
   cliModel = PortChannelLimits

BasicCli.addShowCommandClass( PortChannelLimitsCmd )

class PortChannelLoadBalanceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel [ INTFS ] load-balance'
   data = {
      'port-channel': nodePortChannel,
      'INTFS': matcherChannelIds,
      'load-balance': 'Display traffic distribution of port-channel members',
   }
   handler = LagCli.doShowPortChannelTraffic
   cliModel = PortChannelsTraffic

BasicCli.addShowCommandClass( PortChannelLoadBalanceCmd )

class PortChannelSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel summary'
   data = {
      'port-channel': nodePortChannel,
      'summary': 'Display summary of port-channel status',
   }
   handler = LagCli.doShowPortChannelSummary
   cliModel = PortChannelsSummary

BasicCli.addShowCommandClass( PortChannelSummaryCmd )

class PortChannelTrafficCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel [ INTFS ] traffic'
   data = {
      'port-channel': nodePortChannel,
      'INTFS': matcherChannelIds,
      'traffic': 'Display traffic distribution of port-channel members',
   }
   handler = LagCli.doShowPortChannelTraffic
   cliModel = PortChannelsTraffic

BasicCli.addShowCommandClass( PortChannelTrafficCmd )

class RecircChannelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show recirc-channel [ INTFS ] '
              '[ ( ( brief | detailed ) [ active-ports | all-ports ] ) | '
              '  ( ( active-ports | all-ports ) [ brief | detailed ] ) ] ' )
   data = {
      'recirc-channel': nodeRecircChannelParams,
      'INTFS': matcherRecircChannelIds, 
      'brief': matcherBriefRecirc, 
      'detailed': matcherDetailedRecirc, 
      'active-ports': matcherActivePorts, 
      'all-ports': matcherAllPortsInactive, 
   }
   handler = RecircCli.doShowRecircChannel
   cliModel = PortChannels

BasicCli.addShowCommandClass( RecircChannelCmd )

class RecircChannelLimitsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show recirc-channel limits'
   data = {
      'recirc-channel': nodeRecircChannelParams,
      'limits': 'Display limits on recirc-channel membership',
   }
   handler = RecircCli.doShowRecircChannelLimits
   cliModel = RecircChannelLimits

BasicCli.addShowCommandClass( RecircChannelLimitsCmd )

class RecircChannelTrafficCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show recirc-channel [ INTFS ] traffic'
   data = {
      'recirc-channel': nodeRecircChannelParams,
      'INTFS': matcherRecircChannelIds, 
      'traffic': 'Display traffic distribution of recirc-channel members',
   }
   handler = RecircCli.doShowRecircChannelTraffic
   cliModel = PortChannelsTraffic

BasicCli.addShowCommandClass( RecircChannelTrafficCmd )

def Plugin( entityManager ):
   global bridgingHwCapabilities

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
