#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import time
import textwrap
import Ark
import Arnet
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
import IntfCli
import IntfModel
from LacpLib import printList
import Tac
from TableOutput import TableFormatter
from TableOutput import Headings
from TableOutput import Format
from Toggles.LagToggleLib import toggleLacpTimeoutMultiplierEnabled

PortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" )

# Convert Tac.now() time to UTC time
def toUtc( timestamp ):
   utcTime = timestamp + Tac.utcNow() - Tac.now() if timestamp else 0.0
   return utcTime

# Convert UTC time to string representation. If relative is set give relative
# time
def utcToStr( utcThen ):
   timeDiff = Tac.utcNow() - utcThen
   tacTimeThen = Tac.now() - timeDiff if utcThen else 0
   return Ark.timestampToStr( tacTimeThen, False )

def timeString( utcThen, neverString="Never" ):
   if utcThen == 0.0:
      return( neverString )
   elif utcThen < 0.0:
      return( 'Unknown' )
   now = time.localtime()
   then = time.localtime( utcThen )
   elapsed = Tac.utcNow() - utcThen
   tstr = "%d:%02d:%02d" % ( then.tm_hour, then.tm_min, then.tm_sec )
   if(( elapsed < 6*24*60*60 ) and ( now.tm_mday != then.tm_mday )):
      weekday = [ 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun',  ]
      tstr = weekday[ then.tm_wday ] + " " + tstr
   elif( now.tm_year != then.tm_year or
         now.tm_mon != then.tm_mon or
         now.tm_mday != then.tm_mday ):
      tstr = ("%d/%d/%02d " % (then.tm_mon, then.tm_mday, (then.tm_year%100))
                + tstr )
   return( tstr )

def generateFlagKey():
   # Print Flags Key Table
   t = TableFormatter()
   f = Format( justify="left" ) 
   headings = ( "Flags", "" )
   th = Headings( headings )
   th.doApplyHeaders( t )
   t.startRow( )
   t.newFormattedCell( "a - LACP Active", format=f )
   t.newFormattedCell( "p - LACP Passive", format=f )
   t.newFormattedCell( "* - static fallback", format=f )
   t.startRow( )
   t.newFormattedCell( "F - Fallback enabled", format=f )
   t.newFormattedCell( "f - Fallback configured", format=f )
   t.newFormattedCell( "^ - individual fallback", format=f )
   t.startRow( )
   t.newFormattedCell( "U - In Use", format=f )
   t.newFormattedCell( "D - Down", format=f )
   t.startRow( )
   t.newFormattedCell( "+ - In-Sync ", format=f )
   t.newFormattedCell( "- - Out-of-Sync", format=f )
   t.newFormattedCell( "i - incompatible with agg", format=f )
   t.startRow( )
   t.newFormattedCell( "P - bundled in Po", format=f )
   t.newFormattedCell( "s - suspended", format=f )
   t.newFormattedCell( "G - Aggregable", format=f )
   t.startRow( )
   t.newFormattedCell( "I - Individual", format=f )
   t.newFormattedCell( "S - ShortTimeout", format=f )
   t.newFormattedCell( "w - wait for agg", format=f )
   inactiveFlagExplanation = ( "  E - Inactive. The number of configured port "
                               "channels exceeds the config limit\n" )
   return t.output( ) + inactiveFlagExplanation


# Display info shared by show lacp internal and show lacp interface
def portHeadings( brief, start ):
   headings = []
   if start:
      headings = [ ( "Port", "lh" ), ]
      if brief:
         headings += [ ( "Status", "lhb" ), ]
      else:
         headings += [ ( "Status", "lhb", ( ( "+ = h/w", "c" ), ) ),
                       ( "Select", "lb" ), ]
      headings += [ ( "Partner", "l", ( ( "Sys-id", "l" ), ) ) ]
   if brief:
      subheadings = ( ( "Port#", ( "State", "l" ),
                        "OperKey",
                      ( "PortPriority", "r" ) ) )
   else:
      subheadings = ( ( "Port#", ( "State", "l" ),
                        "OperKey",
                        "AdminKey",
                      ( "PortPriority", "r" ) ) )
   headings += [ ( "Actor", "c", subheadings ), ]
   if not brief:
      headings += [ ( "Churn", "l" ),
                    ( "Last", ( ( "RxTime", "c" ), ) ),
                    ( "State Machines",
                      ( ( "Rx", "l" ),
                      ( "mux", "l" ) ) ),
                    ( "MuxReason", "l" ) ]
   if toggleLacpTimeoutMultiplierEnabled():
      headings += [ "TimeoutMultiplier" ]
   return( headings )


# Display info shared by show lacp neighbor and show lacp interface
def neighborHeadings( brief, start ):
   headings = []
   if start:
      headings = [ ( "Port", "lh" ), ( "Status", "lhb" ), ]
      if not brief:
         headings += [ ( "Select", "lb" ), ]
   subheadings = ( "Port#",
                   ( "State", "l" ),
                     "OperKey",
                   ( "PortPri", "r" ),)
   if start:
      subheadings = ( ( "Sys-id", "l" ), ) + subheadings
   headings += [ ( "Partner", "c", subheadings ) ]
   if not brief:
      headings += [ ( "Partner", "l", ( ( "Churn", "l" ), ) ),
                    ( "Collector", ( ( "MaxDelay" ), ) ) ]
   return( headings )


def describeLacpState( brief ):
   print "State: A = Active, P = Passive; S=ShortTimeout, L=LongTimeout;"
   print "       G = Aggregable, I = Individual; s+=InSync, s-=OutOfSync;"
   desc1 = ( ( not brief==True ) and " (aggregating incoming frames)" ) or ""
   desc2 = " (aggregating outgoing frames),\n" + " " * len( "State:" )
   desc2 = ( ( not brief==True ) and desc2 ) or ","
   print "       C = Collecting%s, X = state machine expired," % desc1
   print "       D = Distributing%s d = default neighbor state" % desc2


class LacpAnnotationMarkerSet( Model ):
   markers = List( valueType=str,
                   help="Set of LACP annotation markers" )

   def toStr( self ):
      return "".join( self.markers )


class LacpMarkerMessage( Model ):
   marker = Str( help="LACP annotation marker" )
   msg = Str( help="LACP annotation marker message" )

   def render( self ):
      lines = textwrap.wrap( self.msg, 70 )
      print "%s - %s" % ( self.marker, lines[ 0 ] )
      for line in lines[ 1: ]:
         print "    %s" % line


class LacpMarkerMessages( Model ):

   markerMessages = List( valueType=LacpMarkerMessage,
                          help="LACP annotation markers and messages" )

   def render( self ):
      for markerMsg in self.markerMessages:
         markerMsg.render()


lacpPortStatusMap = {
    "idle"        :   "LACP idle",
    "noAgg"       :   "No Aggregate",
    "bundled"     :   "Bundled",
    "standby"     :   "Standby",
    "mismatchAgg" :   "Mismatched Agg!",
    "negotiation" :   "In Negotiation",
    "unselected"  :   "Unselected",
    "unknown"     :   "Unknown"
   }

class LacpPortStatus( Model ):
   status = Enum( values=[ 'idle',
                           'noAgg',
                           'bundled',
                           'standby',
                           'mismatchAgg',
                           'negotiation',
                           'unselected',
                           'unknown' ],
                  help="LACP port status" )

def lacpPortStatusStr( status ):
   return lacpPortStatusMap[ status ]

class LacpModeOptional( Model ):
   mode = Enum( values=[ 'off',
                         'passive',
                         'active' ],
                help='LACP active or passive mode',
                optional=True )

def lacpModeFlag( mode ):
   if mode == 'active':
      return 'a'
   elif mode == 'passive':
      return 'p'
   else:
      return 'N/A'

def lacpModeStr( mode ):
   if mode == 'active':
      return 'Active'
   elif mode == 'passive':
      return 'Passive'
   else:
      return 'N/A'


class FallbackTypeOptional( Model ):
   mode = Enum( values=[ 'fallbackStatic',
                         'fallbackIndividual' ],
                help='Lag fallback type',
                optional=True )

def fallbackTypeFlag( mode ):
   return '*' if mode == 'fallbackStatic' else '^'

class LagMode( Model ):
   mode = Enum( values=[ 'unconfigured',
                         'static',
                         'lacp' ],
                help='Lag mode configuration' )

def lagModeStr( mode ):
   modeStr = None
   if mode == 'static':
      modeStr = 'Static'
   elif mode == 'lacp':
      modeStr = 'LACP'
   else:
      modeStr = 'Unconfigured'
   return modeStr

class LagModeOptional( Model ):
   mode = Enum( values=[ 'unconfigured',
                         'static',
                         'lacp' ],
                help='Lag mode configuration',
                optional=True )


class FallbackStateOptional( Model ):
   state = Enum( values=[ 'unconfigured',
                          'configured',
                          'initialized',
                          'passive',
                          'monitoring',
                          'enabled',
                          'locallyOnly' ],
                 help="Port Channel fallback state",
                 optional=True )

def fallbackStateStr( state ):
   if state == 'locallyOnly':
      return 'Locally Only'
   else:
      return state.capitalize()

class AggregatorSelectedState( Model ):
   state = Enum( values=[ 'selectedStateSelected',
                          'selectedStateUnselected',
                          'selectedStateStandby' ],
                 help="Interface aggregator selection state" )

def aggregatorSelectedStateStr( state ):
   return str( state )[ len( 'selectedState' ): ]

class LacpPortChurnState( Model ):
   state = Enum( values=[ 'churnMonitor',
                          'noChurn',
                          'churnDetected' ],
                 help="LACP port churn state" )

def lacpPortChurnStateStr( state ):
   return str( state )

class LacpRxSmState( Model ):
   state = Enum( values=[ 'rxSmCurrent',
                          'rxSmExpired',
                          'rxSmDefaulted',
                          'rxSmInitialize',
                          'rxSmLacpDisabled',
                          'rxSmPortDisabled' ],
                 help="LACP Receive state machine state" )

def lacpRxSmStateStr( state ):
   return str( state )[ len( 'rxSm' ): ]

class LacpMuxSmState( Model ):
   state = Enum( values=[ 'muxSmDetached',
                          'muxSmWaiting',
                          'muxSmAttached',
                          'muxSmCollecting',
                          'muxSmDistributing',
                          'muxSmCollectingDistributing',
                          'muxSmDisablingInProgress' ],
                 help="LACP MUX state machine state" )

def lacpMuxSmStateStr( state ):
   return str( state )[ len( 'muxSm' ): ]


class LacpPortState( Model ):
   activity = Bool( help="LACP active or passive" )
   timeout = Bool( help="LACP long (False) or short (True) timeout" )
   aggregation = Bool( help="Link aggregateable or individual" )
   synchronization = Bool( help="Link in sync with the correct LAG" )
   collecting = Bool( help="Collection of incoming frames enabled" )
   distributing = Bool( help="Distribution of outgoing frames enabled" )
   defaulted = Bool( help="Using defaulted operational partner information" )
   expired = Bool( help="Receive machine in expired state" )

   def toStr( self ):
      s = ""
      s += self.activity and "A" or "P"
      s += self.timeout and "S" or "L"
      s += self.aggregation and "G" or "I"
      s += self.synchronization and "s+" or "s-"
      s += self.collecting and "C" or ""
      s += self.distributing and "D" or ""
      s += self.defaulted and "d" or ""
      s += self.expired and "X" or ""
      return s

#-------------------------------------------------------------------------------
#
# cmd: show port-channel limits
#
#-------------------------------------------------------------------------------

class LagGroupLimits( Model ):
   maxPortChannels = Int( help="Maximum number of active port-channels." )
   maxPortsPerPortChannel = Int( help='Maximum number of active ports'
                                 ' allowed in a port-channel.' )
   portList = List( valueType=IntfModel.Interface,
                    help="Compatible ports." )


class PortChannelLimits( Model ):
   # This collection is keyed by lagGroup name
   portChannelLimits = Dict( valueType=LagGroupLimits,
                             help='Map a LAG Group name to the ports it applies to, '
                             'and the hardware lag limits applicable to them.' )

   def render( self ):
      for lagGroup, limits in sorted( self.portChannelLimits.iteritems() ):
         print "LAG Group: %s" % lagGroup
         print 74*'-'
         fmt = "  Max port-channels per group: %d, Max ports per port-channel: %d"
         print fmt % ( limits.maxPortChannels, limits.maxPortsPerPortChannel )
         printList( limits.portList,
                    "  %d compatible ports: " % len( limits.portList ), 74)
         print 74*'-'
         print


class RecircChannelLimits( Model ):
   recircChannelLimits = PortChannelLimits.portChannelLimits

   def render( self ):
      for lagGroup, limits in sorted( self.recircChannelLimits.iteritems() ):
         print "LAG Group: %s" % lagGroup
         print 74*'-'
         fmt = "  Max recirc-channels per group: %d, Max ports per "\
               "recirc-channel: %d"
         print fmt % ( limits.maxPortChannels, limits.maxPortsPerPortChannel )
         printList( limits.portList,
                    "  %d compatible ports: " % len( limits.portList ), 74)
         print 74*'-'
         print


#-------------------------------------------------------------------------------
#
# cmd: show port-channel dense
#
#-------------------------------------------------------------------------------

class LacpMisconfig( Model ):
   status = Enum( values=[ 'idle',
                           'noAgg',
                           'bundled',
                           'standby',
                           'mismatchAgg',
                           'negotiation',
                           'unknown' ],
                  help="LACP misconfiguration flags" )
   def toFlag( self ):
      flag = ''
      if self.status == 'mismatchAgg':
         flag = 'i'
      elif self.status == 'negotiation':
         flag = 'w'
      return flag

class PortChannelMemberSummary( Model ):
   intf = IntfModel.Interface( help="Interface name" )
   lagMember = Bool( help="Active member of the LAG" )
   linkDown = Bool( help="Is the interface down?" )
   suspended = Bool( help="Interface is not active in the "
                          "Port-Channel (wrong speed)" )
   lacpMisconfig = Submodel( valueType=LacpMisconfig,
                             help="LACP misconfiguration state",
                             optional=True )
   lacpState = Submodel( valueType=LacpPortState,
                         help="LACP port state",
                         optional=True )
   staticLag = Bool( help="Port is configured in a static LAG" )

   def toFlags( self ):
      flag = ''
      if self.linkDown:
         # Check to see if the interface is on, if not print D and exit
         flag += 'D'
         return flag
      if self.suspended:
         # If the link is suspended, don't print any other flags 
         flag += 's'
         return flag
      if self.lagMember:
         flag += 'P'
      if self.staticLag:
         # Leave at this point if Static because we don't want lacp flags
         return flag
      else:
         if self.lacpMisconfig:
            flag += self.lacpMisconfig.toFlag()
         if self.lacpState:
            sflags = self.lacpState.toStr()
            pattern = "S|G|I|\+|-"
            allMatches = re.finditer( pattern, sflags )
            for match in allMatches:
               flag += match.group(0)
      return flag

class FallbackConfigStatus( Model ):
   config = FallbackTypeOptional.mode
   status = FallbackTypeOptional.mode

   def toFlag( self ):
      flag = ''
      if self.config:
         flag = 'f' + fallbackTypeFlag( self.config )
      elif self.status:
         flag = 'F' + fallbackTypeFlag( self.status )
      return flag


class PortChannelSummary( Model ):
   linkState = Enum( values=( 'up', 'down' ),
                     help="state - port channels is up or down" )
   protocol = LagMode.mode
   fallback = Submodel( valueType=FallbackConfigStatus,
                        help="Lag fallback status",
                        optional=True )
   lacpMode = LacpModeOptional.mode
   ports = Dict( keyType=IntfModel.Interface, 
                 valueType=PortChannelMemberSummary,
                 help="Interfaces in port channel" )
   inactive = Bool( help="Exceeds config limit" )

class PortChannelsSummary( Model ):
   numberOfChannelsInUse = Int( help="Number of channels in use" )
   numberOfAggregators = Int( help="Number of aggregators" )
   portChannels = Dict( keyType=IntfModel.Interface,
                        valueType=PortChannelSummary,
                        help='Port-channels summary' )

   def render( self ):
      def portInfoToFlag( info ):
         if info.inactive:
            return "E"
         if info.linkState == "up":
            return "U"
         return "D"

      print
      print generateFlagKey()
      print 'Number of channels in use:', self.numberOfChannelsInUse
      print 'Number of aggregators:', self.numberOfAggregators
      print
      t = TableFormatter()
      headers = ( ( "Port-Channel", "l" ),
                  ( "Protocol", "l" ),
                  ( "Ports", "l" ) )
      th = Headings( headers )
      th.doApplyHeaders( t )
      for port in Arnet.sortIntf( self.portChannels ):
         portInfo = self.portChannels[ port ]
         t.startRow()
         members = ''
         getNewLine = 0
         for m in Arnet.sortIntf( portInfo.ports ):
            portSummary = portInfo.ports[ m ]
            getNewLine += 1
            shortName = IntfCli.Intf.getShortname( portSummary.intf )
            members = members + shortName + '(' + portSummary.toFlags() + ')' + ' '
            if getNewLine % 4 == 0:
               members += '\n'
         portName = IntfCli.Intf.getShortname( port )
         portFlag = portName + '(' + portInfoToFlag( portInfo ) + ')'
         protocolFlag = lagModeStr( portInfo.protocol )
         if portInfo.lacpMode or portInfo.fallback:
            protocolFlag += '('
            if portInfo.lacpMode:
               protocolFlag += lacpModeFlag( portInfo.lacpMode )
            if portInfo.fallback:
               protocolFlag += portInfo.fallback.toFlag()
            protocolFlag += ')'
         t.newCell( portFlag , protocolFlag, members )
      print t.output()


#-------------------------------------------------------------------------------
#
# cmd: show port-channel [MEMBERS] traffic
#
#-------------------------------------------------------------------------------

# per port-channel member traffic
class InterfaceTraffic( Model ):
   inUcastPkts = Float( help='Unicast packets received on this physical port, '
                             'expressed as a percentage of total port-channel '
                             'unicast packets.' )
   outUcastPkts = Float( help='Unicast packets sent on this physical port, '
                              'expressed as a percentage of total port-channel '
                              'unicast packets.' )
   inMulticastPkts = Float( help='Multicast packets received on this physical port, '
                              'expressed as a percentage of total port-channel '
                              'multicast packets.' )
   outMulticastPkts = Float( help='Multicast packets sent on this physical port, '
                             'expressed as a percentage of total port-channel '
                             'multicast packets.' )
   inBroadcastPkts = Float( help='Broadcast packets received on this physical port, '
                             'expressed as a percentage of total port-channel '
                             'broadcast packets.' )
   outBroadcastPkts = Float( help='Broadcast packets sent on this physical port, '
                             'expressed as a percentage of total port-channel '
                             'broadcast packets.' )

# per port-channel traffic
class PortChannelTraffic( Model ):
   interfaceTraffic = Dict(
      keyType=IntfModel.Interface,
      valueType=InterfaceTraffic,
      help='Map from physical interface name to packet categories.' )

# Traffic for all port-channels
class PortChannelsTraffic( Model ):
   portChannels = Dict(
      keyType=IntfModel.Interface,
      valueType=PortChannelTraffic,
      help='Map the port-channel to interface traffic distribution.' )

   def render( self ):
      cntAttrList = [ "inUcastPkts", "outUcastPkts", "inMulticastPkts",
                      "outMulticastPkts", "inBroadcastPkts", "outBroadcastPkts"]
      dashedLine = "------ --------- ------- ------- ------- ------- ------- -------"
      print "ChanId      Port Rx-Ucst Tx-Ucst Rx-Mcst Tx-Mcst Rx-Bcst Tx-Bcst"
      print dashedLine

      skipOnFirstLag = True
      for portChannelName in Arnet.sortIntf( self.portChannels ):
         portChannel = self.portChannels[ portChannelName ]
         if skipOnFirstLag:
            skipOnFirstLag = False
         else:
            print dashedLine
         for intfName in Arnet.sortIntf( portChannel.interfaceTraffic ):
            intfCounters = portChannel.interfaceTraffic[ intfName ]
            print "%6s %9s" % ( PortChannelIntfId.portChannel( portChannelName ),
                                IntfCli.Intf.getShortname( intfName ) ),
            for key in cntAttrList:
               print "%6.2f%1c" % ( getattr( intfCounters, key ), '%' ),
            print # End of Op for intf


#-------------------------------------------------------------------------------
#
# cmd: show port-channel [MEMBERS] [PORT_LIST] [INFO_LEVEL]
# cmd: show etherchannel [MEMBERS] [PORT_LIST] [INFO_LEVEL]
#
#      PORT_LIST  --> all-ports, active-ports, None
#      INFO_LEVEL --> brief, detailed or None
#-------------------------------------------------------------------------------
class ActivePort( Model ):
   timeBecameActive = Float( help="UTC time the port became active",
                             optional=True )
   protocol = LagModeOptional.mode
   lacpMode = LacpModeOptional.mode

class InactivePort( Model ):
   timeBecameInactive = Float( help="UTC time the interface was no longer active",
                               optional=True )
   reasonUnconfigured = Str( help='Reason interface is not part of the LAG',
                             optional=True )

class PortChannel( Model ):
   fallbackState = FallbackStateOptional.state
   activePorts = Dict( keyType=IntfModel.Interface,
                       valueType=ActivePort,
          help="Map active ports to time they became active, protocol, and mode" )
   inactivePorts = Dict( keyType=IntfModel.Interface,
                         valueType=InactivePort,
          help="Map inactive ports to time and reason they became inactive" )
   recircFeature = List( valueType=str,
          help="Contains a list of features that uses recirc-channel" )
   inactiveLag = Bool( help="Exceeds the port channel config limit" )
   _formerlyActiveMembers = List( valueType=IntfModel.Interface,
                                  help="Formerly active members" )
   def getFormerlyActiveMembers( self ):
      return self._formerlyActiveMembers


class PortChannels( Model ):
   portChannels = Dict( keyType=IntfModel.Interface,
                        valueType=PortChannel,
                        help="Map port-channels to their member information" )
   _brief = Bool( help="Brief or detailed information" )
   _allPorts = Bool( help="Information about all ports, including orphan ports" )

   def render( self ):
      brief = self._brief
      allPorts = self._allPorts
      hasLacpChannels = False
      if self.portChannels:
         temp = self.portChannels[ self.portChannels.keys()[ 0 ] ].activePorts
         if temp:
            if temp[ temp.keys()[ 0 ] ]:
               if temp[ temp.keys()[ 0 ] ].lacpMode:
                  hasLacpChannels = True
      for portChannel in Arnet.sortIntf( self.portChannels ):
         portList = self.portChannels[ portChannel ]
         activeMembers = portList.activePorts.keys()
         inactiveMembers = portList.inactivePorts.keys()
         formerlyActiveMembers = \
             self.portChannels[ portChannel ].getFormerlyActiveMembers()
         if portList.activePorts:
            activeMembers = portList.activePorts.keys()
         if portList.inactivePorts:
            inactiveMembers = portList.inactivePorts.keys()

         if brief:
            print "Port Channel %s:" % ( portChannel )
         else:
            fallback = ''
            if portList.fallbackState:
               fallback = fallbackStateStr( portList.fallbackState )
            print "Port Channel %s (Fallback State: %s):" % ( portChannel, fallback )
         if portList.recircFeature:
            print "  Features: %s" % ' '.join( portList.recircFeature )
         if portList.activePorts:
            if brief:
               printList( portList.activePorts.keys(), "  Active Ports: ", 74 )
            else:
               t = TableFormatter( indent=4 )
               headings = ( ( "Port", "l" ),
                          ( "Time became active", "l"  ) )
               headings = headings + ( ( "Protocol", "l" ), )
               if hasLacpChannels:
                  headings = headings + ( ( "Mode", "l" ), )
               th = Headings( headings )
               th.doApplyHeaders( t )

               for activePortKey in Arnet.sortIntf( portList.activePorts ):
                  activePortValue = portList.activePorts[ activePortKey ]
                  if activePortValue is not None:
                     t.startRow()
                     t.newCell( activePortKey )
                     # after hitless restart, timeBecameActive (derived from
                     # epils.memberTime) is 0
                     t.newCell( timeString( activePortValue.timeBecameActive,
                                            'Unknown' ) )
                     t.newCell( lagModeStr( activePortValue.protocol ) )
                     if hasLacpChannels:
                        if activePortValue.lacpMode:
                           t.newCell( lacpModeStr( activePortValue.lacpMode ) )
                        else:
                           t.newCell( "Unknown" )

         if portList.inactiveLag:
            configLimit = Tac.Type( "Lag::PortChannelConfigLimit" ).limit
            print ( "  Inactive. The number of configured port channels exceeds the "
                    "config limit %d." % configLimit )
         elif [ p for p in activeMembers if p not in formerlyActiveMembers ]:
            # Handle the case where all activeMembers become inactive
            if not brief:
               print "  Active Ports:"
               print t.output()
         else:
            print  "  No Active Ports"

         if allPorts:
            if( len( inactiveMembers ) == 0 ):
            # No unconfigured members, go to next LAG
               continue
            t = TableFormatter( indent=4 )
            if brief:
               headings = ( ( "Port", "lh" ),
                            ( "Reason unconfigured", "l" ) )
            else:
               headings = ( ( "Port", "lh" ),
                            ( "Time became inactive", "l" ),
                            ( "Reason unconfigured", "l" ) )
            th = Headings( headings )
            th.doApplyHeaders( t )

            for inactivePortKey in Arnet.sortIntf( portList.inactivePorts ):
               inactivePortValue = portList.inactivePorts[ inactivePortKey ]
               t.startRow()
               t.newCell( inactivePortKey )
               if not brief:
                  t.newCell( timeString( inactivePortValue.timeBecameInactive,
                                            'Always' ) )
               t.newCell( inactivePortValue.reasonUnconfigured )

            if inactiveMembers:
               print "  Configured, but inactive ports:"
               print t.output()


#-------------------------------------------------------------------------------
#
# cmd: show lacp sys-id [INFO_LEVEL]
#
#      INFO_LEVEL -> brief/detailed
#-------------------------------------------------------------------------------

class LacpSysIdBase( Model ):
   def lacpSysId( self, lacpConfig ):
      raise NotImplementedError


class LacpSysId( Model ):
   class Details( Model ):
      priority = Int( help="System priority." )
      address = Str( help="Switch MAC address." )

   # Other entities like mlag may provide additional info
   miscLacpSysIds = List( valueType=LacpSysIdBase,
                          help="Misc System-Id" )
   details = Submodel( valueType=Details,
                       help="LACP System-Id details",
                       optional=True )
   systemId = Str( help="System identifier." )


   def render( self ):
      if self.details:
         print "System Identifier used by LACP:"
         print "System priority: %d" % self.details.priority
         print "Switch MAC Address: %s" % self.details.address
         print "  802.11.43 representation: %s" % self.systemId
      else:
         print "LACP System-identifier: ", self.systemId

      for miscLacpSysId in self.miscLacpSysIds:
         miscLacpSysId.render()


#-------------------------------------------------------------------------------
#
# cmd: show lacp [PORT_LIST] counters [PORT_LEVEL] [INFO_LEVEL]
# cmd: show lacp [PORT_LIST] counters [INFO_LEVEL] [PORT_LEVEL]
#
#      INFO_LEVEL -> brief/detailed/None
#      PORT_LEVEL -> all-ports/None
#-------------------------------------------------------------------------------

class InterfaceLacpCountersDetails( Model ):
   lastRxTime = Float( help="UTC timestamp of when the actor last received "
                            "LACP packet" )
   actorChurnCount = Int( help="Actor port churn count" )
   actorChangeCount = Int( help="Actor port change count" )
   actorSyncTransitionCount = Int( help="Actor port sync transition count" )
   partnerChurnCount = Int( help="Partner port churn count" )
   partnerChangeCount = Int( help="Partner port change count" )
   partnerSyncTransitionCount = Int( help=" Partner port sync transition count" )

class InterfaceLacpCounters( Model ):
   actorPortStatus = LacpPortStatus.status
   lacpdusRxCount = Int( help="LACP PDUs Rx count" )
   lacpdusTxCount = Int( help="LACP PDUs Tx count" )
   markersRxCount = Int( help="Markers Rx count" )
   markersTxCount = Int( help="Markers Tx count" )
   markerResponseRxCount = Int( help="Marker response Rx count" )
   markerResponseTxCount = Int( help="Marker response Tx count" )
   illegalRxCount = Int( help="Illegal Rx packet count" )

   details = Submodel( valueType=InterfaceLacpCountersDetails,
                       help="Additional LACP counter information",
                       optional=True )

class PortChannelLacpCounters( Model ):
   markers = Submodel( valueType=LacpAnnotationMarkerSet,
                       help="annotation brief marker",
                       optional=True )
   interfaces = Dict( keyType=IntfModel.Interface,
                      valueType=InterfaceLacpCounters,
                      help="Map ports to traffic values" )

class LacpCounters( Model ):
   portChannels = Dict( keyType=IntfModel.Interface,
                        valueType=PortChannelLacpCounters,
                        help="Map port-channel LACP traffic statistics" )
   _brief = Bool( help="Brief or detailed information" )
   _allPorts = Bool( help="Information about all ports, including orphan ports" )
   markerMessages = Submodel( valueType=LacpMarkerMessages,
                              help="LACP annotation markers and messages",
                              optional=True )
   orphanPorts = Dict( keyType=IntfModel.Interface,
                       valueType=InterfaceLacpCounters,
                       help="LACP counters for orphaned ports" )

   def _renderPorts( self, portsDict, table ):
      t = table
      for portName in Arnet.sortIntf( portsDict ):
         portInfo = portsDict[ portName ]
         t.startRow()
         t.newCell( IntfCli.Intf.getShortname( portName ) )
         t.newCell( lacpPortStatusStr( portInfo.actorPortStatus ) )
         t.newCell( portInfo.lacpdusRxCount )
         t.newCell( portInfo.lacpdusTxCount )
         t.newCell( portInfo.markersRxCount )
         t.newCell( portInfo.markersTxCount )
         t.newCell( portInfo.markerResponseRxCount )
         t.newCell( portInfo.markerResponseTxCount )
         t.newCell( portInfo.illegalRxCount )
         if not self._brief:
            t.newCell( timeString( portInfo.details.lastRxTime ) )
            t.newCell( portInfo.details.actorChurnCount )
            t.newCell( portInfo.details.partnerChurnCount )
            t.newCell( portInfo.details.actorSyncTransitionCount )
            t.newCell( portInfo.details.partnerSyncTransitionCount )
            t.newCell( portInfo.details.actorChangeCount )
            t.newCell( portInfo.details.partnerChangeCount )

   def render( self ):
      brief = self._brief
      haveOutput = False

      if self.portChannels or self.orphanPorts:
         t = TableFormatter()
         headers = [ ( "Port", "lh" ), ( "Status", "lh" ),
                     ( "LACPDUs",         "c", ( "RX", "TX" ) ),
                     ( "Markers",         "c", ( "RX", "TX" ) ),
                     ( "Marker Response", "c", ( "RX", "TX" ) ),
                     "Illegal" ]
         if not brief:
            headers += [ "LastRxTime",
                 ( "Churn",          "c", ( ( "Actor", "r" ), ( "Partner", "r" ) ) ),
                 ( "SyncTransition", "c", ( ( "Actor", "r" ), ( "Partner", "r" ) ) ),
                 ( "Change",         "c", ( ( "Actor", "r" ), ( "Partner", "r" ) ) ),
                 ]
         th = Headings( headers )
         th.doApplyHeaders( t )

      if self.orphanPorts:
         haveOutput = True
         t.startRow()
         f = Format( justify="left", noBreak=True, terminateRow=True )
         f.noPadLeftIs( True )
         t.newFormattedCell("Orphaned LACP ports: ", nCols=6, format=f )
         self._renderPorts( self.orphanPorts, t )

      if self.portChannels:
         for portChannelName in Arnet.sortIntf( self.portChannels ):
            portChannel = self.portChannels[ portChannelName ]
            t.startRow()
            f = Format( justify="left", noBreak=True, terminateRow=True )
            f.noPadLeftIs( True )
            mStr = portChannel.markers.toStr() if portChannel.markers else ""
            t.newFormattedCell( "Port Channel %s%s: " % ( portChannelName, mStr ),
                                nCols=6, format=f )
            if portChannel.interfaces:
               haveOutput = True
               self._renderPorts( portChannel.interfaces, t )

      if haveOutput:
         print t.output()
         if self.markerMessages:
            self.markerMessages.render()



#-------------------------------------------------------------------------------
#
# cmd: show lacp [PORT_LIST] aggregates [PORT_LEVEL] [INFO_LEVEL]
# cmd: show lacp [PORT_LIST] aggregates [INFO_LEVEL] [PORT_LEVEL]
#
#      INFO_LEVEL -> brief/detailed/None
#      PORT_LEVEL -> all-ports/None
#-------------------------------------------------------------------------------

class IncompatiblePorts( Model ):
   ports = List( valueType=IntfModel.Interface,
                 help='Ports configured to be in port-channel, aggregable, but '
                      'not in agg',
                 optional=True )

class PortChannelAggregate( Model ):
   markers = Submodel( valueType=LacpAnnotationMarkerSet,
                       help="annotation brief marker",
                       optional=True )
   aggregateId = Str( help="Aggregate ID",
                      optional=True )
   bundledPorts = List( valueType=IntfModel.Interface,
                        help="Ports in aggregate" )
   otherPorts = List( valueType=IntfModel.Interface,
                      help="Ports that are selected and/or standby "
                           "but not part of an aggregate" )
   standbyPorts = List( valueType=IntfModel.Interface,
                        help="Standby Ports" )
   negotiationPorts = List( valueType=IntfModel.Interface,
                            help= "Ports still in negotiation" )
   # keyed with aggId
   incompatiblePortsDetail = Dict( valueType=IncompatiblePorts,
                             help="Ports aggregable, but in incompatible aggregate "
                                  "indexed by their aggregate ID" )

class LacpAggregates( Model ):
   portChannels = Dict( keyType=IntfModel.Interface,
                        valueType=PortChannelAggregate,
             help="Map port-channel to aggregate IDs and list of bundled ports" )
   # This collection is keyed by aggregateId
   otherAggregates = Dict( valueType=PortChannelAggregate,
                           help="Aggregates not bound to any port-channel" )
   _brief = Bool( help="Brief or detailed information" )
   _allPorts = Bool( help="Information about all ports, including other interfaces "
                          "and ports in unbound aggregate ")
   markerMessages = Submodel( valueType=LacpMarkerMessages,
                              help="LACP annotation markers and messages",
                              optional=True )

   def _renderAggregate( self, portChannelName, agg ):
      mStr = agg.markers.toStr() if agg.markers else ""
      if portChannelName:
         print "Port Channel %s%s:" % ( portChannelName, mStr )

      if agg.aggregateId:
         print " Aggregate ID: " + agg.aggregateId

      if agg.bundledPorts:
         printList( agg.bundledPorts, "  Bundled Ports: ", 74 )
      else:
         print( "  No bundled ports" )

      if self._allPorts:
         if self._brief:
            if agg.otherPorts:
               printList( agg.otherPorts, "  Other ports in agg: ", 74 )
         else:
            if agg.standbyPorts:
               printList( agg.standbyPorts, "  Standby Ports: ", 74 )
            if agg.negotiationPorts:
               title = "  Ports still in negotiation: "
               printList( agg.negotiationPorts, title, 74 )

      if agg.incompatiblePortsDetail:
         print "  The remote side negotiated some ports into"
         print "  different aggregates.  This should only occur if the"
         print "  two sides are configured inconsistently."
         if self._brief:
            incompatiblePorts = \
                   [ port for ports in agg.incompatiblePortsDetail.values()
                     for port in ports ]
            title = "  Ports in incompatible aggregates: "
            printList( Arnet.sortIntf( incompatiblePorts ), title, 74 )
         else:
            for aggId, incompatiblePorts in sorted(
                      agg.incompatiblePortsDetail.iteritems() ):
               print " Other aggregate ID: " + aggId
               printList( incompatiblePorts.ports, "   ports in agg:", 74 )

   def render( self ):
      if self.portChannels:
         for portChannelName in Arnet.sortIntf( self.portChannels ):
            agg = self.portChannels[ portChannelName ]
            self._renderAggregate( portChannelName, agg )

      if self._allPorts:
         if self.otherAggregates:
            print "Other aggregates:"
            for _, otherAggregate in sorted( self.otherAggregates.items() ):
               self._renderAggregate( None, otherAggregate )

      if self.markerMessages:
         # Extra line to separate output from detail msgs
         print
         self.markerMessages.render()


#-------------------------------------------------------------------------------
#
# cmd: show lacp [PORT_LIST] internal [PORT_LEVEL] [INFO_LEVEL]
# cmd: show lacp [PORT_LIST] internal [INFO_LEVEL] [PORT_LEVEL]
#
#      INFO_LEVEL -> brief/detailed/None
#      PORT_LEVEL -> all-ports/None
#-------------------------------------------------------------------------------

class InterfaceLacpInternalDetails( Model ):
   aggSelectionState = AggregatorSelectedState.state
   actorAdminKey = Str( help="LACP admin key" )
   actorChurnState = LacpPortChurnState.state
   lastRxTime = Float( help="UTC timestamp of when the actor last received "
                            "LACP packet" )
   actorRxSmState = LacpRxSmState.state
   actorMuxSmState = LacpMuxSmState.state
   actorMuxReason = Str( help="LACP actor mux reason" )
   allowToWarmup = Bool( help="Is interface allowed to warm up?",
                        optional=True )
   if toggleLacpTimeoutMultiplierEnabled():
      actorTimeoutMultiplier = Int( help="LACP actor timeout multiplier",
                                    optional=True )

class InterfaceLacpInternal( Model ):
   actorPortStatus = LacpPortStatus.status
   partnerSystemId = Str( help="LACP partner system identifier" )
   actorPortId = Int( help="LACP actor port ID" )
   actorPortState = Submodel( valueType=LacpPortState,
                              help="LACP actor port state" )
   actorOperKey = Str( help="LACP actor operational key used to form aggregate" )
   actorPortPriority = Int( help="LACP actor port priority" )
   details = Submodel( valueType=InterfaceLacpInternalDetails,
                       help="Detailed LACP port information",
                       optional=True )

class PortChannelLacpInternal( Model ):
   markers = Submodel( valueType=LacpAnnotationMarkerSet,
                       help="annotation brief marker",
                       optional=True )
   interfaces = Dict( keyType=IntfModel.Interface,
                      valueType=InterfaceLacpInternal,
                      help="Map ports to LACP protocol information" )

class LacpInternal( Model ):
   portChannels = Dict( keyType=IntfModel.Interface,
                        valueType=PortChannelLacpInternal,
                        help="Map channel to local LACP state" )
   # Other entities like mlag may provide additional info
   miscLacpSysIds = List( valueType=LacpSysIdBase,
                          help="Misc System Id",
                          optional=True )
   systemId = Str( help="LACP System-identifier", optional=True )
   _brief = Bool( help="Brief or detailed information" )
   _allPorts = Bool( help="Information about all ports, including orphan ports" )
   markerMessages = Submodel( valueType=LacpMarkerMessages,
                              help="LACP annotation markers and messages",
                              optional=True )
   orphanPorts = Dict( keyType=IntfModel.Interface,
                       valueType=InterfaceLacpInternal,
                       help="LACP internal information for orphaned ports" )

   def _renderPorts( self, portsDict, table ):
      t = table
      for portName in Arnet.sortIntf( portsDict ):
         portInfo = portsDict[ portName ]
         t.startRow()
         t.newCell( IntfCli.Intf.getShortname( portName ) )
         statusStr = lacpPortStatusStr( portInfo.actorPortStatus )
         if not self._brief and portInfo.details.allowToWarmup:
            statusStr += '+'
         t.newCell( statusStr )
         if not self._brief:
            t.newCell( aggregatorSelectedStateStr( 
               portInfo.details.aggSelectionState ) )
         t.newCell( portInfo.partnerSystemId )
         t.newCell( portInfo.actorPortId )
         t.newCell( portInfo.actorPortState.toStr() )
         t.newCell( portInfo.actorOperKey )
         if not self._brief:
            t.newCell( portInfo.details.actorAdminKey )
         t.newCell( portInfo.actorPortPriority )
         if not self._brief:
            t.newCell( lacpPortChurnStateStr( portInfo.details.actorChurnState ) )
            t.newCell( timeString( portInfo.details.lastRxTime ) )
            t.newCell( lacpRxSmStateStr( portInfo.details.actorRxSmState ) )
            t.newCell( lacpMuxSmStateStr( portInfo.details.actorMuxSmState ) )
            t.newCell( portInfo.details.actorMuxReason )
            if toggleLacpTimeoutMultiplierEnabled():
               t.newCell( portInfo.details.actorTimeoutMultiplier )

   def render( self ):
      brief = self._brief
      allPorts = self._allPorts
      channels = []
      haveOutput = False

      if self.orphanPorts or self.portChannels:
         if self.systemId:
            print "LACP System-identifier: %s" % self.systemId
            if self.miscLacpSysIds:
               for miscLacpSysId in self.miscLacpSysIds:
                  miscLacpSysId.render()

         t = TableFormatter()
         th = Headings( portHeadings( brief, True ))
         th.doApplyHeaders( t )

      if self.orphanPorts:
         haveOutput = True
         t.startRow()
         f = Format( justify="left", noBreak=True, terminateRow=True )
         f.noPadLeftIs( True )
         t.newFormattedCell("Orphaned LACP ports: ", nCols=6, format=f )
         self._renderPorts( self.orphanPorts, t )

      if self.portChannels:
         for portChannelName in Arnet.sortIntf( self.portChannels ):
            portChannel = self.portChannels[ portChannelName ]
            t.startRow()
            f = Format( justify="left", noBreak=True, terminateRow=True )
            f.noPadLeftIs( True )
            channels.append( portChannelName )
            mStr = portChannel.markers.toStr() if portChannel.markers else ""
            t.newFormattedCell( "Port Channel %s%s: " % ( portChannelName, mStr ),
                                nCols=6, format=f)
            if portChannel.interfaces:
               haveOutput = True
               self._renderPorts( portChannel.interfaces, t )
            else:
               t.startRow()
               t.newCell("")
               t.newFormattedCell("No ports in aggregate", nCols=5, format=f)

      if haveOutput:
         describeLacpState( brief )
         print t.output()
         if self.markerMessages:
            self.markerMessages.render()
      elif allPorts and not brief:
         print """Channels %s are configured for LACP, but have no ports
         'show port-channel all-ports' may show more information.""" % \
         repr( channels )


#-------------------------------------------------------------------------------
#
# cmd: show lacp [PORT_LIST] neighbor [PORT_LEVEL] [INFO_LEVEL]
# cmd: show lacp [PORT_LIST] neighbor [INFO_LEVEL] [PORT_LEVEL]
#
#      INFO_LEVEL -> brief/detailed/None
#      PORT_LEVEL -> all-ports/None
#-------------------------------------------------------------------------------
class InterfaceLacpNeighborsDetails( Model ):
   aggSelectionState = AggregatorSelectedState.state
   partnerChurnState = LacpPortChurnState.state
   collectorMaxDelay = Int( help="Collector max delay in tens of microseconds" )

class InterfaceLacpNeighbors( Model ):
   actorPortStatus = LacpPortStatus.status
   partnerSystemId = Str( help="LACP partner system identifier" )
   partnerPortId = Int( help="LACP partner port ID" )
   partnerPortState = Submodel( valueType=LacpPortState,
                            help="LACP partner port state" )
   partnerOperKey = Str( help="LACP partner operational key" )
   partnerPortPriority = Int( help="LACP partner port priority" )
   details = Submodel( valueType=InterfaceLacpNeighborsDetails,
                       help="Detailed LACP port neighbor information",
                       optional=True )

class PortChannelLacpNeighbors( Model ):
   markers = Submodel( valueType=LacpAnnotationMarkerSet,
                       help="annotation brief marker",
                       optional=True )

   interfaces = Dict( keyType=IntfModel.Interface,
                      valueType=InterfaceLacpNeighbors,
                      help="Map ports to LACP neighbor information" )

class LacpNeighbors( Model ):
   portChannels = Dict( keyType=IntfModel.Interface,
                        valueType=PortChannelLacpNeighbors,
     help="Map the port-channels to their remote neighbor LACP protocol state" )
   _brief = Bool( help="Brief or detailed information" )
   _allPorts = Bool( help="Information about all ports, including orphan ports" )
   markerMessages = Submodel( valueType=LacpMarkerMessages,
                              help="LACP annotation markers and messages",
                              optional=True )
   orphanPorts = Dict( keyType=IntfModel.Interface,
                       valueType=InterfaceLacpNeighbors,
                       help="LACP neighbor information for orphaned ports" )

   def _renderPorts( self, portsDict, table ):
      t = table
      for portName in Arnet.sortIntf( portsDict ):
         portInfo = portsDict[ portName ]
         t.startRow()
         t.newCell( IntfCli.Intf.getShortname( portName ) )
         t.newCell( lacpPortStatusStr( portInfo.actorPortStatus ) )
         if not self._brief:
            t.newCell( aggregatorSelectedStateStr(
               portInfo.details.aggSelectionState ) )
         t.newCell( portInfo.partnerSystemId )
         t.newCell( portInfo.partnerPortId )
         t.newCell( portInfo.partnerPortState.toStr() )
         t.newCell( portInfo.partnerOperKey )
         t.newCell( portInfo.partnerPortPriority )
         if not self._brief:
            t.newCell( lacpPortChurnStateStr( 
               portInfo.details.partnerChurnState ) )
            t.newCell( portInfo.details.collectorMaxDelay )

   def render( self ):
      brief = self._brief
      allPorts = self._allPorts
      t = TableFormatter()
      neighbor = neighborHeadings( brief, True )
      th = Headings( neighbor )
      th.doApplyHeaders( t )
      channels = []
      haveOutput = False

      if self.orphanPorts:
         haveOutput = True
         t.startRow()
         f = Format( justify="left", noBreak=True, terminateRow=True )
         f.noPadLeftIs( True )
         t.newFormattedCell("Orphaned LACP ports: ", nCols=6, format=f )
         self._renderPorts( self.orphanPorts, t )

      if self.portChannels:
         for portChannelName in Arnet.sortIntf( self.portChannels ):
            portChannel = self.portChannels[ portChannelName ]
            t.startRow()
            f = Format( justify="left", noBreak=True, terminateRow=True )
            f.noPadLeftIs( True )
            channels.append( portChannelName )
            mStr = portChannel.markers.toStr() if portChannel.markers else ""
            t.newFormattedCell( "Port Channel %s%s: " % ( portChannelName, mStr ),
                                nCols=6, format=f )
            if portChannel.interfaces:
               haveOutput = True
               self._renderPorts( portChannel.interfaces, t )
            elif not brief:
               t.startRow()
               t.newCell("")
               t.newFormattedCell( "No ports in aggregate", nCols=5, format=f )

      if haveOutput:
         describeLacpState( brief )
         print t.output()
         if self.markerMessages:
            self.markerMessages.render()
      elif allPorts and not brief:
         desc = "no active ports matching your specification\n"
         desc +=   " 'show port-channel all-ports' may show more information."
         if len( channels ) == 1:
            desc1 = "Channel %s is configured for LACP, but has " % channels[0]
         else:
            desc1 = "Channels %s are configured for LACP, but have " % channels
         print desc1+desc

#-------------------------------------------------------------------------------
#
# cmd: show lacp interface [INTERFACE_PORT] [PORT_LEVEL] [INFO_LEVEL]
# cmd: show lacp interface [INTERFACE_PORT] [PORT_LEVEL] [INFO_LEVEL]
#
#      INFO_LEVEL -> brief/detailed/None
#      PORT_LEVEL -> all-ports/None
#-------------------------------------------------------------------------------

class InterfaceLacpDetails( Model ):
   aggSelectionState = AggregatorSelectedState.state
   partnerChurnState = LacpPortChurnState.state
   partnerCollectorMaxDelay = Int( help="Partner Collector max delay in "
                                        "tens of microseconds" )
   actorAdminKey = Str( help="Actor adminstrative key" )
   actorChurnState = LacpPortChurnState.state
   actorLastRxTime = Float( help="UTC timestamp of when the actor last received "
                                 "LACP packet" )
   actorRxSmState = LacpRxSmState.state
   actorMuxSmState = LacpMuxSmState.state
   actorMuxReason = Str( help="LACP actor mux reason" )
   if toggleLacpTimeoutMultiplierEnabled():
      actorTimeoutMultiplier = Int( help="LACP actor timeout multiplier",
                                    optional=True )

class InterfaceLacp( Model ):
   actorPortStatus = LacpPortStatus.status
   partnerSysId = Str( help="System Identifier" )
   partnerPortId = Int( help="Partner port number" )
   partnerPortState = Submodel( valueType=LacpPortState,
                          help="LACP partner port state" )
   partnerOperKey = Str( help="LACP partner operational key" )
   partnerPortPriority = Int( help="Partner port priority" )
   actorPortId = Int( help="Actor port number" )
   actorPortState = Submodel( valueType=LacpPortState,
                          help="LACP actor port state" )
   actorOperKey = Str( help="LACP actor operational key" )
   actorPortPriority = Int( help="Actor port priority" )
   details = Submodel( valueType=InterfaceLacpDetails,
                       help="Additional LACP port information",
                       optional=True )

class PortChannelLacp( Model ):
   markers = Submodel( valueType=LacpAnnotationMarkerSet,
                       help="annotation brief marker",
                       optional=True )
   interfaces = Dict( keyType=IntfModel.Interface,
                      valueType=InterfaceLacp,
                      help="Map ports to port LACP information" )

class LacpPort( Model ):
   portChannels = Dict( keyType=IntfModel.Interface,
                        valueType=PortChannelLacp,
                        help="Map port-channels to their LACP information" )
   interface = IntfModel.Interface( help="LACP information for this interface only ",
                                    optional=True )
   _brief = Bool( help="Brief or detailed information" )
   _allPorts = Bool( help="Information about all ports, including orphan ports" )
   markerMessages = Submodel( valueType=LacpMarkerMessages,
                              help="LACP annotation markers and messages",
                              optional=True )
   orphanPorts = Dict( keyType=IntfModel.Interface,
                       valueType=InterfaceLacp,
                       help="LACP information for orphaned ports" )

   def _renderPorts( self, portsDict, table ):
      t = table
      for portName in Arnet.sortIntf( portsDict ):
         portInfo = portsDict[ portName ]
         t.startRow()
         t.newCell( IntfCli.Intf.getShortname( portName ) )
         t.newCell( lacpPortStatusStr( portInfo.actorPortStatus ) )
         if not self._brief:
            t.newCell( aggregatorSelectedStateStr(
               portInfo.details.aggSelectionState ) )
         t.newCell( portInfo.partnerSysId )
         t.newCell( portInfo.partnerPortId )
         t.newCell( portInfo.partnerPortState.toStr() )
         t.newCell( portInfo.partnerOperKey )
         t.newCell( portInfo.partnerPortPriority )
         if not self._brief:
            t.newCell( lacpPortChurnStateStr(
               portInfo.details.partnerChurnState ) )
            t.newCell( portInfo.details.partnerCollectorMaxDelay )
         t.newCell( portInfo.actorPortId )
         t.newCell( portInfo.actorPortState.toStr() )
         t.newCell( portInfo.actorOperKey )
         if not self._brief:
            t.newCell( portInfo.details.actorAdminKey )
         t.newCell( portInfo.actorPortPriority )
         if not self._brief:
            t.newCell( lacpPortChurnStateStr( 
               portInfo.details.actorChurnState ) )
            t.newCell( timeString( portInfo.details.actorLastRxTime ) )
            t.newCell( lacpRxSmStateStr( portInfo.details.actorRxSmState ) )
            t.newCell( lacpMuxSmStateStr( portInfo.details.actorMuxSmState ) )
            t.newCell( portInfo.details.actorMuxReason )
            if toggleLacpTimeoutMultiplierEnabled():
               t.newCell( portInfo.details.actorTimeoutMultiplier )

   def render( self ):
      brief = self._brief
      haveOutput = False

      if self.portChannels or self.orphanPorts:
         t = TableFormatter()
         headings = neighborHeadings( brief, True )
         headings += portHeadings( brief, False )
         th = Headings( headings )
         th.doApplyHeaders( t )

      if self.orphanPorts:
         haveOutput = True
         t.startRow()
         f = Format( justify="left", noBreak=True, terminateRow=True )
         f.noPadLeftIs( True )
         t.newFormattedCell("Orphaned LACP ports: ", nCols=6, format=f )
         self._renderPorts( self.orphanPorts, t )

      if self.portChannels:
         for portChannelName in Arnet.sortIntf( self.portChannels ):
            portChannel = self.portChannels[ portChannelName ]
            if portChannel.interfaces:
               haveOutput = True
               t.startRow()
               f = Format( justify="left", noBreak=True, terminateRow=True )
               f.noPadLeftIs( True )
               mStr = portChannel.markers.toStr() if portChannel.markers else ""
               t.newFormattedCell( "Port Channel %s%s: " % ( portChannelName, mStr ),
                                   nCols=6, format=f )
               self._renderPorts( portChannel.interfaces, t )

      if haveOutput:
         describeLacpState( brief )
         print t.output()
         if self.markerMessages:
            self.markerMessages.render()
      elif self.portChannels and self.interface:
         print "Interface " + self.interface + " is not configured for LACP"
