# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Model
import EthIntfLib
import EthIntfModel

_fallbackModeMap = { 'unconfigured' : 'Unconfigured',
                     'configured' : 'Configured',
                     'initialized' : 'Initialized',
                     'passive' : 'Passive',
                     'monitoring' : 'Monitoring',
                     'enabled' : 'Enabled',
                     'enabledLocally' : 'Enabled Locally only (overridden)', 
                     'na' : 'NA'}

_fallbackTypeMap = { 'fallbackNone' : 'off',
                     'fallbackStatic' : 'static',
                     'fallbackIndividual' : 'individual' }

class LagMemberInformation( Model ):
   duplex = EthIntfModel.EthPhyInterfaceStatus.duplex
   bandwidth = EthIntfModel.EthPhyInterfaceStatus.bandwidth
   # these private fields should go away once the bugs have been resolved
   # pylint: disable-msg=W0212
   _autoNegotiateActive = EthIntfModel.EthPhyInterfaceStatus._autoNegotiateActive
   # pylint: enable-msg=W0212
   _config = Bool( help='autoneg-related output here (BUG7630).')

class LagEthInterfaceStatus( EthIntfModel.EthInterfaceStatus ):
   memberInterfaces = Dict( valueType=LagMemberInformation, 
                            help='The interfaces that are in this LAG. None if no '
                                 'information is available on the interface.' )
   fallbackEnabled = Bool( help='LAG fallback configured' )
   fallbackEnabledType = Enum( values=_fallbackTypeMap.keys(), optional=True,
                               help='The type of LAG fallback configured' )
   fallbackMode = Enum( values=_fallbackModeMap.keys(), optional=True, 
                        help='The status of LAG fallback' )
   fallbackTimeout = Float( help='The timeout for fallback to occur', optional=True )

   def getMemberDuplexAndBandwidth( self, lagMemberInformation ):
      if lagMemberInformation is None:
         return ( '???', '???' )
      
      # Probably a peer interface. For now we don't display
      # autoneg-related output here (BUG7630).
      # pylint: disable-msg=W0212
      autoneg = lagMemberInformation._autoNegotiateActive if \
                                             lagMemberInformation._config else False
      # pylint: enable-msg=W0212
      duplex = EthIntfLib.generateOperDuplex( 'Format1', autoneg, 
                                              lagMemberInformation.duplex )
      bandwidth = EthIntfLib.getSpeedString( lagMemberInformation.bandwidth, 
                                             autoneg, 'Format1' )

      return ( duplex, bandwidth )
   def renderLoopbackMode( self ):
      pass

   def renderLinkTypeSpecific( self ):
      print '  Full-duplex, %s ' % EthIntfLib.getSpeedString( self.bandwidth,
                                                              False, 
                                                              'Format1' )
      print '  Active members in this channel: %d' % len( self.memberInterfaces )
      for intfName in Arnet.sortIntf( self.memberInterfaces ):
         ( duplex, bandwidth ) = self.getMemberDuplexAndBandwidth( 
                                                 self.memberInterfaces[ intfName ] )
         print '  ... %s , %s, %s' % ( intfName, duplex, bandwidth)
         
      # would like to print out non-members and why.

      print '  Fallback mode is: %s' % _fallbackTypeMap[ self.fallbackEnabledType ]
      
      if self.fallbackMode is not None:
         fallbackModeStr = _fallbackModeMap[ self.fallbackMode ]
         print '  Fallback mode status: %s' % ( fallbackModeStr )
      
      if self.fallbackTimeout is not None:
         print '  Fallback timeout is %d seconds' % ( self.fallbackTimeout )
