# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Lag interface type.  In particular, it provides
# the EthLagIntf class.
#-------------------------------------------------------------------------------
import Tac, CliCommand, CliParser, CliMatcher
import IntfCli, EthIntfCli, VirtualIntfRule
import Ethernet
import LazyMount, EbraEthIntfCli, Arnet, ConfigMount
import Cell
from IntfModel import InterfaceCountersRate
from LagIntfModel import LagEthInterfaceStatus 
from LagIntfModel import LagMemberInformation
from IntfRangePlugin.LagIntf import LagAutoIntfType, portChannelRangeFn
from LagCli import maybeDeleteEthPhyIntfLagConfig, isPortChannelIntfId, \
                   portChannel, isRecirc, updateLacpPortIdRangeForLag, \
                   deleteLacpPortIdRangeForLag
import SharedMem
import Smash
import SubIntfCli
from Intf.IntfRange import subintfSupportedGuard
import Toggles.IntfToggleLib

entMan = None

lagIntfConfigDir = None
lagIntfStatusDir = None
lagGroupDir = None
lagIntfCounterDir = None
lagConfigCli = None
lagCheckpointCounterDir = None
bridgingHwCapabilities = None
lacpCliOverrideDir = None
ethPhyIntfStatusDir = None

PortChannelNum = Tac.Type( "Lag::PortChannelNum" )
LacpFallbackTimeoutValue = Tac.Type( "Interface::LacpFallbackTimeoutValue" )

def lagSupportedGuard( mode, token ):
   if bridgingHwCapabilities.lagSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# A subclass of the EthIntf class for Lag interfaces.
#-------------------------------------------------------------------------------
class EthLagIntf( EthIntfCli.EthIntf ):

   #----------------------------------------------------------------------------
   # Creates a new EthLagIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      self.lagId = portChannel( name ) 
      EthIntfCli.EthIntf.__init__( self, name, mode )
      self.ethIntfConfigDir = lagIntfConfigDir
      self.ethIntfStatusDir = lagIntfStatusDir

   #----------------------------------------------------------------------------
   # Returns the shortname of the interface.
   #----------------------------------------------------------------------------   
   def createShortname( self ):
      return 'Po%d' % self.lagId

   #----------------------------------------------------------------------------
   # Creates the Interface::EthLagIntfConfig object for this interface if it does
   # not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      self.ethIntfConfigDir.intfConfig.newMember( self.name )
      updateLacpPortIdRangeForLag( self.name )

   #----------------------------------------------------------------------------
   # Deletes the Interface::EthLagIntfConfig object for this interface
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      lag = self.ethIntfConfigDir.intfConfig.get( self.name )
      lagId = None
      if lag:
         lagId = lag.intfId
         del self.ethIntfConfigDir.intfConfig[ self.name ]
         deleteLacpPortIdRangeForLag( self.name )
      # Make sure no phy intfs point to this port-channel (or any other 
      # obsolete channel)
      self.cleanEthPhyIntfLagConfigs( lagId )

   def cleanEthPhyIntfLagConfigs( self, lagId ):
      for intf in lagConfigCli.phyIntf.itervalues():
         # snapshot lag, so another process clearing intf.lag doesn't cause
         # us to try to take an attribute of None
         lag = intf.lag 
         # Cleanup lag ptr and lacpMode if epilc.lag.intfId matches the
         # requested lagId in the argument
         if lag and lag.intfId == lagId:
            intf.lag = None
            intf.mode = 'lacpModeOff'
         if not intf.lag:
            maybeDeleteEthPhyIntfLagConfig( intf, None )

   def setDefault( self ):
      elic = self.ethIntfConfigDir.intfConfig[ self.name ]
      elic.minLinks = 0
      elic.fallback = 'fallbackNone'
      elic.fallbackTimeout = elic.fallbackTimeoutDefault
      elic.lacpLoopEnable = False
      if lacpCliOverrideDir.systemId.get( self.name ):
         del lacpCliOverrideDir.systemId[ self.name ]
      EthIntfCli.EthIntf.setDefault( self )
      self.setMacAddr( None )
 
   #----------------------------------------------------------------------------
   # EthIntf is a subclass of PhysicalIntf, but we are actually a virtual
   # interface and we're allowed to be deleted.
   #----------------------------------------------------------------------------
   def noInterface( self ):
      self.destroy()

   #----------------------------------------------------------------------------
   # EthIntfCli.EthIntf.getIntfCounterDir assumes the counter dir leaf name
   # is the same as the status dir leaf name, which is not true for the status
   # dir at 'lag/input/interface/mlag'.
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      assert self.status()
      parent = self.status().parent
      if not parent:
         return None
      return lagCheckpointCounterDir

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def addrStr( self ):
      if self.status().forwardingModel == 'intfForwardingModelRouted':
         addr = self.routedAddr()
      else:
         addr = self.addr()
      return "address is %s" % addr

   def counter( self ):
      shmemEm = SharedMem.entityManager( sysdbEm=entMan )
      smi = Smash.mountInfo( 'reader' )
      smashCountersDir = shmemEm.doMount( 'interface/counter/lag/current',
                                          'Smash::Interface::AllIntfCounterDir',
                                          smi )
      return Tac.newInstance( 'Interface::SmashIntfCounter', 'current',
                              self.name,
                              smashCountersDir )

   def hardware( self ):
      return "portChannel"

   def bandwidth( self ):
      return self.status().speed * 1000
   
   def getIntfDuplexStr( self ):
      return "duplexFull"
   
   def xcvrTypeStr( self ):
      return "N/A"

   def autonegActive( self ):
      return False

   def flowcontrolRxPause( self ):
      return None

   def flowcontrolTxPause( self ):
      return None
   
   def getIntfStatusModel( self ):
      return LagEthInterfaceStatus( name=self.status().intfId )

   def showLoopbackMode( self, intfStatusModel ):
      pass
   
   def showLinkTypeSpecific( self, intfStatusModel ):
      for m in Arnet.sortIntf( self.status().member ):
         member = EthIntfCli.EthPhyIntf( m, self.mode_ )
         if member and member.lookupPhysical():
            config = member.config() is not None
            autoNeg = member.autonegActive() if config else False

            lagMemberInfo = LagMemberInformation( duplex=member.status().duplex, 
                                                bandwidth=member.bandwidth() * 1000, 
                                                _config=config, 
                                                _autoNegotiateActive=autoNeg )
            intfStatusModel.memberInterfaces[ m ] =  lagMemberInfo
         
      # output fallback mode config and status
      elic = self.ethIntfConfigDir.intfConfig.get( self.name )
      elis = self.ethIntfStatusDir.intfStatus.get( self.name )
      fallbackEnabledStr = "na"
      if elis:
         fallbackEnabledStr = elis.debugFallbackSmState
         fallbackEnabledStr = fallbackEnabledStr[ len( "fallbackState" ): ].lower()
      if fallbackEnabledStr == "enabled" and elis.fallbackEnabled == 'fallbackNone':
         fallbackEnabledStr = "enabledLocally"
      if( elic ):
         intfStatusModel.fallbackEnabled = elic.fallback != 'fallbackNone'
         intfStatusModel.fallbackEnabledType = elic.fallback
         if(( elic.fallback != 'fallbackNone' ) or 
            ( elic.fallbackTimeout != elic.fallbackTimeoutDefault )):
            intfStatusModel.fallbackMode = fallbackEnabledStr
            intfStatusModel.fallbackTimeout = elic.fallbackTimeout
         if fallbackEnabledStr == "enabled" and \
               elic.minLinks != elic.minLinksDefault:
            msg = "min-links config will not be honored when fallback is active"
            self.mode_.addWarning( msg )
      else:
         intfStatusModel.fallbackEnabled = False
  
      intfStatusModel.bandwidth = self.bandwidth() * 1000

   #----------------------------------------------------------------------------
   # Is this port eligible for "switchport" commands?
   #----------------------------------------------------------------------------
   def switchportEligible( self ):
      return True

   #----------------------------------------------------------------------------
   # The rule for matching Lag interface names.  When this pattern matches, it
   # returns an instance of the EthLagIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = VirtualIntfRule.VirtualIntfMatcher( 'Port-Channel', PortChannelNum.min,
                                       PortChannelNum.max,
                                       helpdesc="Link Aggregation Group (LAG)",
                                       rangeFunc=portChannelRangeFn,
                                       value=lambda mode, intf:
                                       EthLagIntf( intf, mode ) )

   #----------------------------------------------------------------------------
   # Returns an unsorted list of LagEthIntf objects representing all the existing
   # Interface::LagEthIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in lagIntfStatusDir.intfStatus.keys():
         if not isPortChannelIntfId( name ) or isRecirc( name ):
            continue
         intf = EthLagIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   def setL2Mtu( self, value ):
      cfg = self.config()
      cfg.l2Mtu = value

   def updateInterfaceCountersRateModel( self ):
      if self.counter() is None: 
         return None

      elis = self.ethIntfStatusDir.intfStatus.get( self.name )
      if elis and len( elis.member ) == 1 and \
            elis.member.keys()[0] in ethPhyIntfStatusDir:
         intfObj = EthIntfCli.EthPhyIntf( elis.member.keys()[ 0 ], self.mode_ )
         icrm = intfObj.updateInterfaceCountersRateModel()
         if icrm:
            intfCountersRateModel = InterfaceCountersRate( _name=self.name )
            intfCountersRateModel.description = self.description()
            intfCountersRateModel.interval = icrm.interval
            intfCountersRateModel.inBpsRate = icrm.inBpsRate
            intfCountersRateModel.inPpsRate = icrm.inPpsRate
            intfCountersRateModel.inPktsRate = icrm.inPktsRate
            intfCountersRateModel.outBpsRate = icrm.outBpsRate
            intfCountersRateModel.outPpsRate = icrm.outPpsRate
            intfCountersRateModel.outPktsRate = icrm.outPktsRate
            return intfCountersRateModel

      return super( EthLagIntf, self ).updateInterfaceCountersRateModel()



#-------------------------------------------------------------------------------
# Register the EthLagIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( EthLagIntf, LagAutoIntfType,
                                  withIpSupport=True )
EbraEthIntfCli.switchPortMatcher |= EthLagIntf.matcher

#-------------------------------------------------------------------------------
# A subclass of the SubIntf class for Lag Sub-interfaces.
#-------------------------------------------------------------------------------
class LagSubIntf( SubIntfCli.SubIntf ):
   def vlanSubIntfSupported( self ):
      # Enable "vlan ..." CLI commands in the port-channel sub-interface
      # if-mode.
      return True
   
   @staticmethod
   def getAllPhysical( mode ):
      # parent class returns *all* sub-interfaces, nothing to do here.
      return []

def lagSubIntfRuleValueFunc( mode, intf ):
   if SubIntfCli.theSubintfSupportedGuard( mode, intf ) is not None:
      raise CliParser.InvalidInputError()
   return LagSubIntf( intf, mode )

dynamicIfIndexEnabled = Toggles.IntfToggleLib.toggleDynamicIfIndexEnabled()
if dynamicIfIndexEnabled:
   SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
   subLowerBound = SubIntfRangeDetails.subIntfExtendedIdMin
   subUpperBound = SubIntfRangeDetails.subIntfExtendedIdMax
else:
   # BUG490186: The sub-interface range expansion for lags numbered 1-127,
   # originally tracked with BUG269825, did not change this code - the
   # simple rule/matcher, as opposed to the range rule, does not support
   # extended subinterfaces - all Lag subinterfaces have an upper bound
   # of 4094.  This means, e.g., if you create an interface Po1.5000,
   # that "show ipv6 neighbor Po1.5000" will not parse.
   #
   # This bug only exists when the DynamicIfIndex toggle is off, so
   # we can plausibly ignore it.
   subLowerBound = CliParser.subLowerBound
   subUpperBound = CliParser.subUpperBound

LagAutoIntfType.cliSubIntfClazzIs( LagSubIntf )
subMatcher = VirtualIntfRule.VirtualIntfMatcher( 'Port-Channel',
                                           PortChannelNum.min, PortChannelNum.max,
                                           helpdesc="Port-Channel Sub Interface",
                                           subIntf=True,
                                           subIntfGuard=subintfSupportedGuard,
                                           rangeFunc=portChannelRangeFn,
                                           value=lagSubIntfRuleValueFunc,
                                           subLbound=subLowerBound,
                                           subUbound=subUpperBound )
IntfCli.Intf.addSubIntf( subMatcher, LagSubIntf, withIpSupport=True )

def portChannelSubintfParentCfg( parentName ):
   if not parentName.startswith( 'Po' ):
      return None
   return lagIntfConfigDir.get( parentName )

SubIntfCli.subintfParentCfgHook.append( portChannelSubintfParentCfg )

#-------------------------------------------------------------------------------
# Register the EthLagIntf class as able to participate in interface ranges.
#-------------------------------------------------------------------------------
EbraEthIntfCli.switchPortIntfTypes.append( LagAutoIntfType )

#-------------------------------------------------------------------------------
# Enable Lag-specific commands to be added to "config-if" mode.
#-------------------------------------------------------------------------------
class LagIntfConfigModelet( CliParser.Modelet ):
   
   modeletParseTree = CliParser.ModeletParseTree()
   
   def __init__( self, mode ):   # pylint: disable-msg=W0231
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, EthLagIntf ) and not mode.intf.isSubIntf()

IntfCli.IntfConfigMode.addModelet( LagIntfConfigModelet )

#-------------------------------------------------------------------------------
# Register the EthLagIntf class as a type that supports L2 mtu.
#-------------------------------------------------------------------------------
EthIntfCli.registerL2MtuValidIntfType( EthLagIntf )

#-------------------------------------------------------------------------------
# The "[no|default] lacp max-bundle N" command, in "config-if" mode.
# This command is only valid on a Port-Channel interface.
# This is not yet supported.
#-------------------------------------------------------------------------------
matcherMaxBundle = CliMatcher.KeywordMatcher(
   'max-bundle',
   helpdesc='Set the maximum number of concurrent active links' )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel min-links N" command, in "config-if" mode.
# This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
description = 'Minimum no of ports required up before bringing up a port-channel'

def _rangeFn( mode ):
   # This code is called when the startup-config is being parsed,
   # which happens before the FruAgent has discovered all the hardware
   # and populated Sysdb.
   #
   # Therefore, we don't know the upper limit of this value yet, so we
   # need to accept *any* value, which we accomplish by returning a
   # ridiculously upper-end-of-the-range value in that case. I've
   # chosen 500, rather than some power-of-2, to show that this is a
   # meaningless, arbitrary value, rather than some actual limitation
   # in our SW or HW. Oh, and I guess this comment does that, too. :)
   if lagGroupDir is None:
      return ( 0, 500 )
   hwMax = reduce( max,
                   [ lg.maxPortsPerLagIntf for lg in
                     lagGroupDir.lagGroup.itervalues() ], 0 )
   superMax = hwMax if hwMax else 500
   return ( 0, superMax )

def setMinLinks( mode, args ):
   minLinksVal= args[ 'MINLINKS' ]
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.minLinks = minLinksVal

def defaultMinLinks( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.minLinks = elic.minLinksDefault

matcherPortChannel = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'port-channel', 
         helpdesc='Set port-channel parameters' ),
      guard=lagSupportedGuard )

class LagMinLinksCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel min-links MINLINKS'
   noOrDefaultSyntax = 'port-channel min-links ...'
   data = {
      'port-channel' : matcherPortChannel,
      'min-links' : description,
      'MINLINKS' : CliMatcher.DynamicIntegerMatcher( 
         rangeFn=_rangeFn, helpname=None, helpdesc=description )
   }
   handler = setMinLinks
   noOrDefaultHandler = defaultMinLinks

LagIntfConfigModelet.addCommandClass( LagMinLinksCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac-address router MAC_ADDR
#--------------------------------------------------------------------------------
LagIntfConfigModelet.addCommandClass( IntfCli.MacAddressRouterAddrCmd )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel lacp fallback [static|individual]" command,
# in "config-if" mode. This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def setFallback( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   if 'individual' in args:
      elic.fallback = 'fallbackIndividual'
   else:
      # Not specifying individual or static will default to static fallback
      elic.fallback = 'fallbackStatic'

def noFallback( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.fallback = 'fallbackNone'

matcherLacp = CliMatcher.KeywordMatcher( 'lacp', helpdesc='Set LACP parameter' )
matcherFallback = CliMatcher.KeywordMatcher(
   'fallback',
   helpdesc='Enable fallback to static LAG mode when LACP active mode times out' )

class LacpFallbackCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel lacp fallback [ individual | static ]'
   noOrDefaultSyntax = syntax
   data = {
      'port-channel' : matcherPortChannel,
      'lacp' : matcherLacp,
      'fallback' : matcherFallback,
      'individual' : 'Fallback to individual ports',
      'static' : 'Fallback to static LAG mode'
   }
   handler = setFallback
   noOrDefaultHandler = noFallback

LagIntfConfigModelet.addCommandClass( LacpFallbackCmd )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel lacp fallback timeout <N>" command,
# in "config-if" mode. This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def setFallbackTimeout( mode, args ):
   timeoutSeconds = args[ 'TIMEOUT' ]
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.fallbackTimeout = timeoutSeconds
   if not elic.fallback:
      mode.addWarning( 
         'LACP fallback not configured yet. Configuring timeout does not ' +
         'enable LACP fallback automatically.' )

def defaultFallbackTimeout( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.fallbackTimeout = elic.fallbackTimeoutDefault

class LacpFallbackTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel lacp fallback timeout TIMEOUT'
   noOrDefaultSyntax = 'port-channel lacp fallback timeout ...'
   data = {
      'port-channel' : matcherPortChannel,
      'lacp' : matcherLacp,
      'fallback' : matcherFallback,
      'timeout' : 'Set LACP fallback timeout (not enabling fallback automatically)',
      'TIMEOUT' : CliMatcher.IntegerMatcher(
         LacpFallbackTimeoutValue.min,
         LacpFallbackTimeoutValue.max,
         helpdesc='LACP fallback timeout in seconds (Default: 90)' )
   }
   handler = setFallbackTimeout
   noOrDefaultHandler = defaultFallbackTimeout

LagIntfConfigModelet.addCommandClass( LacpFallbackTimeoutCmd )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel lacp loopback" command,
# in "config-if" mode. This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def setLacpLoopback( mode, args ):
   lagIntfConfigDir.intfConfig[ mode.intf.name ].lacpLoopEnable = True

def noLacpLoopback( mode, args ):
   lagIntfConfigDir.intfConfig[ mode.intf.name ].lacpLoopEnable = False

class LacpLoopbackCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel lacp loopback'
   noOrDefaultSyntax = 'port-channel lacp loopback'
   data = {
      'port-channel' : matcherPortChannel,
      'lacp' : matcherLacp,
      'loopback' : 'Enable LACP loopback negotiation'
   }
   handler = setLacpLoopback
   noOrDefaultHandler = noLacpLoopback

LagIntfConfigModelet.addCommandClass( LacpLoopbackCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp system-id aaaa.bbbb.cccc" command,
# in "config-if" mode. This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def setLacpSystemId( mode, args ):
   systemId = args[ 'ADDR' ]
   intfId = mode.intf.name
   ethAddr = Tac.Value( 'Arnet::EthAddr', stringValue=systemId )
   lacpCliOverrideDir.systemId[ intfId ] = ethAddr

def noLacpSystemId( mode, args ):
   intfId = mode.intf.name
   if lacpCliOverrideDir.systemId.get( intfId, None ):
      del lacpCliOverrideDir.systemId[ intfId ]

class LacpSystemIdCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp system-id ADDR'
   noOrDefaultSyntax = 'lacp system-id ...'
   data = {
      'lacp' : matcherLacp,
      'system-id' : 'Configure LACP system ID',
      'ADDR' : CliMatcher.PatternMatcher( Ethernet.macAddrPattern, 
         helpname='H.H.H', helpdesc='LACP system ID' ) 
   }
   handler = setLacpSystemId
   noOrDefaultHandler = noLacpSystemId

LagIntfConfigModelet.addCommandClass( LacpSystemIdCmd )

#-------------------------------------------------------------------------------
# [ no | default ] l2 mtu MTU
# [ no | default ] l2 mru MRU
#-------------------------------------------------------------------------------
LagIntfConfigModelet.addCommandClass( EthIntfCli.L2MtuCmd )
LagIntfConfigModelet.addCommandClass( EthIntfCli.L2MruCmd )

# TODO Changes here needs to be reflected in the CliSavePlugin and ArosTest Plugin
# Add fallback priority to the show command

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global lagIntfConfigDir, lagIntfStatusDir, lagGroupDir, lagCheckpointCounterDir
   global entMan, lagConfigCli, bridgingHwCapabilities
   global lacpCliOverrideDir
   global ethPhyIntfStatusDir

   entMan = entityManager

   lagIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/eth/lag",
                                         "Interface::EthLagIntfConfigDir", "w" )
   lagConfigCli = ConfigMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "w" )
   lacpCliOverrideDir = ConfigMount.mount(entityManager,
                                 "lag/input/lacpoverride/cli",
                                 "Lag::Input::LacpOverrideDir", "w" )
   lagIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                       "Interface::EthLagIntfStatusDir", "r" )
   lagGroupDir = LazyMount.mount( entityManager, "hardware/lag/internalConfig",
                                  "Hardware::Lag::InternalConfig", "r" )
   lagCheckpointCounterDir = entityManager.mount( Cell.path( "lag/input/counters" ),
                                    "Interface::AllIntfCounterDir", "w" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   ethPhyIntfStatusDir = LazyMount.mount( entityManager,
                                          'interface/status/eth/phy/all',
                                          'Interface::AllEthPhyIntfStatusDir', 'r' )
