#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.LagCli as LagCli
import BasicCliModes
from CliToken.LoadBalance import loadBalanceMatcherForConfig
from CliToken.PortChannel import portChannelMatcherForConfig
import Tac

tacMinLinksTimeoutBase = Tac.Type( "Lag::MinLinksTimeoutBase" )

matcherLacp = CliMatcher.KeywordMatcher( 'lacp', 
      helpdesc='Set Link Aggregation Control Protocol (LACP) parameters' )
matcherRateLimit = CliMatcher.KeywordMatcher( 
      'rate-limit', helpdesc='LACPDU Rate Limit Options' )
nodeLacp = CliCommand.Node( 
      matcher=matcherLacp,
      guard=LagCli.lagSupportedGuard )
matcherPortIdRange = CliMatcher.IntegerMatcher( 1, 65535, 
      helpdesc='LACP port-ID range' )

#--------------------------------------------------------------------------------
# [ no | default ] lacp audit
#--------------------------------------------------------------------------------
class LacpAuditCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp audit'
   noOrDefaultSyntax = 'lacp audit ...'
   data = {
      'lacp' : nodeLacp,
      'audit': 'Audit the internal invariants of the LACP state machines',
   }
   handler = LagCli.doEnableLacpAudit
   noOrDefaultHandler = LagCli.noLacpAudit

   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( LacpAuditCmd )

#--------------------------------------------------------------------------------
# [ no | default ] lacp port-id range MIN MAX
#--------------------------------------------------------------------------------
class LacpPortIdRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp port-id range MIN MAX'
   noOrDefaultSyntax = 'lacp port-id range ...'
   data = {
      'lacp': nodeLacp,
      'port-id': 'LACP port-id range configuration',
      'range': 'LACP port-id range configuration',
      'MIN' : matcherPortIdRange,
      'MAX' : matcherPortIdRange
   }
   handler = LagCli.setPortIdRange
   noOrDefaultHandler = LagCli.setPortIdRange

BasicCliModes.GlobalConfigMode.addCommandClass( LacpPortIdRangeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] lacp system-priority SYSTEMPRIORITY
#--------------------------------------------------------------------------------
class LacpSystemPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp system-priority SYSTEMPRIORITY'
   noOrDefaultSyntax = 'lacp system-priority ...'
   data = {
      'lacp': nodeLacp,
      'system-priority': 'Set local system LACP priority',
      'SYSTEMPRIORITY': CliMatcher.IntegerMatcher( 0, 65535, 
         helpdesc='LACP System Priority' ),
   }
   handler = LagCli.setSystemPriority
   noOrDefaultHandler = LagCli.setSystemPriority

BasicCliModes.GlobalConfigMode.addCommandClass( LacpSystemPriorityCmd )

#--------------------------------------------------------------------------------
# load-balance policies
#--------------------------------------------------------------------------------
class LoadBalancePoliciesCmd( CliCommand.CliCommandClass ):
   syntax = 'load-balance policies'
   data = {
      'load-balance': loadBalanceMatcherForConfig,
      'policies': CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 
            'policies', helpdesc='Configure load-balance policies' ), 
         guard=LagCli.lagLBPoliciesSupportedGuard ),
   }
   handler = LagCli.gotoConfigLoadBalancePoliciesMode
   
BasicCliModes.GlobalConfigMode.addCommandClass( LoadBalancePoliciesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] logging event port-channel member-status global ...
#--------------------------------------------------------------------------------
class LoggingEventPortChannelCmd( CliCommand.CliCommandClass ):
   syntax = 'logging event port-channel member-status global ...'
   noOrDefaultSyntax = syntax
   data = {
      'logging': 'Configure logging',
      'event': 'Configure logging events',
      'port-channel': CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 
            'port-channel', helpdesc='Configure port-channel messages' ), 
         guard=LagCli.lagSupportedGuard ),
      'member-status': ( 'Enable system logging for interface joining/leaving'
                         ' a port-channel' ),
      'global': 'Configure global port channel member status',
   }
   handler = LagCli.doEnableLagMemberStatusLog
   noHandler = LagCli.noLagMemberStatusLog
   defaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( LoggingEventPortChannelCmd )

#--------------------------------------------------------------------------------
# [ no | default ] port-channel min-links review interval <timeout>
#--------------------------------------------------------------------------------
class MinLinksReviewIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel min-links review interval TIMEOUT'
   noOrDefaultSyntax = 'port-channel min-links review interval ...'
   data = {
      'port-channel': portChannelMatcherForConfig,
      'min-links': 'Configure min-links',
      'review': 'Review',
      'interval': 'Port-channel min-links review time in seconds (default = 180)',
      'TIMEOUT': CliMatcher.IntegerMatcher( tacMinLinksTimeoutBase.minTimeout,
                                            tacMinLinksTimeoutBase.maxTimeout,
                                            helpdesc='Seconds' ),
   }
   handler = LagCli.setMinLinksReviewInterval
   noOrDefaultHandler = LagCli.setMinLinksReviewInterval

BasicCliModes.GlobalConfigMode.addCommandClass( MinLinksReviewIntervalCmd )

#--------------------------------------------------------------------------------
# lacp rate-limit default
#--------------------------------------------------------------------------------
class LacpRateLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp rate-limit default'
   noOrDefaultSyntax = syntax
   data = {
      'lacp' : nodeLacp,
      'rate-limit' : 'Set LACPDU Rate Limit Options',
      'default' : 'Enable LACPDU rate limiting by default on all ports',
   }
   handler = LagCli.enableGlobalRateLimit
   defaultHandler = handler
   noHandler = LagCli.noGlobalRateLimit

BasicCliModes.GlobalConfigMode.addCommandClass( LacpRateLimitCmd )
