# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import deque
import Tac, Arnet, CliParser, BasicCli, IntfCli, IntfRangeCli, EthIntfCli, Tracing
import CliCommand, CliMatcher
import LazyMount, ConfigMount
import types
from LacpConstants import *
from LacpLib import *
import WaitForWarmupCli
import CliToken.PortChannel, CliToken.LoadBalance, CliToken.Switch
import CliExtensions
import CommonGuards
import Intf.Log
from CliMode.LagLoadBalanceMode import LBPoliciesMode
import LagModel
from EthIntfCli import ethPhyIntfCounterDir
import EbraCliLib
from IntfRangePlugin import LagIntf
from Toggles.LagToggleLib import toggleLacpTimeoutMultiplierEnabled
import MultiRangeRule

# pkgdeps: library Lag

traceHandle = Tracing.Handle( 'LagCli' )
t0 = traceHandle.trace0

# Globals written by the Plugin function at the end of this file
lagIntfConfigDir = None
lagIntfStatusDir = None
lagConfigCli = None
lagConfig = None
lagStatus = None
lacpConfig = None
lacpCliConfig = None
lacpStatus = None
lacpCounters = None
lacpCountersCheckpoint = None
lagGroupDir = None
bridgingConfig = None
bridgingCliConfig = None
ethIntfStatusDir = None
lagConfigDir = None
ethPhyIntfConfigSliceDir = None
agentStatus = None
recircHwConfig = None
bridgingHwCapabilities = None
lacpCliOverrideDir = None

lagConfigValidCallbacks = []
lagConfigAllowCallbacks = []

lagPrefix = "Port-Channel"
recircPrefix = "Recirc-Channel"

MinLinksTimeoutBase = Tac.Type( "Lag::MinLinksTimeoutBase" )
PortChannelNum = Tac.Type( "Lag::PortChannelNum" )
PortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" )
isPortChannelIntfId = PortChannelIntfId.isPortChannelIntfId
isRecirc = PortChannelIntfId.isRecirc
portChannel = PortChannelIntfId.portChannel
portChannelConfigLimit = Tac.Type( "Lag::PortChannelConfigLimit" ).limit
LacpTimeoutMultiplier = Tac.Type( "Lacp::TimeoutMultiplier" )
configLimitExceededMsg = ( "The number of configured port channels exceeds "
                           "the config limit %d." % portChannelConfigLimit )

__defaultTraceHandle__ = Tracing.Handle( "EthCli" )

# Used for annotating Port-Channels with additional information
annotateMarkers = ( "*" )

# showLacpAnnotateHook allows another package to annotate Port-Channels with
# additional information. Use registerLacpAnnotateHook to register the hook.
showLacpAnnotateHook = CliExtensions.CliHook()

def recircChannelRangeFn( mode ):
   return ( PortChannelNum.min, PortChannelNum.max )

# guard for recirc channel group
def recircGuard( mode, token ):
   if not recircHwConfig.recircSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None 

def lagOutputNonUnicastGuard( mode, token ):
   if not bridgingHwCapabilities.lagShowOutputIntfNonUnicastSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

# guard for 'show LAG output member' command
def lagOutputIntfGuard( mode, token ):
   if not bridgingHwCapabilities.lagShowOutputIntfSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def lagSupportedGuard( mode, token ):
   if bridgingHwCapabilities.lagSupported:
      return None
   return CliParser.guardNotThisPlatform

def extendedlagIdSupportedGuard( mode, token ):
   if bridgingHwCapabilities.extendedLagIdSupported:
      return None
   return CliParser.guardNotThisPlatform

def lagLBPoliciesSupportedGuard( mode, token ):
   if bridgingHwCapabilities.lagLoadBalancePoliciesSupported:
      return None
   return CliParser.guardNotThisPlatform

def portChannelFilter( intfId ):
   return isPortChannelIntfId( intfId ) and not isRecirc( intfId )

def registerLagConfigCallback( callback ):
   # Any function that wants to get called back when a physical
   # port is configured as a lag member should register here.
   # a callback can allow or deny the configuration of the
   # physical member as a lag.
   #
   # callback parameters:
   # @modes: interface modes for the Cli
   # @newLagName: the new lag name the ports are put into, or None
   # @oldLagNames: the old lag names the ports were in, or None
   lagConfigAllowCallbacks.append( callback )

def registerLagConfigValidCallback( callback ):
   # Any function that wants to make sure that after the addition/removal
   # of a physical port to a lag, the configuration is valid registers here.
   # On denial addition/deletion of lag member will be reverted back.
   #
   # callback parameters:
   # @mode: mode for the Cli
   # @portName: name of the physical port
   # @newLagName: the new lag name the port is put into, or None
   # @oldLagName: the old lag name the port was in, or None
   lagConfigValidCallbacks.append( callback )

def tryWaitForWarmup( mode ):
   if( mode ):
      WaitForWarmupCli.tryWaitForWarmupAfterVlanPortChange( mode )

# Association marker-->detail message
markerMsgs = {}
# List of tuples (marker, annotateFunc) for registered showLacpAnnotateHook 
# extensions
afmAssoc = []
def genMarkers( portChannel ):
   # Function to generate annotation markers for a port channel
   markers = [ marker for af, marker in afmAssoc if af( portChannel ) ]
   return markers

def registerLacpAnnotateHook( hook ):
   # Any package that wants to annotate a port-channel for 'show lacp'
   # commands (excluding show lacp aggregate) registers its hook here.
   # The hook should return a tuple ( message, annotateFunc )
   # with the detail message description and function that returns True if
   # the passed Port-Channel name should be annotated.
   markerIndex = len( showLacpAnnotateHook.extensions() )
   assert markerIndex < len( annotateMarkers )
   showLacpAnnotateHook.addExtension( hook )

   ( msg, af ) = hook()
   marker = annotateMarkers[ markerIndex ]

   markerMsgs[ marker ] = msg   
   afmAssoc.append( ( af, marker ) )

def markersToModelSet( markerStrings ):
   if not markerStrings or len( markerStrings ) == 0:
      return None
   else:
      return LagModel.LacpAnnotationMarkerSet( markers=markerStrings )

def usedDetailMarkerMessages( usedMarkers, markerMsgs ):
   mmList = []
   for marker in annotateMarkers:
      if marker in usedMarkers:
         mmList.append( LagModel.LacpMarkerMessage(
                              marker=marker,
                              msg=markerMsgs[ marker ] ) )
   if len( mmList ):
      return LagModel.LacpMarkerMessages( markerMessages=mmList )
   else:
      return None

# explain why we may have disabled bridging, if a Cli command needs an
# explanation. Also return legend for show vlan configured-ports.
def ethPhyIntfInLag( intfName, mode ):
   epilc = ethPhyIntfLagConfigDir( mode ).get( intfName )
   lag = epilc and epilc.lag
   if lag:
      return( "in " + IntfCli.Intf.getShortname( lag.intfId ),
              "Member of " + lag.intfId,
              "port channel configuration" )
   else:
      return( None, None, None )

def ethPhyIntfInLagRecirculation( intfName, mode ):
   epilc = ethPhyIntfLagConfigDir( mode ).get( intfName )
   lag = epilc and epilc.lag
   if lag:
      return( "in " + IntfCli.Intf.getShortname( lag.intfId ),
              "Member of " + lag.intfId + " recirc",
              "port channel recirculation configuration" )
   else:
      return( None, None, None )

IntfCli.dataLinkExplanationHook.addExtension( ethPhyIntfInLag )
IntfCli.recirculationExplanationHook.addExtension( ethPhyIntfInLagRecirculation )

lagConfigLimitReachedMsg = ( "Port channel config limit %d reached. "
                             "No interfaces were created." % portChannelConfigLimit )

def lagConfigLimitReachedExtension( intfs ):
   # Extension only applies to port channels
   if intfs.type_ is not LagIntf.LagAutoIntfType:
      return None
   # If the number of configs being added is greater than the config limit or
   # the number of configs that exist plus the number of configs being added
   # is less than the config limit, no set logic is needed
   numIntfs = len( intfs )
   if numIntfs > portChannelConfigLimit:
      return lagConfigLimitReachedMsg
   if numIntfs + len( lagIntfConfigDir ) < portChannelConfigLimit:
      return None
   # If the number of configs that either exist or are being added
   # is greater than the config limit, return an error message
   if len( set( intfs.intfNames() ) |
           set( lagIntfConfigDir ) ) > portChannelConfigLimit:
      return lagConfigLimitReachedMsg
   return None

IntfRangeCli.canCreateIntfsHook.addExtension( lagConfigLimitReachedExtension )

def ethPhyIntfLagConfigDir( mode ):
   return lagConfigDir.phyIntf

def ethPhyIntfLagCliConfigDir( mode ):
   return lagConfigCli.phyIntf

def _ethPhyIntfLagConfig( mode, create, name, configDir):
   if not name:
      name = mode.intf.name
   epilc = configDir.get( name )
   if not epilc:
      if create:
         epilc = configDir.newMember( name )
      else:
         return None
   return epilc

def ethPhyIntfLagConfig( mode, create=False, name=None ):
   return _ethPhyIntfLagConfig( mode, create, name,
                                ethPhyIntfLagConfigDir( mode ) )

def ethPhyIntfLagCliConfig( mode, create=False, name=None ):
   return _ethPhyIntfLagConfig( mode, create, name,
                                ethPhyIntfLagCliConfigDir( mode ))

def inactiveLag( intfId ):
   return ( lagIntfConfigDir.get( intfId, None ) and
            not lagIntfStatusDir.get( intfId, None ) and
            len( lagIntfStatusDir ) >= portChannelConfigLimit )

# pass a channelGroups and the mode in, and return a list of
# channel-groups that match the channelGroups *AND* are configured
# for LACP.  channelGroups of None means list all appropriate
# channel-groups
def portchannelList( mode, channelGroups, lacpOnly=True, inactiveWarn=True, 
                     filterFunc=portChannelFilter ):
   if channelGroups is None:
      explicit = False
      candidateChannelList = lagIntfConfigDir.intfConfig
   else:
      explicit = True
      if not isinstance( channelGroups, types.ListType ):
         channelGroups = [ channelGroups ]
      # Filter out unconfigured channels
      candidateChannelSet = set( c.idStr for c in channelGroups )
      configuredChannels = set( lagIntfConfigDir.intfConfig )
      unconfiguredChannels = candidateChannelSet.difference( configuredChannels )
      candidateChannelList = list( candidateChannelSet.difference(
         unconfiguredChannels ) )
   candidateChannelList = deque( Arnet.sortIntf( candidateChannelList ) )
   inactiveLagIds, filteredChannels = [], []
   while candidateChannelList:
      channel = candidateChannelList.popleft()
      if not filterFunc( channel ):
         if explicit:
            print "%s not configured as LAG" % channel
         continue
      lagIntfConfig = lagIntfConfigDir.get( channel )
      # Handle inactive port channels
      if inactiveLag( channel ):
         if inactiveWarn:
            # Save inactive lag ID for aggregated warning
            inactiveLagIds.append( portChannel( channel ) )
         else:
            # If inactiveWarn == False, we want the inactive channel
            # in our command's output
            filteredChannels.append( channel )
         continue
      # If lacpOnly, filter out non-LACP port channels
      if lacpOnly and lagIntfConfig.mode == 'lagModeOn':
         if explicit:
            print "%s not configured for LACP" % channel
         continue
      # Otherwise, give channel back to command to render
      filteredChannels.append( channel )
   if inactiveWarn and inactiveLagIds:
      print "Port-Channel%s%s %s inactive. %s" % (
            "s" if len( inactiveLagIds ) > 1 else "",
            MultiRangeRule.multiRangeToCanonicalString( inactiveLagIds ),
            "are" if len( inactiveLagIds ) > 1 else "is",
            configLimitExceededMsg )
   if explicit and unconfiguredChannels:
      unconfiguredChannels = Arnet.sortIntf( unconfiguredChannels )
      idStrToId = { c.idStr: c.id for c in channelGroups }
      unconfiguredLagIds = ( idStrToId[ c ] for c in unconfiguredChannels )
      print "Port-Channel%s%s not configured as LAG" % (
            "s" if len( unconfiguredChannels ) > 1 else "",
            MultiRangeRule.multiRangeToCanonicalString( unconfiguredLagIds ) )
   return filteredChannels

class ChannelGroup( object ):
   def __init__( self, id ):
      self.id_ = id
      self.newborn = 0
      self.lagPrefix_ = lagPrefix

   id = property( lambda self: self.id_ )
   idStr = property( lambda self: "%s%d" % ( self.lagPrefix_, self.id ) )

   # Create the port-channel interface corresponding to this object.
   # This function is called whenever the config-if channel-group N
   # command is used, or when handling the "interface Port-ChannelN"
   # mode.
   def create( self, mode ):
      lagDir = lagIntfConfigDir
      if not lagDir.has_key( self.idStr ):
         if len( lagIntfConfigDir ) >= portChannelConfigLimit:
            return None
         lagDir.intfConfig.newMember( self.idStr )
         self.newborn = 1
         updateLacpPortIdRangeForLag( self.idStr )
      else:
         self.newborn = 0
      return lagDir[ self.idStr ]

   def get( self, mode ):
      lagDir = lagIntfConfigDir
      return lagDir.get( self.idStr, None )

class RecircChannelGroup( ChannelGroup ):
   def __init__( self, id ):
      self.id_ = id
      self.newborn = 0
      self.lagPrefix_ = recircPrefix

def getChannelGroupIdList( intfs ):
   if intfs is None:
      return None
   return map( ChannelGroup, intfs.values() )

def getRecircGroupIdList( intfs ):
   if intfs is None:
      return None
   return map( RecircChannelGroup, intfs.values() )

class EthPhyLagIntfDependent( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      epilc = ethPhyIntfLagCliConfig( mode=None, create=False, 
                                      name=self.intf_.name )
      if epilc:
         del lacpCliConfig.rateLimitIntfEnable[ epilc.intfId ]
         del lagConfigCli.phyIntf[ epilc.intfId ]

# Only pass in mode when the mode represents a single EthPhyIntf
def maybeDeleteEthPhyIntfLagConfig( epilc, mode, inputDir=None ):
   if mode and not epilc:
      epilc = ethPhyIntfLagCliConfig( mode=mode, 
                                      create=False, name=mode.intf.name )
   if epilc:
      if inputDir:
         epilcd = inputDir
      else:
         epilcd = lagConfigCli
      name = epilc.intfId
      if( epilc.lag == None and
          epilc.mode == "lacpModeOff" and
          epilc.priority == PortPriorityDefault and
          epilc.portId == LacpPortIdDefault and
          epilc.timeout == 'lacpLongTimeout' and
          epilc.timeoutMult == LacpTimeoutMultiplier.defaultValue and
          epilc.fwdEnabled == True ):
         del epilcd.phyIntf[ name ]

class ChannelGroupModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return intfSupportsLag( mode.intf.name )

   modeletParseTree = CliParser.ModeletParseTree()

#-------------------------------------------------------------------------------
# Rule to match the token class <channelGroup_id>, returning a corresponding
# ChannelGroup object, or <channelGroup_id list>, returning a list of
# corresponding ChannelGroup objects.
#-------------------------------------------------------------------------------
channelGroupMatcher = CliMatcher.DynamicIntegerMatcher( LagIntf.portChannelRangeFn,
                                                        helpdesc='Channel Group ID' )

recircGroupMatcher = CliMatcher.DynamicIntegerMatcher( recircChannelRangeFn,
                                                       helpdesc='Channel Group ID' )

def channelGroups( match ):
   # match is idList
   return( map( ChannelGroup, match ))

#-------------------------------------------------------------------------------
# Rule to match the token class <recircChannelGroup_id>, returning a corresponding
# RecircChannelGroup object, or <recirCchannelGroup_id list>, returning a list of
# corresponding RecircChannelGroup objects.
#-------------------------------------------------------------------------------

def recircChannelGroups( match ):
   # match is idList
   return( map( RecircChannelGroup, match ))

#-------------------------------------------------------------------------------
# The "[no] lacp system-priority N" command, in "config" mode.
#-------------------------------------------------------------------------------

canSetSystemPriorityHook = CliExtensions.CliHook()

nodeLacp = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      'lacp', helpdesc='Set Link Aggregation Control Protocol (LACP) parameters' ),
   guard=lagSupportedGuard )

def setSystemPriority( mode, args ):
   systemPriority = args.get( 'SYSTEMPRIORITY', SystemPriorityDefault )
   for hook in canSetSystemPriorityHook.extensions():
      if not hook( mode, systemPriority ):
         return
   lacpCliConfig.priority = systemPriority

def getCurrentSystemPriority():
   with ConfigMount.ConfigMountDisabler( disable=True ):
      return lacpCliConfig.priority

#-------------------------------------------------------------------------
# The "[no|default] lacp rate-limit default" command, in "config" mode.
#-------------------------------------------------------------------------

def enableGlobalRateLimit( mode, args ):
   lacpCliConfig.rateLimitEnable = True

def noGlobalRateLimit( mode, args ):
   lacpCliConfig.rateLimitEnable = False

#-----------------------------------------------------------------------------
# The "[no|default] lacp port-id range <min> <max>" command, in "config" mode.
#------------------------------------------------------------------------------

def updateLacpPortIdRangeForLag( poName ):
   if lacpCliConfig.portIdRange != Tac.Value( 'Lacp::PortIdRange' ):
      lacpCliOverrideDir.portIdRange[ poName ] = lacpCliConfig.portIdRange

def deleteLacpPortIdRangeForLag( poName ):
   del lacpCliOverrideDir.portIdRange[ poName ]

def updateLacpOverridePortIdRange():
   if lacpCliConfig.portIdRange == Tac.Value( 'Lacp::PortIdRange' ):
      for poName in lagIntfConfigDir:
         deleteLacpPortIdRangeForLag( poName )
   else:
      for poName in lagIntfConfigDir:
         updateLacpPortIdRangeForLag( poName )

def setPortIdRange( mode, args ):
   portIdMin = args.get( 'MIN', LacpPortIdDefault )
   portIdMax = args.get( 'MAX', LacpPortIdDefault )
   if portIdMin > portIdMax:
      mode.addError( "Invalid LACP port-id range values" )
      return
   lacpCliConfig.portIdRange = Tac.Value( 'Lacp::PortIdRange', portIdMin, portIdMax )
   updateLacpOverridePortIdRange()

#----------------------------------------------------------------------
# The "[no] logging event port-channel member-status global N" command, 
# in "config" mode.
#----------------------------------------------------------------------

def doEnableLagMemberStatusLog( mode, args ):
   lagConfigCli.memberStatusLogEnabled = True

def noLagMemberStatusLog( mode, args ):
   lagConfigCli.memberStatusLogEnabled = False

#------------------------------------------------------------------------------
# The "[no|default] port-channel min-links review interval <timeout>" command, 
# in "config" mode.
#------------------------------------------------------------------------------

def setMinLinksReviewInterval( mode, args ):
   interval = args.get( 'TIMEOUT', MinLinksTimeoutBase.timeoutDefault )
   lagConfigCli.minLinksTimeoutBase = interval

#--------------------------
# Load-Balance polices mode
#--------------------------

class ConfigLoadBalancePoliciesMode( BasicCli.ConfigModeBase, LBPoliciesMode ):
   name = "LoadBalance Policies"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      LBPoliciesMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------------------------
# "load-balance policies" in global config mode
#-----------------------------------------------------------

def gotoConfigLoadBalancePoliciesMode( mode, args ):
   childMode = mode.childMode( ConfigLoadBalancePoliciesMode )
   mode.session_.gotoChildMode( childMode )

#-------------------------------------------------------------------------------
# The "port-channel load-balance [src-mac|dst-mac]" command, in "config" mode.
# Implemented in FocalPoint and Sand.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# The "[no|default] channel-protocol lacp" command, in "config-if" mode.
# This command is only valid on an aggregable interface, and is a
# no-op; we only support LACP.
#-------------------------------------------------------------------------------

class ChannelProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'channel-protocol lacp'
   noOrDefaultSyntax = 'channel-protocol ...'
   data = {
      'channel-protocol' : (
         'Set the channel protocol associated with this interface' ),
      'lacp' : nodeLacp
   }
   hidden = True
   handler = lambda mode, args: None
   noOrDefaultHandler = handler

ChannelGroupModelet.addCommandClass( ChannelProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp port-priority N" command, in "config-if" mode.
# This command is only valid on an aggregable interface.
#-------------------------------------------------------------------------------

def setPortPriority( mode, args ):
   portPriority = args.get( 'PRIORITY', PortPriorityDefault )
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   epilc.priority = portPriority
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class LacpPortPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp port-priority PRIORITY'
   noOrDefaultSyntax = 'lacp port-priority ...'
   data = {
      'lacp' : nodeLacp,
      'port-priority' : 'Set the port prioriy associated with this interface',
      'PRIORITY' : CliMatcher.IntegerMatcher(
         PortPriorityMin, PortPriorityMax, helpdesc='LACP Port Priority' )
   }
   handler = setPortPriority
   noOrDefaultHandler = setPortPriority

ChannelGroupModelet.addCommandClass( LacpPortPriorityCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp port-id N" command, in "config-if" mode.
#-------------------------------------------------------------------------------

def setPortId( mode, args ):
   portId = args.get( 'ID', LacpPortIdDefault )
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   epilc.portId = portId
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class LacpPortIdCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp port-id ID'
   noOrDefaultSyntax = 'lacp port-id ...'
   data = {
      'lacp' : nodeLacp,
      'port-id' : 'Set the LACP port number associated with this interface ',
      'ID' : CliMatcher.IntegerMatcher(
         LacpPortIdMin, LacpPortIdMax, helpdesc='LACP Port Id' )
   }
   handler = setPortId
   noOrDefaultHandler = setPortId

ChannelGroupModelet.addCommandClass( LacpPortIdCmd )

#----------------------------------------------------------------------------
# The "[no|default] lacp rate-limit" command, in "config-if" mode.
#----------------------------------------------------------------------------

def setRateLimit( mode, args ):
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   lacpCliConfig.rateLimitIntfEnable[ mode.intf.name ] = True
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

def defaultRateLimit( mode, args ):
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   rateLimit = lacpCliConfig.rateLimitIntfEnable
   del rateLimit[ mode.intf.name ]
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

def noRateLimit( mode, args ):
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   lacpCliConfig.rateLimitIntfEnable[ mode.intf.name ] = False
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class LacpRateLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp rate-limit'
   noOrDefaultSyntax = 'lacp rate-limit'
   data = {
      'lacp' : nodeLacp,
      'rate-limit' : 'Enable LACPDU rate limiting on this port',
   }
   handler = setRateLimit
   defaultHandler = defaultRateLimit
   noHandler = noRateLimit

ChannelGroupModelet.addCommandClass( LacpRateLimitCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp timer (normal | fast)" command, in "config-if" mode.
# This command is only valid on an aggregable interface.
#
# legacy:
# "[no|default] lacp rate (normal | fast)""
#-------------------------------------------------------------------------------

def setLacpRate( mode, lacpRate='normal' ):
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   if lacpRate == 'fast':
      epilc.timeout = 'lacpShortTimeout'
   else:
      epilc.timeout = 'lacpLongTimeout'
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

matcherRate = CliMatcher.EnumMatcher( {
   'normal' : '30 seconds while synchronized, 1 second while synchronizing',
   'fast' : '1 second whether synchronized or not',
} )
matcherTimer = CliMatcher.KeywordMatcher( 
   'timer', helpdesc='Set the rate at which the peer sends us LACP PDUs' )

class LacpTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp timer RATE'
   noOrDefaultSyntax = 'lacp timer ...'
   data = {
      'lacp' : nodeLacp,
      'timer' : matcherTimer,
      'RATE' : matcherRate
   }
   @staticmethod
   def handler( mode, args ):
      setLacpRate( mode, lacpRate=args[ 'RATE' ] )   
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setLacpRate( mode )

class LacpRateCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp rate RATE'
   noOrDefaultSyntax = 'lacp rate ...'
   data = {
      'lacp' : nodeLacp,
      'rate' : CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 'rate',
            helpdesc='Set the rate at which the peer sends us LACP PDUs' ),
         deprecatedByCmd='lacp timer' ),
      'RATE' : matcherRate
   }
   @staticmethod
   def handler( mode, args ):
      setLacpRate( mode, lacpRate=args[ 'RATE' ] )
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setLacpRate( mode )

ChannelGroupModelet.addCommandClass( LacpRateCmd )
ChannelGroupModelet.addCommandClass( LacpTimerCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp timer multiplier <multiplier>" command, in "config-if" mode.
# This command is only valid on an aggregable interface.
#-------------------------------------------------------------------------------

def setLacpTimeoutMultiplier( mode, args ):
   timeoutMultiplier = args.get( 'MULTIPLIER',
                                 LacpTimeoutMultiplier.defaultValue )
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   epilc.timeoutMult = timeoutMultiplier
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class LacpTimeoutMultiplierCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp timer multiplier MULTIPLIER'
   noOrDefaultSyntax = 'lacp timer multiplier ...'
   data = {
      'lacp' : nodeLacp,
      'timer' : matcherTimer,
      'multiplier' :
         'Set the multiplier rate of missed LACP PDUs time out a member',
      'MULTIPLIER' : CliMatcher.IntegerMatcher(
         LacpTimeoutMultiplier.min, LacpTimeoutMultiplier.max,
         helpdesc=
            'Consecutively missed LACP PDUs to time out a member (default=3)' )
   }
   handler = setLacpTimeoutMultiplier
   noOrDefaultHandler = setLacpTimeoutMultiplier

if toggleLacpTimeoutMultiplierEnabled():
   ChannelGroupModelet.addCommandClass( LacpTimeoutMultiplierCmd )

#-------------------------------------------------------------------------------
# The "[no|default] channel-group N mode {on|active|passive}" command, 
# in "config-if" mode. This command is only valid on an aggregable interface.
#-------------------------------------------------------------------------------

nodeChannelGroup = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'channel-group',
      helpdesc='Set link aggregation group' ),
   guard=lagSupportedGuard )

def copySwitchportParams( frm, to ):
   '''Copy switchport parameters.  May really belong in Ebra/VlanCli.'''
   to.enabled = frm.enabled
   to.switchportMode = frm.switchportMode
   to.accessVlan = frm.accessVlan
   to.trunkNativeVlan = frm.trunkNativeVlan
   to.trunkAllowedVlans = frm.trunkAllowedVlans
   to.blockUnknownUnicast = frm.blockUnknownUnicast
   to.blockUnknownMulticast = frm.blockUnknownMulticast
   to.trunkGroup.clear()
   for group in frm.trunkGroup:
      to.trunkGroup[ group ] = True

def getChannelModeFromLacpMode( lagMode ):
   if lagMode == 'lacpModeOff':
      return "on"
   elif lagMode == 'lacpModeActive':
      return "active"
   elif lagMode == 'lacpModePassive':
      return "passive"

def reqSetChannelGroupMode( mode, args ):
   channelGroupId = ChannelGroup( args[ 'ID' ] )
   channelGroupMode = args[ 'MODE' ]
   setChannelGroupMode( mode, channelGroupId, channelGroupMode )

def reqSetRecircChannelGroupMode( mode, args ):
   recircChannelGroupId = RecircChannelGroup( args[ 'ID' ] )
   if mode.intf.config() and mode.intf.config().loopbackMode == 'loopbackNone':
      warning = 'Interface %s is not configured for system loopback' % \
         mode.intf.name
      mode.addWarning( warning )
   setChannelGroupMode( mode, recircChannelGroupId, 'on' )
   
def setChannelGroupMode( mode, channelGroupId, channelGroupMode,
                         checkValidity=True ):
   # Old configuration values in case we have to rolback the current config
   oldMode = None
   def _oldLagState( _mode ):
      _oldLag = None
      _oldLagId = None
      _oldepilc = ethPhyIntfLagCliConfig( _mode, create=False )
      if _oldepilc:
         _oldLag = _oldepilc.lag
         if _oldLag:
            _oldLagId = portChannel( _oldLag.intfId )
      return (_oldepilc, _oldLag, _oldLagId)

   oldLag = {}
   with ConfigMount.ConfigMountDisabler( disable=True ):
      oldLag["Sysdb"] = _oldLagState( mode )
   oldLag["ConfigSession"] = _oldLagState( mode )

   def _lagConfigHookArgs( args ):
      args.append( channelGroupId )
      for src in [ 'Sysdb', 'ConfigSession' ]:
         oldLagIds = [ oldLag[ src ][ 2 ] ]
         intfModes = args[0]
         if len( intfModes ) > 1:
            for im in intfModes[1:]:
               if src == 'Sysdb':
                  with ConfigMount.ConfigMountDisabler( disable=True ):
                     _, _, _oldLagId = _oldLagState( im )
               else:
                  _, _, _oldLagId = _oldLagState( im )
            if _oldLagId:
               oldLagIds.append( _oldLagId )
         args.append( oldLagIds )

   for lagConfigCallback in lagConfigAllowCallbacks:
      if not mode.maybeIntfRangeCliHook(
         lagConfigCallback, lagConfigCallback.func_name, _lagConfigHookArgs ):
         return

   # the config should be written to in session if running in session
   oldepilc, oldLag, oldLagId = oldLag["ConfigSession"]
   # "no channel-group"
   if channelGroupId is None:
      noChannelGroupMode( oldepilc, mode)
      return

   lag = channelGroupId.create( mode )
   if not lag:
      mode.addError( lagConfigLimitReachedMsg )
      return
   # if there are any current members, warn if we're changing the mode
   if channelGroupMode == 'on':
      desiredLagMode = 'lagModeOn'
   else:
      if mode.intf.name in bridgingConfig.switchIntfConfig:
         switchportMode  = bridgingConfig.switchIntfConfig[ mode.intf.name ]. \
             switchportMode
         if switchportMode in [ 'tap', 'tool' ]:
            warningMessage = 'The interface switchport mode of %s is not ' \
                'compatible with LACP protocol, and will not be able to join ' \
                'the LAG.' % switchportMode
            mode.addWarning( warningMessage )
      desiredLagMode = 'lagModeLacp'
   if lag.mode != 'lagModeUnconfigured' and lag.mode != desiredLagMode:
      #XXX This limitation sucks, but is present in the industry standard.
      # Really, we should set all members to desiredLagMode
      # if it is different, and warn about that.
      members = 0
      for intf in ethPhyIntfLagCliConfigDir( mode ).itervalues():
         if intf.lag == lag:
            members += 1
      if members:
         mode.addError( "Cannot change mode; remove all members and try again." )
         return
   lag.mode = desiredLagMode
   # Create/get LacpIntfConfig for this interface.
   # Note that all interfaces get LacpConfigs, even those that are
   # statically configured, so that the Marker protocol can run.
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   oldLag = epilc.lag
   oldMode = epilc.mode
   
   # BUG2537: switchIntfConfig copying should be done in sysdb plugin
   if channelGroupId.newborn:
      switchIntfConfigs = bridgingCliConfig.switchIntfConfig
      lagName = lag.intfId
      intfName = mode.intf.name
      if intfName in switchIntfConfigs:
         if lagName not in switchIntfConfigs:
            switchIntfConfigs.newMember( lagName, 'access' )
         copySwitchportParams( switchIntfConfigs[ intfName ],
                               switchIntfConfigs[ lagName ] )

   # add this interface as a member of lag
   # Set lag attribute first before mode to avoid attrlog race with 
   # SysdbReactors for mode attribute
   epilc.lag = lag
   if channelGroupMode == 'active':
      epilc.mode = 'lacpModeActive'
   elif channelGroupMode == 'passive':
      epilc.mode = 'lacpModePassive'
   else:
      epilc.mode = 'lacpModeOff'

   tryWaitForWarmup( mode )
   if checkValidity:
      for lagConfigValidCallback in lagConfigValidCallbacks:
         ( status, err ) = lagConfigValidCallback(
            mode, epilc.intfId, lag.intfId, oldLag.intfId if oldLag else None )
         if not status:
            if not err:
               err = "Unknown"

            # Revert back the config
            if oldLag:
               oldLagId = portChannel( oldLag.intfId )
               channelGroupId = ChannelGroup( oldLagId )
               channelGroupMode = getChannelModeFromLacpMode( oldMode )
               setChannelGroupMode( mode, channelGroupId, channelGroupMode,
                     checkValidity=False )
            else:
               noChannelGroupMode( epilc, mode )
            mode.addError( 'Cannot add %s to a port channel (%s)' %
                           ( mode.intf.shortname, err ) )
            break
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

def noChannelGroupMode( epilc, mode ):
   if epilc:
      name = epilc.intfId
      epilc.lag = None
      epilc.mode = 'lacpModeOff'
   elif mode:
      name = mode.intf.name
   else:
      return
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )
   tryWaitForWarmup( mode )

def reqNoChannelGroupMode( mode, args ):
   setChannelGroupMode( mode, None, None )

class ChannelGroupModeCmd( CliCommand.CliCommandClass ):
   syntax = 'channel-group ID mode MODE'
   noOrDefaultSyntax = 'channel-group ...'
   data = {
      'channel-group' : nodeChannelGroup,
      'ID' : channelGroupMatcher,
      'mode' : 'Select LACP or static link aggregation',
      'MODE' : CliMatcher.EnumMatcher( {
         'on' : 'Enable static link aggregation',
         'active' : 'Enable LACP in active mode',
         'passive' : 'Enable LACP in passive mode',
      } )
   }
   handler = reqSetChannelGroupMode
   noOrDefaultHandler = reqNoChannelGroupMode

class RecircChannelGroupModeCmd( CliCommand.CliCommandClass ):
   syntax = 'channel-group recirculation ID'
   data = {   
      'channel-group' : nodeChannelGroup,
      'recirculation' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'recirculation',
            helpdesc='Enable recirculation with static link aggregation' ),
         guard=recircGuard ),
      'ID' : recircGroupMatcher
   }
   handler = reqSetRecircChannelGroupMode

ChannelGroupModelet.addCommandClass( ChannelGroupModeCmd )
ChannelGroupModelet.addCommandClass( RecircChannelGroupModeCmd )

#-------------------------------------------------------------------------------
# The "[no|default] forwarding port-channel disabled" command, in "config-if" mode.
# This command is only valid on an aggregable interface.
#-------------------------------------------------------------------------------
def setLagPortForwardingDisabled( mode, args ):
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   epilc.fwdEnabled = False
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

def setNoLagPortForwardingDisabled( mode, args ):
   epilc = ethPhyIntfLagCliConfig( mode, create=True )
   epilc.fwdEnabled = True
   maybeDeleteEthPhyIntfLagConfig( epilc, mode )

class ForwardingDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding port-channel disabled'
   noOrDefaultSyntax = syntax
   data = {
         'forwarding' : CliToken.Switch.matcherForwarding,
         'port-channel' : CliCommand.guardedKeyword(
            'port-channel',
            'Set forwarding when part of a port channel',
            lagSupportedGuard ),
         'disabled' : 'Disable port forwarding when part of a port channel'
   }
   handler = setLagPortForwardingDisabled
   noOrDefaultHandler = setNoLagPortForwardingDisabled

ChannelGroupModelet.addCommandClass( ForwardingDisabledCmd )

#-------------------------------------------------------------------------------
# Associate the ChannelGroupModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( ChannelGroupModelet )

portChannelNodeForShow = CliCommand.Node(
      matcher=CliToken.PortChannel.portChannelMatcherForShow,
      guard=lagSupportedGuard )

# Platform packages register the actual "show load-balance destination
# port-channel" command using the above tokens and rules

def lacpPortStateModel( state ):
   def stateFlag( flag ):
      return( True if ( state & flag ) else False )

   return LagModel.LacpPortState(
      activity = stateFlag( StateActivity ),
      timeout = stateFlag( StateTimeout ),
      aggregation = stateFlag( StateAggregation ),
      synchronization = stateFlag( StateSynchronization ),
      collecting = stateFlag( StateCollecting ),
      distributing = stateFlag( StateDistributing ),
      defaulted = stateFlag( StateDefaulted ),
      expired = stateFlag( StateExpired )
      )

# returns list of qualifying ports in the port-channel.
# Trampoline to channelPorts C++ implementation for faster execution.
def _channelPorts( tacLagCliSupport, channel,
                   lacpOnly=True, status=Tac.Type( "Lag::LagStatusFilter" ).\
                      filterOnActiveAndInactive, useLagConfig=False ):
   tacLagCliSupport.portList.clear()
   tacLagCliSupport.channelPorts( channel, lacpOnly, status, useLagConfig )
   return tacLagCliSupport.portList

# If channel is singleton, return list of ports in channel.  If channel is a list,
# returns a dict of channel:ports Status can be 'filterOnActiveAndInactive' (return
# both active and inactive), or 'filterOnActive' or 'filterOnInactive'.
# "active" is always defined consistently, whether we are talking about the
# lag as a whole (lacpOnly == False) or only about LACP (lacpOnly ==
# True). "inactive" is defined relative to whether you are dealing with the
# LAG as a whole, or LACP only.  A port is "active" in the LAG in the sense of being
# a member of the LAG, which (if the port-channel is in lacpMode) is the same as
# being a member of stat->agg->aggregate in LACP. When lacpOnly==False, then
# "inactive" means any port configured to be in the port-channel, but is not in
# lagStatus.member.  In the case lacpOnly==True, then "inactive" means those ports
# considered by LACP, (e.g. the LAG hasn't rejected them as incompatible) but not
# aggregated, e.g. it only considers those that have been added to
# lacpIntfLagConfig.member.  The inactive ports are those that are either in
# stat->agg->standby, stat->otherIntf, or stat->agg->selected but *not* in
# stat->agg->aggregate. 
# It is possible that a port is considered by LACP but is in none of those
# collections.
#  useLagConfig:
#     When this option is True, the mbrList is composed from lag/config/phyIntf/
def channelPorts( mode, channel, lacpOnly=True, 
                  status=Tac.Type( "Lag::LagStatusFilter" ).\
                     filterOnActiveAndInactive,
                  useLagConfig=False ):
   tacLagCliSupport = Tac.newInstance( "Lag::LagCliSupport",
                                       lagConfigDir.force(),
                                       lagIntfConfigDir.force(),
                                       lagIntfStatusDir.force(),
                                       lacpStatus.force() )
   if type( channel ) is not list:
      return _channelPorts( tacLagCliSupport, channel,
                            lacpOnly, status, useLagConfig )
   else:
      return dict( [ ( port, 
                       _channelPorts( tacLagCliSupport,
                                      port, lacpOnly, status, useLagConfig ).keys() )
                     for port in channel ] )

EbraCliLib.getLagMembersCallBack = channelPorts

# This should return {} unless there is some bug, or we catch it in the middle
# of reactors deleting ports from a channel that has been unconfigured.
# Trampolines to lacpOrphanPorts C++ implementation for faster execution.
def lacpOrphanPorts( mode ):
   """Find ports in lls.portStatus whose lag is not in lagIntfConfigDir"""
   
   # Complete mounts if the required entities are not yet resolved.
   tacLagCliSupport = Tac.newInstance( "Lag::LagCliSupport",
                                       lagConfigDir.force(),
                                       lagIntfConfigDir.force(),
                                       lagIntfStatusDir.force(),
                                       lacpStatus.force() )
   tacLagCliSupport.lacpOrphanPorts()
   return list( tacLagCliSupport.portMap.items() )

# return ( brief, all, channelPortTree )
def lacpChannelsAndPorts( mode, channelGroups, intf, brief, all ):
   assert not ( intf and channelGroups ), \
       "Cannot specify both Interface and Channel" 
   channelPortTree = []
   if brief is None:
      brief = True
   ls = lacpStatus
   if intf and intf.name:
      epilc = ethPhyIntfLagConfig( mode, name=intf.name )
      elic = lagIntfConfigDir.get( intf.name )
      lpc = ls.portStatus.get( intf.name )
      elag = epilc and epilc.lag
      if not elag:
         print "Interface %s is not a member of a port channel." % intf.name
         return ( brief, all, channelPortTree )
      if not lpc:
         print ( "Interface %s is not a member of a port channel configured for "
                 "LACP." % intf.name )
         return ( brief, all, channelPortTree )
      if inactiveLag( elag.intfId ):
         print ( "Interface %s is a member of an inactive LACP port channel. "
                 "%s" ) % ( intf.name, configLimitExceededMsg )
         return ( brief, all, channelPortTree )
      channels = [ elag.intfId ]
      all = True
   else:
      channels = portchannelList( mode, channelGroups )
   if channels:
      ls = lacpStatus
      activePorts = channelPorts( mode, channels, True, 
                                  status=Tac.Type( "Lag::LagStatusFilter" ).\
                                     filterOnActive )   
      for channel in channels:
         ports = cliLacpPorts( mode, channel,
                               channels, all, activePorts[ channel ] )
         if intf and intf.name:
            if intf.name in ports:
               ports = [ intf.name ]
            else:
               ports = []
         channelPortTree.append( ( channel, ports, ) )
   if(( channelGroups is None ) and all and not brief ):
      # Find ports not associated with any active channels
      channelPortTree += lacpOrphanPorts( mode )
   return( brief, all, channelPortTree )

LagProtocolMap = { "lagModeUnconfigured": "Unconfigured",
                   "lagModeOn":"Static",
                   "lagModeLacp":"LACP" } 
LacpModeMap = { None:"Unknown",
                "lacpModeOff":"off",
                "lacpModePassive":"passive",
                "lacpModeActive":"active" }


def getInterfaceSummary( elis, port, elic ):
   linkDown = False
   suspended = False
   lacpMisconfig = None
   lacpState = None
   staticLag = False
   lagMember = False

   ethIntfstatus = ethIntfStatusDir.get( port )
   # Check to see if the interface is on, if not print D and exit
   if (not ethIntfstatus or (ethIntfstatus.linkStatus != "linkUp" or 
           ethIntfstatus.operStatus != "intfOperUp") ):
      linkDown = True
   lacpIntfConf = lacpConfig.lacpIntfLagConfig.get( port )
   # Check to see if part of the port channel
   if ( elis and elis.member ):
      if port in elis.member:
         lagMember = True
      else:
         # If the link is suspended, don't print any other flags 
         # Caller verifies that elic is not None
         # A channel could be suspended if it is the wrong speed for the channel
         # (among other reasons)
         if ( LagProtocolMap[ elic.mode ] == "LACP" ):
            if( lacpIntfConf and
                lacpIntfConf.individualLagKey.individual == True and 
                lacpIntfConf.individualLagKey.lagIntfId == "" ):
               suspended = True
         elif ( LagProtocolMap[ elic.mode ] == "Static" ): 
            suspended = True

   if ( LagProtocolMap[ elic.mode ] == "Static" ):
      staticLag = True
   elif ( LagProtocolMap[ elic.mode ] == "LACP" ):
      misConfig = None
      lss = lacpStatus.lagStatus.get( elic.intfId )
      if not lss:
         misConfig = 'idle'        # LACP Idle- not flagged
      elif not lss.agg:
         misConfig = 'noAgg'       # LACP No Aggregate- not flagged yet
      elif port in lss.agg.aggregate:
         misConfig = 'bundled'     # LACP Bundled- already flagged above
      elif port in lss.agg.standby:
         misConfig = 'standby'     # LACP Standby- not flagged yet
      elif port in lss.otherIntf:
         misConfig = 'mismatchAgg' # LACP Mismatched Aggregate (incompatible)
      elif port in lss.agg.selected:
         misConfig = 'negotiation' # LACP In Negotation
      else:
         misConfig = 'unknown'     # LACP Unknown error- not flagged yet

      lacpMisconfig = LagModel.LacpMisconfig( status=misConfig )

      s = lacpStatus.portStatus.get( port )
      if s:
         lacpState = lacpPortStateModel( s.actorState )

   return ( linkDown, suspended, lacpMisconfig, lacpState, staticLag, lagMember )

def getPortChannelLinkState( elis, elic ):
   portFlags = ""
   # Check to see if the port channel is active
   if( elis and elis.member ):
      portFlags += "up"
   else:
      portFlags += "down"
   return portFlags

def generateSummaryStats(lags, lagPortTable):
   # A channel is considered to be an aggregator if:
   # 1.) LACP channel in lss.agg
   # 2.) LACP channel in lss.otherIntf
   # 3.) Static configured channel
   numAggs = 0
   for elic in lags:
      lss = lacpStatus.lagStatus.get( elic.intfId )
      if( lss and lss.agg ):
         numAggs += 1         
         numAggs += len( set( map( lambda p: p.lagId, lss.otherIntf.itervalues() )))
      # A static channel should always be considered an aggregator
      elif( LagProtocolMap[ elic.mode ] == "Static" ):
         numAggs += 1
   numChannels = 0
   for elic in lags:
      elis = lagIntfStatusDir.get( elic.intfId, None )
      if( elis and elis.member ):
         numChannels += 1
   return ( numChannels, numAggs )

def lagModeToProtocol( mode ):
   assert mode in LagProtocolMap
   return LagProtocolMap[ mode ].lower()

def doShowPortChannelSummary( mode, args ):
   elicd = lagIntfConfigDir
   elisd = lagIntfStatusDir
   channels = portchannelList( mode, None, lacpOnly=False, inactiveWarn=False )
   lags = map( elicd.get, channels )
   epils = lagStatus.phyIntf

   # Build up table of all ports in a LAG
   lagPortTable = {}
   for phyIntf in ethPhyIntfLagConfigDir( mode ).itervalues():
      if( not phyIntf.lag or ( phyIntf.lag not in lags )):
         continue
      lagPortTable.setdefault( phyIntf.lag.intfId, [] ).append( phyIntf.intfId )

   numberOfChannelsInUse, numberOfAggregators = generateSummaryStats( lags,
                                                               lagPortTable )
   portChannels = {}
   for elic in lags:
      # Don't print any RecircChannels
      if isRecirc( elic.intfId ):
         continue

      elis = elisd.get( elic.intfId, None )

      # Determine Port-channel name and flags
      portChannelLinkState = getPortChannelLinkState( elis, elic )
      portName = elic.intfId

      intfs = lagPortTable.get( elic.intfId, [] )

      memberDict = {}
      for intf in intfs:
         cfg = ethPhyIntfLagConfig( mode, name=intf )
         linkDown, suspended, lacpMisconfig, lacpState, staticLag, lagMember = \
             getInterfaceSummary( elis, intf, elic )
         memberDict[ intf ] = \
               LagModel.PortChannelMemberSummary( intf=intf,
                                                  linkDown=linkDown,
                                                  lagMember=lagMember,
                                                  suspended=suspended,
                                                  lacpMisconfig=lacpMisconfig,
                                                  lacpState=lacpState,
                                                  staticLag=staticLag )
      if memberDict:
         lacpMode = None
         fallback = None
         cfg = ethPhyIntfLagConfig( mode, name=min( intfs, key=Arnet.intfNameKey ) )
         if cfg and ( cfg.mode == 'lacpModeActive' or
                      cfg.mode == 'lacpModePassive' ):
            lacpMode = LacpModeMap[ cfg.mode ] 
         if elis and elis.fallbackEnabled != 'fallbackNone':
            if elis.fallbackEnabled == 'fallbackStatic' or \
                   elis.fallbackEnabled == 'fallbackIndividual':
               fallback = LagModel.FallbackConfigStatus( 
                     status=elis.fallbackEnabled )
         elif elic.fallback != 'fallbackNone':
            if elic.fallback == 'fallbackStatic' or \
                   elic.fallback == 'fallbackIndividual':
               fallback = LagModel.FallbackConfigStatus( config=elic.fallback )
         protocol = lagModeToProtocol( elic.mode )
         inactive = inactiveLag( elic.intfId )
         portChannels[ portName ] = LagModel.PortChannelSummary(
                                          linkState=portChannelLinkState,
                                          protocol=protocol,
                                          lacpMode=lacpMode,
                                          fallback=fallback,
                                          ports=memberDict,
                                          inactive=inactive )
   return LagModel.PortChannelsSummary( numberOfChannelsInUse=numberOfChannelsInUse,
                                        numberOfAggregators=numberOfAggregators,
                                        portChannels=portChannels )


# getLagMemberDict
#  This returns a Dictionary of Lists:
#     Key   - Lag Name
#     Value - List of Members in the Lag
def getLagMemberDict ( mode, channelGroupIdList=None, 
                       filterFunc=portChannelFilter ):
   channels = portchannelList( mode, channelGroupIdList, lacpOnly=False,
                               filterFunc=filterFunc )
   # Add the Member Lists from the info in lag/config/phyIntf/
   return channelPorts( mode, channels, useLagConfig=True )

def doShowChannelGroupTraffic( mode, lagDict ):
   cntAttrList = [ "inUcastPkts", "outUcastPkts", "inMulticastPkts", \
                   "outMulticastPkts", "inBroadcastPkts", "outBroadcastPkts"]

   if not lagDict:
      return LagModel.PortChannelsTraffic()

   firstLagToPrint = True
   portChannelDict = {}
   # Each lag is a List of Members
   for ( lag, lagMembers ) in sorted( lagDict.items() ):
      portDict = {}
      if not lagMembers:
         continue
      # Build the counters:
      #   lagCntrDict --> Dictionary
      #        Key: attribute name
      #        Val: Sum of all interfaces per attribute
      #   lagMemberCntrDict - Dictionary of Dictionaries
      #        Key : Interface Name
      #        Val : Dictionary of Counters for the interface (intfCntrDict)
      #   intfCntrDict - Dictionary
      #        Key : Attribute name
      #        Val : Latest counter Value
      lagCntrDict = {}
      lagMemberCntrDict = {}
      for attr in cntAttrList:
         lagCntrDict[ attr ] = 0

      def stat(attr):
         sysdbValue = getattr( curIntfCounter.statistics, attr, None )
         if sysdbValue is None:
            return 'n/a'
         checkpointValue = 0
         if ckpt:
            checkpointValue = getattr( ckpt.statistics, attr, 0 )
         return sysdbValue - checkpointValue

      for intf in lagMembers:
         intfCounters = ethPhyIntfCounterDir().intfCounterDir.get( intf )
         if intfCounters is None:
            # Mlag peer interfaces don't have counters.
            continue
         curIntfCounter = intfCounters[ "current" ]
         intfObj = EthIntfCli.EthPhyIntf( intf, mode )
         ckpt = intfObj.getLatestCounterCheckpoint()
         phyIntf = lagStatus.phyIntf.get( intf )
         if phyIntf is None:
            # It left the lag between the beginning of the function and now.
            continue
         lagMemberCkpt = phyIntf.lagMembershipCounterCheckpoint
         # access statsUpdateTime by 1st reading the nominal object rates
         # to get the latest shared memory values
         if not ckpt or ( ckpt.rates.statsUpdateTime < 
               lagMemberCkpt.rates.statsUpdateTime ): 
            ckpt = lagMemberCkpt
         intfCntrDict = {}
         lagMemberCntrDict[ intf ] = intfCntrDict
         for attr in cntAttrList:
            counter = stat( attr )
            intfCntrDict[ attr ] = counter
            lagCntrDict[ attr ] = lagCntrDict[ attr ] + counter

      lagId = portChannel( lag ) 
      for intf in lagMembers:
         intfCntrDict = lagMemberCntrDict.get( intf )
         if intfCntrDict is None:
            continue
         percentDict = {}
         for attr in cntAttrList:
            p = 0.0
            lagVal = lagCntrDict[ attr ]
            intfVal = intfCntrDict[ attr ]
            if ( lagVal > 0 ) :
               p = ( float( intfVal ) / float( lagVal ) ) * 100
            percentDict[ attr ] = p
         tempPortDict = LagModel.InterfaceTraffic( **percentDict )
         portDict[ intf ] = tempPortDict
      portChannelDict[ lag ] = LagModel.PortChannelTraffic(
                                           interfaceTraffic=portDict )
   return LagModel.PortChannelsTraffic( portChannels=portChannelDict )


def doShowPortChannelTraffic( mode, args ):
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   lagDict = getLagMemberDict( mode, channelGroupIdList )
   return doShowChannelGroupTraffic( mode, lagDict )

def doShowChannelGroups( mode, channels,
                         brief=True, allPorts=True ):
   if brief == None:
      brief = True
   if allPorts == None:
      allPorts = True
   elicd = lagIntfConfigDir
   elisd = lagIntfStatusDir
   lags = set( map( elicd.get, channels ) )
   if allPorts or not brief:
      epilsd = lagStatus.phyIntf

   # Build up table of unconfigured ports in a LAG
   if( allPorts ):
      lagPortTable = {}
      for phyIntf in ethPhyIntfLagConfigDir( mode ).itervalues():
         pLag = phyIntf.lag
         if( not pLag or ( pLag not in lags )):
            continue
         lagPortTable.setdefault( pLag.intfId, [] ).append( phyIntf.intfId )

   # return True only if at least one port channel is configured to use
   # LACP *and* it has active members (We want to know whether we'll need
   # the "Lacp Mode" column in the output.
   def channelLacpActive( elic ):
      if elic.mode == "lagModeLacp":
         elis = elisd.get( elic.intfId, None )
         return( elis and ( len( elis.member ) != 0 ))
      else:
         return( False )
   hasLacpChannels = reduce( lambda x, elic: x or channelLacpActive( elic ),
                             filter( lambda x: x in lags, elicd.itervalues()),
                             False )
   hasFallback = reduce( lambda x, elic: x or ( elic.fallback or 
                             elic.fallbackTimeout != elic.fallbackTimeoutDefault),
                             filter( lambda p: p in lags, elicd.itervalues()),
                             False )

   portChannelDict = {}
   for elic in lags:
      activePortsDict = {}
      inactivePortsDict = {}
      elis = elisd.get( elic.intfId, None )

      activeMembers = {}
      formerlyActiveMembers = []
      fallbackState = None
      checkMinLinks = False
      if hasFallback and elis and not brief:
         fallbackState = elis.debugFallbackSmState
         fallbackState = fallbackState[ len("fallbackState"): ]
         if fallbackState == "Enabled":
            checkMinLinks = True
         if( fallbackState == "Enabled" and not elis.fallbackEnabled ):
            fallbackState = "locallyOnly"
         else:
            fallbackState = fallbackState.lower()

      if checkMinLinks and elic.minLinks != elic.minLinksDefault:
         msg = "min-links config will not be honored when fallback is active"
         mode.addWarning( msg )

      if( elis and elis.member ):
         activeMembers = elis.member
         for port in Arnet.sortIntf( activeMembers ):
            if brief:
               activePortsDict[ port ] = LagModel.ActivePort()
            else:
               epils = epilsd.get( port )
               if epils:
                  time = LagModel.toUtc( epils.memberTime )
               else:
                  formerlyActiveMembers.append( port )
                  activePortsDict[ port ] = LagModel.ActivePort()
                  continue

               tempProtocol = None
               tempMode = None
               cfg = ethPhyIntfLagConfig( mode, name=port )
               tempProtocol = lagModeToProtocol( elic.mode )
               if hasLacpChannels and cfg:
                  tempMode = LacpModeMap[ cfg.mode ]
               activePortsDict[ port ] = LagModel.ActivePort( timeBecameActive=time,
                                                              protocol=tempProtocol,
                                                              lacpMode=tempMode )

      if allPorts:
         inactiveMembers = [ p for p in lagPortTable.get( elic.intfId, [] )
                             if not p in activeMembers or
                             ( p in formerlyActiveMembers ) ]

         for port in Arnet.sortIntf( inactiveMembers ):
            epils = epilsd.get( port )
            if epils:
               reason = epils.reason
               time = LagModel.toUtc( epils.memberTime )
            elif not ethIntfStatusDir.get( port ):
               reason = 'Interface unavailable'
               time = -1.0
            else:
               reason = 'unknown'
               time = 0.0
            if brief:
               time = None
            inactivePortsDict[ port ] = LagModel.InactivePort(
               timeBecameInactive=time,
               reasonUnconfigured=reason )
      portChannelDict[ elic.intfId ] = LagModel.PortChannel(
         fallbackState=fallbackState,
         activePorts=activePortsDict,
         inactivePorts=inactivePortsDict,
         recircFeature=elic.recircFeature.keys(),
         inactiveLag=inactiveLag( elic.intfId ),
         _formerlyActiveMembers=formerlyActiveMembers )
   return LagModel.PortChannels( portChannels=portChannelDict,
                                 _brief=brief,
                                 _allPorts=allPorts )

def doShowPortChannel( mode, args ):
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   brief = 'detailed' not in args
   allPorts = 'active-ports' not in args
   channels = portchannelList( mode, channelGroupIdList, lacpOnly=False,
                               inactiveWarn=False )
   return doShowChannelGroups( mode, channels, brief, allPorts )

def doShowChannelLimits( mode, channelType='port-channel', limits=None ):
   compatiblePorts = {}
   for intf in lagGroupDir.phyIntf.itervalues():
      lg = intf.lagGroup
      if lg:
         compatiblePorts.setdefault( lg.name, []).append( intf.name )
   groups = list( lagGroupDir.lagGroup.itervalues() )
   groups.sort( key=lambda lg: lg.name )
   portChannelLimits = {}
   for lagGroup in groups:
      if compatiblePorts.get( lagGroup.name ) is None:
         continue
      portList = compatiblePorts.get( lagGroup.name )
      portChannelLimits[ lagGroup.name ] = LagModel.LagGroupLimits(
                       maxPortChannels=lagGroup.maxLagIntfs,
                       maxPortsPerPortChannel=lagGroup.maxPortsPerLagIntf,
                       portList=Arnet.sortIntf( portList ) )
   if channelType == 'port-channel':
      return LagModel.PortChannelLimits( portChannelLimits=portChannelLimits )
   else:
      return LagModel.RecircChannelLimits( recircChannelLimits=portChannelLimits )

def doShowPortChannelLimits( mode, args ):
   return doShowChannelLimits( mode, limits=None )
   
#-------------------------------------------------------------------------------
# The "show lacp" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show lacp <channelGroupId>
#        [ counters|internal|peer|sys-id|aggregates ]
#        [brief|detailed]
#        [all-ports]
#   legacy:
#    show lacp <channelGroupId> neighbor
#   deprecated by:
#    show lacp <channelGroupId> peer
# So far this is basically a clone of the industry standard interface, but
#  we may want to expose a bit more.
#-------------------------------------------------------------------------------

def cliLacpPorts( mode, channel, channelGroupId, allPorts, activePorts ):
   # Maybe if channelGroupId not = none, then allPorts = True should mean
   # ports in Lag, but not in LACP.  And allPorts == False should mean
   # *all* ports in LACP.
   if allPorts:
      ports = channelPorts( mode, channel, True )
   else:
      ports = activePorts
   return( ports )
   
# showLacpSysIdHook allows another package to provide extra information
# about system ids used by LACP. The hook should take the lacpConfig
# and the brief parameters and print out the relevant output.
showLacpSysIdHook = CliExtensions.CliHook()

def doShowLacpSysId( mode, args ):
   brief = 'detailed' not in args
   lc = lacpConfig
   # LACP System priority:
   priority = lc.priority
   # Switch MAC Address
   bc = bridgingConfig
   addr = bc.bridgeMacAddr
   sysIdRep = dot43sysid( priority, addr )
 
   sysIdList = []
   for hook in showLacpSysIdHook.extensions():
      miscLacpSysId = hook( lc, brief )
      if miscLacpSysId:
         sysIdList.append( miscLacpSysId )
 
   details = None
   if not brief:
      details = LagModel.LacpSysId.Details( priority=priority, address=addr )
   return LagModel.LacpSysId( details=details,
                              miscLacpSysIds=sysIdList,
                              systemId=sysIdRep )

def lacpPortStatus( port, lacpLagStatus ):
   agg = lacpLagStatus and lacpLagStatus.agg
   if not lacpLagStatus:
      status = 'idle'
   elif not agg:
      status = 'noAgg'
   elif port in agg.aggregate:
      status = 'bundled'
   elif port in agg.standby:
      status = 'standby'
   elif port in lacpLagStatus.otherIntf:
      status = 'mismatchAgg'
   elif port in agg.selected:
      status = 'negotiation'
   else:
      s = lacpStatus.portStatus.get( port )
      if s and s.selected == "selectedStateUnselected":
         status = 'unselected'
      else:
         status = 'unknown'
   return status

def portCounters( port, channelStatus, lacpCounters, lacpCountersCheckpoint, brief ):
   c = lacpCounters.portCounters.get( port )
   cc = lacpCountersCheckpoint.portCounters.get( port )
   if not c or not cc:
      return

   tempCounter = Tac.newInstance( "Lacp::LacpIntfCounters", "")
   # copy then subtract
   tempCounter.doCopy( c )
   tempCounter.doSubtract( cc )

   details = None
   if not brief:
      details = LagModel.InterfaceLacpCountersDetails(
         lastRxTime = LagModel.toUtc( tempCounter.debugLastRxTime ),
         actorChurnCount = getattr( tempCounter,
                                    "debugActorChurnCount" ),
         actorChangeCount = getattr( tempCounter, "debugActorChangeCount" ),
         actorSyncTransitionCount = getattr( tempCounter,
                                             "debugActorSyncTransitionCount" ),
         partnerChurnCount = getattr( tempCounter,
                                      "debugPartnerChurnCount" ),
         partnerSyncTransitionCount = getattr( tempCounter,
                                               "debugPartnerSyncTransitionCount" ),
         partnerChangeCount = getattr( tempCounter,
                                       "debugPartnerChangeCount" ) )

   return LagModel.InterfaceLacpCounters(
         actorPortStatus = lacpPortStatus( port, channelStatus ),
         lacpdusRxCount = getattr( tempCounter, "statsLACPDUsRx" ),
         lacpdusTxCount = getattr( tempCounter, "statsLACPDUsTx" ),
         markersRxCount = getattr( tempCounter, "statsMarkerPDUsRx" ),
         markersTxCount = getattr( tempCounter, "statsMarkerPDUsTx" ),
         markerResponseRxCount = getattr( tempCounter,
                                          "statsMarkerResponsePDUsRx" ),
         markerResponseTxCount = getattr( tempCounter,
                                          "statsMarkerResponsePDUsTx" ),
         illegalRxCount = tempCounter.statsIllegalRx,
         details = details )


def portsCountersDict( portChannel, ports, lacpStatus, lacpCounters,
                       lacpCountersCheckpoint, brief ):
   portDict = {}
   if ports:
      channelStatus = portChannel and lacpStatus.lagStatus.get( portChannel )
      for port in ports:
         pn = portCounters( port, channelStatus, lacpCounters,
                           lacpCountersCheckpoint, brief )
         if pn:
            portDict[ port ] = pn
   return portDict


def portChannelCounters( portChannel, ports, markers, lacpStatus, lacpCounters,
                         lacpCountersCheckpoint, brief ):
   portsDict = portsCountersDict( portChannel, ports, lacpStatus, lacpCounters,
                                  lacpCountersCheckpoint, brief )
   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelLacpCounters( interfaces=portsDict,
                                   markers=lacpAnnotationMarkers )

# Hmm... do I show info on channel-groups that have a lacpIntfConfig, but
# have mode lacpModeOff?
# Maybe we should, but only if detailed is true... or maybe
# only if specifically mentioned the port-channel *and* detailed is true.

def portNeighbor( port, lacpStatus, channelStatus, brief ):
   s = lacpStatus.portStatus.get( port )
   if not s:
      return

   # s.lagId.{remote,local} are almost the same as
   # s.{partner,actor}Port, except that the port priority
   # and port number are zero in s.lagId if aggregation
   # is occurring.  (43.3.6.1/f)

   details = None
   if not brief:
      details = LagModel.InterfaceLacpNeighborsDetails(
       aggSelectionState = s.selected,
       partnerChurnState = s.partnerChurnState,
       collectorMaxDelay = s.partnerCollectorMaxDelay )

   r = s.partnerPort
   return LagModel.InterfaceLacpNeighbors(
                  actorPortStatus = lacpPortStatus( port, channelStatus ),
                  partnerSystemId = dot43sysid( r.sysPriority, r.sysId ),
                  partnerPortId = r.portId,
                  partnerPortState = lacpPortStateModel( s.partnerState ),
                  partnerOperKey = "0x%04x" % r.key,
                  partnerPortPriority = r.portPriority,
                  details = details )


def portsNeighborDict( portChannel, ports, lacpStatus, brief ):
   portDict = {}
   if ports:
      channelStatus = portChannel and lacpStatus.lagStatus.get( portChannel )
      for port in ports:
         pn = portNeighbor( port, lacpStatus, channelStatus, brief )
         if pn:
            portDict[ port ] = pn
   return portDict


def portChannelNeighbor( portChannel, ports, markers, lacpStatus, brief ):
   portsDict = portsNeighborDict( portChannel, ports, lacpStatus, brief )
   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelLacpNeighbors( interfaces=portsDict,
                                    markers=lacpAnnotationMarkers )


def portInternal( port, lacpStatus, channelStatus, lacpCounters, brief ):
   s = lacpStatus.portStatus.get( port )
   if not s:
      return
   lacpIntfConf = lacpConfig.lacpIntfConfig.get( port )
   if not lacpIntfConf:
      return
   l = s.actorPort
   r = s.partnerPort

   details = None
   if not brief:
      c = lacpCounters.portCounters.get( port )
      details = LagModel.InterfaceLacpInternalDetails(
         aggSelectionState = s.selected,
         actorAdminKey = "0x%04x" % l.key,
         actorChurnState = s.actorChurnState,
         lastRxTime = LagModel.toUtc( c.debugLastRxTime ) if c else 0.0,
         actorRxSmState = s.rxSmState,
         actorMuxSmState = s.muxSmState,
         actorMuxReason = c.debugMuxReason if c else '',
         allowToWarmup = s.allowToWarmup )
      if toggleLacpTimeoutMultiplierEnabled():
         details.actorTimeoutMultiplier = lacpIntfConf.timeoutMult

   return LagModel.InterfaceLacpInternal(
         actorPortStatus = lacpPortStatus( port, channelStatus ),
         partnerSystemId = dot43sysid( r.sysPriority, r.sysId ),
         actorPortId = l.portId,
         actorPortState = lacpPortStateModel( s.actorState ),
         actorOperKey = "0x%04x" % l.key,
         actorPortPriority = l.portPriority,
         details = details )


def portsInternalDict( portChannel, ports, lacpStatus, lacpCounters, brief ):
   portDict = {}
   if ports:
      channelStatus = portChannel and lacpStatus.lagStatus.get( portChannel )
      for port in ports:
         pn = portInternal( port, lacpStatus, channelStatus, lacpCounters, brief )
         if pn:
            portDict[ port ] = pn
   return portDict


def portChannelInternal( portChannel, ports, markers, lacpStatus, lacpCounters,
                         brief ):
   portsDict = portsInternalDict( portChannel, ports, lacpStatus, lacpCounters,
                                  brief )
   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelLacpInternal( interfaces=portsDict,
                                   markers=lacpAnnotationMarkers )


#
# Request the agent to update counters into sysdb
#
def updatePortCounters( mode ):
   # On standby supe, return counter values in Sysdb without waiting for
   # countersUpdated().
   if CommonGuards.standbyGuard( mode, "" ) == CliParser.guardNotThisSupervisor:
      t0( "updatePortCounters called on standby supervisor" )
      return

   config = lacpCliConfig
   status = lacpStatus
   config.counterUpdateRequestTime = Tac.now() # monotonic time
   def countersUpdated():
      return status.counterUpdateTime >= config.counterUpdateRequestTime
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.1,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      print "Warning: displaying stale counters"

def doShowLacpNeighbors( mode, args ):
   brief = 'detailed' not in args
   allPorts = 'all-ports' in args
   intf = args.get( 'INTERFACE' )
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   
   updatePortCounters( mode )
   brief, allPorts, channelPortTree = \
       lacpChannelsAndPorts( mode, channelGroupIdList, intf, brief, allPorts )

   portChannelDict = {}
   orphanPorts = {}
   usedMarkerMessages = None

   if channelPortTree:
      haveOutput = False
      usedMarkers = set()

      for channel, ports in channelPortTree:
         haveOutput = True if ports else haveOutput
         if channel:
            markers = genMarkers( channel )
            usedMarkers = usedMarkers.union( set( markers ) )
            portChannelDict[ channel ] = portChannelNeighbor( channel, ports,
                                                 markers, lacpStatus, brief )
         else:
            orphanPorts = portsNeighborDict( None, ports, lacpStatus, brief )

      if haveOutput:
         usedMarkerMessages = usedDetailMarkerMessages( usedMarkers, markerMsgs )

   return LagModel.LacpNeighbors( portChannels=portChannelDict,
                                  _brief=brief,
                                  _allPorts=allPorts,
                                  markerMessages=usedMarkerMessages,
                                  orphanPorts=orphanPorts )

def doShowLacpCounters( mode, args ):
   brief = 'detailed' not in args
   allPorts = 'all-ports' in args
   intf = args.get( 'INTERFACE' )
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   updatePortCounters( mode )
   syncCounterCheckpoint( lacpCountersCheckpoint, lacpCounters )
   brief, allPorts, channelPortTree = \
       lacpChannelsAndPorts( mode, channelGroupIdList, intf, brief, allPorts )

   portChannelDict = {}
   orphanPorts = {}
   usedMarkerMessages = None

   if channelPortTree:
      haveOutput = False
      usedMarkers = set()

      for channel, ports in channelPortTree:
         haveOutput = True if ports else haveOutput
         if channel:
            markers = genMarkers( channel )
            usedMarkers = usedMarkers.union( set( markers ) )
            portChannelDict[ channel ] = portChannelCounters( channel, ports,
                                           markers, lacpStatus, lacpCounters,
                                           lacpCountersCheckpoint, brief )
         else:
            orphanPorts = portsCountersDict( None, ports, lacpStatus, lacpCounters,
                                             lacpCountersCheckpoint, brief )
      if haveOutput:
         usedMarkerMessages = usedDetailMarkerMessages( usedMarkers, markerMsgs )

   return LagModel.LacpCounters( portChannels=portChannelDict,
                                 _brief=brief,
                                 _allPorts=allPorts,
                                 markerMessages=usedMarkerMessages,
                                 orphanPorts=orphanPorts )

# clear lacp [channel-group] counters
def clearLacpCounters( mode, args ):
   channelGroupId = None
   if 'LACP' in args:
      channelGroupId = ChannelGroup( args.get( 'LACP' ) )
   syncCounterCheckpoint( lacpCountersCheckpoint, lacpCounters)
   if channelGroupId:
      Intf.Log.logClearCounters( "lacp", "channel-group " + str( channelGroupId ) )
      portNames = channelPorts( mode, portchannelList( mode, channelGroupId )[ 0 ] )
   else:
      Intf.Log.logClearCounters( "lacp", "all ports" )
      portNames = lacpCountersCheckpoint.portCounters.keys()

   for port in portNames:
      counters = lacpCounters.portCounters.get( port )
      countersCheckpoint = lacpCountersCheckpoint.portCounters.get( port )

      if counters is None or countersCheckpoint is None:
         continue
      countersCheckpoint.doCopy( counters )


def syncCounterCheckpoint( toDir, fromDir ):

   toIntfNames = set( toDir.portCounters.keys() )
   fromIntfNames = set( fromDir.portCounters.keys() )

   for intf in toIntfNames:
      if intf not in fromIntfNames:
         del toDir.portCounters[ intf ]

   toIntfNames = set( toDir.portCounters.keys() )

   for intf in fromIntfNames:
      if intf not in toIntfNames:
         toDir.portCounters.newMember( intf )

# showLacpInternalSysIdHook allows another package to provide extra information
# about system ids used by LACP. The hook takes the lacpConfig as a parameter
# and is responsible for printing out the relevant information.
showLacpInternalSysIdHook = CliExtensions.CliHook()

def doShowLacpInternal( mode, args ): 
   intf = args.get( 'INTERFACE' )
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   brief = 'detailed' not in args
   allPorts = 'all-ports' in args
   updatePortCounters( mode )
   brief, allPorts, channelPortTree = \
          lacpChannelsAndPorts( mode, channelGroupIdList, intf, brief, allPorts )
 
   portChannelDict = {}
   orphanPorts = {}
   systemIdentifier = None
   usedMarkerMessages = None
   sysIdList = None
 
   if channelPortTree:
      haveOutput = False
      usedMarkers = set()
 
      for channel, ports in channelPortTree:
         haveOutput = True if ports else haveOutput
         if channel:
            markers = genMarkers( channel )
            usedMarkers = usedMarkers.union( set( markers ) )
            portChannelDict[ channel ] = \
                  portChannelInternal( channel, ports, markers, 
                                       lacpStatus, lacpCounters, brief )
         else:
            orphanPorts = portsInternalDict( None, ports, lacpStatus, lacpCounters,
                                              brief )
 
      # LACP System priority:
      priority = lacpConfig.priority
      addr = bridgingConfig.bridgeMacAddr
      systemIdentifier = dot43sysid( priority, addr )
      sysIdList = []
      for hook in showLacpInternalSysIdHook.extensions():
         internalSysId = hook( lacpConfig )
         if internalSysId:
            sysIdList.append( internalSysId )
      if haveOutput:
         usedMarkerMessages = usedDetailMarkerMessages( usedMarkers, markerMsgs )
 
   return LagModel.LacpInternal( portChannels=portChannelDict,
                                 systemId=systemIdentifier,
                                 miscLacpSysIds=sysIdList,
                                 _brief=brief,
                                 _allPorts=allPorts,
                                 markerMessages=usedMarkerMessages,
                                 orphanPorts=orphanPorts )


def aggNegotiationPorts( agg, lagStatus ):
   ports = [ i for i in agg.selected
             if not ( i in agg.aggregate or
                      i in agg.standby or
                      ( lagStatus and i in lagStatus.otherIntf ) ) ]
   return Arnet.sortIntf( ports )


def aggOtherPorts( agg ):
   ports = set( agg.standby.keys() ).union( agg.selected.keys() )
   return Arnet.sortIntf( port for port in ports if port not in agg.aggregate )


def incompatiblePortsDetail( lacpStatus, lagStatus ):
   aggDict = {}
   incompatPortsDetail = {}
   for portName in lagStatus.otherIntf.keys():
      port = lacpStatus.portStatus.get( portName )
      if not port:
         # Can this happen if the port is in the process of being
         # removed from the lag?
         continue
      aggDict.setdefault( port.lagId, [ ] ).append( portName )
   for aggId, ports in aggDict.iteritems():
      incompatPortsDetail[ dot43lagId( aggId ) ] = \
                         LagModel.IncompatiblePorts( ports=Arnet.sortIntf( ports ) )
   return incompatPortsDetail


def describeAgg( agg, lagStatus, allPorts, brief ):
   aggregateId = dot43lagId( agg.lagId )
   bundledPorts = Arnet.sortIntf( agg.aggregate )

   ( otherPorts, standbyPorts, negotiationPorts ) = ( [], [], [] )
   if allPorts:
      if not brief:
         otherPorts = aggOtherPorts( agg )
         standbyPorts = Arnet.sortIntf( agg.standby )
      negotiationPorts = aggNegotiationPorts( agg, lagStatus )
   return ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts )


def lacpAggregateInfo( agg, lagStatus, allPorts, brief ):
   ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts,
     incompatPortsDetail ) = '', [], [], [], [], {}
   if agg:
      ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts ) = \
                                       describeAgg( agg, lagStatus, allPorts, brief )
   if ( lagStatus and lagStatus.otherIntf ):
      incompatPortsDetail = incompatiblePortsDetail( lacpStatus, lagStatus )
   return ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts,
            incompatPortsDetail )


def otherLacpAggregates( lacpStatus, allPorts, brief ):
   otherAggsDict = {}
   otherAggs = dict( lacpStatus.aggStatus )
   for lls in lacpStatus.lagStatus.itervalues():
      if lls.agg and lls.agg.lagId in otherAggs:
         del otherAggs[ lls.agg.lagId ]

   if otherAggs:
      for agg in otherAggs.itervalues():
         ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts, \
           incompatPortsDetail ) = lacpAggregateInfo( agg, 
                                                      None, # no lls for otherAggs
                                                      allPorts, brief )

         otherAggsDict[ aggregateId ] = LagModel.PortChannelAggregate(
                                      aggregateId = aggregateId,
                                      bundledPorts = bundledPorts,
                                      otherPorts = otherPorts,
                                      standbyPorts = standbyPorts,
                                      negotiationPorts = negotiationPorts,
                                      incompatiblePortsDetail = incompatPortsDetail,
                                      markers = None )
   return otherAggsDict


def portChannelAggregate( portChannel, markers, lacpStatus, allPorts, brief ):
   lagStatus = lacpStatus.lagStatus.get( str( portChannel ) )
   agg = lagStatus and lagStatus.agg

   if lagStatus and ( agg or lagStatus.otherIntf ):
      ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts,
        incompatPortsDetail ) = lacpAggregateInfo( agg, lagStatus,
                                                   allPorts, brief )
   else:
      ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts,
        incompatPortsDetail ) = '', [], [], [], [], {}

   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelAggregate( aggregateId = aggregateId,
                                bundledPorts = bundledPorts,
                                otherPorts = otherPorts,
                                standbyPorts = standbyPorts,
                                negotiationPorts = negotiationPorts,
                                incompatiblePortsDetail = incompatPortsDetail,
                                markers = lacpAnnotationMarkers )


def doShowLacpAggregates( mode, args ):
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   allPorts = 'all-ports' in args
   brief = 'detailed' not in args
   updatePortCounters( mode )
   brief = True if brief is None else brief
   channels = portchannelList( mode, channelGroupIdList )

   usedMarkers = set()
   usedMarkerMessages = None
   portChannelDict = {}
   otherAggs = {}

   for channel in channels:
      markers = genMarkers( channel )
      usedMarkers = usedMarkers.union( set( markers ) )
      portChannelDict[ channel ] = portChannelAggregate( channel, markers,
                                                 lacpStatus, allPorts, brief )

   if allPorts:
      otherAggs = otherLacpAggregates( lacpStatus, allPorts, brief )

   if len( usedMarkers ):
      usedMarkerMessages = usedDetailMarkerMessages( usedMarkers, markerMsgs )

   return LagModel.LacpAggregates( portChannels=portChannelDict,
                                   otherAggregates=otherAggs,
                                   _brief=brief,
                                   _allPorts=allPorts,
                                   markerMessages=usedMarkerMessages )

def portLacp( port, lacpStatus, lacpCounters, channelStatus, brief ):
   s = lacpStatus.portStatus.get( port )
   if not s:
      return
   lacpIntfConf = lacpConfig.lacpIntfConfig.get( port )
   if not lacpIntfConf:
      return

   r = s.partnerPort
   l = s.actorPort

   details = None
   if not brief:
      c = lacpCounters.portCounters.get( port )
      details = LagModel.InterfaceLacpDetails(
        aggSelectionState = s.selected,
        partnerChurnState = s.partnerChurnState,
        partnerCollectorMaxDelay = s.partnerCollectorMaxDelay,
        actorAdminKey = "0x%04x" % l.key,
        actorChurnState = s.actorChurnState,
        actorLastRxTime = LagModel.toUtc( c.debugLastRxTime ) if c else 0.0,
        actorRxSmState = s.rxSmState,
        actorMuxSmState = s.muxSmState,
        actorMuxReason = c.debugMuxReason if c else '' )
      if toggleLacpTimeoutMultiplierEnabled():
         details.actorTimeoutMultiplier = lacpIntfConf.timeoutMult

   return LagModel.InterfaceLacp(
      actorPortStatus = lacpPortStatus( port, channelStatus ),
      partnerSysId = dot43sysid( r.sysPriority, r.sysId ),
      partnerPortId = r.portId,
      partnerPortState = lacpPortStateModel( s.partnerState ),
      partnerOperKey = "0x%04x" % r.key,
      partnerPortPriority = r.portPriority,
      actorPortId = l.portId,
      actorPortState = lacpPortStateModel( s.actorState ),
      actorOperKey = "0x%04x" % l.key,
      actorPortPriority = l.portPriority,
      details = details )


def portsLacpDict( portChannel, ports, lacpStatus, lacpCounters, brief ):
   portDict = {}
   if ports:
      channelStatus = portChannel and lacpStatus.lagStatus.get( portChannel )
      for port in ports:
         pn = portLacp( port, lacpStatus, lacpCounters, channelStatus, brief )
         if pn:
            portDict[ port ] = pn
   return portDict


def portChannelLacp( portChannel, ports, markers, lacpStatus, lacpCounters, brief ):
   portsDict = portsLacpDict( portChannel, ports, lacpStatus, lacpCounters, brief )
   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelLacp( interfaces=portsDict,
                                    markers=lacpAnnotationMarkers )


def doShowLacpPort( mode, args ):
   intf = args.get( 'INTERFACE' )
   brief = 'detailed' not in args
   allPorts = 'all-ports' in args
   updatePortCounters( mode )
   brief, allPorts, channelPortTree = \
       lacpChannelsAndPorts( mode, None, intf, brief, allPorts )

   portChannelDict = {}
   orphanPorts = {}
   usedMarkerMessages = None
   interface = intf.name if intf else None

   if channelPortTree:
      haveOutput = False
      usedMarkers = set()

      for channel, ports in channelPortTree:
         haveOutput = True if ports else haveOutput
         if channel:
            markers = genMarkers( channel )
            usedMarkers = usedMarkers.union( set( markers ) )
            portChannelDict[ channel ] = portChannelLacp( channel, ports, markers,
                                                 lacpStatus, lacpCounters, brief )
         else:
            orphanPorts = portsLacpDict( None, ports, lacpStatus, lacpCounters,
                                         brief )

      if haveOutput:
         usedMarkerMessages = usedDetailMarkerMessages( usedMarkers, markerMsgs )

   return LagModel.LacpPort( portChannels=portChannelDict,
                             interface=interface,
                             _brief=brief,
                             _allPorts=allPorts,
                             markerMessages=usedMarkerMessages,
                             orphanPorts=orphanPorts )

#
# LAG hidden command
# 
def doShowLagInitializationTime( mode, args ):
   print 'Lag initialization time: '
   for name, value in agentStatus.initTime.iteritems():
      print '   %s : %6.3f seconds' % ( name, value )

def doEnableLacpAudit( mode, args ):
   lacpCliConfig.enableAudit = True

def noLacpAudit( mode, args ):
   lacpCliConfig.enableAudit = False

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global lagIntfConfigDir, lagIntfStatusDir, lagStatus
   global lacpConfig, lacpCliConfig, lacpStatus, lacpCounters
   global lagGroupDir, lagConfigCli, lagConfigDir
   global lacpCountersCheckpoint
   global bridgingConfig, bridgingCliConfig 
   global ethIntfStatusDir, agentStatus
   global ethPhyIntfConfigSliceDir
   global recircHwConfig
   global bridgingHwCapabilities
   global lacpCliOverrideDir

   lagIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/eth/lag",
                                       "Interface::EthLagIntfConfigDir", "w" )
   lagIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                       "Interface::EthLagIntfStatusDir", "r" )
   lagConfigCli = ConfigMount.mount( entityManager, "lag/input/config/cli",
                                       "Lag::Input::Config", "w" )
   lagConfigDir = LazyMount.mount( entityManager, "lag/config",
                                   "Lag::Config", "r" )
   lagStatus = LazyMount.mount( entityManager, "lag/status", "Lag::Status", "r" )
   ethPhyIntfConfigSliceDir = ConfigMount.mount(
                         entityManager,
                         "interface/config/eth/phy/slice",
                         "Tac::Dir", "wi" )
   lacpConfig = LazyMount.mount( entityManager, "lag/lacp/config",
                                 "Lacp::Config", "r" )
   lacpCliConfig = ConfigMount.mount( entityManager, "lag/lacp/input/config/cli",
                                      "Lacp::CliConfig", "w" )
   lacpStatus = LazyMount.mount( entityManager, "lag/lacp/status",
                                 "Lacp::LacpStatus", "r" )
   lacpCounters = LazyMount.mount( entityManager, "lag/lacp/counters",
                                   "Lacp::LacpCounters", "r" )
   lacpCountersCheckpoint = LazyMount.mount( entityManager, 
                                             "lag/lacp/countersCheckpoint",
                                             "Lacp::LacpCounters", "w" )
   lagGroupDir = LazyMount.mount( entityManager, "hardware/lag/internalConfig",
                                  "Hardware::Lag::InternalConfig", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   bridgingCliConfig = ConfigMount.mount( entityManager, 
                                          "bridging/input/config/cli", 
                                          "Bridging::Input::CliConfig", "w" )
   ethIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/intf",
                                          "Interface::EthIntfStatusDir", "r")
   recircHwConfig = LazyMount.mount( entityManager, "lag/recirc/hwconfig", 
                                     "Lag::Recirc::HwConfig", "r" )
   agentStatus = LazyMount.mount( entityManager, "agent/lag/status", 
                                  "Lag::Agent::InitStatus", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   lacpCliOverrideDir = ConfigMount.mount(entityManager,
                                 "lag/input/lacpoverride/cli",
                                 "Lag::Input::LacpOverrideDir", "w" )
   IntfCli.Intf.registerDependentClass( EthPhyLagIntfDependent )

